﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Base.h>
#if defined(NN_BUILD_CONFIG_OS_WIN32)
//#define WIN32_LEAN_AND_MEAN
//#define NOMINMAX
#include <nn/nn_Windows.h>
#endif
#include <nn/diag/diag_Config.h>

#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#include <nn/gfx/util/detail/gfx_DebugFontGpuBuffer.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_CharacterEncoding.h>
#include <nn/gfx/detail/gfx_Log.h>
#include <nn/os.h>
#include <nn/mem/mem_StandardAllocator.h>

#define EXTRA_MARGIN_ENABLED


namespace nn {
namespace gfx {
namespace util {

namespace detail {
namespace {

#if NN_GFX_IS_TARGET_VK
#include "..\..\..\..\Resources\GfxUtil\DebugFontBuildinShader_vk.h"
#elif defined(NN_BUILD_CONFIG_SPEC_NX) && NN_GFX_IS_TARGET_GL
#include "..\..\..\..\Resources\GfxUtil\DebugFontBuildinShader_gl.h"
#elif defined(NN_BUILD_CONFIG_SPEC_NX)
#include "..\..\..\..\Resources\GfxUtil\DebugFontBuildinShader_nx.h"
#elif defined(NN_BUILD_CONFIG_OS_WIN)
#include "..\..\..\..\Resources\GfxUtil\DebugFontBuildinShader_win32.h"
#else
static uint8_t DebugFontBuildinShader[1] = { 0 };
#endif

#include "..\..\..\..\Resources\GfxUtil\DebugFontBuildinFontImadeData.h"
#include "..\..\..\..\Resources\GfxUtil\DebugFontBuildinFontGlyphData.h"

static const size_t DebugFontFormatBufferLength = 1024;
static const size_t DebugFontTextWriterImplAlignment = 16;
static const int VertexCountByLetter = 4;  //!< 文字単位の頂点数です。
static const int IndexCountByLetter = 6;   //!< 文字単位のインデックス数です。
static const int VertexAttributeCount = 3; //!< 頂点属性の数です。

struct Rectangle
{
    float left;
    float top;
    float right;
    float bottom;

    float GetWidth() const { return right - left; }
    float GetHeight() const { return bottom - top; }
};

struct CharAttribute
{
    nn::util::Float4 pos;
    nn::util::Color4u8Type color;
    nn::util::Float4 tex;
};

struct Vertex
{
    nn::util::Float3 xyz;
    nn::util::Color4u8Type color;
    nn::util::Float2 uv;
    void Set(float x, float y, float z, const nn::util::Color4u8Type& c, float u, float v)
    {
        xyz.v[0] = x;
        xyz.v[1] = y;
        xyz.v[2] = z;
        color = c;
        uv.v[0] = u;
        uv.v[1] = v;
    }
};

struct ShaderParam
{
    float m_Mtx[4][4];
    float m_Alpha;
};

size_t GetBuildinShaderResourceAlignment()
{
    nn::util::BinaryFileHeader* pHeader = reinterpret_cast<nn::util::BinaryFileHeader*>(DebugFontBuildinShader);
    return pHeader->GetAlignment();
}

template<typename T, size_t S>
inline int GetArrayLength(const T(&array)[S]
    )
{
    NN_UNUSED(array);
    return static_cast<int>(S);
}

template <typename T>
void SwapEndian(T& value)
{
#if defined(NN_BUILD_CONFIG_SPEC_CAFE)
    GX2EndianSwap(&value, sizeof(T));
#else
    NN_UNUSED(value);
#endif
}

size_t CalculateVertexBufferSize(int charCountMax)
{
    return sizeof(Vertex) * VertexCountByLetter * charCountMax;
}

size_t CalculateIndexBufferSize(int charCountMax)
{
    return sizeof(uint16_t) * IndexCountByLetter * charCountMax;
}

size_t GetRequiredVertexBufferSize(nn::gfx::Device* pDevice, int charCountMax)
{
    const size_t vtxBufSize = CalculateVertexBufferSize(charCountMax);

    nn::gfx::Buffer::InfoType info;
    info.SetDefault();
    info.SetGpuAccessFlags(nn::gfx::GpuAccess_VertexBuffer);

    return nn::util::align_up(vtxBufSize, nn::gfx::Buffer::GetBufferAlignment(pDevice, info));
}

size_t GetRequiredIndexBufferSize(nn::gfx::Device* pDevice, int charCountMax)
{
    const size_t idxBufSize = CalculateIndexBufferSize(charCountMax);

    nn::gfx::Buffer::InfoType info;
    info.SetDefault();
    info.SetGpuAccessFlags(nn::gfx::GpuAccess_IndexBuffer);

    return nn::util::align_up(idxBufSize, nn::gfx::Buffer::GetBufferAlignment(pDevice, info));
}

size_t GetRequiredConstantBufferSize(nn::gfx::Device* pDevice)
{
    nn::gfx::Buffer::InfoType info;
    info.SetDefault();
    info.SetGpuAccessFlags(nn::gfx::GpuAccess_ConstantBuffer);

    size_t  constantBufferSize = nn::util::align_up(sizeof(ShaderParam), nn::gfx::Buffer::GetBufferAlignment(pDevice, info));

    return constantBufferSize;
}

class DebugFontFormatStringOutputBuffer{
public:
    DebugFontFormatStringOutputBuffer(char* pBuffer, int length) NN_NOEXCEPT
        : m_pBegin(pBuffer),
        m_pEnd(pBuffer + length - 1),
        m_pCurrent(pBuffer)
    {
        NN_UNUSED(m_pBegin);
        *m_pCurrent = '\0';
    }

    static void Output(uintptr_t arg, const char* pCharacters, int count) NN_NOEXCEPT
    {
        reinterpret_cast<DebugFontFormatStringOutputBuffer*>(arg)->PutCharacters(pCharacters, count);
    }

private:
    void PutCharacters(const char* pCharacters, size_t count) NN_NOEXCEPT
    {
        while (count > 0)
        {
            const size_t bufferLeft = m_pEnd - m_pCurrent;
            size_t copyCount = std::min(bufferLeft, count);
            std::memcpy(m_pCurrent, pCharacters, copyCount);

            m_pCurrent += copyCount;
            pCharacters += copyCount;
            count -= copyCount;

            *m_pCurrent = '\0';
        }
    }

    char const* m_pBegin;   // 出力用バッファの先頭
    char const* m_pEnd;     // 出力用バッファの末尾
    char* m_pCurrent;       // 出力用バッファの現在位置
};

bool GetCharCodeUtf8(uint32_t& code, int& charBytes, const char*str)
{
    uint8_t c0 = static_cast<uint8_t>(str[0]);
    if(c0 < 0x80)
    {
        code = static_cast<uint32_t>(c0);
        charBytes = 1;
        return true;
    }
    else if(c0 < 0xe0)
    {
        uint8_t c1 = static_cast<uint8_t>(str[1]);
        code = (static_cast<uint32_t>(c0) | static_cast<uint32_t>(c1 << 8));
        charBytes = 2;
        return true;
    }
    else if(c0 < 0xf0)
    {
        uint8_t c1 = static_cast<uint8_t>(str[1]);
        uint8_t c2 = static_cast<uint8_t>(str[2]);
        code = (static_cast<uint32_t>(c0) | static_cast<uint32_t>(c1 << 8) | static_cast<uint32_t>(c2 << 16));
        charBytes = 3;
        return true;
    }
    else if(c0 < 0xf8)
    {
        uint8_t c1 = static_cast<uint8_t>(str[1]);
        uint8_t c2 = static_cast<uint8_t>(str[2]);
        uint8_t c3 = static_cast<uint8_t>(str[3]);
        code = (static_cast<uint32_t>(c0) | static_cast<uint32_t>(c1 << 8) | static_cast<uint32_t>(c2 << 16) | static_cast<uint32_t>(c3 << 24));
        charBytes = 4;
        return true;
    }

    return false;
}

bool GetCharCodeCp932(uint16_t& code, int& charBytes, const char*str)
{
    uint8_t c0 = static_cast<uint8_t>(str[0]);
    if((0x81 <= c0 && c0 <= 0x9f) || (0xe0 <= c0 && c0 <= 0xfc))
    {
        uint8_t c1 = static_cast<uint8_t>(str[1]);
        code = (static_cast<uint16_t>(c0) | (static_cast<uint16_t>(c1) << 8));
        charBytes = 2;
    }
    else
    {
        code = static_cast<uint16_t>(c0);
        charBytes = 1;
    }

    return true;
}

} // unnamed namespace


class DebugFontTextWriterImpl
{
    NN_DISALLOW_COPY(DebugFontTextWriterImpl);

public:
    // コンストラクタ
    DebugFontTextWriterImpl();

    // デストラクタ
    virtual ~DebugFontTextWriterImpl();

    static size_t GetRequiredMemorySize(
        nn::gfx::Device* pDevice,
        const DebugFontTextWriterInfo &info
        );

    static size_t GetRequiredMemoryPoolSize(
        nn::gfx::Device* pDevice,
        const DebugFontTextWriterInfo &info
        );

    void Initialize(
        nn::gfx::Device* pDevice,
        const DebugFontTextWriterInfo &info,
        void* pMemory,
        size_t memorySize,
        nn::gfx::MemoryPool* pMemoryPool,
        ptrdiff_t memoryPoolOffset,
        size_t memoryPoolSize
        );

    void Finalize();

    bool IsInitialized() const { return m_IsInitialized; }
    int GetCharCountMax() const { return m_CharCountMax; }
    int GetBufferCount() const { return m_BufferCount; }
    bool IsUserMemoryPoolEnabled() const { return m_IsUserMemoryPoolEnable; }
    nn::gfx::MemoryPool* GetMemoryPool() const { return IsUserMemoryPoolEnabled() ? m_pMemoryPool : nullptr; }
    ptrdiff_t GetMemoryPoolOffset() const { return IsUserMemoryPoolEnabled() ? m_MemoryPoolOffset : 0; }
    int GetSamplerDescriptorIndexSlot() const { return m_SamplerDescriptorIndexSlot; }
    int GetTextureDescriptorIndexSlot() const { return m_TextureDescriptorIndexSlot; }

    void SetDisplayWidth(int displayWidth);
    void SetDisplayHeight(int displayHeight);
    void SetTextureDescriptor(nn::gfx::DescriptorPool* pTextureDescriptorPool, int textureDescriptorIndexSlot);
    void SetSamplerDescriptor(nn::gfx::DescriptorPool* pSamplerDescriptorPool, int samplerDescriptorIndexSlot);
    void SetLineHeight(float height);
    float GetLineHeight() const { return g_DebugFontLineFeed * GetScaleY() + m_LineSpace; }
    void SetTabWidth(int tabWidth);
    int GetTabWidth() const     { return m_TabWidth; }
    void SetScale(float hScale, float vScale);
    float GetScaleX() const { return m_Scale.x; }
    float GetScaleY() const { return m_Scale.y; }
    void SetFontSize(float width, float height);
    void SetFontSize(float height);
    float GetFontWidth() const  { return g_DebugFontCharWidth * GetScaleX(); }
    float GetFontHeight() const { return g_DebugFontLineFeed * GetScaleY(); }
    void SetCursor(float x, float y);
    void SetCursorX(float x);
    void SetCursorY(float y);
    float GetCursorX() const { return m_CursorPos.x; }
    float GetCursorY() const { return m_CursorPos.y; }
    void SetTextColor(const nn::util::Color4u8Type &color);
    const nn::util::Color4u8Type& GetTextColor() const { return m_TextColor; }

    void Draw(nn::gfx::CommandBuffer* pCommandBuffer);
    void PrintUtf8(bool defaultLocaleCharset, const char* str);
    void PrintUtf8(const char* str);
    void PrintCp932(const char* str);
    void PrintUtf16(const uint16_t* str);

    void SetFixedWidthEnabled(bool isFixed) { m_IsWidthFixed = isFixed; }
    bool IsFixedWidthEnabled() const        { return m_IsWidthFixed; }
    void SetFixedWidth(float width)         { m_FixedWidth = width; }
    float GetFixedWidth() const             { return m_FixedWidth; }
    void CalculateStringRectUtf8(Rectangle* pRect, bool defaultLocaleCharset, const char* str);
    void CalculateStringRectUtf8(Rectangle* pRect, const char* str);
    void CalculateStringRectCp932(Rectangle* pRect, const char* str);
    void CalculateStringRectUtf16(Rectangle* pRect, const uint16_t* str);

private:
    static size_t GetRequiredMemorySizeVertexState();
    static size_t GetRequiredMemorySizeBlendState();
    static size_t GetRequiredMemorySizeGpuBuffer(int bufferCount);
    static size_t GetRequiredMemoryPoolSize(nn::gfx::Device* pDevice, int charCountMax, int bufferCount);

    void InitializeGfxShader();
    void InitializeGfxVertexState();
    void InitializeGfxSampler();
    void InitializeGfxBlendState();
    void InitializeGfxDepthStencilState();
    void InitializeGfxRasterizerState();

    void InitializeFontMemoryPoolObject();
    void InitializeFontCharAttributeBuffer();
    void CreateFontConstantBuffer(nn::gfx::MemoryPool* pMemoryPool, ptrdiff_t memoryPoolOffset, size_t memoryPoolSize, int bufferCount);
    void CreateFontVertexBuffer(nn::gfx::MemoryPool* pMemoryPool, ptrdiff_t memoryPoolOffset, size_t memoryPoolSize, int bufferCount);
    void CreateFontIndexBuffer(nn::gfx::MemoryPool* pMemoryPool, ptrdiff_t memoryPoolOffset, size_t memoryPoolSize);
    void CreateFontTexture(nn::gfx::MemoryPool* pMemoryPool, ptrdiff_t memoryPoolOffset, size_t memoryPoolSize);

    void FinalizeGfx();
    void FinalizeFont();

    void PrintGlyph(float x, const DebugFontGlyph& glyph);
    bool GetGlyphUtf8(const DebugFontGlyph** pGlyph, uint32_t code);
    bool GetGlyphCp932(const DebugFontGlyph** pGlyph, uint16_t code);
    bool GetGlyphUtf16(const DebugFontGlyph** pGlyph, uint16_t code);
    void BuildConstantBuffer();
    void BuildVertexBuffer();

    void* AllocateFromHeap(size_t size, size_t alignment = 4);
    void DeallocateHeap(void* ptr);
    static void *GpuBufferAllocateFunction(size_t size, size_t alignment, void* pUserData);
    static void GpuBufferDeallocateFunction(void* ptr, void* pUserData);


    nn::gfx::Device* m_pDevice;
    int m_DisplayWidth;
    int m_DisplayHeight;
    nn::gfx::DescriptorPool* m_pTextureDescriptorPool;
    int m_TextureDescriptorIndexSlot;
    nn::gfx::DescriptorPool* m_pSamplerDescriptorPool;
    int m_SamplerDescriptorIndexSlot;
    int m_CharCountMax;

    nn::os::MutexType m_Mutex;
    nn::util::BytePtr m_pMemoryHeap;
    nn::mem::StandardAllocator m_Allocator;
    bool m_IsInitialized;

    nn::util::Color4u8Type m_TextColor; //!< 文字色
    nn::util::Float2 m_Scale;           //!< 文字拡大率
    nn::util::Float2 m_CursorPos;       //!< カーソル位置

    nn::gfx::RasterizerState m_RasterizerState;
    nn::gfx::DepthStencilState m_DepthStencilState;
    nn::gfx::BlendState m_BlendState;

    nn::gfx::Texture m_Texture;
    nn::gfx::TextureView m_TextureView;
    nn::gfx::DescriptorSlot m_DescriptorSlotForTexture;

    nn::util::BytePtr m_pMemoryPoolPtr;
    nn::gfx::MemoryPool m_MemoryPool;
    nn::gfx::MemoryPool* m_pMemoryPool;
    ptrdiff_t m_MemoryPoolOffset;
    size_t m_MemoryPoolSize;
    bool m_IsUserMemoryPoolEnable;
    detail::GpuBuffer m_ConstantBuffer;
    detail::GpuBuffer m_VertexBuffer;
    nn::gfx::Buffer m_IndexBuffer;
    int m_VertexBufferIndex;
    int m_ConstantBufferIndex;
    int m_BufferCount;

    nn::gfx::ResShaderFile* m_pResShaderFile;
    nn::gfx::ShaderCodeType m_ShaderCodeType;
    int m_VertexShaderSlot;
    int m_PixelShaderSlot;
    int m_TextureSlot;
    nn::gfx::VertexState m_VertexState;
    nn::gfx::Sampler m_Sampler;
    nn::gfx::DescriptorSlot m_DescriptorSlotForSampler;
    size_t  m_VertexBufferOffset;       //! 頂点バッファの GpuBuffer 上での先頭ポインタからのオフセット
    size_t  m_ConstantBufferOffset;     //! コンスタントバッファの GpuBuffer 上での先頭ポインタからのオフセット
    size_t  m_VertexBufferSize;         //! 頂点バッファのサイズ
    int m_CharCount;                    //!< 格納した文字数
    CharAttribute* m_CharAttrs;         //!< 頂点属性配列

    bool m_IsWidthFixed;
    float m_FixedWidth;
    float m_LineSpace;
    int m_TabWidth;
};

DebugFontTextWriterImpl::DebugFontTextWriterImpl() :
m_pDevice(nullptr),
m_DisplayWidth(0),
m_DisplayHeight(0),
m_CharCountMax(0),
m_pMemoryHeap(nullptr),
m_IsInitialized(false),
m_pMemoryPoolPtr(nullptr),
m_pMemoryPool(nullptr),
m_MemoryPoolOffset(0),
m_MemoryPoolSize(0),
m_IsUserMemoryPoolEnable(false),
m_VertexBufferIndex(0),
m_ConstantBufferIndex(0),
m_BufferCount(0),
m_pResShaderFile(nullptr),
m_VertexShaderSlot(0),
m_PixelShaderSlot(0),
m_TextureSlot(0),
m_VertexBufferOffset(0),
m_ConstantBufferOffset(0),
m_VertexBufferSize(0),
m_CharCount(0),
m_CharAttrs(nullptr),
m_IsWidthFixed(false),
m_FixedWidth(0),
m_LineSpace(0),
m_TabWidth(4)
{}

DebugFontTextWriterImpl::~DebugFontTextWriterImpl()
{}

size_t GetMemorySizeStandardAllocator(size_t alignment, size_t size)
{
    return ((alignment > 4 * 1024) ? alignment : 4 * 1024) + ((size > 16 * 1024) ? size : 16 * 1024);
}

size_t DebugFontTextWriterImpl::GetRequiredMemorySize(nn::gfx::Device* pDevice, const DebugFontTextWriterInfo& info)
{
    size_t requiredSize = 0;
    size_t alignment;
    size_t size;

    // シェーダ
    {
        alignment = GetBuildinShaderResourceAlignment();
        size = sizeof(DebugFontBuildinShader);
        requiredSize += GetMemorySizeStandardAllocator(alignment, size);
    }

    // 頂点ステート
    requiredSize += GetRequiredMemorySizeVertexState();

    // ブレンドステート
    requiredSize += GetRequiredMemorySizeBlendState();

    // 文字バッファ
    {
        alignment = 4;
        size = sizeof(CharAttribute) * info.GetCharCountMax();
        requiredSize += GetMemorySizeStandardAllocator(alignment, size);
    }

    // 定数バッファ・頂点バッファ・インデクスバッファ・テクスチャ
    if (info.IsUserMemoryPoolEnabled() == false)
    {
        requiredSize += GetRequiredMemoryPoolSize(pDevice, info.GetCharCountMax(), info.GetBufferCount());
    }

    // GpuBuffer
    requiredSize += GetRequiredMemorySizeGpuBuffer(info.GetBufferCount());

    return requiredSize;
}

size_t DebugFontTextWriterImpl::GetRequiredMemoryPoolSize(nn::gfx::Device* pDevice, const DebugFontTextWriterInfo& info)
{
    size_t requiredSize = 0;

    // 定数バッファ・頂点バッファ・インデクスバッファ・テクスチャ
    if (info.IsUserMemoryPoolEnabled() == true)
    {
        requiredSize += GetRequiredMemoryPoolSize(pDevice, info.GetCharCountMax(), info.GetBufferCount());
    }

    return requiredSize;
}

size_t DebugFontTextWriterImpl::GetRequiredMemorySizeVertexState()
{
    nn::gfx::VertexState::InfoType info;
    info.SetDefault();
    nn::gfx::VertexAttributeStateInfo attribs[VertexAttributeCount];
    {
        attribs[0].SetDefault();
        attribs[0].SetNamePtr("aVertex");
        attribs[0].SetBufferIndex(0);
        attribs[0].SetFormat(nn::gfx::AttributeFormat_32_32_32_Float);
        attribs[0].SetOffset(offsetof(Vertex, xyz));

        attribs[1].SetDefault();
        attribs[1].SetNamePtr("aColor");
        attribs[1].SetBufferIndex(0);
        attribs[1].SetFormat(nn::gfx::AttributeFormat_8_8_8_8_UintToFloat);
        attribs[1].SetOffset(offsetof(Vertex, color));

        attribs[2].SetDefault();
        attribs[2].SetNamePtr("aTexCoord0");
        attribs[2].SetBufferIndex(0);
        attribs[2].SetFormat(nn::gfx::AttributeFormat_32_32_Float);
        attribs[2].SetOffset(offsetof(Vertex, uv));
    }
    nn::gfx::VertexBufferStateInfo buffers[1];
    {
        buffers[0].SetDefault();
        buffers[0].SetStride(sizeof(Vertex));
    }
    info.SetVertexAttributeStateInfoArray(attribs, GetArrayLength(attribs));
    info.SetVertexBufferStateInfoArray(buffers, GetArrayLength(buffers));

    size_t alignment = nn::gfx::VertexState::RequiredMemoryInfo_Alignment;
    size_t size = nn::gfx::VertexState::GetRequiredMemorySize(info);
    return GetMemorySizeStandardAllocator(alignment, size);
}

size_t DebugFontTextWriterImpl::GetRequiredMemorySizeBlendState()
{
    nn::gfx::BlendState::InfoType info;
    info.SetDefault();
    nn::gfx::BlendTargetStateInfo targetInfo;
    {
        targetInfo.SetDefault();
        targetInfo.SetChannelMask(
            nn::gfx::ChannelMask_Red
            | nn::gfx::ChannelMask_Green
            | nn::gfx::ChannelMask_Blue);
        targetInfo.SetBlendEnabled(true);
        targetInfo.SetColorBlendFunction(nn::gfx::BlendFunction_Add);
        targetInfo.SetDestinationColorBlendFactor(nn::gfx::BlendFactor_OneMinusSourceAlpha);
        targetInfo.SetSourceColorBlendFactor(nn::gfx::BlendFactor_SourceAlpha);
    }
    info.SetBlendTargetStateInfoArray(&targetInfo, 1);

    size_t alignment = 4;
    size_t size = nn::gfx::BlendState::GetRequiredMemorySize(info);
    return GetMemorySizeStandardAllocator(alignment, size);
}

size_t DebugFontTextWriterImpl::GetRequiredMemorySizeGpuBuffer(int bufferCount)
{
    size_t alignment;
    size_t size;
    size_t requiredSize = 0;

    // GpuBuffer 用
    alignment = 4;
    size = sizeof(nn::gfx::Buffer) * bufferCount;
    requiredSize += GetMemorySizeStandardAllocator(alignment, size);
    alignment = 4;
    size = sizeof(std::atomic<size_t>);
    requiredSize += GetMemorySizeStandardAllocator(alignment, size);

    alignment = 4;
    size = sizeof(nn::gfx::Buffer) * bufferCount;
    requiredSize += GetMemorySizeStandardAllocator(alignment, size);
    alignment = 4;
    size = sizeof(std::atomic<size_t>);
    requiredSize += GetMemorySizeStandardAllocator(alignment, size);

    return requiredSize;
}

size_t DebugFontTextWriterImpl::GetRequiredMemoryPoolSize(nn::gfx::Device* pDevice, int charCountMax, int bufferCount)
{
    nn::gfx::MemoryPoolInfo memPoolInfo;
    memPoolInfo.SetDefault();
    memPoolInfo.SetMemoryPoolProperty(nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuCached);
    nn::gfx::Buffer::InfoType info;

    // コンスタントバッファ
    const size_t constantBufferSize = GetRequiredConstantBufferSize(pDevice);
    info.SetDefault();
    info.SetGpuAccessFlags(nn::gfx::GpuAccess_ConstantBuffer);
    info.SetSize(constantBufferSize);
    const size_t constantBufferAlignment = nn::gfx::Buffer::GetBufferAlignment(pDevice, info);

    // 頂点バッファ
    const size_t vertexBufferSize = GetRequiredVertexBufferSize(pDevice, charCountMax);
    info.SetDefault();
    info.SetGpuAccessFlags(nn::gfx::GpuAccess_VertexBuffer);
    info.SetSize(vertexBufferSize);
    const size_t vertexBufferAlignment = nn::gfx::Buffer::GetBufferAlignment(pDevice, info);

    // インデクスバッファ
    const size_t indexBufferSize = GetRequiredIndexBufferSize(pDevice, charCountMax);
    info.SetDefault();
    info.SetGpuAccessFlags(nn::gfx::GpuAccess_IndexBuffer);
    info.SetSize(indexBufferSize);
    const size_t indexBufferAlignment = nn::gfx::Buffer::GetBufferAlignment(pDevice, info);

    // テクスチャ
    const size_t textureSize = g_DebugFontImageWidth * g_DebugFontImageHeight * 1;
    nn::gfx::Texture::InfoType texinfo;
    texinfo.SetDefault();
    texinfo.SetGpuAccessFlags(nn::gfx::GpuAccess_Texture);
    texinfo.SetWidth(static_cast<int>(g_DebugFontImageWidth));
    texinfo.SetHeight(static_cast<int>(g_DebugFontImageHeight));
    texinfo.SetDepth(1);
    texinfo.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
    texinfo.SetImageFormat(nn::gfx::ImageFormat_R8_Unorm);
    texinfo.SetTileMode(nn::gfx::TileMode_Linear);
    texinfo.SetMipCount(1);
    texinfo.SetArrayLength(1);
    const size_t textureAlignment = std::max(nn::gfx::Texture::CalculateMipDataAlignment(pDevice, texinfo), nn::gfx::MemoryPool::GetPoolMemoryAlignment(pDevice, memPoolInfo));

    // メモリプール
    const size_t memoryPoolGranularity = nn::gfx::MemoryPool::GetPoolMemorySizeGranularity(
        pDevice,
        memPoolInfo);

    size_t requiredSize = nn::util::align_up(
        constantBufferAlignment + constantBufferSize * bufferCount
        + vertexBufferAlignment + vertexBufferSize * bufferCount
        + indexBufferAlignment + indexBufferSize
        + textureAlignment + textureSize,
        memoryPoolGranularity);

    return requiredSize;
}

void DebugFontTextWriterImpl::Initialize(
    nn::gfx::Device* pDevice,
    const DebugFontTextWriterInfo& info,
    void* pMemory,
    size_t memorySize,
    nn::gfx::MemoryPool* pMemoryPool,
    ptrdiff_t memoryPoolOffset,
    size_t memoryPoolSize
    )
{
    NN_UNUSED(pMemoryPool);
    NN_UNUSED(memoryPoolOffset);
    NN_UNUSED(memoryPoolSize);

    NN_SDK_ASSERT(m_IsInitialized == false);

    // mutex
    nn::os::InitializeMutex(&m_Mutex, false, 0);
    nn::os::LockMutex(&m_Mutex);
    {
        m_pMemoryHeap.Reset(pMemory);
        m_Allocator.Initialize(m_pMemoryHeap.Get(), memorySize);

        m_pDevice = pDevice;
        m_CharCountMax = info.GetCharCountMax();
        m_BufferCount = info.GetBufferCount();

        // メモリプール
        if (info.IsUserMemoryPoolEnabled())
        {
            m_pMemoryPool = pMemoryPool;
            m_MemoryPoolOffset = memoryPoolOffset;
            m_MemoryPoolSize = memoryPoolSize;
            m_IsUserMemoryPoolEnable = true;
        }
        else {
            m_pMemoryPool = &m_MemoryPool;
            m_MemoryPoolOffset = 0;
            m_MemoryPoolSize = 0;
            m_IsUserMemoryPoolEnable = false;
        }

        // gfx
        InitializeGfxShader();
        InitializeGfxVertexState();
        InitializeGfxSampler();
        InitializeGfxBlendState();
        InitializeGfxDepthStencilState();
        InitializeGfxRasterizerState();

        // font
        InitializeFontMemoryPoolObject();
        InitializeFontCharAttributeBuffer();

        // writer
        const nn::util::Color4u8Type white = { { std::numeric_limits<uint8_t>::max(), std::numeric_limits<uint8_t>::max(), std::numeric_limits<uint8_t>::max(), std::numeric_limits<uint8_t>::max() } };
        m_TextColor = white;
        m_Scale.x = 1.0f;
        m_Scale.y = 1.0f;
        m_CursorPos.x = 0.0f;
        m_CursorPos.y = 0.0f;

        m_IsInitialized = true;
    }
    nn::os::UnlockMutex(&m_Mutex);
}

void DebugFontTextWriterImpl::Finalize()
{
    if (!m_IsInitialized){
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return;
    }
    nn::os::LockMutex(&m_Mutex);
    {
        FinalizeFont();
        FinalizeGfx();

        m_Allocator.Finalize();
        m_pMemoryHeap.Reset(nullptr);

        m_pDevice = nullptr;
        m_DisplayWidth = 0;
        m_DisplayHeight = 0;
        m_CharCountMax = 0;
    }
    nn::os::UnlockMutex(&m_Mutex);

    nn::os::FinalizeMutex(&m_Mutex);
    m_IsInitialized = false;
}

void DebugFontTextWriterImpl::InitializeGfxShader()
{
    {
        const size_t sdrAlign = GetBuildinShaderResourceAlignment();
        const size_t sdrSize = sizeof(DebugFontBuildinShader);
        void* pShader = AllocateFromHeap(sdrSize, sdrAlign);

        std::memcpy(pShader, DebugFontBuildinShader, sizeof(DebugFontBuildinShader));

        m_pResShaderFile = nn::gfx::ResShaderFile::ResCast(pShader);
    }
    nn::gfx::ResShaderContainer* pContainer = m_pResShaderFile->GetShaderContainer();
    NN_SDK_ASSERT_NOT_NULL(pContainer);
    pContainer->Initialize(m_pDevice);

    {
        nn::gfx::ResShaderVariation* pVariation = pContainer->GetResShaderVariation(0);
        NN_SDK_ASSERT_NOT_NULL(pVariation);

        // バイナリで初期化
        nn::gfx::ResShaderProgram* pBinaryProgram = pVariation->GetResShaderProgram(nn::gfx::ShaderCodeType_Binary);
        m_ShaderCodeType = nn::gfx::ShaderCodeType_Binary;

        nn::gfx::ShaderInitializeResult shaderResult = pBinaryProgram != NULL ? pBinaryProgram->Initialize(m_pDevice) : nn::gfx::ShaderInitializeResult_SetupFailed;

        if (shaderResult != nn::gfx::ShaderInitializeResult_Success)
        {
            // 中間表現で初期化
            nn::gfx::ResShaderProgram* pIrProgram = pVariation->GetResShaderProgram(nn::gfx::ShaderCodeType_Ir);
            m_ShaderCodeType = nn::gfx::ShaderCodeType_Ir;

            shaderResult = pIrProgram != NULL ? pIrProgram->Initialize(m_pDevice) : nn::gfx::ShaderInitializeResult_SetupFailed;
        }

        if (shaderResult != nn::gfx::ShaderInitializeResult_Success)
        {
            // ソースで初期化
            nn::gfx::ResShaderProgram* pSourceProgram = pVariation->GetResShaderProgram(nn::gfx::ShaderCodeType_Source);
            NN_SDK_ASSERT_NOT_NULL(pSourceProgram);
            m_ShaderCodeType = nn::gfx::ShaderCodeType_Source;
            shaderResult = pSourceProgram->Initialize(m_pDevice);
        }
        NN_SDK_ASSERT(shaderResult == nn::gfx::ShaderInitializeResult_Success);
    }

    nn::gfx::Shader* pVertexShader = m_pResShaderFile->GetShaderContainer()->GetResShaderVariation(0)->GetResShaderProgram(m_ShaderCodeType)->GetShader();
    nn::gfx::Shader* pPixelShader = m_pResShaderFile->GetShaderContainer()->GetResShaderVariation(0)->GetResShaderProgram(m_ShaderCodeType)->GetShader();
    m_VertexShaderSlot = pVertexShader->GetInterfaceSlot(nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_ConstantBuffer, "ShaderParam");
    m_PixelShaderSlot = pPixelShader->GetInterfaceSlot(nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_ConstantBuffer, "ShaderParam");
    m_TextureSlot = pPixelShader->GetInterfaceSlot(nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, "uTextureSrc");
}

void DebugFontTextWriterImpl::InitializeGfxVertexState()
{
    nn::gfx::VertexState::InfoType info;
    info.SetDefault();
    nn::gfx::VertexAttributeStateInfo attribs[VertexAttributeCount];
    {
        attribs[0].SetDefault();
        attribs[0].SetNamePtr("aVertex");
        attribs[0].SetBufferIndex(0);
        attribs[0].SetFormat(nn::gfx::AttributeFormat_32_32_32_Float);
        attribs[0].SetOffset(offsetof(Vertex, xyz));

        attribs[1].SetDefault();
        attribs[1].SetNamePtr("aColor");
        attribs[1].SetBufferIndex(0);
        attribs[1].SetFormat(nn::gfx::AttributeFormat_8_8_8_8_UintToFloat);
        attribs[1].SetOffset(offsetof(Vertex, color));

        attribs[2].SetDefault();
        attribs[2].SetNamePtr("aTexCoord0");
        attribs[2].SetBufferIndex(0);
        attribs[2].SetFormat(nn::gfx::AttributeFormat_32_32_Float);
        attribs[2].SetOffset(offsetof(Vertex, uv));
    }
    nn::gfx::VertexBufferStateInfo buffers[1];
    {
        buffers[0].SetDefault();
        buffers[0].SetStride(sizeof(Vertex));
    }
    info.SetVertexAttributeStateInfoArray(attribs, GetArrayLength(attribs));
    info.SetVertexBufferStateInfoArray(buffers, GetArrayLength(buffers));
    void* memory;
    size_t memorySize = nn::gfx::VertexState::GetRequiredMemorySize(info);

    memory = AllocateFromHeap(memorySize, nn::gfx::VertexState::RequiredMemoryInfo_Alignment);
    m_VertexState.SetMemory(memory, memorySize);

    nn::gfx::Shader* pVertexShader = m_pResShaderFile->GetShaderContainer()->GetResShaderVariation(0)->GetResShaderProgram(m_ShaderCodeType)->GetShader();
    m_VertexState.Initialize(m_pDevice, info, pVertexShader);
}

void DebugFontTextWriterImpl::InitializeGfxSampler()
{
    // サンプラ
    nn::gfx::Sampler::InfoType info;
    info.SetDefault();
    info.SetAddressU(nn::gfx::TextureAddressMode_Repeat);
    info.SetAddressV(nn::gfx::TextureAddressMode_Repeat);
    info.SetFilterMode(nn::gfx::FilterMode_MinLinear_MagLinear_MipPoint);
    m_Sampler.Initialize(m_pDevice, info);
}

void DebugFontTextWriterImpl::InitializeGfxBlendState()
{
    nn::gfx::BlendState::InfoType info;
    info.SetDefault();
    nn::gfx::BlendTargetStateInfo targetInfo;
    {
        targetInfo.SetDefault();
        targetInfo.SetChannelMask(
            nn::gfx::ChannelMask_Red
            | nn::gfx::ChannelMask_Green
            | nn::gfx::ChannelMask_Blue);
        targetInfo.SetBlendEnabled(true);
        targetInfo.SetColorBlendFunction(nn::gfx::BlendFunction_Add);
        targetInfo.SetDestinationColorBlendFactor(nn::gfx::BlendFactor_OneMinusSourceAlpha);
        targetInfo.SetSourceColorBlendFactor(nn::gfx::BlendFactor_SourceAlpha);
    }
    info.SetBlendTargetStateInfoArray(&targetInfo, 1);
    {
        size_t memorySize = nn::gfx::BlendState::GetRequiredMemorySize(info);
        m_BlendState.SetMemory(AllocateFromHeap(memorySize), memorySize);
    }
    m_BlendState.Initialize(m_pDevice, info);
}

void DebugFontTextWriterImpl::InitializeGfxDepthStencilState()
{
    nn::gfx::DepthStencilState::InfoType info;
    info.SetDefault();
    info.SetDepthTestEnabled(false);
    info.SetDepthWriteEnabled(false);
    m_DepthStencilState.Initialize(m_pDevice, info);
}

void DebugFontTextWriterImpl::InitializeGfxRasterizerState()
{
    nn::gfx::RasterizerState::InfoType info;
    info.SetDefault();
    info.SetCullMode(nn::gfx::CullMode_None);
    info.SetScissorEnabled(true);
    info.SetDepthClipEnabled(false);
    m_RasterizerState.Initialize(m_pDevice, info);
}

void DebugFontTextWriterImpl::InitializeFontMemoryPoolObject()
{
    // 定数バッファ・頂点バッファ・インデクスバッファ
    nn::gfx::MemoryPoolInfo memPoolInfo;
    memPoolInfo.SetDefault();
    memPoolInfo.SetMemoryPoolProperty(nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuCached);
    nn::gfx::Buffer::InfoType info;


    // コンスタントバッファ
    const size_t constantBufferSize = GetRequiredConstantBufferSize(m_pDevice);
    info.SetDefault();
    info.SetGpuAccessFlags(nn::gfx::GpuAccess_ConstantBuffer);
    info.SetSize(constantBufferSize);
    const size_t constantBufferAlignment = nn::gfx::Buffer::GetBufferAlignment(m_pDevice, info);

    // 頂点バッファ
    const size_t vertexBufferSize = GetRequiredVertexBufferSize(m_pDevice, m_CharCountMax);
    info.SetDefault();
    info.SetGpuAccessFlags(nn::gfx::GpuAccess_VertexBuffer);
    info.SetSize(vertexBufferSize);
    const size_t vertexBufferAlignment = nn::gfx::Buffer::GetBufferAlignment(m_pDevice, info);

    // インデクスバッファ
    const size_t indexBufferSize = GetRequiredIndexBufferSize(m_pDevice, m_CharCountMax);
    info.SetDefault();
    info.SetGpuAccessFlags(nn::gfx::GpuAccess_IndexBuffer);
    info.SetSize(indexBufferSize);
    const size_t indexBufferAlignment = nn::gfx::Buffer::GetBufferAlignment(m_pDevice, info);

    // テクスチャ (線形タイルモードなので CpuUncached で作成可能)
    const size_t textureSize = g_DebugFontImageWidth * g_DebugFontImageHeight * 1;
    nn::gfx::Texture::InfoType texinfo;
    texinfo.SetDefault();
    texinfo.SetGpuAccessFlags(nn::gfx::GpuAccess_Texture);
    texinfo.SetWidth(static_cast<int>(g_DebugFontImageWidth));
    texinfo.SetHeight(static_cast<int>(g_DebugFontImageHeight));
    texinfo.SetDepth(1);
    texinfo.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
    texinfo.SetImageFormat(nn::gfx::ImageFormat_R8_Unorm);
    texinfo.SetTileMode(nn::gfx::TileMode_Linear);
    texinfo.SetMipCount(1);
    texinfo.SetArrayLength(1);
    const size_t textureAlignment = std::max(nn::gfx::Texture::CalculateMipDataAlignment(m_pDevice, texinfo), nn::gfx::MemoryPool::GetPoolMemoryAlignment(m_pDevice, memPoolInfo));

    // メモリプールを作成
    if (m_IsUserMemoryPoolEnable == false)
    {
        const size_t memoryPoolGranularity = nn::gfx::MemoryPool::GetPoolMemorySizeGranularity(
            m_pDevice,
            memPoolInfo);

        const size_t memoryPoolAlignment = nn::gfx::MemoryPool::GetPoolMemoryAlignment(
            m_pDevice,
            memPoolInfo);

        const size_t memoryPoolSize = nn::util::align_up(
            constantBufferAlignment + constantBufferSize * m_BufferCount
            + vertexBufferAlignment + vertexBufferSize * m_BufferCount
            + indexBufferAlignment + indexBufferSize
            + textureAlignment + textureSize,
            memoryPoolGranularity);

        void* pMemoryPoolMemory = AllocateFromHeap(memoryPoolSize, memoryPoolAlignment);
        m_pMemoryPoolPtr.Reset(pMemoryPoolMemory);
        memPoolInfo.SetPoolMemory(m_pMemoryPoolPtr.Get(), memoryPoolSize);

        m_MemoryPool.Initialize(m_pDevice, memPoolInfo);
        m_MemoryPoolOffset = 0;
        m_MemoryPoolSize = memoryPoolSize;
    }

    // コンスタントバッファを作成
    ptrdiff_t offset = nn::util::align_up(m_MemoryPoolOffset, constantBufferAlignment);
    CreateFontConstantBuffer(m_pMemoryPool, offset, constantBufferSize, m_BufferCount);
    offset += constantBufferSize * m_BufferCount;

    // 頂点バッファを作成
    offset = nn::util::align_up(offset, vertexBufferAlignment);
    CreateFontVertexBuffer(m_pMemoryPool, offset, vertexBufferSize, m_BufferCount);
    offset += vertexBufferSize * m_BufferCount;

    // インデクスバッファを作成
    offset = nn::util::align_up(offset, indexBufferAlignment);
    CreateFontIndexBuffer(m_pMemoryPool, offset, indexBufferSize);
    offset += indexBufferSize;

    // テクスチャを作成
    offset = nn::util::align_up(offset, textureAlignment);
    CreateFontTexture(m_pMemoryPool, offset, textureSize);
    offset += textureSize;

    m_VertexBufferSize = CalculateVertexBufferSize(m_CharCountMax);
}

void DebugFontTextWriterImpl::InitializeFontCharAttributeBuffer()
{
    // 文字バッファ
    const size_t drawBufSize = sizeof(CharAttribute) * m_CharCountMax;
    m_CharAttrs = static_cast<CharAttribute*>(AllocateFromHeap(drawBufSize));
    m_CharCount = 0;
}

void DebugFontTextWriterImpl::CreateFontConstantBuffer(nn::gfx::MemoryPool* pMemoryPool, ptrdiff_t memoryPoolOffset, size_t memoryPoolSize, int bufferCount)
{
    detail::GpuBuffer::InitializeArg arg;
    arg.SetGpuAccessFlag(nn::gfx::GpuAccess_ConstantBuffer);
    arg.SetBufferSize(memoryPoolSize);
    arg.SetBufferCount(bufferCount);
    arg.SetMemoryPool(pMemoryPool);
    arg.SetMemoryPoolOffset(memoryPoolOffset);
    arg.SetAllocator(GpuBufferAllocateFunction, this);
    arg.SetAllocateSyncFlag(false);

    m_ConstantBuffer.Initialize(m_pDevice, arg);
}

void DebugFontTextWriterImpl::CreateFontVertexBuffer(nn::gfx::MemoryPool* pMemoryPool, ptrdiff_t memoryPoolOffset, size_t memoryPoolSize, int bufferCount)
{
    detail::GpuBuffer::InitializeArg arg;
    arg.SetGpuAccessFlag(nn::gfx::GpuAccess_VertexBuffer);
    arg.SetBufferSize(memoryPoolSize);
    arg.SetBufferCount(bufferCount);
    arg.SetMemoryPool(pMemoryPool);
    arg.SetMemoryPoolOffset(memoryPoolOffset);
    arg.SetAllocator(GpuBufferAllocateFunction, this);
    arg.SetAllocateSyncFlag(false);

    m_VertexBuffer.Initialize(m_pDevice, arg);
}

void DebugFontTextWriterImpl::CreateFontIndexBuffer(nn::gfx::MemoryPool* pMemoryPool, ptrdiff_t memoryPoolOffset, size_t memoryPoolSize)
{
    nn::gfx::Buffer::InfoType info;
    info.SetDefault();
    info.SetSize(memoryPoolSize);
    info.SetGpuAccessFlags(nn::gfx::GpuAccess_IndexBuffer);

    if (NN_STATIC_CONDITION(nn::gfx::Buffer::IsMemoryPoolRequired))
    {
        m_IndexBuffer.Initialize(m_pDevice, info, pMemoryPool, memoryPoolOffset, info.GetSize());
    }
    else
    {
        m_IndexBuffer.Initialize(m_pDevice, info, NULL, 0, 0);
    }

    // インデクスバッファ初期化
    {
        // バッファ初期化
        uint16_t* buffer = m_IndexBuffer.Map<uint16_t>();
        for (int i = 0; i < m_CharCountMax; i++)
        {
            uint32_t  indexOffset = i * IndexCountByLetter;
            uint32_t  vertexOffset = i * VertexCountByLetter;
            buffer[indexOffset + 0] = static_cast<uint16_t>(vertexOffset + 0);
            buffer[indexOffset + 1] = static_cast<uint16_t>(vertexOffset + 2);
            buffer[indexOffset + 2] = static_cast<uint16_t>(vertexOffset + 3);
            buffer[indexOffset + 3] = static_cast<uint16_t>(vertexOffset + 3);
            buffer[indexOffset + 4] = static_cast<uint16_t>(vertexOffset + 1);
            buffer[indexOffset + 5] = static_cast<uint16_t>(vertexOffset + 0);
        }
        m_IndexBuffer.FlushMappedRange( 0, memoryPoolSize );
        m_IndexBuffer.Unmap();
    }
}

void DebugFontTextWriterImpl::CreateFontTexture(nn::gfx::MemoryPool* pMemoryPool, ptrdiff_t memoryPoolOffset, size_t memoryPoolSize)
{
    // メモリプールへテクスチャデータをコピー
    void* poolMemory = pMemoryPool->Map();
    nn::util::BytePtr texMemory(poolMemory);
    texMemory += memoryPoolOffset;
    std::memcpy(texMemory.Get(), DebugFontImage, g_DebugFontImageSize);
    pMemoryPool->Unmap();

    // テクスチャ作成
    nn::gfx::Texture::InfoType texinfo;
    texinfo.SetDefault();
    texinfo.SetGpuAccessFlags(nn::gfx::GpuAccess_Texture);
    texinfo.SetWidth(static_cast<int>(g_DebugFontImageWidth));
    texinfo.SetHeight(static_cast<int>(g_DebugFontImageHeight));
    texinfo.SetDepth(1);
    texinfo.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
    texinfo.SetImageFormat(nn::gfx::ImageFormat_R8_Unorm);
    texinfo.SetTileMode(nn::gfx::TileMode_Linear);
    texinfo.SetMipCount(1);
    texinfo.SetArrayLength(1);

    m_Texture.Initialize(m_pDevice, texinfo, pMemoryPool, memoryPoolOffset, memoryPoolSize);

    // TextureView
    nn::gfx::TextureView::InfoType texviewinfo;
    texviewinfo.SetDefault();
    texviewinfo.SetChannelMapping(nn::gfx::ChannelMapping_One, nn::gfx::ChannelMapping_One, nn::gfx::ChannelMapping_One, nn::gfx::ChannelMapping_Red);
    texviewinfo.SetImageFormat(nn::gfx::ImageFormat_R8_Unorm);
    texviewinfo.SetImageDimension(nn::gfx::ImageDimension_2d);
    texviewinfo.SetTexturePtr(&m_Texture);
    m_TextureView.Initialize(m_pDevice, texviewinfo);
}

void DebugFontTextWriterImpl::FinalizeGfx()
{
    void* pBlendMemory = m_BlendState.GetMemory();
    void* pVertexMemory = m_VertexState.GetMemory();

    m_RasterizerState.Finalize(m_pDevice);
    m_DepthStencilState.Finalize(m_pDevice);
    m_BlendState.Finalize(m_pDevice);

    m_Sampler.Finalize(m_pDevice);
    m_VertexState.Finalize(m_pDevice);

    nn::gfx::ResShaderContainer* pShaderContainer = m_pResShaderFile->GetShaderContainer();
    NN_SDK_ASSERT_NOT_NULL(pShaderContainer);
    nn::gfx::ResShaderVariation* pVariation = pShaderContainer->GetResShaderVariation(0);
    pVariation->GetResShaderProgram(m_ShaderCodeType)->Finalize(m_pDevice);
    pShaderContainer->Finalize(m_pDevice);
    DeallocateHeap(m_pResShaderFile);

    DeallocateHeap(pBlendMemory);
    DeallocateHeap(pVertexMemory);
}

void DebugFontTextWriterImpl::FinalizeFont()
{
    DeallocateHeap(m_CharAttrs);

    m_ConstantBuffer.Finalize(m_pDevice, GpuBufferDeallocateFunction, this);
    m_VertexBuffer.Finalize(m_pDevice, GpuBufferDeallocateFunction, this);
    m_IndexBuffer.Finalize(m_pDevice);

    m_TextureView.Finalize(m_pDevice);
    m_Texture.Finalize(m_pDevice);

    if (m_IsUserMemoryPoolEnable == false)
    {
        m_MemoryPool.Finalize(m_pDevice);
        DeallocateHeap(m_pMemoryPoolPtr.Get());
    }
}

void DebugFontTextWriterImpl::SetDisplayWidth(int displayWidth)
{
    if (!m_IsInitialized){
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return;
    }

    nn::os::LockMutex(&m_Mutex);
    {
        m_DisplayWidth = displayWidth;
    }
    nn::os::UnlockMutex(&m_Mutex);
}

void DebugFontTextWriterImpl::SetDisplayHeight(int displayHeight)
{
    if (!m_IsInitialized){
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return;
    }

    nn::os::LockMutex(&m_Mutex);
    {
        m_DisplayHeight = displayHeight;
    }
    nn::os::UnlockMutex(&m_Mutex);
}

void DebugFontTextWriterImpl::SetTextureDescriptor(nn::gfx::DescriptorPool* pTextureDescriptorPool, int textureDescriptorIndexSlot)
{
    if (!m_IsInitialized){
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return;
    }

    nn::os::LockMutex(&m_Mutex);
    {
        m_pTextureDescriptorPool = pTextureDescriptorPool;
        m_TextureDescriptorIndexSlot = textureDescriptorIndexSlot;

        // テクスチャデスクリプタスロット
        m_pTextureDescriptorPool->BeginUpdate();
        {
            m_pTextureDescriptorPool->SetTextureView(m_TextureDescriptorIndexSlot, &m_TextureView);
        }
        m_pTextureDescriptorPool->EndUpdate();
        m_pTextureDescriptorPool->GetDescriptorSlot(&m_DescriptorSlotForTexture, m_TextureDescriptorIndexSlot);
    }
    nn::os::UnlockMutex(&m_Mutex);
}

void DebugFontTextWriterImpl::SetSamplerDescriptor(nn::gfx::DescriptorPool* pSamplerDescriptorPool, int samplerDescriptorIndexSlot)
{
    if (!m_IsInitialized){
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return;
    }

    nn::os::LockMutex(&m_Mutex);
    {
        m_pSamplerDescriptorPool = pSamplerDescriptorPool;
        m_SamplerDescriptorIndexSlot = samplerDescriptorIndexSlot;

        // サンプラデスクリプタスロット
        m_pSamplerDescriptorPool->BeginUpdate();
        {
            m_pSamplerDescriptorPool->SetSampler(m_SamplerDescriptorIndexSlot, &m_Sampler);
        }
        m_pSamplerDescriptorPool->EndUpdate();
        m_pSamplerDescriptorPool->GetDescriptorSlot(&m_DescriptorSlotForSampler, m_SamplerDescriptorIndexSlot);
    }
    nn::os::UnlockMutex(&m_Mutex);
}

void DebugFontTextWriterImpl::SetLineHeight(float height)
{
    if (!m_IsInitialized) {
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return;
    }

    nn::os::LockMutex(&m_Mutex);
    {
        m_LineSpace = height - g_DebugFontLineFeed * GetScaleY();
    }
    nn::os::UnlockMutex(&m_Mutex);
}

void DebugFontTextWriterImpl::SetTabWidth(int tabWidth)
{
    if (!m_IsInitialized) {
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return;
    }

    nn::os::LockMutex(&m_Mutex);
    {
        m_TabWidth = tabWidth;
    }
    nn::os::UnlockMutex(&m_Mutex);
}

void DebugFontTextWriterImpl::SetScale(float hScale, float vScale)
{
    if (!m_IsInitialized){
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return;
    }

    nn::os::LockMutex(&m_Mutex);
    {
        m_Scale.v[0] = hScale;
        m_Scale.v[1] = vScale;
    }
    nn::os::UnlockMutex(&m_Mutex);
}

void DebugFontTextWriterImpl::SetFontSize(float width, float height)
{
    const float hScale = width / g_DebugFontCharWidth;
    const float vScale = height / g_DebugFontLineFeed;
    SetScale(hScale, vScale);
}

void DebugFontTextWriterImpl::SetFontSize(float height)
{
    const float scale = height / g_DebugFontLineFeed;
    SetScale(scale, scale);
}

void DebugFontTextWriterImpl::SetCursor(float x, float y)
{
    if (!m_IsInitialized){
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return;
    }

    nn::os::LockMutex(&m_Mutex);
    {
        m_CursorPos.x = x;
        m_CursorPos.y = y;
    }
    nn::os::UnlockMutex(&m_Mutex);
}

void DebugFontTextWriterImpl::SetCursorX(float x)
{
    if (!m_IsInitialized) {
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return;
    }

    nn::os::LockMutex(&m_Mutex);
    {
        m_CursorPos.x = x;
    }
    nn::os::UnlockMutex(&m_Mutex);
}

void DebugFontTextWriterImpl::SetCursorY(float y)
{
    if (!m_IsInitialized) {
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return;
    }

    nn::os::LockMutex(&m_Mutex);
    {
        m_CursorPos.y = y;
    }
    nn::os::UnlockMutex(&m_Mutex);
}

void DebugFontTextWriterImpl::SetTextColor(const nn::util::Color4u8Type& color)
{
    if (!m_IsInitialized){
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return;
    }

    nn::os::LockMutex(&m_Mutex);
    {
        m_TextColor = color;
    }
    nn::os::UnlockMutex(&m_Mutex);
}

void DebugFontTextWriterImpl::Draw(nn::gfx::CommandBuffer* pCommandBuffer)
{
    if (!m_IsInitialized){
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return;
    }
    nn::os::LockMutex(&m_Mutex);
    {
        // gfx state
        pCommandBuffer->SetBlendState(&m_BlendState);
        pCommandBuffer->SetDepthStencilState(&m_DepthStencilState);
        pCommandBuffer->SetRasterizerState(&m_RasterizerState);

        // vertexbuffer
        BuildVertexBuffer();

        // constantbuffer
        BuildConstantBuffer();

        // draw
        {
            nn::gfx::GpuAddress vertexBufferGpuAddr;
            nn::gfx::GpuAddress constantBufferGpuAddr;

            m_VertexBuffer.GetGpuAddress(&vertexBufferGpuAddr);
            vertexBufferGpuAddr.Offset(m_VertexBufferOffset);
            m_ConstantBuffer.GetGpuAddress(&constantBufferGpuAddr);
            constantBufferGpuAddr.Offset(m_ConstantBufferOffset);

            {
                nn::gfx::ResShaderVariation* pVariation = m_pResShaderFile->GetShaderContainer()->GetResShaderVariation(0);
                pCommandBuffer->SetShader(pVariation->GetResShaderProgram(m_ShaderCodeType)->GetShader(), nn::gfx::ShaderStageBit_All);
                pCommandBuffer->SetVertexState(&m_VertexState);

                pCommandBuffer->SetVertexBuffer(m_VertexShaderSlot, vertexBufferGpuAddr, sizeof(Vertex), m_VertexBufferSize);
                pCommandBuffer->SetConstantBuffer(m_VertexShaderSlot, nn::gfx::ShaderStage_Vertex, constantBufferGpuAddr, sizeof(ShaderParam));
                pCommandBuffer->SetConstantBuffer(m_PixelShaderSlot, nn::gfx::ShaderStage_Pixel, constantBufferGpuAddr, sizeof(ShaderParam));
            }

            pCommandBuffer->SetTextureAndSampler(m_TextureSlot, nn::gfx::ShaderStage_Pixel, m_DescriptorSlotForTexture, m_DescriptorSlotForSampler);

            nn::gfx::GpuAddress indexBufferAddr;
            m_IndexBuffer.GetGpuAddress(&indexBufferAddr);

            pCommandBuffer->DrawIndexed(nn::gfx::PrimitiveTopology_TriangleList, nn::gfx::IndexFormat_Uint16, indexBufferAddr, m_CharCount * IndexCountByLetter, 0);
        }

        m_CharCount = 0;
    }
    nn::os::UnlockMutex(&m_Mutex);
}

void DebugFontTextWriterImpl::PrintUtf8(bool defaultLocaleCharset, const char* str)
{
    NN_SDK_ASSERT_NOT_NULL(str);

#if defined(NN_BUILD_CONFIG_OS_WIN32)
    if(defaultLocaleCharset && (GetConsoleOutputCP() != CP_UTF8))
    {
        PrintCp932(str);
    }
    else
    {
        PrintUtf8(str);
    }
#else
    NN_UNUSED(defaultLocaleCharset);
    PrintUtf8(str);
#endif
}

void DebugFontTextWriterImpl::PrintUtf8(const char* str)
{
    NN_SDK_ASSERT_NOT_NULL(str);

    if (!m_IsInitialized) {
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return;
    }

    nn::os::LockMutex(&m_Mutex);
    {
        float xOrigin = m_CursorPos.x;
        float yOrigin = m_CursorPos.y;

        m_CursorPos.y = yOrigin + (g_DebugFontAscent * m_Scale.v[1]);
        float yCursorAdj = yOrigin - m_CursorPos.y;

        const char* cur = str;
        while (*cur)
        {
            uint32_t code;
            int bytes;
            if( GetCharCodeUtf8(code, bytes, cur) )
            {
                cur += bytes;
            }
            else
            {
                break;
            }

            if (code < ' ')
            {
                //---- タグ処理
                if (code == '\n')
                {
                    float lineHeight = GetLineHeight();
                    m_CursorPos.x = xOrigin;
                    m_CursorPos.y += lineHeight;
                }
                else if (code == '\t')
                {
                    if (m_TabWidth > 0)
                    {
                        const float aCharWidth = m_IsWidthFixed ? m_FixedWidth : (g_DebugFontCharWidth * m_Scale.v[0]);
                        const float dx = m_CursorPos.x - xOrigin;
                        const float tabPixel = m_TabWidth * aCharWidth;
                        const int numTab = static_cast<int>(dx / tabPixel) + 1;
                        m_CursorPos.x = xOrigin + tabPixel * numTab;
                    }
                }
            }
            else
            {
                //---- 通常の文字
                const float baseY = m_CursorPos.y;

                const DebugFontGlyph *glyph;
                bool found = GetGlyphUtf8(&glyph, code);
                if (!found) {
                    GetGlyphUtf8(&glyph, '?');
                }

                //---- カーソルはベースラインにあるのでセル上端へ移動する
                const float adj = -(g_DebugFontBaselinePos) * m_Scale.v[1];
                m_CursorPos.y += adj;

                if (m_IsWidthFixed)
                {
                    if (code != ' ')
                    {
                        float margin = (m_FixedWidth - glyph->charWidth * m_Scale.v[0]) / 2;
                        PrintGlyph(m_CursorPos.x + static_cast<int>(margin) + (glyph->left * m_Scale.v[0]), *glyph);
                    }
                    m_CursorPos.x += m_FixedWidth;
                }
                else
                {
                    if (code != ' ')
                    {
                        PrintGlyph(m_CursorPos.x + (glyph->left * m_Scale.v[0]), *glyph);
                    }
                    m_CursorPos.x += (glyph->charWidth * m_Scale.v[0]);
                }

                // 戻す
                m_CursorPos.y = baseY;
            }
        }

        m_CursorPos.y += yCursorAdj;
    }
    nn::os::UnlockMutex(&m_Mutex);
}

void DebugFontTextWriterImpl::PrintCp932(const char* str)
{
    NN_SDK_ASSERT_NOT_NULL(str);

    if (!m_IsInitialized) {
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return;
    }

    nn::os::LockMutex(&m_Mutex);
    {
        float xOrigin = m_CursorPos.x;
        float yOrigin = m_CursorPos.y;

        m_CursorPos.y = yOrigin + (g_DebugFontAscent * m_Scale.v[1]);
        float yCursorAdj = yOrigin - m_CursorPos.y;

        const char* cur = str;
        while (*cur)
        {
            uint16_t code;
            int bytes;
            if( GetCharCodeCp932(code, bytes, cur) )
            {
                cur += bytes;
            }
            else
            {
                break;
            }

            if (code < ' ')
            {
                //---- タグ処理
                if (code == '\n')
                {
                    float lineHeight = GetLineHeight();
                    m_CursorPos.x = xOrigin;
                    m_CursorPos.y += lineHeight;
                }
                else if (code == '\t')
                {
                    if (m_TabWidth > 0)
                    {
                        const float aCharWidth = m_IsWidthFixed ? m_FixedWidth : (g_DebugFontCharWidth * m_Scale.v[0]);
                        const float dx = m_CursorPos.x - xOrigin;
                        const float tabPixel = m_TabWidth * aCharWidth;
                        const int numTab = static_cast<int>(dx / tabPixel) + 1;
                        m_CursorPos.x = xOrigin + tabPixel * numTab;
                    }
                }
            }
            else
            {
                //---- 通常の文字
                const float baseY = m_CursorPos.y;

                const DebugFontGlyph *glyph;
                bool found = GetGlyphCp932(&glyph, code);
                if (!found) {
                    GetGlyphCp932(&glyph, '?');
                }

                //---- カーソルはベースラインにあるのでセル上端へ移動する
                const float adj = -(g_DebugFontBaselinePos) * m_Scale.v[1];
                m_CursorPos.y += adj;

                if (m_IsWidthFixed)
                {
                    if (code != ' ')
                    {
                        float margin = (m_FixedWidth - glyph->charWidth * m_Scale.v[0]) / 2;
                        PrintGlyph(m_CursorPos.x + static_cast<int>(margin) + (glyph->left * m_Scale.v[0]), *glyph);
                    }
                    m_CursorPos.x += m_FixedWidth;
                }
                else
                {
                    if (code != ' ')
                    {
                        PrintGlyph(m_CursorPos.x + (glyph->left * m_Scale.v[0]), *glyph);
                    }
                    m_CursorPos.x += (glyph->charWidth * m_Scale.v[0]);
                }

                // 戻す
                m_CursorPos.y = baseY;
            }
        }

        m_CursorPos.y += yCursorAdj;
    }
    nn::os::UnlockMutex(&m_Mutex);
}

void DebugFontTextWriterImpl::PrintUtf16(const uint16_t* str)
{
    NN_SDK_ASSERT_NOT_NULL(str);

    if (!m_IsInitialized) {
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return;
    }

    nn::os::LockMutex(&m_Mutex);
    {
        int length = 0;
        for (const uint16_t* cur = str; *cur != 0; ++cur, ++length) {}
        NN_SDK_ASSERT(length >= 0);

        float xOrigin = m_CursorPos.x;
        float yOrigin = m_CursorPos.y;

        m_CursorPos.y = yOrigin + (g_DebugFontAscent * m_Scale.v[1]);
        float yCursorAdj = yOrigin - m_CursorPos.y;

        for (int i = 0; i<length; i++)
        {
            uint16_t code = str[i];
            if (code < ' ')
            {
                //---- タグ処理
                if (code == '\n')
                {
                    float lineHeight = GetLineHeight();
                    m_CursorPos.x = xOrigin;
                    m_CursorPos.y += lineHeight;
                }
                else if (code == '\t')
                {
                    if (m_TabWidth > 0)
                    {
                        const float aCharWidth = m_IsWidthFixed ? m_FixedWidth : (g_DebugFontCharWidth * m_Scale.v[0]);
                        const float dx = m_CursorPos.x - xOrigin;
                        const float tabPixel = m_TabWidth * aCharWidth;
                        const int numTab = static_cast<int>(dx / tabPixel) + 1;
                        m_CursorPos.x = xOrigin + tabPixel * numTab;
                    }
                }
            }
            else
            {
                //---- 通常の文字
                const float baseY = m_CursorPos.y;

                const DebugFontGlyph *glyph;
                bool found = GetGlyphUtf16(&glyph, code);
                if (!found) {
                    GetGlyphUtf16(&glyph, L'?');
                }

                //---- カーソルはベースラインにあるのでセル上端へ移動する
                const float adj = -(g_DebugFontBaselinePos) * m_Scale.v[1];
                m_CursorPos.y += adj;

                if (m_IsWidthFixed)
                {
                    if (code != ' ')
                    {
                        float margin = (m_FixedWidth - glyph->charWidth * m_Scale.v[0]) / 2;
                        PrintGlyph(m_CursorPos.x + static_cast<int>(margin) + (glyph->left * m_Scale.v[0]), *glyph);
                    }
                    m_CursorPos.x += m_FixedWidth;
                }
                else
                {
                    if (code != ' ')
                    {
                        PrintGlyph(m_CursorPos.x + (glyph->left * m_Scale.v[0]), *glyph);
                    }
                    m_CursorPos.x += (glyph->charWidth * m_Scale.v[0]);
                }

                // 戻す
                m_CursorPos.y = baseY;
            }
        }

        m_CursorPos.y += yCursorAdj;
    }
    nn::os::UnlockMutex(&m_Mutex);
}

void DebugFontTextWriterImpl::PrintGlyph(float x, const DebugFontGlyph& glyph)
{
    float texLeft;
    float texRight;

    // cellYは左上原点の値が入っているので、これをOpenGLの左下原点の
    // テクスチャ座標に変換してセットする。
    float texTop;
    float texBottom;

    float px;
    float py;
    float width;
    float height;

    {
        uint32_t texWidth = g_DebugFontImageWidth;
        uint32_t texHeight = g_DebugFontImageHeight;
        NN_SDK_ASSERT(texWidth >= 1);
        NN_SDK_ASSERT(texHeight >= 1);

#if defined(EXTRA_MARGIN_ENABLED)
        const float margin = 0.5f;

        texLeft = (1.0f * glyph.cellX - margin) / texWidth;
        texRight = (1.0f * (glyph.cellX + glyph.glyphWidth) + margin) / texWidth;

        texTop = 1.f - (static_cast<float>(glyph.cellY) - margin) / texHeight;
        texBottom = 1.f - (static_cast<float>(glyph.cellY + glyph.height) + margin) / texHeight;

        px = x - margin * m_Scale.v[0];
        py = m_CursorPos.y - margin * m_Scale.v[1];
        width = (glyph.glyphWidth + margin * 2.0f) * m_Scale.v[0];
        height = (glyph.height + margin * 2.0f) * m_Scale.v[1];
#else
        texLeft = (1.0f * glyph.cellX) / texWidth;
        texRight = (1.0f * (glyph.cellX + glyph.glyphWidth)) / texWidth;

        texTop = 1.f - static_cast<float>(glyph.cellY) / texHeight;
        texBottom = 1.f - static_cast<float>(glyph.cellY + glyph.height) / texHeight;

        px = x;
        py = m_CursorPos.y;
        width = glyph.glyphWidth * m_Scale.v[0];
        height = glyph.height * m_Scale.v[1];
#endif
    }

    int charIdx = m_CharCount;
    if (charIdx >= m_CharCountMax)
    {
        NN_DETAIL_GFX_WARN("nn::gfx::util::DebugFontTextWriter : Vertex Buffer Over.\n");
        return;
    }
    m_CharCount++;

    CharAttribute* pCharAttrs = &m_CharAttrs[charIdx];

    // ポジションのセット
    // カーソルが進んだら下に行くようにyを反転する
    pCharAttrs->pos.v[0] = width;
    pCharAttrs->pos.v[1] = height;
    pCharAttrs->pos.v[2] = px;
    pCharAttrs->pos.v[3] = py;

    // カラーのセット
    pCharAttrs->color = m_TextColor;

    // テクスチャ座標のセット
    pCharAttrs->tex.v[0] = texLeft;
    pCharAttrs->tex.v[1] = texTop;
    pCharAttrs->tex.v[2] = texRight;
    pCharAttrs->tex.v[3] = texBottom;
}

bool DebugFontTextWriterImpl::GetGlyphUtf8(const DebugFontGlyph** pGlyph, uint32_t code)
{
    if (code <= 0x7e)
    {
        int index = code - 0x20;
        NN_SDK_ASSERT(index >= 0);
        *pGlyph = &g_DebugFontGlyphs[index];
        return true;
    }

    int start = (0x7e + 1) - 0x20;
    int end = static_cast<int>(g_DebugFontGlyphCount);
    int middle;

    // 二分探索で探す
    for (;;)
    {
        middle = (start + end) / 2;
        NN_SDK_ASSERT(middle >= ((0x7e + 1) - 0x20));
        NN_SDK_ASSERT(middle < g_DebugFontGlyphCount);
        uint32_t middle_code = g_DebugFontGlyphTableUtf8[middle].code;
        if (middle_code == code)
        {
            *pGlyph = &g_DebugFontGlyphs[ g_DebugFontGlyphTableUtf8[middle].index ];
            return true;
        }
        else if (middle_code < code)
        {
            if (start == middle)
            {
                break;
            }
            start = middle;
        }
        else
        {
            if (end == middle)
            {
                break;
            }
            end = middle;
        }
    }

    return false;
}

bool DebugFontTextWriterImpl::GetGlyphCp932(const DebugFontGlyph** pGlyph, uint16_t code)
{
    if (code <= 0x7e)
    {
        int index = code - 0x20;
        NN_SDK_ASSERT(index >= 0);
        *pGlyph = &g_DebugFontGlyphs[index];
        return true;
    }

    int start = (0x7e + 1) - 0x20;
    int end = static_cast<int>(g_DebugFontGlyphCount);
    int middle;

    // 二分探索で探す
    for (;;)
    {
        middle = (start + end) / 2;
        NN_SDK_ASSERT(middle >= ((0x7e + 1) - 0x20));
        NN_SDK_ASSERT(middle < g_DebugFontGlyphCount);
        uint16_t middle_code = g_DebugFontGlyphTableCp932[middle].code;
        if (middle_code == code)
        {
            *pGlyph = &g_DebugFontGlyphs[ g_DebugFontGlyphTableCp932[middle].index ];
            return true;
        }
        else if (middle_code < code)
        {
            if (start == middle)
            {
                break;
            }
            start = middle;
        }
        else
        {
            if (end == middle)
            {
                break;
            }
            end = middle;
        }
    }

    return false;
}

bool DebugFontTextWriterImpl::GetGlyphUtf16(const DebugFontGlyph** pGlyph, uint16_t code)
{
    if (code <= 0x7e)
    {
        int index = code - 0x20;
        NN_SDK_ASSERT(index >= 0);
        *pGlyph = &g_DebugFontGlyphs[index];
        return true;
    }

    int start = (0x7e + 1) - 0x20;
    int end = static_cast<int>(g_DebugFontGlyphCount);
    int middle;

    // 二分探索で探す
    for (;;)
    {
        middle = (start + end) / 2;
        NN_SDK_ASSERT(middle >= ((0x7e + 1) - 0x20));
        NN_SDK_ASSERT(middle < g_DebugFontGlyphCount);
        uint16_t middle_code = g_DebugFontGlyphTableUtf16[middle].code;
        if (middle_code == code)
        {
            *pGlyph = &g_DebugFontGlyphs[middle];
            return true;
        }
        else if (middle_code < code)
        {
            if (start == middle)
            {
                break;
            }
            start = middle;
        }
        else
        {
            if (end == middle)
            {
                break;
            }
            end = middle;
        }
    }

    return false;
}

void DebugFontTextWriterImpl::CalculateStringRectUtf8(Rectangle* pRect, bool defaultLocaleCharset, const char* str)
{
    NN_SDK_ASSERT_NOT_NULL(str);

#if defined(NN_BUILD_CONFIG_OS_WIN32)
    if(defaultLocaleCharset && (GetConsoleOutputCP() != CP_UTF8))
    {
        CalculateStringRectCp932(pRect, str);
    }
    else
    {
        CalculateStringRectUtf8(pRect, str);
    }
#else
    NN_UNUSED(defaultLocaleCharset);
    CalculateStringRectUtf8(pRect, str);
#endif
}

void DebugFontTextWriterImpl::CalculateStringRectUtf8(Rectangle* pRect, const char* str)
{
    NN_SDK_ASSERT_NOT_NULL(str);

    if (!m_IsInitialized) {
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return;
    }

    nn::os::LockMutex(&m_Mutex);
    {
        float lineHeight = GetLineHeight();
        pRect->left = 0;
        pRect->right = 0;
        pRect->top = std::min(0.0f, lineHeight);
        pRect->bottom = std::max(0.0f, lineHeight);

        float currentPosX = 0;
        float currentPosY = 0;
        float xOrigin = currentPosX;
        float yOrigin = currentPosY;

        currentPosY = yOrigin + (g_DebugFontAscent * m_Scale.v[1]);

        const char* cur = str;
        while (*cur)
        {
            uint32_t code;
            int bytes;
            if( GetCharCodeUtf8(code, bytes, cur) )
            {
                cur += bytes;
            }
            else
            {
                break;
            }

            if (code < ' ')
            {
                //---- タグ処理
                if (code == '\n')
                {
                    currentPosX = xOrigin;
                    currentPosY += lineHeight;

                    pRect->top = std::min(pRect->top, currentPosY);
                    pRect->bottom = std::max(pRect->bottom, currentPosY);
                }
                else if (code == '\t')
                {
                    if (m_TabWidth > 0)
                    {
                        const float aCharWidth = m_IsWidthFixed ? m_FixedWidth : (g_DebugFontCharWidth * m_Scale.v[0]);
                        const float dx = currentPosX - xOrigin;
                        const float tabPixel = m_TabWidth * aCharWidth;
                        const int numTab = static_cast<int>(dx / tabPixel) + 1;
                        currentPosX = xOrigin + tabPixel * numTab;
                    }

                    pRect->left = std::min(pRect->left, currentPosX);
                    pRect->right = std::max(pRect->right, currentPosX);
                }
            }
            else
            {
                //---- 通常の文字
                const float baseY = currentPosY;

                const DebugFontGlyph *glyph;
                bool found = GetGlyphUtf8(&glyph, code);
                if (!found) {
                    GetGlyphUtf8(&glyph, '?');
                }

                //---- カーソルはベースラインにあるのでセル上端へ移動する
                const float adj = -(g_DebugFontBaselinePos) * m_Scale.v[1];
                currentPosY += adj + lineHeight;

                if (m_IsWidthFixed)
                {
                    currentPosX += m_FixedWidth;
                }
                else
                {
                    currentPosX += (glyph->charWidth * m_Scale.v[0]);
                }

                pRect->left = std::min(pRect->left, currentPosX);
                pRect->right = std::max(pRect->right, currentPosX);
                pRect->top = std::min(pRect->top, currentPosY);
                pRect->bottom = std::max(pRect->bottom, currentPosY);

                // 戻す
                currentPosY = baseY;
            }
        }
    }
    nn::os::UnlockMutex(&m_Mutex);
}

void DebugFontTextWriterImpl::CalculateStringRectCp932(Rectangle* pRect, const char* str)
{
    NN_SDK_ASSERT_NOT_NULL(str);

    if (!m_IsInitialized) {
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return;
    }

    nn::os::LockMutex(&m_Mutex);
    {
        float lineHeight = GetLineHeight();
        pRect->left = 0;
        pRect->right = 0;
        pRect->top = std::min(0.0f, lineHeight);
        pRect->bottom = std::max(0.0f, lineHeight);

        float currentPosX = 0;
        float currentPosY = 0;
        float xOrigin = currentPosX;
        float yOrigin = currentPosY;

        currentPosY = yOrigin + (g_DebugFontAscent * m_Scale.v[1]);

        const char* cur = str;
        while (*cur)
        {
            uint16_t code;
            int bytes;
            if( GetCharCodeCp932(code, bytes, cur) )
            {
                cur += bytes;
            }
            else
            {
                break;
            }

            if (code < ' ')
            {
                //---- タグ処理
                if (code == '\n')
                {
                    currentPosX = xOrigin;
                    currentPosY += lineHeight;

                    pRect->top = std::min(pRect->top, currentPosY);
                    pRect->bottom = std::max(pRect->bottom, currentPosY);
                }
                else if (code == '\t')
                {
                    if (m_TabWidth > 0)
                    {
                        const float aCharWidth = m_IsWidthFixed ? m_FixedWidth : (g_DebugFontCharWidth * m_Scale.v[0]);
                        const float dx = currentPosX - xOrigin;
                        const float tabPixel = m_TabWidth * aCharWidth;
                        const int numTab = static_cast<int>(dx / tabPixel) + 1;
                        currentPosX = xOrigin + tabPixel * numTab;
                    }

                    pRect->left = std::min(pRect->left, currentPosX);
                    pRect->right = std::max(pRect->right, currentPosX);
                }
            }
            else
            {
                //---- 通常の文字
                const float baseY = currentPosY;

                const DebugFontGlyph *glyph;
                bool found = GetGlyphCp932(&glyph, code);
                if (!found) {
                    GetGlyphCp932(&glyph, '?');
                }

                //---- カーソルはベースラインにあるのでセル上端へ移動する
                const float adj = -(g_DebugFontBaselinePos) * m_Scale.v[1];
                currentPosY += adj + lineHeight;

                if (m_IsWidthFixed)
                {
                    currentPosX += m_FixedWidth;
                }
                else
                {
                    currentPosX += (glyph->charWidth * m_Scale.v[0]);
                }

                pRect->left = std::min(pRect->left, currentPosX);
                pRect->right = std::max(pRect->right, currentPosX);
                pRect->top = std::min(pRect->top, currentPosY);
                pRect->bottom = std::max(pRect->bottom, currentPosY);

                // 戻す
                currentPosY = baseY;
            }
        }
    }
    nn::os::UnlockMutex(&m_Mutex);
}

void DebugFontTextWriterImpl::CalculateStringRectUtf16(Rectangle* pRect, const uint16_t* str)
{
    NN_SDK_ASSERT_NOT_NULL(str);

    if (!m_IsInitialized) {
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return;
    }

    nn::os::LockMutex(&m_Mutex);
    {
        float lineHeight = GetLineHeight();
        pRect->left = 0;
        pRect->right = 0;
        pRect->top = std::min(0.0f, lineHeight);
        pRect->bottom = std::max(0.0f, lineHeight);

        int length = 0;
        for (const uint16_t* cur = str; *cur != 0; ++cur, ++length) {}
        NN_SDK_ASSERT(length >= 0);

        float currentPosX = 0;
        float currentPosY = 0;
        float xOrigin = currentPosX;
        float yOrigin = currentPosY;

        currentPosY = yOrigin + (g_DebugFontAscent * m_Scale.v[1]);

        for (int i = 0; i<length; i++)
        {
            uint16_t code = str[i];
            if (code < ' ')
            {
                //---- タグ処理
                if (code == '\n')
                {
                    currentPosX = xOrigin;
                    currentPosY += lineHeight;

                    pRect->top = std::min(pRect->top, currentPosY);
                    pRect->bottom = std::max(pRect->bottom, currentPosY);
                }
                else if (code == '\t')
                {
                    if (m_TabWidth > 0)
                    {
                        const float aCharWidth = m_IsWidthFixed ? m_FixedWidth : (g_DebugFontCharWidth * m_Scale.v[0]);
                        const float dx = currentPosX - xOrigin;
                        const float tabPixel = m_TabWidth * aCharWidth;
                        const int numTab = static_cast<int>(dx / tabPixel) + 1;
                        currentPosX = xOrigin + tabPixel * numTab;
                    }

                    pRect->left = std::min(pRect->left, currentPosX);
                    pRect->right = std::max(pRect->right, currentPosX);
                }
            }
            else
            {
                //---- 通常の文字
                const float baseY = currentPosY;

                const DebugFontGlyph *glyph;
                bool found = GetGlyphUtf16(&glyph, code);
                if (!found) {
                    GetGlyphUtf16(&glyph, L'?');
                }

                //---- カーソルはベースラインにあるのでセル上端へ移動する
                const float adj = -(g_DebugFontBaselinePos) * m_Scale.v[1];
                currentPosY += adj + lineHeight;

                if (m_IsWidthFixed)
                {
                    currentPosX += m_FixedWidth;
                }
                else
                {
                    currentPosX += (glyph->charWidth * m_Scale.v[0]);
                }

                pRect->left = std::min(pRect->left, currentPosX);
                pRect->right = std::max(pRect->right, currentPosX);
                pRect->top = std::min(pRect->top, currentPosY);
                pRect->bottom = std::max(pRect->bottom, currentPosY);

                // 戻す
                currentPosY = baseY;
            }
        }
    }
    nn::os::UnlockMutex(&m_Mutex);
}

void DebugFontTextWriterImpl::BuildVertexBuffer()
{
    // 文字数が足りなければ、頂点バッファの作成は行いません。
    const int  charCount = m_CharCount;
    if (charCount == 0)
    {
        return;
    }
    else if (m_CharCountMax < charCount)
    {
        return;
    }

    m_VertexBuffer.Map(m_VertexBufferIndex);
    {
        m_VertexBufferOffset = m_VertexBuffer.Allocate(nn::util::align_up(m_VertexBufferSize, m_VertexBuffer.GetBufferAlignment()));

        nn::util::BytePtr   pVertexBuffer(m_VertexBuffer.GetMappedPointer());
        void* pDestPointer = pVertexBuffer.Advance(m_VertexBufferOffset).Get();

        Vertex* vtxArray = static_cast<Vertex*>(pDestPointer);
        for (int i = 0; i < charCount; ++i)
        {
            CharAttribute* atr = &m_CharAttrs[i];
            Vertex* vtx = &vtxArray[i * VertexCountByLetter];

            const float width = atr->pos.v[0];
            const float height = atr->pos.v[1];
            const float x = atr->pos.v[2];
            const float y = atr->pos.v[3];
            const nn::util::Color4u8Type& color = atr->color;

            vtx[0].Set(x, y, 0, color, atr->tex.v[0], atr->tex.v[1]);
            vtx[1].Set(x + width, y, 0, color, atr->tex.v[2], atr->tex.v[1]);
            vtx[2].Set(x, y + height, 0, color, atr->tex.v[0], atr->tex.v[3]);
            vtx[3].Set(x + width, y + height, 0, color, atr->tex.v[2], atr->tex.v[3]);
        }
    }
    m_VertexBuffer.Unmap();
    m_VertexBuffer.SetGpuAccessBufferIndex(m_VertexBufferIndex);

    ++m_VertexBufferIndex;
    m_VertexBufferIndex %= m_BufferCount;
}

void DebugFontTextWriterImpl::BuildConstantBuffer()
{
    m_ConstantBuffer.Map(m_ConstantBufferIndex);
    {
        size_t  constantBufferSize = nn::util::align_up(sizeof(ShaderParam), m_ConstantBuffer.GetBufferAlignment());
        size_t  requiredConstantBufferSize = constantBufferSize;

        m_ConstantBufferOffset = m_ConstantBuffer.Allocate(requiredConstantBufferSize);

        nn::util::BytePtr   pConstantBuffer(m_ConstantBuffer.GetMappedPointer());
        ShaderParam* shaderParam = static_cast<ShaderParam*>(pConstantBuffer.Advance(m_ConstantBufferOffset).Get());

        {
            // フチ二回描画なしの通常のコンスタントバッファ作成。

            // プロジェクション行列の計算
            nn::util::MatrixT4x4fType projection;
            nn::util::MatrixOrthographicOffCenterRightHanded(
                &projection,
                0 /*left*/,
                static_cast<float>(m_DisplayWidth) /*right*/,
                static_cast<float>(m_DisplayHeight) /*bottom*/,
                0 /*top*/,
                0.0f /*near*/,
                300.0f /*far*/);
            nn::util::MatrixStore(reinterpret_cast<nn::util::FloatT4x4*>(&shaderParam->m_Mtx), projection);
            SwapEndian(shaderParam->m_Mtx[0][0]);
            SwapEndian(shaderParam->m_Mtx[0][1]);
            SwapEndian(shaderParam->m_Mtx[0][2]);
            SwapEndian(shaderParam->m_Mtx[0][3]);
            SwapEndian(shaderParam->m_Mtx[1][0]);
            SwapEndian(shaderParam->m_Mtx[1][1]);
            SwapEndian(shaderParam->m_Mtx[1][2]);
            SwapEndian(shaderParam->m_Mtx[1][3]);
            SwapEndian(shaderParam->m_Mtx[2][0]);
            SwapEndian(shaderParam->m_Mtx[2][1]);
            SwapEndian(shaderParam->m_Mtx[2][2]);
            SwapEndian(shaderParam->m_Mtx[2][3]);
            SwapEndian(shaderParam->m_Mtx[3][0]);
            SwapEndian(shaderParam->m_Mtx[3][1]);
            SwapEndian(shaderParam->m_Mtx[3][2]);
            SwapEndian(shaderParam->m_Mtx[3][3]);

            shaderParam->m_Alpha = 1.0f;
            SwapEndian(shaderParam->m_Alpha);
        }
    }
    m_ConstantBuffer.Unmap();
    m_ConstantBuffer.SetGpuAccessBufferIndex(m_ConstantBufferIndex);

    ++m_ConstantBufferIndex;
    m_ConstantBufferIndex %= m_BufferCount;
}

void* DebugFontTextWriterImpl::AllocateFromHeap(size_t size, size_t alignment)
{
    return m_Allocator.Allocate(size, alignment);
}

void DebugFontTextWriterImpl::DeallocateHeap(void* ptr)
{
    if (ptr == nullptr) return;
    m_Allocator.Free(ptr);
}

void* DebugFontTextWriterImpl::GpuBufferAllocateFunction(size_t size, size_t alignment, void* pUserData)
{
    DebugFontTextWriterImpl *pWriter = static_cast<DebugFontTextWriterImpl *>(pUserData);
    return pWriter->AllocateFromHeap(size, alignment);
}

void DebugFontTextWriterImpl::GpuBufferDeallocateFunction(void* ptr, void* pUserData)
{
    DebugFontTextWriterImpl *pWriter = static_cast<DebugFontTextWriterImpl *>(pUserData);
    pWriter->DeallocateHeap(ptr);
}

} // namespace detail

DebugFontTextWriter::DebugFontTextWriter() :
m_pImpl(nullptr),
m_pMemory(nullptr)
{}

DebugFontTextWriter::~DebugFontTextWriter()
{}

size_t DebugFontTextWriter::GetRequiredMemorySize(nn::gfx::Device* pDevice, const DebugFontTextWriterInfo& info)
{
    size_t implSize = sizeof(detail::DebugFontTextWriterImpl) + detail::DebugFontTextWriterImplAlignment;
    return implSize + detail::DebugFontTextWriterImpl::GetRequiredMemorySize(pDevice, info);
}

size_t DebugFontTextWriter::GetRequiredMemoryPoolSize(nn::gfx::Device* pDevice, const DebugFontTextWriterInfo& info)
{
    return detail::DebugFontTextWriterImpl::GetRequiredMemoryPoolSize(pDevice, info);
}

void DebugFontTextWriter::Initialize(
    nn::gfx::Device* pDevice,
    const DebugFontTextWriterInfo& info,
    void* pMemory,
    size_t memorySize,
    nn::gfx::MemoryPool* pMemoryPool,
    ptrdiff_t memoryPoolOffset,
    size_t memoryPoolSize
    )
{
    NN_SDK_ASSERT(m_pImpl == nullptr);

    m_pMemory = pMemory;
    nn::util::BytePtr heap(pMemory);
    heap.AlignUp(detail::DebugFontTextWriterImplAlignment);

    m_pImpl = static_cast<detail::DebugFontTextWriterImpl *>(heap.Get());
    new(m_pImpl)detail::DebugFontTextWriterImpl;

    size_t implSize = sizeof(detail::DebugFontTextWriterImpl);
    heap += implSize;

    m_pImpl->Initialize(
        pDevice,
        info,
        heap.Get() ,
        memorySize - (implSize + detail::DebugFontTextWriterImplAlignment),
        pMemoryPool,
        memoryPoolOffset,
        memoryPoolSize
        );
}

void DebugFontTextWriter::Finalize()
{
    if (m_pImpl == nullptr){
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return;
    }

    m_pImpl->Finalize();
    m_pImpl = nullptr;
    m_pMemory = nullptr;
}

bool DebugFontTextWriter::IsInitialized()
{
    if (m_pImpl == nullptr) {
        return false;
    }
    return m_pImpl->IsInitialized();
}

int DebugFontTextWriter::GetCharCountMax()
{
    if (m_pImpl == nullptr) {
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return 0;
    }
    return m_pImpl->GetCharCountMax();
}

int DebugFontTextWriter::GetBufferCount()
{
    if (m_pImpl == nullptr) {
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return 0;
    }
    return m_pImpl->GetBufferCount();
}

bool DebugFontTextWriter::IsUserMemoryPoolEnabled()
{
    if (m_pImpl == nullptr) {
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return false;
    }
    return m_pImpl->IsUserMemoryPoolEnabled();
}

void* DebugFontTextWriter::GetMemory()
{
    if (m_pImpl == nullptr) {
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return nullptr;
    }
    return m_pMemory;
}

nn::gfx::MemoryPool* DebugFontTextWriter::GetMemoryPool()
{
    if (m_pImpl == nullptr) {
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return nullptr;
    }
    return m_pImpl->GetMemoryPool();
}

ptrdiff_t DebugFontTextWriter::GetMemoryPoolOffset()
{
    if (m_pImpl == nullptr) {
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return 0;
    }
    return m_pImpl->GetMemoryPoolOffset();
}

int DebugFontTextWriter::GetSamplerDescriptorIndexSlot()
{
    if (m_pImpl == nullptr) {
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return 0;
    }
    return m_pImpl->GetSamplerDescriptorIndexSlot();
}

int DebugFontTextWriter::GetTextureDescriptorIndexSlot()
{
    if (m_pImpl == nullptr) {
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return 0;
    }
    return m_pImpl->GetTextureDescriptorIndexSlot();
}

void DebugFontTextWriter::SetDisplayWidth(int displayWidth)
{
    if (m_pImpl == nullptr){
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return;
    }
    m_pImpl->SetDisplayWidth(displayWidth);
}

void DebugFontTextWriter::SetDisplayHeight(int displayHeight)
{
    if (m_pImpl == nullptr){
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return;
    }
    m_pImpl->SetDisplayHeight(displayHeight);
}

void DebugFontTextWriter::SetTextureDescriptor(nn::gfx::DescriptorPool* pTextureDescriptorPool,
    int textureDescriptorIndexSlot
    )
{
    if (m_pImpl == nullptr){
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return;
    }
    m_pImpl->SetTextureDescriptor(pTextureDescriptorPool, textureDescriptorIndexSlot);
}

void DebugFontTextWriter::SetSamplerDescriptor(nn::gfx::DescriptorPool* pSamplerDescriptorPool,
    int samplerDescriptorIndexSlot
    )
{
    if (m_pImpl == nullptr){
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return;
    }
    m_pImpl->SetSamplerDescriptor(pSamplerDescriptorPool, samplerDescriptorIndexSlot);
}

void DebugFontTextWriter::SetScale(float hScale, float vScale)
{
    if (m_pImpl == nullptr){
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return;
    }
    m_pImpl->SetScale(hScale, vScale);
}

float DebugFontTextWriter::GetScaleX() const
{
    if (m_pImpl == nullptr) {
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return 1.0f;
    }
    return m_pImpl->GetScaleX();
}

float DebugFontTextWriter::GetScaleY() const
{
    if (m_pImpl == nullptr) {
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return 1.0f;
    }
    return m_pImpl->GetScaleY();
}

void DebugFontTextWriter::SetFontSize(float width, float height)
{
    if (m_pImpl == nullptr) {
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return;
    }
    m_pImpl->SetFontSize(width, height);
}

void DebugFontTextWriter::SetFontSize(float height)
{
    if (m_pImpl == nullptr) {
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return;
    }
    m_pImpl->SetFontSize(height);
}

float DebugFontTextWriter::GetFontWidth() const
{
    if (m_pImpl == nullptr) {
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return 0.0f;
    }
    return m_pImpl->GetFontWidth();
}

float DebugFontTextWriter::GetFontHeight() const
{
    if (m_pImpl == nullptr) {
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return 0.0f;
    }
    return m_pImpl->GetFontHeight();
}

void DebugFontTextWriter::SetCursor(float x, float y)
{
    if (m_pImpl == nullptr){
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return;
    }
    m_pImpl->SetCursor(x, y);
}

void DebugFontTextWriter::SetCursorX(float x)
{
    if (m_pImpl == nullptr) {
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return;
    }
    m_pImpl->SetCursorX(x);
}

void DebugFontTextWriter::SetCursorY(float y)
{
    if (m_pImpl == nullptr) {
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return;
    }
    m_pImpl->SetCursorY(y);
}

float DebugFontTextWriter::GetCursorX() const
{
    if (m_pImpl == nullptr) {
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return 0.0f;
    }
    return m_pImpl->GetCursorX();
}

float DebugFontTextWriter::GetCursorY() const
{
    if (m_pImpl == nullptr) {
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return 0.0f;
    }
    return m_pImpl->GetCursorY();
}

void DebugFontTextWriter::SetTextColor(const nn::util::Color4u8Type& color)
{
    if (m_pImpl == nullptr){
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return;
    }
    m_pImpl->SetTextColor(color);
}

const nn::util::Color4u8Type& DebugFontTextWriter::GetTextColor() const
{
    if (m_pImpl == nullptr) {
        static nn::util::Color4u8Type color = { { 255, 255, 255, 255 } };
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return color;
    }
    return m_pImpl->GetTextColor();
}

void DebugFontTextWriter::SetLineHeight(float height)
{
    if (m_pImpl == nullptr) {
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return;
    }
    m_pImpl->SetLineHeight(height);
}

float DebugFontTextWriter::GetLineHeight() const
{
    if (m_pImpl == nullptr) {
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return 0;
    }
    return m_pImpl->GetLineHeight();
}

void DebugFontTextWriter::SetTabWidth(int tabWidth)
{
    if (m_pImpl == nullptr) {
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return;
    }
    m_pImpl->SetTabWidth(tabWidth);
}

int DebugFontTextWriter::GetTabWidth() const
{
    if (m_pImpl == nullptr) {
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return 0;
    }
    return m_pImpl->GetTabWidth();
}

void DebugFontTextWriter::Draw(nn::gfx::CommandBuffer* pCommandBuffer)
{
    if (m_pImpl == nullptr){
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return;
    }
    m_pImpl->Draw(pCommandBuffer);
}

void DebugFontTextWriter::SetFixedWidthEnabled(bool isFixed)
{
    if (m_pImpl == nullptr) {
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return;
    }
    m_pImpl->SetFixedWidthEnabled(isFixed);
}

bool DebugFontTextWriter::IsFixedWidthEnabled() const
{
    if (m_pImpl == nullptr) {
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return false;
    }
    return m_pImpl->IsFixedWidthEnabled();
}

void DebugFontTextWriter::SetFixedWidth(float width)
{
    if (m_pImpl == nullptr) {
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return;
    }
    m_pImpl->SetFixedWidth(width);
}

float DebugFontTextWriter::GetFixedWidth() const
{
    if (m_pImpl == nullptr) {
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return 0.0f;
    }
    return m_pImpl->GetFixedWidth();
}

void DebugFontTextWriter::VPrintUtf8(bool defaultLocaleCharset, const char* format, std::va_list formatArg)
{
    if (m_pImpl == nullptr){
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        return;
    }
    char outputBuffer[detail::DebugFontFormatBufferLength];
    detail::DebugFontFormatStringOutputBuffer out(outputBuffer, static_cast<int>(detail::DebugFontFormatBufferLength));
    nn::util::VFormatString(out.Output, reinterpret_cast<uintptr_t>(&out), format, formatArg);

    m_pImpl->PrintUtf8(defaultLocaleCharset, outputBuffer);
}

void DebugFontTextWriter::VCalculateFormatStringWidthHeightUtf8(float& width, float& height, bool defaultLocaleCharset, const char* format, std::va_list formatArg)
{
    if (m_pImpl == nullptr) {
        NN_DETAIL_GFX_ERROR("DebugFontTextWriter is not initialized.\n");
        width = 0;
        height = 0;
        return;
    }
    char outputBuffer[detail::DebugFontFormatBufferLength];
    detail::DebugFontFormatStringOutputBuffer out(outputBuffer, static_cast<int>(detail::DebugFontFormatBufferLength));
    nn::util::VFormatString(out.Output, reinterpret_cast<uintptr_t>(&out), format, formatArg);

    detail::Rectangle rect;
    m_pImpl->CalculateStringRectUtf8(&rect, defaultLocaleCharset, outputBuffer);
    width = rect.GetWidth();
    height = rect.GetHeight();
}


} // namespace util
} // namespace gfx
} // namespace nn
