﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>

#include <nn/os/os_Mutex.h>

#if !defined( NN_BUILD_CONFIG_OS_WIN32 )
    #error
#endif

#include "gfx_CommonHelper.h"
#include "gfx_WGLHelper.h"

#define WGL_CONTEXT_MAJOR_VERSION_ARB 0x2091 // NOLINT
#define WGL_CONTEXT_MINOR_VERSION_ARB 0x2092 // NOLINT
#define WGL_CONTEXT_FLAGS_ARB 0x2094 // NOLINT
#define WGL_CONTEXT_PROFILE_MASK_ARB 0x9126 // NOLINT
#define WGL_CONTEXT_CORE_PROFILE_BIT_ARB 0x00000001 // NOLINT
#define WGL_CONTEXT_DEBUG_BIT_ARB 0x0001 // NOLINT

namespace nn {
namespace gfx {
namespace detail {

typedef HGLRC (WINAPI* PFNWGLCREATECONTEXTATTRIBSARBPROC)
    ( HDC hDc, HGLRC hShareContext, const int* attribList );
typedef BOOL (WINAPI* PFNWGLSWAPINTERVALEXTPROC)( int interval );
typedef int (WINAPI* PFNWGLGETSWAPINTERVALEXTPROC)();

void* Wgl::GetProcAddress( const char* name ) NN_NOEXCEPT
{
    return ::wglGetProcAddress( name );
}

HGLRC Wgl::CreateGlRc( HDC hDc, HGLRC hSharedRc, int majorVersion, int minorVersion, bool debugMode ) NN_NOEXCEPT
{
    if( !SetPixelFormat( hDc ) )
    {
        return NULL;
    }

    HGLRC hRc = ::wglCreateContext( hDc );
    if( hRc == NULL )
    {
        return NULL;
    }

    int flag = debugMode ? WGL_CONTEXT_DEBUG_BIT_ARB : 0;
    int attribs[] =
    {
        WGL_CONTEXT_MAJOR_VERSION_ARB, majorVersion,
        WGL_CONTEXT_MINOR_VERSION_ARB, minorVersion,
        WGL_CONTEXT_FLAGS_ARB, flag,
        WGL_CONTEXT_PROFILE_MASK_ARB, WGL_CONTEXT_CORE_PROFILE_BIT_ARB,
        0
    };

    MakeCurrent( hDc, hRc );

    const PFNWGLCREATECONTEXTATTRIBSARBPROC wglCreateContextAttribsARB =
        (PFNWGLCREATECONTEXTATTRIBSARBPROC)wglGetProcAddress( "wglCreateContextAttribsARB" );
    if( wglCreateContextAttribsARB )
    {
        HGLRC hNewRc = wglCreateContextAttribsARB( hDc, hSharedRc, attribs );
        if( hNewRc )
        {
            ::wglMakeCurrent( NULL, NULL );
            ::wglDeleteContext( hRc );
            return hNewRc;
        }
    }

    ::wglMakeCurrent( NULL, NULL );
    if( hSharedRc )
    {
        ::wglShareLists( hSharedRc, hRc );
    }

    return hRc;
}

void Wgl::DeleteGlRc( HGLRC hRc ) NN_NOEXCEPT
{
    if( hRc )
    {
        ::wglDeleteContext( hRc );
    }
}

int Wgl::SetPixelFormat( HDC hDc ) NN_NOEXCEPT
{
    PIXELFORMATDESCRIPTOR pfd =
    {
        sizeof( PIXELFORMATDESCRIPTOR ),
        1,
        PFD_DRAW_TO_WINDOW | PFD_SUPPORT_OPENGL | PFD_DOUBLEBUFFER, // フラグ
        PFD_TYPE_RGBA, // フレームバッファータイプ
        24, // フレームバッファーのカラービット幅
        0, 0, 0, 0, 0, 0,
        0,
        0,
        0,
        0, 0, 0, 0,
        0, // 深度バッファーのビット幅
        0, // ステンシルバッファーのビット幅
        0,
        PFD_MAIN_PLANE,
        0,
        0, 0, 0
    };

    return Wgl::SetPixelFormat( hDc, pfd );
}

int Wgl::SetPixelFormat( HDC hDc, const PIXELFORMATDESCRIPTOR& pfd ) NN_NOEXCEPT
{
    // 同じデバイスコンテキストに対して一度しか設定できない
    int pixelFormat = ::GetPixelFormat( hDc );
    if( pixelFormat == 0 )
    {
        pixelFormat = ::ChoosePixelFormat( hDc, &pfd );
        if( pixelFormat )
        {
            if( ::SetPixelFormat( hDc, pixelFormat, &pfd ) )
            {
                return pixelFormat;
            }
        }

        return 0;
    }

    return pixelFormat;
}

bool Wgl::SetSwapInterval( int interval ) NN_NOEXCEPT
{
    // MEMO: interval = -1 で適応 Vsync
    static const PFNWGLSWAPINTERVALEXTPROC wglSwapIntervalEXT =
        (PFNWGLSWAPINTERVALEXTPROC)wglGetProcAddress( "wglSwapIntervalEXT" );
    return wglSwapIntervalEXT( interval ) == TRUE;
}

int Wgl::GetSwapInterval() NN_NOEXCEPT
{
    static const PFNWGLGETSWAPINTERVALEXTPROC wglGetSwapIntervalEXT =
        (PFNWGLGETSWAPINTERVALEXTPROC)wglGetProcAddress( "wglGetSwapIntervalEXT" );
    return wglGetSwapIntervalEXT();
}

void Wgl::GetWidthAndHeight( int* pOutWidth, int* pOutHeight, HDC hDc ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pOutWidth );
    NN_SDK_ASSERT_NOT_NULL( pOutHeight );
    NN_SDK_ASSERT_NOT_NULL( hDc );

    HBITMAP hBitmap = static_cast< HBITMAP >( ::GetCurrentObject( hDc, OBJ_BITMAP ) );
    BITMAP bitmap;
    ::GetObject( hBitmap, sizeof( BITMAP ), &bitmap );
    *pOutWidth = static_cast< int >( bitmap.bmWidth );
    *pOutHeight = static_cast< int >( bitmap.bmHeight );
}

void Wgl::MakeCurrent( HDC hDc, HGLRC hGlRc ) NN_NOEXCEPT
{
    BOOL result = ::wglMakeCurrent( static_cast< HDC >( hDc ), static_cast< HGLRC >( hGlRc ) );
    NN_SDK_ASSERT( result == TRUE, "GetLastError: 0x%08X\n", GetLastError() );
    NN_UNUSED( result );
}

}
}
}
