﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>

#include <nn/util/util_BitPack.h>
#include <nn/util/util_BytePtr.h>
#include <nn/util/util_BitArray.h>

#include <nn/gfx/gfx_Enum.h>
#include <nn/gfx/gfx_SwapChainInfo.h>
#include <nn/gfx/gfx_TextureInfo.h>
#include <nn/gfx/gfx_GpuAddress.h>
#include <nn/gfx/gfx_CommandBufferInfo.h>

#include <nn/gfx/detail/gfx_Common-api.vk.h>
#include <nn/gfx/detail/gfx_Device-api.vk.1.h>
#include <nn/gfx/detail/gfx_Declare.h>
#include <nn/gfx/detail/gfx_Misc.h>
#if defined(NN_BUILD_CONFIG_OS_WIN32)
#define VK_USE_PLATFORM_WIN32_KHR
#endif
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
#define VK_USE_PLATFORM_VI_NN
#endif
#include <vulkan/vulkan.h>

#define NN_GFX_CALL_VK_FUNCTION(func) (func)

#include <nn/lmem/lmem_FrameHeap.h>
#include <new>

namespace nn {
namespace gfx {
namespace detail {

enum VkDeviceFeature
{
    VkDeviceFeature_RobustBufferAccess,
    VkDeviceFeature_FullDrawIndexUint32,
    VkDeviceFeature_ImageCubeArray,
    VkDeviceFeature_IndependentBlend,
    VkDeviceFeature_GeometryShader,
    VkDeviceFeature_TessellationShader,
    VkDeviceFeature_SampleRateShading,
    VkDeviceFeature_DualSrcBlend,
    VkDeviceFeature_LogicOp,
    VkDeviceFeature_MultiDrawIndirect,
    VkDeviceFeature_DrawIndirectFirstInstance,
    VkDeviceFeature_DepthClamp,
    VkDeviceFeature_DepthBiasClamp,
    VkDeviceFeature_FillModeNonSolid,
    VkDeviceFeature_DepthBounds,
    VkDeviceFeature_WideLines,
    VkDeviceFeature_LargePoints,
    VkDeviceFeature_AlphaToOne,
    VkDeviceFeature_MultiViewport,
    VkDeviceFeature_SamplerAnisotropy,
    VkDeviceFeature_TextureCompressionETC2,
    VkDeviceFeature_TextureCompressionASTC_LDR,
    VkDeviceFeature_TextureCompressionBC,
    VkDeviceFeature_OcclusionQueryPrecise,
    VkDeviceFeature_PipelineStatisticsQuery,
    VkDeviceFeature_VertexPipelineStoresAndAtomics,
    VkDeviceFeature_FragmentStoresAndAtomics,
    VkDeviceFeature_ShaderTessellationAndGeometryPointSize,
    VkDeviceFeature_ShaderImageGatherExtended,
    VkDeviceFeature_ShaderStorageImageExtendedFormats,
    VkDeviceFeature_ShaderStorageImageMultisample,
    VkDeviceFeature_ShaderStorageImageReadWithoutFormat,
    VkDeviceFeature_ShaderStorageImageWriteWithoutFormat,
    VkDeviceFeature_ShaderUniformBufferArrayDynamicIndexing,
    VkDeviceFeature_ShaderSampledImageArrayDynamicIndexing,
    VkDeviceFeature_ShaderStorageBufferArrayDynamicIndexing,
    VkDeviceFeature_ShaderStorageImageArrayDynamicIndexing,
    VkDeviceFeature_ShaderClipDistance,
    VkDeviceFeature_ShaderCullDistance,
    VkDeviceFeature_ShaderFloat64,
    VkDeviceFeature_ShaderInt64,
    VkDeviceFeature_ShaderInt16,
    VkDeviceFeature_ShaderResourceResidency,
    VkDeviceFeature_ShaderResourceMinLod,
    VkDeviceFeature_SparseBinding,
    VkDeviceFeature_SparseResidencyBuffer,
    VkDeviceFeature_SparseResidencyImage2D,
    VkDeviceFeature_SparseResidencyImage3D,
    VkDeviceFeature_SparseResidency2Samples,
    VkDeviceFeature_SparseResidency4Samples,
    VkDeviceFeature_SparseResidency8Samples,
    VkDeviceFeature_SparseResidency16Samples,
    VkDeviceFeature_SparseResidencyAliased,
    VkDeviceFeature_VariableMultisampleRate,
    VkDeviceFeature_InheritedQueries,

    VkDeviceFeature_End
};

enum VkDeviceExtension
{
    // Define flag only for the extension which is referred to in runtime.
    VkDeviceExtension_KhrMaintenance1,
    VkDeviceExtension_KhrMaintenance2,
    VkDeviceExtension_NvGlslShader,
    VkDeviceExtension_NvDedicatedAllocation,

    VkDeviceExtension_End
};

class ReferenceCount
{
public:
    ReferenceCount() NN_NOEXCEPT : m_Count( 0 ){}

    bool IsZero() NN_NOEXCEPT
    {
        return GetCount() == 0;
    }

    int GetCount() NN_NOEXCEPT
    {
        return m_Count;
    }

    int Increment() NN_NOEXCEPT
    {
        return ++m_Count;
    }

    int Decrement() NN_NOEXCEPT
    {
        return --m_Count;
    }

private:
    int     m_Count;

};

template< typename T > class DestroyingObjectInfo
{
public:
    explicit DestroyingObjectInfo( T handle ) NN_NOEXCEPT : m_Handle( handle ) {}

    T& GetHandle() NN_NOEXCEPT
    {
        return m_Handle;
    }

    ReferenceCount& GetReferenceCount() NN_NOEXCEPT
    {
        return m_ReferenceCount;
    }

private:
    T m_Handle;
    ReferenceCount m_ReferenceCount;
};

struct DestroyingShaderObjectHandle
{
    VkShaderModule handle[ ShaderStage_End ];
};

typedef DestroyingObjectInfo< DestroyingShaderObjectHandle > DestroyingShaderObjectInfo;

struct DestroyingImageObjectHandle
{
    VkImage image;
    VkDeviceMemory memory;
};

typedef DestroyingObjectInfo< DestroyingImageObjectHandle > DestroyingImageObjectInfo;

struct DestroyingImageViewObjectHandle
{
    VkImageView imageViewHandle;
    DestroyingImageObjectInfo* pDestroyingImageObjectInfo;
};

typedef DestroyingObjectInfo< DestroyingImageViewObjectHandle > DestroyingImageViewObjectInfo;

struct GfxVkGpuState;
class BufferMemoryHeap;
class TextureManageData;
class VkManager;

class Vk
{
public:
    static bool IsLayoutManagementEnabled() NN_NOEXCEPT;
    static void CreateInstance( DeviceImpl< ApiVariationVk1 >* pDevice,
        int layerCount, const char** pLayerNames, int extensionCount, const char** pExtensionNames ) NN_NOEXCEPT;
    static void DestroyInstance( DeviceImpl< ApiVariationVk1 >* pDevice ) NN_NOEXCEPT;
    static void InitializePhysicalDevice( DeviceImpl< ApiVariationVk1 >* pDevice ) NN_NOEXCEPT;
    static void CreateLogicalDevice( DeviceImpl< ApiVariationVk1 >* pDevice,
        int layerCount, const char** pLayerNames,
        int extensionCount, const char** pExtensionNames ) NN_NOEXCEPT;
    static void DestroyLogicalDevice( DeviceImpl< ApiVariationVk1 >* pDevice ) NN_NOEXCEPT;
    static void GetDeviceFeatures( nn::util::BitArray* pOutBits, const VkPhysicalDeviceFeatures& features ) NN_NOEXCEPT;
    static void SetupGlobalDynamicState( const VkPhysicalDeviceFeatures& features ) NN_NOEXCEPT;
    static void GetDeviceExtensions( nn::util::BitPack32* pOutBits, int extensionCount, const char** ppExtensionNames ) NN_NOEXCEPT;
    static void UpdateAllocator( DeviceImpl< ApiVariationVk1 >* pDevice ) NN_NOEXCEPT;
    static void* AllocDriverMemory( size_t size, size_t alignment ) NN_NOEXCEPT;
    static void FreeDriverMemory( void* pMemory ) NN_NOEXCEPT;

    static void SetTicksPerNanosecond( double ticksPerNanosecond ) NN_NOEXCEPT;
    static VkFormat GetImageFormat( ImageFormat format ) NN_NOEXCEPT;
    static VkFormat GetAttributeFormat( AttributeFormat format ) NN_NOEXCEPT;
    static VkImageViewType GetImageTarget( ImageDimension dimension ) NN_NOEXCEPT;
    static VkImageType GetImageType( ImageStorageDimension dimension ) NN_NOEXCEPT;
    static VkCompareOp GetCompareOperation( ComparisonFunction compare ) NN_NOEXCEPT;
    static VkStencilOp GetStencilOperation( StencilOperation operation ) NN_NOEXCEPT;
    static VkBlendOp GetBlendEquation( BlendFunction function ) NN_NOEXCEPT;
    static VkBlendFactor GetBlendFunction( BlendFactor factor ) NN_NOEXCEPT;
    static VkLogicOp GetLogicOperation( LogicOperation operation ) NN_NOEXCEPT;
    static VkFrontFace GetFrontFace( FrontFace face ) NN_NOEXCEPT;
    static VkPolygonMode GetFillMode( FillMode mode ) NN_NOEXCEPT;
    static VkPrimitiveTopology GetDrawPrimitive( PrimitiveTopology topology ) NN_NOEXCEPT;
    static VkFilter GetMinFilter( FilterMode filterMode ) NN_NOEXCEPT;
    static VkFilter GetMagFilter( FilterMode filterMode ) NN_NOEXCEPT;
    static VkSamplerMipmapMode GetMipmapMode( FilterMode filterMode ) NN_NOEXCEPT;
    static bool IsMipMapUsed( FilterMode filterMode ) NN_NOEXCEPT;
    static VkSamplerAddressMode GetWrapMode( TextureAddressMode wrapMode ) NN_NOEXCEPT;
    static VkShaderStageFlagBits GetShaderStage( ShaderStage stage ) NN_NOEXCEPT;
    static int GetShaderStageBits( int shaderStageBits ) NN_NOEXCEPT;
    static VkCompareOp GetRComparisonFunction( ComparisonFunction compare ) NN_NOEXCEPT;
    static VkIndexType GetIndexFormat( IndexFormat indexFormat ) NN_NOEXCEPT;
    static VkCullModeFlags GetCullMode( CullMode face ) NN_NOEXCEPT;
    static int GetMemoryPoolFlags( int memoryPoolProperty ) NN_NOEXCEPT;
    static VkDeviceMemory * GetBufferAddress( const GpuAddress gpuAddress ) NN_NOEXCEPT;
    static VkBufferUsageFlags GetBufferUsageFlags( int gpuAccessFlags ) NN_NOEXCEPT;
    static VkImageUsageFlags GetImageUsageFlags( int gpuAccessFlags ) NN_NOEXCEPT;
    static VkImageLayout GetImageLayout( int textureState ) NN_NOEXCEPT;
    static VkAccessFlags GetBufferStateAccessFlags( int bufferState ) NN_NOEXCEPT;
    static VkAccessFlags GetGpuAccessFlags( int gpuAccessFlags ) NN_NOEXCEPT;
    static VkAccessFlags GetTextureStateAccessFlags( int textureState ) NN_NOEXCEPT;
    static VkComponentSwizzle GetComponentSwizzle( nn::gfx::ChannelMapping mapping ) NN_NOEXCEPT;
    static VkPipelineStageFlags GetPipelineStageBits( int pipelineStageBits, bool isDepth ) NN_NOEXCEPT;
    static VkPipelineBindPoint GetPipelineBindPoint( PipelineType pipelineType ) NN_NOEXCEPT;
    static VkFormatFeatureFlags GetFormatFeatureFlagsFromGpuFlags( int gpuAccessFlags ) NN_NOEXCEPT;
    static void SetupScanBufferTextureInfo( TextureInfo* pOutInfo, const SwapChainInfo& info ) NN_NOEXCEPT;

    static nn::TimeSpan ToTimeSpan( int64_t timestamp ) NN_NOEXCEPT;

    static ImageFormat GetGfxImageFormat( VkFormat vkFormat ) NN_NOEXCEPT;
    static void GetImageFormatProperty( ImageFormatProperty* pOutImageFormatProperty,
        DeviceImpl< ApiVariationVk1 >* pDevice, VkFormat vkFormat ) NN_NOEXCEPT;
    static VkBool32 IsDepthFormat( VkFormat vkFormat ) NN_NOEXCEPT;
    static VkBool32 IsDepthOnlyFormat( VkFormat vkFormat ) NN_NOEXCEPT;
    static VkBool32 IsStencilOnlyFormat( VkFormat vkFormat ) NN_NOEXCEPT;
    static VkImageAspectFlags GetImageAspect( ImageFormat imageFormat ) NN_NOEXCEPT;
    static VkBool32 IsColor( VkImageAspectFlags flags ) NN_NOEXCEPT;
    static VkBool32 IsDepth( VkImageAspectFlags flags ) NN_NOEXCEPT;
    static VkBool32 IsStencil( VkImageAspectFlags flags ) NN_NOEXCEPT;

    static VkResult SelectMemoryType( uint32_t* pOutMemoryTypeIndex,
        uint32_t memoryTypeCount, const uint32_t* pMemoryPropertyFlags,
        uint32_t typeBits, uint32_t requirementsMask ) NN_NOEXCEPT;

    static size_t InitializeOptimalTexture( DeviceImpl< ApiVariationVk1 >* pDevice, VkImage image,
        const nn::gfx::ImageFormat& imageFormat, uint32_t arrayLevel, uint32_t mipLevel, uint32_t width,
        uint32_t height, uint32_t depth, const void* pSourceMem ) NN_NOEXCEPT;

    static void TransitionTextureLayout( DeviceImpl< ApiVariationVk1 >* pDevice, VkImage image,
        const nn::gfx::ImageFormat& imageFormat, VkImageLayout srcLayout, VkAccessFlags srcAccess,
        VkImageLayout dstLayout, VkAccessFlags dstAccess ) NN_NOEXCEPT;

    static size_t CopyCpuMemoryToTexture( void* pOutMem, VkDevice device, VkImage image,
        const nn::gfx::ImageFormat& imageFormat, uint32_t arrayLevel, uint32_t mipLevel,
        uint32_t width, uint32_t height, uint32_t depth, const void* pSourceMem ) NN_NOEXCEPT;

    static void CreateDebugCallback( DeviceImpl< ApiVariationVk1 >* pDevice ) NN_NOEXCEPT;
    static void DestroyDebugCallback( DeviceImpl< ApiVariationVk1 >* pDevice ) NN_NOEXCEPT;

    static bool QueryLayer(const char* pExtensionName) NN_NOEXCEPT;
    static bool QueryExtension(const char* pExtensionName) NN_NOEXCEPT;

    static void EnumerateLayers() NN_NOEXCEPT;
    static void EnumerateInstanceExtensions() NN_NOEXCEPT;
    static void EnumerateDeviceExtensions( DeviceImpl< ApiVariationVk1 >* pDevice ) NN_NOEXCEPT;
    static bool VerifyLayers( const char** pLayerNames, size_t layerNameSize ) NN_NOEXCEPT;
    static VkBuffer GetVkBufferFromGpuAddress( const GpuAddress& address ) NN_NOEXCEPT;
    static uint64_t GetVkOffsetFromGpuAddress( const GpuAddress& address ) NN_NOEXCEPT;

    static void CreatePipelineCache( DeviceImpl< ApiVariationVk1 >* pDevice ) NN_NOEXCEPT;
    static void DestroyPipelineCache( DeviceImpl< ApiVariationVk1 >* pDevice ) NN_NOEXCEPT;
    static void CreateCommandBuffer( CommandBufferImpl< ApiVariationVk1 >* pCommandBuffer ) NN_NOEXCEPT;
    static void CreateQueryPool( VkQueryPool* pOutQueryPool, DeviceImpl< ApiVariationVk1 >* pDevice,
        QueryTarget queryTarget, uint32_t queryCount ) NN_NOEXCEPT;
    static void CreateGraphicsPipelineStatisticsQueriesPool( VkQueryPool* pOutQueryPool,
        DeviceImpl< ApiVariationVk1 >* pDevice, uint32_t queryCount ) NN_NOEXCEPT;
    static void DestroyQueryPool( DeviceImpl< ApiVariationVk1 >* pDevice, VkQueryPool queryPool ) NN_NOEXCEPT;
    static VkQueryType GetQueryType( QueryTarget queryTarget ) NN_NOEXCEPT;
    static VkQueryPipelineStatisticFlags GetQueryPipelineStatisticFlag( QueryTarget queryTarget ) NN_NOEXCEPT;
    static void DestroyCommandBuffer( DeviceImpl< ApiVariationVk1 >* pDevice,
        CommandBufferImpl< ApiVariationVk1 >* pCommandBuffer ) NN_NOEXCEPT;
    static void BeginCommandBuffer( VkCommandBuffer commandBuffer ) NN_NOEXCEPT;
    static void EndCommandBuffer( VkCommandBuffer commandBuffer ) NN_NOEXCEPT;
    static void EndRenderPass( VkCommandBuffer commandBuffer ) NN_NOEXCEPT;
    static void CreateRenderPass( VkRenderPass* pOutRenderPass, DeviceImpl< ApiVariationVk1 >* pDevice,
        VkFormat format, int samples, bool isDepth ) NN_NOEXCEPT;
    static void CreateFramebuffer( VkFramebuffer* pOutFramebuffer, DeviceImpl< ApiVariationVk1 >* pDevice,
        VkRenderPass renderPass, VkImageView imageView, uint32_t width, uint32_t height, uint32_t layers ) NN_NOEXCEPT;
    static void CreateComputePipeline( VkPipeline* pOutPipeline, DeviceImpl< ApiVariationVk1 >* pDevice,
        const GfxVkGpuState& gpuState, VkPipelineLayout pipelineLayout ) NN_NOEXCEPT;
    static void CreateGraphicsPipeline( VkPipeline* pOutPipeline, const DeviceImpl< ApiVariationVk1 >* pDevice,
        const GfxVkGpuState& gpuState, VkPipelineLayout pipelineLayout, VkRenderPass renderPass ) NN_NOEXCEPT;
    static void CreateRenderPass( VkRenderPass* pOutRenderPass,
        DeviceImpl< ApiVariationVk1 >* pDevice, const GfxVkGpuState& gpuState ) NN_NOEXCEPT;
    static void CreateDescriptorSetLayout( VkDescriptorSetLayout* pOutDescriptorSetLayout,
        DeviceImpl< ApiVariationVk1 >* pDevice, const GfxVkGpuState& gpuState ) NN_NOEXCEPT;
    static void CreatePipelineLayout( VkPipelineLayout* pOutPipelineLayout, VkDescriptorSetLayout descriptorSetLayout,
        DeviceImpl< ApiVariationVk1 >* pDevice ) NN_NOEXCEPT;
    static void CreateDescriptorSet( VkDescriptorPool* pOutDescriptorPool, VkDescriptorSet* pOutDescriptorSet,
        DeviceImpl< ApiVariationVk1 >* pDevice, const GfxVkGpuState& gpuState, VkDescriptorSetLayout descriptorSetLayout ) NN_NOEXCEPT;
    static void CreateFramebuffer( VkFramebuffer* pOutFramebuffer, DeviceImpl< ApiVariationVk1 >* pDevice,
        const GfxVkGpuState& gpuState, VkRenderPass renderPass ) NN_NOEXCEPT;
    static void SetupPipelineShaderStageCreateInfo( VkPipelineShaderStageCreateInfo* pOutStages, const GfxVkGpuState& gpuState ) NN_NOEXCEPT;
    static void SetupPipelineVertexInputStateCreateInfo( VkPipelineVertexInputStateCreateInfo* pOutVertexInputState,
        VkVertexInputBindingDescription pOutVertexInputBinding[], int vertexInputBindingCount,
        VkVertexInputAttributeDescription pOutVertexInputAttribute[], int vertexInputAttributeCount, const GfxVkGpuState& gpuState ) NN_NOEXCEPT;
    static void SetupPipelineInputAssemblyStateCreateInfo( VkPipelineInputAssemblyStateCreateInfo* pOutInputAssemblyState,
        const GfxVkGpuState& gpuState ) NN_NOEXCEPT;
    static void SetupPipelineTessellationStateCreateInfo( VkPipelineTessellationStateCreateInfo* pOutTessellationState,
        const GfxVkGpuState& gpuState ) NN_NOEXCEPT;
    static void SetupPipelineViewportStateCreateInfo( VkPipelineViewportStateCreateInfo* pOutViewportState,
        const GfxVkGpuState& gpuState ) NN_NOEXCEPT;
    static void SetupPipelineRasterizationStateCreateInfo( VkPipelineRasterizationStateCreateInfo* pOutRasterizationState,
        const GfxVkGpuState& gpuState ) NN_NOEXCEPT;
    static void SetupPipelineMultisampleStateCreateInfo( VkPipelineMultisampleStateCreateInfo* pOutMultisampleState,
        VkSampleMask* pSampleMask, const GfxVkGpuState& gpuState ) NN_NOEXCEPT;
    static void SetupPipelineDepthStencilStateCreateInfo( const DeviceImpl< ApiVariationVk1 >* pDevice,
        VkPipelineDepthStencilStateCreateInfo* pOutDepthStencilState, const GfxVkGpuState& gpuState ) NN_NOEXCEPT;
    static void SetupPipelineColorBlendStateCreateInfo( VkPipelineColorBlendStateCreateInfo* pOutColorBlendState,
        VkPipelineColorBlendAttachmentState pColorBlendAttachmentState[], int colorBlendAttachmentCount,
        const GfxVkGpuState& gpuState ) NN_NOEXCEPT;
    static void SetupPipelineDynamicStateCreateInfo( VkPipelineDynamicStateCreateInfo* pOutDynamicState ) NN_NOEXCEPT;
    static void CreateBufferMemoryHeap( DeviceImpl< ApiVariationVk1 >* pDevice ) NN_NOEXCEPT;
    static void DestroyBufferMemoryHeap( DeviceImpl< ApiVariationVk1 >* pDevice ) NN_NOEXCEPT;
    static void CreateVkManagerList( DeviceImpl< ApiVariationVk1 >* pDevice ) NN_NOEXCEPT;
    static void DestroyVkManagerList( DeviceImpl< ApiVariationVk1 >* pDevice ) NN_NOEXCEPT;
    static void AddToVkManagerList( DeviceImpl< ApiVariationVk1 >* pDevice, VkManager* pVkManager ) NN_NOEXCEPT;
    static void RemoveFromVkManagerList( DeviceImpl< ApiVariationVk1 >* pDevice, VkManager* pVkManager ) NN_NOEXCEPT;
    static bool IsVkManagerNodeIdentical( VkManager* const& pNode, void* pArgument ) NN_NOEXCEPT;
    static void RegisterDestroyingShaderObject( DeviceImpl< ApiVariationVk1 >* pDevice, ShaderImpl< ApiVariationVk1 >* pShader ) NN_NOEXCEPT;
    static void RegisterDestroyingImageObject( DeviceImpl< ApiVariationVk1 >* pDevice, TextureImpl< ApiVariationVk1 >* pTexture ) NN_NOEXCEPT;
    static void DestroyShaderObject( DeviceImpl< ApiVariationVk1 >* pDevice, DestroyingShaderObjectInfo* pDestroyingShaderObjectInfo ) NN_NOEXCEPT;
    static void DestroyImageObject( DeviceImpl< ApiVariationVk1 >* pDevice, DestroyingImageObjectInfo* pDestroyingImageObjectInfo ) NN_NOEXCEPT;
    static void DestroyImageViewObject( DeviceImpl< ApiVariationVk1 >* pDevice, DestroyingImageViewObjectInfo* pDestroyingImageViewObjectInfo ) NN_NOEXCEPT;
    static VkAccessFlags GetLayoutAccessFlags( VkImageLayout layout ) NN_NOEXCEPT;
    static void ClearColorTexture( DeviceImpl< ApiVariationVk1 >* pDevice, VkImage image, const ClearColorValue& clearColor ) NN_NOEXCEPT;

    static VkImageLayout LayoutColorAttachment;
    static VkImageLayout LayoutDepthStencilAttachment;
    static VkImageLayout LayoutCopyImageSrc;
    static VkImageLayout LayoutCopyImageDst;
    static VkImageLayout LayoutBlitImageSrc;
    static VkImageLayout LayoutBlitImageDst;
    static VkImageLayout LayoutResolveImageSrc;
    static VkImageLayout LayoutResolveImageDst;
    static VkImageLayout LayoutClearImage;
    static VkImageLayout LayoutTextureRead;
    static VkImageLayout LayoutTextureDepthStencilRead;

    struct BufferViewDescriptorSlot
    {
        VkHandle buffer;
        uint64_t offset;
        uint64_t size;
    };

    struct TextureViewDescriptorSlot
    {
        enum Flags
        {
            Flag_TexelBuffer
        };
        VkHandle imageView;
        detail::Ptr< TextureManageData > pManager;
        nn::util::BitPack32 flags;
        uint32_t format;
        uint32_t baseArrayLayer;
        uint32_t layerCount;
        uint32_t baseMipLevel;
        uint32_t mipCount;
    };

    struct SamplerDescriptorSlot
    {
        VkHandle sampler;
    };

};

template< typename TObject >
inline bool IsInitialized( const TObject& obj ) NN_NOEXCEPT
{
    return obj.ToData()->state == TObject::DataType::State_Initialized;
}

template< typename VkObject >
inline VkObject CastToVkNonDispatchableObject( VkHandle handle ) NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_OS_WIN ) && !defined( NN_BUILD_CONFIG_CPU_X64 )
    return handle;
#else
    return reinterpret_cast< VkObject >( handle );
#endif
}

template< typename VkObject >
inline VkObject CastToVkDispatchableObject( VkHandle handle ) NN_NOEXCEPT
{
    return reinterpret_cast< VkObject >( handle );
}

template< typename VkObject >
inline VkHandle CastFromVkNonDispatchableObject( VkObject object ) NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_OS_WIN ) && !defined( NN_BUILD_CONFIG_CPU_X64 )
    return object;
#else
    return reinterpret_cast< VkHandle >( object );
#endif
}

template< typename VkObject >
inline VkHandle CastFromVkDispatchableObject( VkObject object ) NN_NOEXCEPT
{
    return reinterpret_cast< VkHandle >( object );
}

#if defined(NN_SDK_BUILD_DEBUG)
#define DEBUG_HASH_LIST_COUNTER
#endif

template < typename T >
class List
{
public:
    List() NN_NOEXCEPT : m_pHead( nullptr ), m_pTail( nullptr ), m_FrameHeap( nullptr ), m_isResetHeap( false )
    #ifdef DEBUG_HASH_LIST_COUNTER
    , nodeCount( 0 )
    #endif
    {
    }
    ~List() NN_NOEXCEPT
    {
        RemoveAll();
    }

    typedef bool( *ConditionFunction )( const T&, void* );

private:
    struct Node
    {
        Node() NN_NOEXCEPT : pNext( nullptr ) {}
        T nodeData;
        Node* pNext;
    };

    Node* m_pHead;
    Node* m_pTail;

    nn::lmem::HeapHandle m_FrameHeap;
    bool m_isResetHeap;

    void LinkBack( Node* pNewNode ) NN_NOEXCEPT
    {
        if (m_pHead == nullptr)
        {
            m_pHead = pNewNode;
            m_pTail = pNewNode;
        }
        else
        {
            m_pTail->pNext = pNewNode;
            m_pTail = pNewNode;
        }
        #ifdef DEBUG_HASH_LIST_COUNTER
        ++nodeCount;
        #endif
    }

    #ifdef DEBUG_HASH_LIST_COUNTER
    int nodeCount;
    #endif

public:

    class Iterator
    {
        Node* m_pNode;
    public:
        Iterator() NN_NOEXCEPT : m_pNode( nullptr ){}
        explicit Iterator( Node* pNode ) : m_pNode( pNode ){}
        T& operator *() NN_NOEXCEPT
        {
            NN_SDK_ASSERT( m_pNode != nullptr );
            return m_pNode->nodeData;
        }
        T& operator->() NN_NOEXCEPT
        {
            NN_SDK_ASSERT( m_pNode != nullptr );
            return m_pNode->nodeData;
        }
        Iterator& operator++() NN_NOEXCEPT
        {
            if ( m_pNode != nullptr )
            {
                m_pNode = m_pNode->pNext;
            }
            return *this;
        }
        bool operator==( Iterator& rhs ) const NN_NOEXCEPT
        {
            return this->m_pNode == rhs.m_pNode;
        }
        bool operator!=( Iterator& rhs ) const NN_NOEXCEPT
        {
            return !( *this == rhs );
        }
        bool IsEmpty() const NN_NOEXCEPT
        {
            return m_pNode == nullptr;
        }
    };

    Iterator Begin() const NN_NOEXCEPT
    {
        return Iterator( m_pHead );
    }

    Iterator End() const NN_NOEXCEPT
    {
        return Iterator( nullptr );
    }

    Iterator BeforeEnd() const NN_NOEXCEPT
    {
        return Iterator( m_pTail );
    }

    bool IsEnd( Iterator& iterator ) const NN_NOEXCEPT
    {
        return iterator.IsEmpty();
    }

    bool IsEmpty() const NN_NOEXCEPT
    {
        return m_pHead == nullptr;
    }

    void SetAllocator( nn::lmem::HeapHandle heapHandle, bool isResetHeap ) NN_NOEXCEPT
    {
        m_FrameHeap = heapHandle;
        m_isResetHeap = isResetHeap;
    }

    T& EmplaceBack() NN_NOEXCEPT
    {
        void* pData = m_FrameHeap != nullptr ? nn::lmem::AllocateFromFrameHeap( m_FrameHeap, sizeof( Node ), 8 ) :
            Vk::AllocDriverMemory( sizeof( Node ), 8 );
        Node* pNewNode = new ( pData ) Node;

        LinkBack( pNewNode );

        return pNewNode->nodeData;
    }

    void RemoveAll() NN_NOEXCEPT
    {
        if ( m_FrameHeap == nullptr )
        {
            Node* pNext;
            for ( Node* pNode = m_pHead; pNode != nullptr; pNode = pNext )
            {
                pNext = pNode->pNext;
                pNode->~Node();
                Vk::FreeDriverMemory( pNode );
            }
        }
        else if ( m_isResetHeap )
        {
            nn::lmem::FreeToFrameHeap( m_FrameHeap, nn::lmem::FreeMode_All );
        }

        m_pHead = m_pTail = nullptr;

        #ifdef DEBUG_HASH_LIST_COUNTER
        nodeCount = 0;
        #endif
    }

    void RemoveIf( ConditionFunction conditionFunction, void* pArgument ) NN_NOEXCEPT
    {
        NN_SDK_ASSERT( m_FrameHeap == nullptr );
        List newList;
        Node* pNext;

        for (Node* pNode = m_pHead; pNode != nullptr; pNode = pNext )
        {
            pNext = pNode->pNext;
            if ( conditionFunction( pNode->nodeData, pArgument ) )
            {
                pNode->~Node();
                Vk::FreeDriverMemory( pNode );
            }
            else
            {
                pNode->pNext = nullptr;
                newList.LinkBack( pNode );
            }
        }
        this->m_pHead = newList.m_pHead;
        this->m_pTail = newList.m_pTail;
        newList.m_pHead = nullptr;
        newList.m_pTail = nullptr;
        #ifdef DEBUG_HASH_LIST_COUNTER
        this->nodeCount = newList.nodeCount;
        #endif
    }

    #ifdef DEBUG_HASH_LIST_COUNTER
    int GetNodeCount() const NN_NOEXCEPT { return nodeCount; }
    #endif
};

template< typename T, typename HashKey, int HashTableSize >
class HashList
{
public:
    HashList() NN_NOEXCEPT
        : m_FrameHeap( nullptr ), m_isResetHeap( false ), m_CompareFunction( nullptr )
    #ifdef DEBUG_HASH_LIST_COUNTER
    , m_TotalCount( 0 ), m_MissCount( 0 ), m_HitCount( 0 )
    #endif
    {}
    virtual ~HashList() NN_NOEXCEPT {}

    typedef uint64_t HashValue;

    struct Node
    {
        Node() NN_NOEXCEPT {}
        T nodeData;
        HashValue hashValue;
    };
    typedef List< Node > HashNode;
    typedef typename List< Node >::Iterator HashNodeIterator;

private:

    typedef bool( *ConditionFunction )( const Node&, void* );

    HashNode m_HashTable[ HashTableSize ];
    nn::lmem::HeapHandle m_FrameHeap;
    bool m_isResetHeap;

    typedef HashValue( *HashFunction )( const HashKey& );
    HashFunction m_HashFunction;
    typedef int( *GetTableIndexFunction )( HashValue, int );
    GetTableIndexFunction m_GetTableIndexFunction;
    typedef bool( *CompareFunction  )( const T&, const HashKey& );
    CompareFunction m_CompareFunction;

    #ifdef DEBUG_HASH_LIST_COUNTER
    int m_TotalCount;
    int m_MissCount;
    int m_HitCount;
    #endif

public:
    void SetFrameHeap( nn::lmem::HeapHandle handle, bool isResetHeap ) NN_NOEXCEPT
    {
        m_FrameHeap = handle;
        m_isResetHeap = isResetHeap;
        for ( int idx = 0; idx < HashTableSize; ++idx )
        {
            HashNode& hashNode = m_HashTable[ idx ];
            hashNode.SetAllocator( handle, false );
        }
    }

    void SetHashFunction( HashFunction function ) NN_NOEXCEPT
    {
        m_HashFunction = function;
    }

    void SetTableIndexFunction( GetTableIndexFunction function ) NN_NOEXCEPT
    {
        m_GetTableIndexFunction = function;
    }

    void SetCompareFunction( CompareFunction function ) NN_NOEXCEPT
    {
        m_CompareFunction = function;
    }

    HashNode& GetHashTableNode( int index ) NN_NOEXCEPT
    {
        NN_SDK_ASSERT( index < HashTableSize );
        return m_HashTable[ index ];
    }

    T* GetNode( bool* pOutIsFound, const HashKey& hashKey ) NN_NOEXCEPT
    {
        NN_SDK_ASSERT( m_HashFunction != nullptr );
        HashValue hashValue = m_HashFunction( hashKey );
        NN_SDK_ASSERT( m_GetTableIndexFunction != nullptr );
        int tableIndex = m_GetTableIndexFunction( hashValue, HashTableSize );
        NN_SDK_ASSERT(tableIndex < HashTableSize);
        HashNode& hashNode = m_HashTable[ tableIndex ];

        NN_SDK_ASSERT( m_CompareFunction != nullptr );
        for ( HashNodeIterator iterator = hashNode.Begin();
            !hashNode.IsEnd( iterator ); ++iterator )
        {
            bool found = ( *iterator ).hashValue == hashValue;
            if ( found && m_CompareFunction( ( *iterator ).nodeData, hashKey ) )
            {
                #ifdef DEBUG_HASH_LIST_COUNTER
                ++m_HitCount;
                #endif
                *pOutIsFound = true;
                return &( *iterator ).nodeData;
            }
        }

        #ifdef DEBUG_HASH_LIST_COUNTER
        ++m_MissCount;
        ++m_TotalCount;
        #endif

        Node& node = hashNode.EmplaceBack();
        node.hashValue = hashValue;
        *pOutIsFound = false;

        return &node.nodeData;
    }

    void RemoveAll() NN_NOEXCEPT
    {
        for ( int idx = 0; idx < HashTableSize; ++idx )
        {
            HashNode& hashNode = m_HashTable[ idx ];

            if ( !hashNode.IsEmpty() )
            {
                hashNode.RemoveAll();
            }
        }

        if ( m_FrameHeap != nullptr && m_isResetHeap )
        {
            nn::lmem::FreeToFrameHeap( m_FrameHeap, nn::lmem::FreeMode_All );
        }
    }

    void RemoveIf( ConditionFunction conditionFunction, void* pArgument ) NN_NOEXCEPT
    {
        NN_SDK_ASSERT( m_FrameHeap == nullptr );
        for ( int idx = 0; idx < HashTableSize; ++idx )
        {
            HashNode& hashNode = m_HashTable[ idx ];

            if ( !hashNode.IsEmpty() )
            {
                hashNode.RemoveIf( conditionFunction, pArgument );
            }
        }
    }

    bool IsEmpty() NN_NOEXCEPT
    {
        for ( int idx = 0; idx < HashTableSize; ++idx )
        {
            HashNode& hashNode = m_HashTable[ idx ];

            if ( !hashNode.IsEmpty() )
            {
                return false;
            }
        }

        return true;
    }

    #ifdef DEBUG_HASH_LIST_COUNTER
    int GetTotalCount() const NN_NOEXCEPT
    {
        return m_TotalCount;
    }

    int GetHitCount() const NN_NOEXCEPT
    {
        return m_HitCount;
    }

    int GetMissCount() const NN_NOEXCEPT
    {
        return m_MissCount;
    }

    int GetEachNodeCount( int idxTable ) const NN_NOEXCEPT
    {
        return m_HashTable[ idxTable ].GetNodeCount();
    }

    void ClearCounter() NN_NOEXCEPT
    {
        m_HitCount = 0;
        m_MissCount = 0;
    }
    #endif
};

/*
 * BufferMemoryHeapでは、VulkanのBuffer用のDeviceMemoryを確保します。
 * 256および512バイトのメモリ要求のみ、予め確保した大領域から確保します。
 * 256および512バイトのメモリは簡易のアロケータアルゴリズムで実装しておきます。
*/
class BufferMemoryHeap
{
    NN_DISALLOW_COPY( BufferMemoryHeap );

public:
    explicit BufferMemoryHeap( DeviceImpl< ApiVariationVk1 >* pDevice ) NN_NOEXCEPT
    {
        /*
         * MemoryRequirmentsを取得するためのダミーのバッファを生成します。
        */
        VkBuffer tmpBuffer;
        VkBufferCreateInfo vkInfo = {};
        vkInfo.sType = VK_STRUCTURE_TYPE_BUFFER_CREATE_INFO;
        vkInfo.pNext = NULL;
        vkInfo.usage = VK_BUFFER_USAGE_VERTEX_BUFFER_BIT | VK_BUFFER_USAGE_INDEX_BUFFER_BIT | VK_BUFFER_USAGE_UNIFORM_BUFFER_BIT | VK_BUFFER_USAGE_TRANSFER_DST_BIT;
        vkInfo.queueFamilyIndexCount = 0;
        vkInfo.pQueueFamilyIndices = NULL;
        vkInfo.sharingMode = VK_SHARING_MODE_EXCLUSIVE;
        vkInfo.flags = 0;
        vkInfo.size = 16;

        VkResult result;
        m_Device = CastToVkDispatchableObject< VkDevice >( pDevice->ToData()->hDevice );
        m_pAllocator = pDevice->ToData()->pAllocationCallback;
        NN_GFX_CALL_VK_FUNCTION( result = vkCreateBuffer( m_Device, &vkInfo, m_pAllocator, &tmpBuffer ) );
        NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );

        VkMemoryRequirements memReqs;
        NN_GFX_CALL_VK_FUNCTION( vkGetBufferMemoryRequirements( m_Device, tmpBuffer, &memReqs ) );

        VkMemoryAllocateInfo allocInfo;
        allocInfo.sType = VK_STRUCTURE_TYPE_MEMORY_ALLOCATE_INFO;
        allocInfo.pNext = NULL;
        allocInfo.memoryTypeIndex = 0;
        allocInfo.allocationSize = BlockMemoryHeapSize;

        /*
         * メモリ属性固定のため、試験済み環境と異なる環境では、何か問題が起こるかもしれません。
         * 現状の実装は、HostVisible | HostCoherent固定です。
        */
        uint32_t memFlags = VK_MEMORY_PROPERTY_HOST_VISIBLE_BIT | VK_MEMORY_PROPERTY_HOST_COHERENT_BIT;

        result = Vk::SelectMemoryType( &allocInfo.memoryTypeIndex, pDevice->ToData()->memoryTypeCount,
            pDevice->ToData()->memoryPropertyFlags, memReqs.memoryTypeBits, memFlags );
        NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );

        NN_GFX_CALL_VK_FUNCTION( result = vkAllocateMemory( m_Device, &allocInfo, m_pAllocator, &m_BlockMemoryHeap ) );
        NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );

        NN_GFX_CALL_VK_FUNCTION( vkDestroyBuffer( m_Device, tmpBuffer, m_pAllocator ) );

        NN_GFX_CALL_VK_FUNCTION( result = vkMapMemory( m_Device, m_BlockMemoryHeap, 0, BlockMemoryHeapSize, 0, &m_pMappedAddress ) );
        NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );

        memset( m_AllocState, 0xff, sizeof( uint64_t ) * AllocStateArraySize );
        m_StartBlock = 0;
        m_StartLine = 0;
        m_IsFullOneBlock = false;
        m_IsFullTwoBlock = false;
    }
    virtual ~BufferMemoryHeap() NN_NOEXCEPT
    {
        NN_GFX_CALL_VK_FUNCTION( vkFreeMemory( m_Device, m_BlockMemoryHeap, m_pAllocator ) );
    }

    bool IsMemoryIndependent( VkDeviceMemory deviceMemory ) const NN_NOEXCEPT
    {
        return deviceMemory != m_BlockMemoryHeap;
    }

private:
    bool AllocateBlock( VkDeviceSize* pOutMemoryOffset, int numBlock ) NN_NOEXCEPT
    {
        if ( numBlock == 1 )
        {
            for ( int line = 0; line < AllocStateArraySize; ++line )
            {
                int currentLine = ( line + m_StartLine ) % AllocStateArraySize;
                uint64_t allocLineState = m_AllocState[ currentLine ];

                if ( allocLineState != 0 )
                {
                    for ( int position = 0; position < 64; ++position )
                    {
                        int currentBlock = ( position + m_StartBlock ) % 64;
                        if ( ( allocLineState >> currentBlock ) & 1 )
                        {
                            /* 空き領域を見つけました。 */
                            m_AllocState[ currentLine ] &= ~( static_cast< uint64_t >( 1 ) << currentBlock );
                            *pOutMemoryOffset = static_cast< VkDeviceSize >( currentLine * LineSize + currentBlock * BlockSize );
                            if ( m_AllocState[ currentLine ] == 0 )
                            {
                                m_StartBlock = 0;
                                m_StartLine = ( currentLine + 1 ) % AllocStateArraySize;
                            }
                            else
                            {
                                m_StartBlock = ( currentBlock + 1 ) % 64;
                                m_StartLine = currentLine;
                            }
                            return true;
                        }
                    }
                }
            }
            m_IsFullOneBlock = true;
            m_IsFullTwoBlock = true;
        }
        else if ( numBlock == 2 )
        {
            for ( int line = 0; line < AllocStateArraySize; ++line )
            {
                int currentLine = ( line + m_StartLine ) % AllocStateArraySize;
                uint64_t allocLineState = m_AllocState[ currentLine ];

                if ( HasSequentialTwoBits( allocLineState ) )
                {
                    for ( int position = 0; position < 64; ++position )
                    {
                        int currentBlock = ( position + m_StartBlock ) % 64;
                        if ( currentBlock == 63 )
                        {
                            continue;
                        }

                        uint64_t twoBits = ( allocLineState >> currentBlock ) & 3;
                        if ( twoBits == 3 )
                        {
                            /* 空き領域を見つけました。 */
                            m_AllocState[ currentLine ] &= ~( static_cast< uint64_t >( 3 ) << currentBlock );
                            *pOutMemoryOffset = static_cast< VkDeviceSize >( currentLine * LineSize + currentBlock * BlockSize );
                            if ( !HasSequentialTwoBits( m_AllocState[ currentLine ] ) )
                            {
                                m_StartBlock = 0;
                                m_StartLine = ( currentLine + 1 ) % AllocStateArraySize;
                            }
                            else
                            {
                                m_StartBlock = ( currentBlock + 2 ) % 63;
                                m_StartLine = currentLine;
                            }
                            return true;
                        }
                        else if ( ( twoBits & 2 ) == 0 )
                        {
                            ++position;
                        }
                    }
                }
            }
            m_IsFullTwoBlock = true;
        }
        else
        {
            NN_SDK_ASSERT( 0 );
        }

        return false;
    }

    bool HasSequentialTwoBits( uint64_t value ) const NN_NOEXCEPT
    {
        return ( ( value & static_cast< uint64_t >( 0x5555555555555555 ) )
            & ( ( value >> 1 ) & static_cast< uint64_t >( 0x5555555555555555 ) ) ) != 0
            || ( ( value & static_cast< uint64_t >( 0xaaaaaaaaaaaaaaaa ) )
            & ( ( value >> 1 ) & static_cast< uint64_t >( 0xaaaaaaaaaaaaaaaa ) ) ) != 0;
    }

public:
    void AllocateMemory( VkDeviceMemory* pOutDeviceMemory, VkDeviceSize* pOutMemoryOffset, const VkMemoryAllocateInfo& allocInfo ) NN_NOEXCEPT
    {
        /* ブロックメモリヒープからメモリを切り出して確保します。 */
        if ( ( allocInfo.allocationSize == 256 && !m_IsFullOneBlock )
            || ( allocInfo.allocationSize == 512 && !m_IsFullTwoBlock ) )
        {
            bool alloced = AllocateBlock( pOutMemoryOffset, static_cast< int >( allocInfo.allocationSize / BlockSize ) );

            if ( alloced )
            {
                *pOutDeviceMemory = m_BlockMemoryHeap;
                return;
            }
        }

        /* ブロックメモリヒープから取得できなかった場合は通常の方法でメモリを確保します。 */
        VkResult result;
        NN_GFX_CALL_VK_FUNCTION( result = vkAllocateMemory( m_Device, &allocInfo, m_pAllocator, pOutDeviceMemory ) );
        NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );
        *pOutMemoryOffset = 0;
    }

    void FreeMemory( VkDeviceMemory deviceMemory, VkDeviceSize memoryOffset, VkDeviceSize memorySize ) NN_NOEXCEPT
    {
        if ( IsMemoryIndependent( deviceMemory ) )
        {
            NN_GFX_CALL_VK_FUNCTION( vkFreeMemory( m_Device, deviceMemory, m_pAllocator ) );
        }
        else
        {
            int line = static_cast< int >( memoryOffset / LineSize );
            int pos = static_cast< int >( ( memoryOffset % LineSize ) / BlockSize );
            uint64_t bits = memorySize == 256 ? 1 : 3;
            NN_SDK_ASSERT( ( ( m_AllocState[ line ] >> pos ) & bits ) == 0 );
            m_AllocState[ line ] |= bits << pos;
            if ( m_IsFullOneBlock || m_IsFullTwoBlock )
            {
                m_StartLine = line;
                m_StartBlock = pos;
                m_IsFullOneBlock = false;
                m_IsFullTwoBlock = false;
            }
        }
    }

    void* MapMemory( VkDeviceMemory memory, VkDeviceSize offset, VkDeviceSize size, VkMemoryMapFlags flags ) NN_NOEXCEPT
    {
        void* pData;

        if ( memory == m_BlockMemoryHeap )
        {
            pData = nn::util::BytePtr( m_pMappedAddress ).Advance( static_cast< ptrdiff_t >( offset ) ).Get();
        }
        else
        {
            VkResult result;
            NN_GFX_CALL_VK_FUNCTION( result = vkMapMemory( m_Device, memory, offset, size, flags, &pData ) );
            NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );
        }

        return pData;
    }

    void UnmapMemory( VkDeviceMemory memory ) const NN_NOEXCEPT
    {
        if ( memory != m_BlockMemoryHeap )
        {
            NN_GFX_CALL_VK_FUNCTION( vkUnmapMemory( m_Device, memory ) );
        }
    }

private:
    #if defined( NN_BUILD_CONFIG_OS_WIN32 )
    static const VkDeviceSize BlockMemoryHeapSize = 8 * 1024 * 1024;        //!< 本クラスで扱うデバイスメモリのサイズです
    #elif defined( NN_BUILD_CONFIG_OS_HORIZON )
    static const VkDeviceSize BlockMemoryHeapSize = 4 * 1024 * 1024;
    #endif

    static const int BlockSize = 256;                                       //!< 1ブロックのサイズ（バイト数）です
    static const int LineSize = BlockSize * 64;                             //!< 64ブロックを1ラインとして管理します
    static const int AllocStateArraySize = BlockMemoryHeapSize / LineSize;  //!< m_AllocStateの配列数です

    VkDevice                m_Device;                                       //!< VkDeviceです
    VkAllocationCallbacks*  m_pAllocator;                                   //!< ユーザアロケータです
    VkDeviceMemory          m_BlockMemoryHeap;                              //!< ブロックメモリ用のデバイスメモリです

    uint64_t                m_AllocState[ AllocStateArraySize ];            //!< 各ビットが各ブロックの確保状況を示し、1の場合は空です
    int                     m_StartBlock;                                   //!< アロケーションの検索開始位置です
    int                     m_StartLine;                                    //!< アロケーションの検索開始ラインです
    bool                    m_IsFullOneBlock;                               //!< 1ブロック確保ができない場合trueです
    bool                    m_IsFullTwoBlock;                               //!< 連続2ブロック確保ができない場合trueです
    void*                   m_pMappedAddress;                               //!< 本クラスで扱うデバイスメモリの先頭をマップしたアドレスです
};

template< typename T >
class ImageViewCache
{
    NN_DISALLOW_COPY( ImageViewCache );

public:
    ImageViewCache() NN_NOEXCEPT {}

    class Node
    {
    public:
        Node() NN_NOEXCEPT{}
        ~Node(){}
        T infoData;
        VkImageView imageView;
    };

    typedef List< Node > NodeList;
    typedef typename List< Node >::Iterator Iterator;

    VkImageView GetFromCache( const T& src ) NN_NOEXCEPT
    {
        for ( Iterator iterator = m_NodeList.Begin(); !m_NodeList.IsEnd( iterator ); ++iterator )
        {
            Node& node = *iterator;
            if ( memcmp( &src, &node.infoData, sizeof( T ) ) == 0 )
            {
                return node.imageView;
            }
        }

        return VK_NULL_HANDLE;
    }

    void AddCache( const T& infoData, VkImageView imageView ) NN_NOEXCEPT
    {
        Node& node = m_NodeList.EmplaceBack();
        node.infoData = infoData;
        node.imageView = imageView;
    }

    void DestroyAllImageView( DeviceImpl< ApiVariationVk1 >* pDevice ) NN_NOEXCEPT
    {
        VkDevice device = CastToVkDispatchableObject< VkDevice >( pDevice->ToData()->hDevice );
        const VkAllocationCallbacks* pAllocator =
            static_cast< const VkAllocationCallbacks* >( pDevice->ToData()->pAllocationCallback.ptr );

        for ( Iterator iterator = m_NodeList.Begin(); !m_NodeList.IsEnd( iterator ); ++iterator )
        {
            Node& node = *iterator;
            NN_GFX_CALL_VK_FUNCTION( vkDestroyImageView( device, node.imageView, pAllocator ) );
        }
    }

    const NodeList* GetNodeList() NN_NOEXCEPT
    {
        return &m_NodeList;
    }

private:
    NodeList m_NodeList;
};

class TextureManageData
{
    NN_DISALLOW_COPY( TextureManageData );

public:
    TextureManageData( uint32_t layers, uint32_t mips ) NN_NOEXCEPT
        : m_ImageHandle( VK_NULL_HANDLE ),
          m_Layers( layers ),
          m_Mips( mips )
    {
        m_ReferenceCount.Increment();
        if ( Vk::IsLayoutManagementEnabled() )
        {
            m_pCurrentLayout = reinterpret_cast< VkImageLayout* >( Vk::AllocDriverMemory( layers * mips * sizeof( VkImageLayout ), 8 ) );
            ResetImageLayout( VK_IMAGE_LAYOUT_UNDEFINED );
        }
        else
        {
            m_pCurrentLayout = nullptr;
        }
    }

    virtual ~TextureManageData() NN_NOEXCEPT {}

    void DestroyAllImageView( DeviceImpl< ApiVariationVk1 >* pDevice ) NN_NOEXCEPT
    {
        m_TextureView.DestroyAllImageView( pDevice );

        // Color target view and depth stencil view will be destroyed after clearing cache in VkManager.
        //  m_ColorTargetView.DestroyAllImageView( pDevice );
        //  m_DepthStencilView.DestroyAllImageView( pDevice );

        if ( Vk::IsLayoutManagementEnabled() )
        {
            Vk::FreeDriverMemory( m_pCurrentLayout );
            m_pCurrentLayout = nullptr;
        }
    }

    void PrepareLayoutTransition( VkImageLayout requiredLayout, uint32_t* pLayoutCount, VkImageMemoryBarrier* pImageBarriers,
        uint32_t maxLayout, uint32_t baseLayer, uint32_t layerCount, uint32_t baseMip, uint32_t mipCount,
        VkPipelineStageFlags* pSrcStages, VkPipelineStageFlags* pDstStages, VkImageAspectFlags imageAspectFlags ) NN_NOEXCEPT
    {
        NN_UNUSED( maxLayout );
        uint32_t barrierCount = *pLayoutCount;

        // Make sure parameters are in-bounds
        layerCount = ( baseLayer + layerCount ) <= m_Layers ? layerCount : ( m_Layers - baseLayer + 1 );
        mipCount = ( baseMip + mipCount ) <= m_Mips ? mipCount : ( m_Mips - baseMip + 1 );

        VkImageMemoryBarrier imageMemoryBarrier;
        imageMemoryBarrier.sType = VK_STRUCTURE_TYPE_IMAGE_MEMORY_BARRIER;
        imageMemoryBarrier.pNext = NULL;
        imageMemoryBarrier.oldLayout = VK_IMAGE_LAYOUT_UNDEFINED;
        imageMemoryBarrier.newLayout = requiredLayout;
        imageMemoryBarrier.srcAccessMask = 0;
        imageMemoryBarrier.dstAccessMask = Vk::GetLayoutAccessFlags( requiredLayout );
        imageMemoryBarrier.srcQueueFamilyIndex = VK_QUEUE_FAMILY_IGNORED;
        imageMemoryBarrier.dstQueueFamilyIndex = VK_QUEUE_FAMILY_IGNORED;
        imageMemoryBarrier.image = m_ImageHandle;
        imageMemoryBarrier.subresourceRange.aspectMask = imageAspectFlags == 0 ? m_ImageAspectFlags : imageAspectFlags;
        imageMemoryBarrier.subresourceRange.baseArrayLayer = 0;
        imageMemoryBarrier.subresourceRange.layerCount = 0;
        imageMemoryBarrier.subresourceRange.baseMipLevel = 0;
        imageMemoryBarrier.subresourceRange.levelCount = 0;

        bool isDirty = false;
        for ( uint32_t layer = baseLayer; layer < baseLayer + layerCount; layer++ )
        {
            for ( uint32_t mip = baseMip; mip < baseMip + mipCount; mip++ )
            {
                VkImageLayout oldLayout = GetImageLayout( layer, mip );
                if ( isDirty )
                {
                    if ( imageMemoryBarrier.oldLayout != oldLayout )
                    {
                        // The old layout changed so the barrier must be flush.
                        NN_SDK_ASSERT_LESS( barrierCount, maxLayout );
                        imageMemoryBarrier.subresourceRange.layerCount++;
                        pImageBarriers[ barrierCount ] = imageMemoryBarrier;
                        barrierCount++;
                        isDirty = false;
                    }
                    else
                    {
                        // Coalesce transitions in the same level
                        imageMemoryBarrier.subresourceRange.levelCount++;

                        // Update tracking information
                        m_pCurrentLayout[ layer * m_Mips + mip ] = requiredLayout;
                    }
                }

                if ( !isDirty && oldLayout != requiredLayout )
                {
                    imageMemoryBarrier.subresourceRange.baseMipLevel = mip;
                    imageMemoryBarrier.subresourceRange.levelCount = 1;
                    imageMemoryBarrier.subresourceRange.baseArrayLayer = layer;
                    imageMemoryBarrier.subresourceRange.layerCount = 0;
                    imageMemoryBarrier.oldLayout = oldLayout;
                    imageMemoryBarrier.srcAccessMask = Vk::GetLayoutAccessFlags( oldLayout );

                    // Update tracking information
                    m_pCurrentLayout[ layer * m_Mips + mip ] = requiredLayout;

                    isDirty = true;
                }
            }

            // Perform each layer separately.
            // TODO: This could be optimized further to do multiple levels.
            if ( isDirty )
            {
                // Flush
                NN_SDK_ASSERT_LESS( barrierCount, maxLayout );
                imageMemoryBarrier.subresourceRange.layerCount++;
                pImageBarriers[ barrierCount ] = imageMemoryBarrier;
                barrierCount++;
                isDirty = false;
            }
        }

        // Update used slots
        *pLayoutCount = barrierCount;

        // FIXME: This should be made to be more accurate based on actual usage.
        *pSrcStages |= VK_PIPELINE_STAGE_ALL_COMMANDS_BIT;
        *pDstStages |= VK_PIPELINE_STAGE_ALL_COMMANDS_BIT;
    }

    void ResetImageLayout( VkImageLayout layout ) NN_NOEXCEPT
    {
        for ( uint32_t idxLayout = 0; idxLayout < m_Mips * m_Layers; idxLayout++ )
        {
            m_pCurrentLayout[ idxLayout ] = layout;
        }
    }

    void SetImageLayout( VkImageLayout layout, const VkImageSubresourceRange& range ) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_LESS_EQUAL( range.baseArrayLayer + range.layerCount, m_Layers );
        NN_SDK_ASSERT_LESS_EQUAL( range.baseMipLevel + range.levelCount, m_Mips );
        for ( uint32_t layer = range.baseArrayLayer; layer < range.layerCount; layer++ )
        {
            for ( uint32_t mip = range.baseMipLevel; mip < range.levelCount; mip++ )
            {
                m_pCurrentLayout[ layer * m_Mips + mip ] = layout;
            }
        }
    }

    void SetImageLayout( VkImageLayout layout, uint32_t layer, uint32_t mip ) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_LESS_EQUAL( layer, m_Layers );
        NN_SDK_ASSERT_LESS_EQUAL( mip, m_Mips );
        m_pCurrentLayout[ layer * m_Mips + mip ] = layout;
    }

    void SetImageAspectFlags( VkImageAspectFlags flags ) NN_NOEXCEPT
    {
        m_ImageAspectFlags = flags;
    }

    void SetImageHandle( VkImage handle ) NN_NOEXCEPT
    {
        m_ImageHandle = handle;
    }

    ReferenceCount& GetReferenceCount() NN_NOEXCEPT
    {
        return m_ReferenceCount;
    }

    ImageViewCache< TextureViewInfoData >& GetTextureView() NN_NOEXCEPT
    {
        return m_TextureView;
    }

    ImageViewCache< ColorTargetViewInfoData >& GetColorTargetView() NN_NOEXCEPT
    {
        return m_ColorTargetView;
    }

    ImageViewCache< DepthStencilViewInfoData >& GetDepthStencilView() NN_NOEXCEPT
    {
        return m_DepthStencilView;
    }

    VkImage GetImageHandle() const NN_NOEXCEPT
    {
        return m_ImageHandle;
    }

    VkImageLayout GetCurrentLayout( uint32_t layer, uint32_t mip ) const NN_NOEXCEPT
    {
        return m_pCurrentLayout[ layer * m_Mips + mip ];
    }

    VkImageAspectFlags GetImageAspectFlags() const NN_NOEXCEPT
    {
        return m_ImageAspectFlags;
    }

    uint32_t GetLayerCount() const NN_NOEXCEPT
    {
        return m_Layers;
    }

    uint32_t GetMipCount() const NN_NOEXCEPT
    {
        return m_Mips;
    }

private:

    VkImageLayout GetImageLayout(uint32_t layers, uint32_t mip) const NN_NOEXCEPT
    {
        return m_pCurrentLayout[ layers * m_Mips + mip ];
    }

    ImageViewCache< TextureViewInfoData > m_TextureView;
    ImageViewCache< ColorTargetViewInfoData > m_ColorTargetView;
    ImageViewCache< DepthStencilViewInfoData > m_DepthStencilView;
    ReferenceCount m_ReferenceCount;
    VkImage m_ImageHandle;
    VkImageAspectFlags m_ImageAspectFlags;
    VkImageLayout* m_pCurrentLayout;
    uint32_t m_Layers;
    uint32_t m_Mips;
};

}
}
}
