﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/nn_SdkLog.h>
#include <nn/util/util_BytePtr.h>

#include <nn/gfx/gfx_TextureInfo.h>

#include <nn/gfx/detail/gfx_Texture-api.vk.1.h>
#include <nn/gfx/detail/gfx_MemoryPool-api.vk.1.h>

#include "gfx_CommonHelper.h"
#include "gfx_VkHelper.h"

namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationVk1 Target;

namespace
{

size_t CalculateMipDataOffsetSize( ptrdiff_t* pMipOffsets, const TextureInfo& info ) NN_NOEXCEPT
{
    ImageStorageDimension imageStorageDimension = info.GetImageStorageDimension();
    NN_SDK_ASSERT_NOT_EQUAL( imageStorageDimension, ImageStorageDimension_Undefined );
    int arrayLength = info.GetArrayLength();

    int width = info.GetWidth();
    int height = ( imageStorageDimension == ImageStorageDimension_1d
        && arrayLength > 1 ) ? arrayLength : info.GetHeight();
    int depth = ( imageStorageDimension == ImageStorageDimension_2d
        && arrayLength > 1 ) ? arrayLength : info.GetDepth();

    int minHeight = 1;
    int minDepth = 1;
    if ( arrayLength > 1 )
    {
        if ( imageStorageDimension == ImageStorageDimension_1d )
        {
            minHeight = height;
        }
        else if ( imageStorageDimension == ImageStorageDimension_2d )
        {
            minDepth = depth;
        }
    }

    ChannelFormat channelFormat = GetChannelFormat( info.GetImageFormat() );
    size_t size = 0;
    for ( size_t mipLevel = 0; mipLevel < static_cast< size_t >( info.GetMipCount() ); ++mipLevel )
    {
        if ( pMipOffsets )
        {
            pMipOffsets[ mipLevel ] = size;
        }
        size += CalculateImageSize( channelFormat,
            std::max NN_PREVENT_MACRO_FUNC( width >> mipLevel, 1 ),
            std::max NN_PREVENT_MACRO_FUNC( height >> mipLevel, minHeight ),
            std::max NN_PREVENT_MACRO_FUNC( depth >> mipLevel, minDepth ) );
    }
    return size;
}

}
size_t TextureImpl< Target >::CalculateMipDataAlignment( DeviceImpl< Target >*, const InfoType& ) NN_NOEXCEPT
{
    return 1;
}

size_t TextureImpl< Target >::CalculateMipDataSize( DeviceImpl< Target >*, const InfoType& info ) NN_NOEXCEPT
{
    return CalculateMipDataOffsetSize( NULL, info );
}

void TextureImpl< Target >::CalculateMipDataOffsets(
    ptrdiff_t* pMipOffsets, DeviceImpl< Target >*, const InfoType& info ) NN_NOEXCEPT
{
    CalculateMipDataOffsetSize( pMipOffsets, info );
}

size_t TextureImpl< Target >::GetRowPitch( DeviceImpl< Target >*, const InfoType& info ) NN_NOEXCEPT
{
    return CalculateRowSize( info.GetWidth(), GetChannelFormat( info.GetImageFormat() ) );
}

TextureImpl< Target >::TextureImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

TextureImpl< Target >::~TextureImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized || this->flags.GetBit( Flag_Shared ) );
}

void TextureImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info,
    MemoryPoolImpl< Target >* pMemoryPool, ptrdiff_t memoryPoolOffset, size_t memoryPoolSize ) NN_NOEXCEPT
{
    bool doMap = false;
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_SDK_REQUIRES( !pMemoryPool || memoryPoolOffset >= 0 );
    // for agl
    //NN_SDK_ASSERT( !pMemoryPool || pMemoryPool->ToData()->pMemory );
    NN_SDK_REQUIRES( !pMemoryPool || memoryPoolSize >= CalculateMipDataSize( pDevice, info ) );
    NN_UNUSED( memoryPoolSize );

    // save for DepthStencilView::Initialize()
    this->format = info.GetImageFormat();
    this->width = info.GetWidth();
    this->height = info.GetImageStorageDimension() == ImageStorageDimension_1d ? 1 : info.GetHeight();
    this->depth = info.GetImageStorageDimension() == ImageStorageDimension_3d ? info.GetDepth() : 1;
    this->samples = info.GetMultisampleCount() ? info.GetMultisampleCount() : 1;
    this->pMemory.ptr = pMemoryPool ? pMemoryPool->ToData()->pMemory.ptr : 0;

    VkResult result;
    VkImageCreateInfo vkInfo;
    vkInfo.sType = VK_STRUCTURE_TYPE_IMAGE_CREATE_INFO;
    vkInfo.pNext = NULL;
    vkInfo.flags = IsCompressedFormat(
        GetChannelFormat( info.GetImageFormat() ) ) ? 0 : VK_IMAGE_CREATE_MUTABLE_FORMAT_BIT;
    vkInfo.imageType = Vk::GetImageType( info.GetImageStorageDimension() );
    vkInfo.format = Vk::GetImageFormat( info.GetImageFormat() );
    vkInfo.extent.width = info.GetWidth();
    vkInfo.extent.height = info.GetImageStorageDimension() == ImageStorageDimension_1d ? 1 : info.GetHeight();
    vkInfo.extent.depth = info.GetImageStorageDimension() == ImageStorageDimension_3d ? info.GetDepth() : 1;
    vkInfo.mipLevels = info.GetMipCount() == 0 ? 1 : info.GetMipCount();
    vkInfo.arrayLayers = info.GetArrayLength() == 0 ? 1 : info.GetArrayLength();
    vkInfo.samples = static_cast< VkSampleCountFlagBits >( this->samples );
    vkInfo.tiling = VK_IMAGE_TILING_OPTIMAL;
    vkInfo.usage = Vk::GetImageUsageFlags( info.GetGpuAccessFlags() ) | VK_IMAGE_USAGE_TRANSFER_SRC_BIT;
    vkInfo.sharingMode = VK_SHARING_MODE_EXCLUSIVE;
    vkInfo.queueFamilyIndexCount = 0;
    vkInfo.pQueueFamilyIndices = NULL;
    vkInfo.initialLayout = VK_IMAGE_LAYOUT_PREINITIALIZED;

    if ( info.GetImageStorageDimension() == ImageStorageDimension_2d
        && info.GetWidth() == info.GetHeight()
        && info.GetArrayLength() >= 6
        && info.GetArrayLength() % 6 == 0 )
    {
        vkInfo.flags |= VK_IMAGE_CREATE_CUBE_COMPATIBLE_BIT;
    }

    if ( info.GetImageStorageDimension() == ImageStorageDimension_3d
        && pDevice->ToData()->availableExtensions.GetBit( VkDeviceExtension_KhrMaintenance1 ) )
    {
        vkInfo.flags |= VK_IMAGE_CREATE_2D_ARRAY_COMPATIBLE_BIT_KHR;
    }

    if ( IsCompressedFormat( GetChannelFormat( info.GetImageFormat() ) )
        && pDevice->ToData()->availableExtensions.GetBit( VkDeviceExtension_KhrMaintenance2 ) )
    {
        vkInfo.flags |= VK_IMAGE_CREATE_MUTABLE_FORMAT_BIT | VK_IMAGE_CREATE_BLOCK_TEXEL_VIEW_COMPATIBLE_BIT;
    }

    VkFormatProperties formatProperties = { 0 };
    VkPhysicalDevice physicalDevice = CastToVkDispatchableObject< VkPhysicalDevice >( pDevice->ToData()->hPhysicalDevice );

    NN_GFX_CALL_VK_FUNCTION( vkGetPhysicalDeviceFormatProperties( physicalDevice, vkInfo.format,
        &formatProperties ) );

    VkFormatFeatureFlags featureFlags = Vk::GetFormatFeatureFlagsFromGpuFlags( info.GetGpuAccessFlags() );

    #if 0   // 下記をコメントアウトすることで、強制的にOptimalにします。
    VkImageFormatProperties imageFormatProperties;
    if ( info.GetTileMode() == nn::gfx::TileMode_Linear
        && ( formatProperties.linearTilingFeatures & featureFlags ) == featureFlags )
    {
        NN_GFX_CALL_VK_FUNCTION( result = vkGetPhysicalDeviceImageFormatProperties(
            physicalDevice, vkInfo.format, vkInfo.imageType,
            VK_IMAGE_TILING_LINEAR, vkInfo.usage, vkInfo.flags, &imageFormatProperties ) );

        // FIXME: Although 980Ti claims to support linear BC3 Unorm it will fail.
        if ( result == VK_SUCCESS &&
            static_cast< uint32_t >( info.GetMipCount() ) <= imageFormatProperties.maxMipLevels
            && static_cast< uint32_t >( info.GetArrayLength() ) <= imageFormatProperties.maxArrayLayers )
        {
            doMap = true;
            vkInfo.tiling = VK_IMAGE_TILING_LINEAR;
        }
    }
    #else
    #if defined( NN_DETAIL_ENABLE_SDK_ASSERT )
    VkImageFormatProperties imageFormatProperties;
    #endif
    #endif

    // Attempt to use a different format if the HW doesn't support it.
    if ( vkInfo.tiling == VK_IMAGE_TILING_OPTIMAL &&
        ( formatProperties.optimalTilingFeatures & featureFlags ) == 0 )
    {
        switch ( this->format )
        {
            case ImageFormat_D24_Unorm_S8_Uint:
            {
                this->format = ImageFormat_D32_Float_S8_Uint_X24;
                vkInfo.format = Vk::GetImageFormat( this->format );
                NN_GFX_CALL_VK_FUNCTION( vkGetPhysicalDeviceFormatProperties( physicalDevice, vkInfo.format,
                    &formatProperties ) );
                break;
            }
            default: break;
        }

    }

    // Indicates that this format is not supported as linear or tiled on this system.
    NN_SDK_ASSERT( vkInfo.tiling != VK_IMAGE_TILING_OPTIMAL
        || ( formatProperties.optimalTilingFeatures & featureFlags ) == featureFlags );
    NN_SDK_ASSERT( VK_SUCCESS == vkGetPhysicalDeviceImageFormatProperties(
        physicalDevice, vkInfo.format, vkInfo.imageType,
        VK_IMAGE_TILING_OPTIMAL, vkInfo.usage, vkInfo.flags, &imageFormatProperties ) );

    if ( vkInfo.tiling == VK_IMAGE_TILING_OPTIMAL )
    {
        vkInfo.usage |= VK_IMAGE_USAGE_TRANSFER_DST_BIT;
        vkInfo.initialLayout = VK_IMAGE_LAYOUT_UNDEFINED;
    }

    bool isNvDedicatedAllocationEnabled = pDevice->ToData()->availableExtensions.GetBit( VkDeviceExtension_NvDedicatedAllocation );
    VkDedicatedAllocationImageCreateInfoNV dedicatedImageInfo;
    if ( isNvDedicatedAllocationEnabled )
    {
        dedicatedImageInfo.sType = VK_STRUCTURE_TYPE_DEDICATED_ALLOCATION_IMAGE_CREATE_INFO_NV;
        dedicatedImageInfo.pNext = NULL;
        dedicatedImageInfo.dedicatedAllocation = VK_TRUE;
        vkInfo.pNext = &dedicatedImageInfo;
    }

    VkDevice device = CastToVkDispatchableObject< VkDevice >( pDevice->ToData()->hDevice );
    VkImage vkTextureImage;
    NN_GFX_CALL_VK_FUNCTION( result = vkCreateImage( device, &vkInfo,
        pDevice->ToData()->pAllocationCallback, &vkTextureImage ) );
    NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );
    this->hImage = CastFromVkNonDispatchableObject< VkImage >( vkTextureImage );

    // Allocate memory
    VkMemoryRequirements vkMemoryRequirements = {};
    NN_GFX_CALL_VK_FUNCTION( vkGetImageMemoryRequirements( device, vkTextureImage, &vkMemoryRequirements ) );

    VkMemoryAllocateInfo allocInfo = {};
    allocInfo.sType = VK_STRUCTURE_TYPE_MEMORY_ALLOCATE_INFO;
    allocInfo.pNext = NULL;
    allocInfo.memoryTypeIndex = 0;
    allocInfo.allocationSize = vkMemoryRequirements.size;

    VkDedicatedAllocationMemoryAllocateInfoNV dedicatedMemoryInfo;
    if ( isNvDedicatedAllocationEnabled )
    {
        dedicatedMemoryInfo.sType = VK_STRUCTURE_TYPE_DEDICATED_ALLOCATION_MEMORY_ALLOCATE_INFO_NV;
        dedicatedMemoryInfo.pNext = NULL;
        dedicatedMemoryInfo.image = vkTextureImage;
        dedicatedMemoryInfo.buffer = VK_NULL_HANDLE;
        allocInfo.pNext = &dedicatedMemoryInfo;
    }

    result = Vk::SelectMemoryType( &allocInfo.memoryTypeIndex, pDevice->ToData()->memoryTypeCount, pDevice->ToData()->memoryPropertyFlags,
        vkMemoryRequirements.memoryTypeBits, doMap ?
        ( VK_MEMORY_PROPERTY_DEVICE_LOCAL_BIT | VK_MEMORY_PROPERTY_HOST_VISIBLE_BIT | VK_MEMORY_PROPERTY_HOST_COHERENT_BIT )
        : VK_MEMORY_PROPERTY_DEVICE_LOCAL_BIT );
    if ( result != VK_SUCCESS )
    {
        // Try non-device local
        result = Vk::SelectMemoryType( &allocInfo.memoryTypeIndex, pDevice->ToData()->memoryTypeCount, pDevice->ToData()->memoryPropertyFlags,
            vkMemoryRequirements.memoryTypeBits, doMap ?
            ( VK_MEMORY_PROPERTY_HOST_VISIBLE_BIT | VK_MEMORY_PROPERTY_HOST_COHERENT_BIT ) : 0 );
    }
    NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );

    VkDeviceMemory vkTextureMemory;
    NN_GFX_CALL_VK_FUNCTION( result = vkAllocateMemory( device, &allocInfo,
        pDevice->ToData()->pAllocationCallback, &vkTextureMemory ) );

    // Try non-device local if we're out of device local memory
    if ( result != VK_SUCCESS )
    {
        result = Vk::SelectMemoryType( &allocInfo.memoryTypeIndex, pDevice->ToData()->memoryTypeCount, pDevice->ToData()->memoryPropertyFlags,
            vkMemoryRequirements.memoryTypeBits, doMap ?
            ( VK_MEMORY_PROPERTY_HOST_VISIBLE_BIT | VK_MEMORY_PROPERTY_HOST_COHERENT_BIT ) : 0 );
        NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );
        NN_GFX_CALL_VK_FUNCTION( result = vkAllocateMemory( device, &allocInfo,
            pDevice->ToData()->pAllocationCallback, &vkTextureMemory ) );
    }
    NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );
    this->hMemory = CastFromVkNonDispatchableObject< VkDeviceMemory >( vkTextureMemory );

    NN_GFX_CALL_VK_FUNCTION( result = vkBindImageMemory( device, vkTextureImage, vkTextureMemory, 0 ) );
    NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );

    if ( doMap && this->samples == 1 && this->pMemory.ptr )
    {
        void* pMapData = NULL;
        NN_GFX_CALL_VK_FUNCTION( result = vkMapMemory( device, vkTextureMemory, 0,
            vkMemoryRequirements.size, 0, ( void** )&pMapData ) );
        NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );

        uint32_t minHeight = ( info.GetImageStorageDimension() == ImageStorageDimension_1d ) ? height : 1;
        uint32_t minDepth = ( info.GetImageStorageDimension() == ImageStorageDimension_2d ) ? depth : 1;
        nn::util::BytePtr addr = pMemoryPool->ToData()->pMemory;
        addr.Advance( memoryPoolOffset );
        for ( size_t mipLevel = 0; mipLevel < vkInfo.mipLevels; ++mipLevel )
        {
            // FIXME: arrayLevel may not be correct for depth textures
            for ( uint32_t arrayLevel = 0; arrayLevel < vkInfo.arrayLayers; ++arrayLevel )
            {
                size_t size = Vk::CopyCpuMemoryToTexture( pMapData, device, vkTextureImage, info.GetImageFormat(),
                    static_cast< uint32_t >( arrayLevel ), static_cast< uint32_t >( mipLevel ),
                    static_cast< uint32_t >( std::max NN_PREVENT_MACRO_FUNC( width >> mipLevel, static_cast< uint32_t >( 1 ) ) ),
                    static_cast< uint32_t >( std::max NN_PREVENT_MACRO_FUNC( height >> mipLevel, minHeight ) ),
                    static_cast< uint32_t >( std::max NN_PREVENT_MACRO_FUNC( depth >> mipLevel, minDepth ) ),
                    addr.Get() );
                addr.Advance( size );
            }
        }
        NN_GFX_CALL_VK_FUNCTION( vkUnmapMemory( device, vkTextureMemory ) );
    }
    else if ( vkInfo.tiling == VK_IMAGE_TILING_OPTIMAL && !Vk::IsDepthFormat( Vk::GetImageFormat( info.GetImageFormat() ) ) && pMemoryPool && this->pMemory.ptr
        && this->samples == 1 )
    {
        uint32_t minHeight = ( info.GetImageStorageDimension() == ImageStorageDimension_1d ) ? height : 1;
        uint32_t minDepth = ( info.GetImageStorageDimension() == ImageStorageDimension_2d ) ? depth : 1;
        nn::util::BytePtr addr = pMemoryPool->ToData()->pMemory;
        addr.Advance( memoryPoolOffset );
        for ( size_t mipLevel = 0; mipLevel < vkInfo.mipLevels; ++mipLevel )
        {
            // FIXME: arrayLevel may not be correct for depth textures
            for ( uint32_t arrayLevel = 0; arrayLevel < vkInfo.arrayLayers; ++arrayLevel )
            {
                size_t size = Vk::InitializeOptimalTexture( pDevice, vkTextureImage, info.GetImageFormat(),
                    static_cast< uint32_t >( arrayLevel ),
                    static_cast< uint32_t >( mipLevel ),
                    static_cast< uint32_t >( std::max NN_PREVENT_MACRO_FUNC( width >> mipLevel, static_cast< uint32_t >( 1 ) ) ),
                    static_cast< uint32_t >( std::max NN_PREVENT_MACRO_FUNC( height >> mipLevel, minHeight ) ),
                    static_cast< uint32_t >( std::max NN_PREVENT_MACRO_FUNC( depth >> mipLevel, minDepth ) ),
                    addr.Get() );
                addr.Advance( size );
            }
        }
    }
    else
    {
        Vk::TransitionTextureLayout( pDevice, vkTextureImage, info.GetImageFormat(),
            VK_IMAGE_LAYOUT_UNDEFINED, 0, Vk::LayoutCopyImageDst, VK_ACCESS_TRANSFER_WRITE_BIT );
    }

    this->layerCount = info.GetArrayLength() ? info.GetArrayLength() : 1;
    this->levelCount = info.GetMipCount() ? info.GetMipCount() : 1;
    this->gpuAccessFlag = info.GetGpuAccessFlags();
    this->memorySize = static_cast< uint32_t >( vkMemoryRequirements.size );

    this->flags.SetBit( Flag_Shared, false );
    this->state = State_Initialized;

    this->pManager = Vk::AllocDriverMemory( sizeof( TextureManageData ), 8 );
    new( this->pManager.ptr ) TextureManageData( info.GetImageStorageDimension() == ImageStorageDimension_3d ? this->depth : this->layerCount, this->levelCount );
    if ( Vk::IsLayoutManagementEnabled() )
    {
        TextureManageData* p_Manager = this->pManager;
        p_Manager->SetImageHandle( vkTextureImage );
        p_Manager->ResetImageLayout( Vk::LayoutCopyImageDst );
        p_Manager->SetImageAspectFlags( Vk::GetImageAspect( info.GetImageFormat() ) );
    }

}//NOLINT(impl/function_size)

void TextureImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsInitialized( *this ) );
    NN_SDK_ASSERT( !this->flags.GetBit( Flag_Shared ) );
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( IsInitialized( *pDevice ) );

    if ( this->pManager.ptr != NULL )
    {
        TextureManageData* pTextureManageData = static_cast< TextureManageData* >( this->pManager );
        pTextureManageData->GetReferenceCount().Decrement();
        if ( pTextureManageData->GetReferenceCount().IsZero() )
        {
            // Register image, color target view, and depth stencil view objects
            // so that they're destroyed after clearing framebuffer cache referring to these objects.
            Vk::RegisterDestroyingImageObject( pDevice, this );

            // Only texture view objects are destroyed immediately in DestroyAllImageView.
            pTextureManageData->DestroyAllImageView( pDevice );
            pTextureManageData->~TextureManageData();
            Vk::FreeDriverMemory( pTextureManageData );
            this->pManager = 0;
        }
    }

    this->state = State_NotInitialized;
}

void TextureImpl< Target >::Vk1Copy( const TextureImpl< ApiVariationVk1 >& srcTexture ) NN_NOEXCEPT
{
    DataType& dst = this->ToData();
    dst = srcTexture.ToData();
    this->flags.SetBit( Flag_Shared, true );

    if ( this->pManager.ptr != NULL )
    {
        TextureManageData* pTextureManageData = static_cast< TextureManageData* >( this->pManager );
        pTextureManageData->GetReferenceCount().Increment();
    }
}

TextureViewImpl< Target >::TextureViewImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

TextureViewImpl< Target >::~TextureViewImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized || this->flags.GetBit( Flag_Shared ) );
}

void TextureViewImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( IsInitialized( *pDevice ) );

    const TextureImpl< Target >* pSourceTexture = info.GetTexturePtr();
    NN_SDK_ASSERT_NOT_NULL( pSourceTexture );
    NN_SDK_ASSERT( IsInitialized( *pSourceTexture ) );
    NN_SDK_ASSERT( pSourceTexture->ToData()->gpuAccessFlag & GpuAccess_Texture );

    TextureManageData* textureManageData = static_cast< TextureManageData* >( pSourceTexture->ToData()->pManager );
    NN_SDK_ASSERT( textureManageData != NULL );
    VkImageView cachedImageView = textureManageData->GetTextureView().GetFromCache( info.ToData() );
    if ( cachedImageView != VK_NULL_HANDLE )
    {
        this->hImageView = CastFromVkNonDispatchableObject< VkImageView >( cachedImageView );
        this->format = Vk::GetImageFormat( info.GetImageFormat() );
    }
    else
    {
        VkDevice device = CastToVkDispatchableObject< VkDevice >( pDevice->ToData()->hDevice );

        VkImageViewCreateInfo vkInfo = {};
        vkInfo.sType = VK_STRUCTURE_TYPE_IMAGE_VIEW_CREATE_INFO;
        vkInfo.pNext = NULL;
        vkInfo.flags = 0;
        vkInfo.image = CastToVkNonDispatchableObject< VkImage >( pSourceTexture->ToData()->hImage );
        vkInfo.viewType = Vk::GetImageTarget( info.GetImageDimension() );
        vkInfo.format = Vk::GetImageFormat( info.GetImageFormat() );
        vkInfo.components.r = Vk::GetComponentSwizzle( info.GetChannelMapping( nn::gfx::ColorChannel_Red ) );
        vkInfo.components.g = Vk::GetComponentSwizzle( info.GetChannelMapping( nn::gfx::ColorChannel_Green ) );
        vkInfo.components.b = Vk::GetComponentSwizzle( info.GetChannelMapping( nn::gfx::ColorChannel_Blue ) );
        vkInfo.components.a = Vk::GetComponentSwizzle( info.GetChannelMapping( nn::gfx::ColorChannel_Alpha ) );
        vkInfo.subresourceRange.aspectMask = Vk::IsDepthFormat( vkInfo.format )
            ? info.GetDepthStencilTextureMode() == DepthStencilFetchMode_DepthComponent
            ? VK_IMAGE_ASPECT_DEPTH_BIT : VK_IMAGE_ASPECT_STENCIL_BIT : VK_IMAGE_ASPECT_COLOR_BIT;
        vkInfo.subresourceRange.baseMipLevel = info.GetSubresourceRange().GetMipRange().GetMinMipLevel();
        vkInfo.subresourceRange.levelCount = info.GetSubresourceRange().GetMipRange().GetMipCount() ?
            info.GetSubresourceRange().GetMipRange().GetMipCount() : 1;
        vkInfo.subresourceRange.baseArrayLayer = info.GetSubresourceRange().GetArrayRange().GetBaseArrayIndex();
        vkInfo.subresourceRange.layerCount = info.GetSubresourceRange().GetArrayRange().GetArrayLength() ?
            info.GetSubresourceRange().GetArrayRange().GetArrayLength() : 1;
        NN_SDK_ASSERT( vkInfo.viewType != VK_IMAGE_VIEW_TYPE_CUBE_ARRAY
            || pDevice->ToData()->supportedFeaturesBitArray.test( VkDeviceFeature_ImageCubeArray ) );

        VkResult result;
        VkImageView imageView;
        NN_GFX_CALL_VK_FUNCTION( result = vkCreateImageView( device, &vkInfo,
            pDevice->ToData()->pAllocationCallback, &imageView ) );
        NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );
        NN_UNUSED( result );

        this->hImageView = CastFromVkNonDispatchableObject< VkImageView >( imageView );
        this->format = vkInfo.format;

        textureManageData->GetTextureView().AddCache( info.ToData(), imageView );
    }
    this->pManager = const_cast< void* >( reinterpret_cast< const void* >( pSourceTexture ) );

    this->hImage = pSourceTexture->ToData()->hImage;
    this->hMemory = pSourceTexture->ToData()->hMemory;
    this->baseMipLevel = info.GetSubresourceRange().GetMipRange().GetMinMipLevel();
    this->levelCount = info.GetSubresourceRange().GetMipRange().GetMipCount() ?
        info.GetSubresourceRange().GetMipRange().GetMipCount() : 1;
    this->baseArrayLayer = info.GetSubresourceRange().GetArrayRange().GetBaseArrayIndex();
    this->layerCount = info.GetSubresourceRange().GetArrayRange().GetArrayLength() ?
        info.GetSubresourceRange().GetArrayRange().GetArrayLength() : 1;
    this->height = pSourceTexture->ToData()->height;
    this->depth = pSourceTexture->ToData()->depth;
    this->samples = pSourceTexture->ToData()->samples;
    this->gpuAccessFlag = pSourceTexture->ToData()->gpuAccessFlag;
    this->originalFormat = pSourceTexture->ToData()->format;
    this->flags.SetBit( Flag_Shared, false );
    this->state = State_Initialized;
}

void TextureViewImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_SDK_ASSERT( !this->flags.GetBit( Flag_Shared ) );
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( IsInitialized( *pDevice ) );

    NN_UNUSED( pDevice );
    // Do nothing here. VkImageView will be destroyed when the image referenced by this is destroyed.

    this->state = State_NotInitialized;
}

ColorTargetViewImpl< Target >::ColorTargetViewImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

ColorTargetViewImpl< Target >::~ColorTargetViewImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized || this->flags.GetBit( Flag_Shared ) );
}

void ColorTargetViewImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( IsInitialized( *pDevice ) );

    const TextureImpl< Target >* pSourceTexture = info.GetTexturePtr();
    NN_SDK_ASSERT_NOT_NULL( pSourceTexture );
    NN_SDK_ASSERT( IsInitialized( *pSourceTexture ) );

    TextureManageData* textureManageData = static_cast< TextureManageData* >( pSourceTexture->ToData()->pManager );
    NN_SDK_ASSERT( textureManageData != NULL );
    VkImageView cachedImageView = textureManageData->GetColorTargetView().GetFromCache( info.ToData() );
    if ( cachedImageView != VK_NULL_HANDLE )
    {
        this->hImageView = CastFromVkNonDispatchableObject< VkImageView >( cachedImageView );
        this->format = Vk::GetImageFormat( info.GetImageFormat() );
    }
    else
    {
        VkDevice device = CastToVkDispatchableObject< VkDevice >( pDevice->ToData()->hDevice );

        VkImageViewCreateInfo vkInfo;
        memset( &vkInfo, 0, sizeof( vkInfo ) );
        vkInfo.sType = VK_STRUCTURE_TYPE_IMAGE_VIEW_CREATE_INFO;
        vkInfo.pNext = NULL;
        vkInfo.flags = 0;
        vkInfo.image = CastToVkNonDispatchableObject< VkImage >( pSourceTexture->ToData()->hImage );
        vkInfo.viewType = Vk::GetImageTarget( info.GetImageDimension() );
        vkInfo.format = Vk::GetImageFormat( info.GetImageFormat() );
        vkInfo.components.r = VK_COMPONENT_SWIZZLE_R;
        vkInfo.components.g = VK_COMPONENT_SWIZZLE_G;
        vkInfo.components.b = VK_COMPONENT_SWIZZLE_B;
        vkInfo.components.a = VK_COMPONENT_SWIZZLE_A;
        vkInfo.subresourceRange.aspectMask = VK_IMAGE_ASPECT_COLOR_BIT;
        vkInfo.subresourceRange.baseMipLevel = info.GetMipLevel();
        vkInfo.subresourceRange.levelCount = 1;
        vkInfo.subresourceRange.baseArrayLayer = info.GetArrayRange().GetBaseArrayIndex();
        vkInfo.subresourceRange.layerCount = info.GetArrayRange().GetArrayLength();
        VkImageView imageView;
        VkResult result;
        NN_GFX_CALL_VK_FUNCTION( result = vkCreateImageView( device, &vkInfo,
            pDevice->ToData()->pAllocationCallback, &imageView ) );
        NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );

        this->hImageView = CastFromVkNonDispatchableObject< VkImageView >( imageView );
        this->format = vkInfo.format;

        textureManageData->GetColorTargetView().AddCache( info.ToData(), imageView );
    }
    this->pManager = const_cast< void* >( reinterpret_cast< const void* >( pSourceTexture ) );

    this->layerCount = pSourceTexture->ToData()->layerCount;
    this->levelCount = pSourceTexture->ToData()->levelCount;
    this->targetMipLevel = info.GetMipLevel();
    this->targetBaseArray = info.GetArrayRange().GetBaseArrayIndex();
    this->targetArrayCount = info.GetArrayRange().GetArrayLength();
    this->width = pSourceTexture->ToData()->width;
    this->height = pSourceTexture->ToData()->height;
    this->depth = pSourceTexture->ToData()->depth;
    this->samples = pSourceTexture->ToData()->samples;
    this->flags.SetBit( Flag_Shared, false );
    this->hImage = pSourceTexture->ToData()->hImage;
    this->hMemory = pSourceTexture->ToData()->hMemory;
    this->memorySize = pSourceTexture->ToData()->memorySize;
    this->originalFormat = pSourceTexture->ToData()->format;
    this->state = State_Initialized;
}

void ColorTargetViewImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_SDK_ASSERT( !this->flags.GetBit( Flag_Shared ) );
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( IsInitialized( *pDevice ) );

    NN_UNUSED( pDevice );
    // Do nothing here. VkImageView will be destroyed when the image referenced by this is destroyed.

    this->state = State_NotInitialized;
}

DepthStencilViewImpl< Target >::DepthStencilViewImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

DepthStencilViewImpl< Target >::~DepthStencilViewImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized || this->flags.GetBit( Flag_Shared ) );
}

void DepthStencilViewImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( IsInitialized( *pDevice ) );

    const TextureImpl< Target >* pSourceTexture = info.GetTexturePtr();
    NN_SDK_ASSERT_NOT_NULL( pSourceTexture );
    NN_SDK_ASSERT( IsInitialized( *pSourceTexture ) );
    this->hImage = pSourceTexture->ToData()->hImage;

    TextureManageData* textureManageData = static_cast< TextureManageData* >( pSourceTexture->ToData()->pManager );
    NN_SDK_ASSERT( textureManageData != NULL );
    VkImageView cachedImageView = textureManageData->GetDepthStencilView().GetFromCache( info.ToData() );
    if ( cachedImageView != VK_NULL_HANDLE )
    {
        this->hImageView = CastFromVkNonDispatchableObject< VkImageView >( cachedImageView );
        this->format = Vk::GetImageFormat( pSourceTexture->ToData()->format );
    }
    else
    {
        VkDevice device = CastToVkDispatchableObject< VkDevice >( pDevice->ToData()->hDevice );

        VkImageViewCreateInfo vkInfo = { };
        vkInfo.sType = VK_STRUCTURE_TYPE_IMAGE_VIEW_CREATE_INFO;
        vkInfo.pNext = NULL;
        vkInfo.flags = 0;
        vkInfo.components.r = VK_COMPONENT_SWIZZLE_R;
        vkInfo.components.g = VK_COMPONENT_SWIZZLE_G;
        vkInfo.components.b = VK_COMPONENT_SWIZZLE_B;
        vkInfo.components.a = VK_COMPONENT_SWIZZLE_A;
        vkInfo.image = CastToVkNonDispatchableObject< VkImage >( this->hImage );
        vkInfo.viewType = Vk::GetImageTarget( info.GetImageDimension() );
        vkInfo.format = Vk::GetImageFormat( pSourceTexture->ToData()->format );
        vkInfo.subresourceRange.aspectMask = Vk::IsDepthOnlyFormat( vkInfo.format ) ? VK_IMAGE_ASPECT_DEPTH_BIT
            : VK_IMAGE_ASPECT_DEPTH_BIT | VK_IMAGE_ASPECT_STENCIL_BIT;
        vkInfo.subresourceRange.baseArrayLayer = info.GetArrayRange().GetBaseArrayIndex();
        vkInfo.subresourceRange.baseMipLevel = info.GetMipLevel();
        vkInfo.subresourceRange.layerCount = info.GetArrayRange().GetArrayLength();
        vkInfo.subresourceRange.levelCount = 1;
        VkImageView imageView;
        VkResult result;
        NN_GFX_CALL_VK_FUNCTION( result = vkCreateImageView( device, &vkInfo,
            pDevice->ToData()->pAllocationCallback, &imageView ) );
        NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );
        NN_UNUSED( result );
        this->hImageView = CastFromVkNonDispatchableObject< VkImageView >( imageView );
        this->format = vkInfo.format;

        textureManageData->GetDepthStencilView().AddCache( info.ToData(), imageView );
    }
    this->pManager = const_cast< void* >( reinterpret_cast< const void* >( pSourceTexture ) );

    this->layerCount = pSourceTexture->ToData()->layerCount;
    this->levelCount = pSourceTexture->ToData()->levelCount;
    this->targetMipLevel = info.GetMipLevel();
    this->targetBaseArray = info.GetArrayRange().GetBaseArrayIndex();
    this->targetArrayCount = info.GetArrayRange().GetArrayLength();
    this->width = pSourceTexture->ToData()->width;
    this->height = pSourceTexture->ToData()->height;
    this->depth = pSourceTexture->ToData()->depth;
    this->samples = pSourceTexture->ToData()->samples;
    this->flags.SetBit( Flag_Shared, false );
    this->state = State_Initialized;
}

void DepthStencilViewImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_SDK_ASSERT( !this->flags.GetBit( Flag_Shared ) );
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( IsInitialized( *pDevice ) );

    NN_UNUSED( pDevice );
    // Do nothing here. VkImageView will be destroyed when the image referenced by this is destroyed.

    this->state = State_NotInitialized;
}

template<>
void GetImageFormatProperty< Target >( ImageFormatProperty* pOutImageFormatProperty,
    DeviceImpl< Target >* pDevice, ImageFormat imageFormat ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pOutImageFormatProperty );

    VkFormat vkFormat = Vk::GetImageFormat( imageFormat );
    Vk::GetImageFormatProperty( pOutImageFormatProperty, pDevice, vkFormat );
}

}
}
}
