﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <memory>

#include <nn/util/util_BytePtr.h>

#include <nn/gfx/gfx_TextureInfo.h>

#include <nn/gfx/detail/gfx_Texture-api.d3d.11.h>
#include <nn/gfx/detail/gfx_MemoryPool-api.d3d.11.h>
#include <nn/gfx/detail/gfx_Device-api.d3d.11.h>

#include "gfx_CommonHelper.h"
#include "gfx_D3dHelper.h"

namespace nn {
namespace gfx {
namespace detail {

namespace {

size_t CalculateMipDataOffsetsImpl( ptrdiff_t* pMipOffsets, const TextureInfo& info ) NN_NOEXCEPT
{
    ImageStorageDimension imageStorageDimension = info.GetImageStorageDimension();
    NN_SDK_ASSERT_NOT_EQUAL( imageStorageDimension, ImageStorageDimension_Undefined );
    int arrayLength = info.GetArrayLength();

    int width = info.GetWidth();
    int height = ( imageStorageDimension == ImageStorageDimension_1d
        && arrayLength > 1 ) ? arrayLength : info.GetHeight();
    int depth = imageStorageDimension== ImageStorageDimension_3d ? info.GetDepth() : arrayLength;

    int minHeight = 1;
    int minDepth = 1;
    if( arrayLength > 1 )
    {
        if( imageStorageDimension == ImageStorageDimension_1d )
        {
            minHeight = height;
        }
        else if( imageStorageDimension == ImageStorageDimension_2d )
        {
            minDepth = depth;
        }
    }

    ChannelFormat channelFormat = GetChannelFormat( info.GetImageFormat() );
    size_t size = 0;
    for( int mipLevel = 0; mipLevel < static_cast< int >( info.GetMipCount() ); ++mipLevel )
    {
        if( pMipOffsets )
        {
            pMipOffsets[ mipLevel ] = size;
        }
        size += CalculateImageSize( channelFormat,
            std::max NN_PREVENT_MACRO_FUNC ( width >> mipLevel, 1 ),
            std::max NN_PREVENT_MACRO_FUNC ( height >> mipLevel, minHeight ),
            std::max NN_PREVENT_MACRO_FUNC ( depth >> mipLevel, minDepth ) );
    }
    return size;
}

template< typename T >
struct FreeDeleter
{
    void operator()( T* ptr )
    {
        if( ptr )
        {
            free( ptr );
        }
    }
};

}

typedef ApiVariationD3d11 Target;

size_t TextureImpl< Target >::CalculateMipDataAlignment( DeviceImpl< Target >*, const InfoType& info ) NN_NOEXCEPT
{
    NN_UNUSED( info );

    return 1;
}

size_t TextureImpl< Target >::CalculateMipDataSize( DeviceImpl< Target >*, const InfoType& info ) NN_NOEXCEPT
{
    return CalculateMipDataOffsetsImpl( NULL, info );
}

void TextureImpl< Target >::CalculateMipDataOffsets(
    ptrdiff_t* pMipOffsets, DeviceImpl< Target >*, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pMipOffsets );

    CalculateMipDataOffsetsImpl( pMipOffsets, info );
}

size_t TextureImpl< Target >::GetRowPitch( DeviceImpl< Target >*, const InfoType& info ) NN_NOEXCEPT
{
    return CalculateRowSize( info.GetWidth(), GetChannelFormat( info.GetImageFormat() ) );
}

TextureImpl< Target >::TextureImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

TextureImpl< Target >::~TextureImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized || this->flags.GetBit( Flag_Shared ) );
}

void TextureImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info,
    MemoryPoolImpl< Target >* pMemoryPool, ptrdiff_t memoryPoolOffset, size_t memoryPoolSize ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_SDK_REQUIRES( !pMemoryPool || memoryPoolOffset >= 0 );
    NN_SDK_ASSERT( !pMemoryPool || pMemoryPool->ToData()->pMemory );
    NN_SDK_REQUIRES( !pMemoryPool || memoryPoolSize >= CalculateMipDataSize( pDevice, info ) );
    NN_UNUSED( memoryPoolSize );

    NN_SDK_ASSERT_NOT_EQUAL( info.GetImageStorageDimension(), ImageStorageDimension_Undefined );

    D3dTextureFormat format = D3d::GetShaderResourceTextureFormat( info.GetImageFormat() );
    int bindFlags = D3d::GetBindFlag( info.GetGpuAccessFlags() );
    int width = info.GetWidth();
    int height = info.GetHeight();
    int depth = info.GetDepth();

    this->imageFormat = info.GetImageFormat();
    this->imageStorageDimension = info.GetImageStorageDimension();
    this->sampleCount = info.GetMultisampleCount() == 0 ? 1 : info.GetMultisampleCount();
    this->arrayLength = info.GetArrayLength() == 0 ? 1 : info.GetArrayLength();
    this->mipCount = info.GetMipCount();

    NN_SDK_ASSERT( format.dxgiFormat != DXGI_FORMAT_UNKNOWN );
    NN_SDK_ASSERT( !IsD3dHandleValid( this->pTexture ) );

    ID3D11Device* pD3dDevice = static_cast< ID3D11Device* >( pDevice->ToData()->renderingContext.hD3dDevice );
    NN_SDK_ASSERT( IsD3dHandleValid( pD3dDevice ) );

    bool hasInitData = false;
    std::unique_ptr< D3D11_SUBRESOURCE_DATA[], FreeDeleter< D3D11_SUBRESOURCE_DATA > > initData(
        static_cast< D3D11_SUBRESOURCE_DATA* >( malloc( sizeof( D3D11_SUBRESOURCE_DATA ) * this->arrayLength * this->mipCount ) ) );

    if( pMemoryPool && this->sampleCount == 1 )
    {
        MemoryPoolImpl< Target >::DataType& memoryPool = pMemoryPool->ToData();
        nn::util::BytePtr pData( memoryPool.pMemory, memoryPoolOffset );

        for( int mipLevel = 0; mipLevel < static_cast< int >( this->mipCount ); ++mipLevel )
        {
            size_t size = D3d::GetImageSize( info.GetImageFormat(),
                std::max NN_PREVENT_MACRO_FUNC ( width >> mipLevel, 1 ),
                std::max NN_PREVENT_MACRO_FUNC ( height >> mipLevel, 1 ),
                std::max NN_PREVENT_MACRO_FUNC ( depth >> mipLevel, 1 ) );

            size_t oneLineSize = D3d::GetImageSize( info.GetImageFormat(),
                std::max NN_PREVENT_MACRO_FUNC ( width >> mipLevel, 1 ),
                1,
                1 );

            size_t oneSliceSize = D3d::GetImageSize( info.GetImageFormat(),
                std::max NN_PREVENT_MACRO_FUNC ( width >> mipLevel, 1 ),
                std::max NN_PREVENT_MACRO_FUNC ( height >> mipLevel, 1 ),
                1 );

            for ( int arrayIndex = 0; arrayIndex < static_cast< int >( arrayLength ); ++arrayIndex )
            {
                // D3D11_SUBRESOURCE_DATA の格納順序は
                // Array0 Mip0, Array0 Mip1, ..., Array0 MipN, Array1 Mip0, Array1 Mip1, ...
                // gfxの格納順序は
                // Array0 Mip0, Array1 Mip0, ..., ArrayN Mip0, Array0 Mip1, Array1 Mip1, ...
                D3D11_SUBRESOURCE_DATA& data = initData[ this->mipCount * arrayIndex + mipLevel ];

                data.pSysMem = pData.Get();
                data.SysMemPitch = static_cast< UINT >( oneLineSize );
                data.SysMemSlicePitch = static_cast< UINT >( oneSliceSize );

                pData.Advance( static_cast< ptrdiff_t >( size ) );
            }
        }
        NN_SDK_ASSERT( nn::util::BytePtr( memoryPool.pMemory.ptr ).Distance(
            pData.Get() ) <= static_cast< ptrdiff_t >( memoryPool.memorySize ) );

        hasInitData = true;
    }

    HRESULT hResult;

    switch( info.GetImageStorageDimension() )
    {
    case ImageStorageDimension_1d:
        {
            D3D11_TEXTURE1D_DESC desc;
            desc.Width = width;
            desc.MipLevels = this->mipCount;
            desc.ArraySize = this->arrayLength;
            desc.Format = format.dxgiFormat;
            desc.Usage = D3D11_USAGE_DEFAULT;
            desc.BindFlags = bindFlags;
            desc.CPUAccessFlags = 0;
            desc.MiscFlags = 0;

            ID3D11Texture1D* pD3dTexture1d;
            hResult = NN_GFX_CALL_D3D_FUNCTION( pD3dDevice->CreateTexture1D(
                &desc, hasInitData ? initData.get() : NULL, &pD3dTexture1d ) );
            this->pTexture = pD3dTexture1d;

            NN_SDK_ASSERT( SUCCEEDED( hResult ) && IsD3dHandleValid( this->pTexture ) );

            desc.Usage = D3D11_USAGE_STAGING;
            desc.BindFlags = 0;
            desc.CPUAccessFlags = D3D11_CPU_ACCESS_READ | D3D11_CPU_ACCESS_WRITE;

            hResult = NN_GFX_CALL_D3D_FUNCTION( pD3dDevice->CreateTexture1D( &desc, NULL, &pD3dTexture1d ) );
            this->pStagingTexture = pD3dTexture1d;

            NN_SDK_ASSERT( SUCCEEDED( hResult ) && IsD3dHandleValid( this->pStagingTexture ) );
        }
        break;
    case ImageStorageDimension_2d:
        {
            D3D11_TEXTURE2D_DESC desc;
            desc.Width = width;
            desc.Height = height;
            desc.MipLevels = this->mipCount;
            desc.ArraySize = this->arrayLength;
            desc.Format = format.dxgiFormat;
            desc.SampleDesc.Count = this->sampleCount;
            desc.SampleDesc.Quality = this->sampleCount > 1 ? D3D11_STANDARD_MULTISAMPLE_PATTERN : 0;
            desc.Usage = D3D11_USAGE_DEFAULT;
            desc.BindFlags = bindFlags;
            desc.CPUAccessFlags = 0;
            desc.MiscFlags = this->arrayLength == 6 ? D3D11_RESOURCE_MISC_TEXTURECUBE : 0;

            ID3D11Texture2D* pD3dTexture2d;
            hResult = NN_GFX_CALL_D3D_FUNCTION( pD3dDevice->CreateTexture2D(
                &desc, hasInitData ? initData.get() : NULL, &pD3dTexture2d ) );
            this->pTexture = pD3dTexture2d;

            NN_SDK_ASSERT( SUCCEEDED( hResult ) && IsD3dHandleValid( this->pTexture ) );

            if ( this->sampleCount == 1 )
            {
                desc.Usage = D3D11_USAGE_STAGING;
                desc.BindFlags = 0;
                desc.CPUAccessFlags = D3D11_CPU_ACCESS_READ | D3D11_CPU_ACCESS_WRITE;

                hResult = NN_GFX_CALL_D3D_FUNCTION( pD3dDevice->CreateTexture2D(
                    &desc, NULL, &pD3dTexture2d ) );
                this->pStagingTexture = pD3dTexture2d;

                NN_SDK_ASSERT( SUCCEEDED( hResult ) && IsD3dHandleValid( this->pStagingTexture ) );
            }
        }
        break;
    case ImageStorageDimension_3d:
        {
            D3D11_TEXTURE3D_DESC desc;
            desc.Width = width;
            desc.Height = height;
            desc.Depth = depth;
            desc.MipLevels = this->mipCount;
            desc.Format = format.dxgiFormat;
            desc.Usage = D3D11_USAGE_DEFAULT;
            desc.BindFlags = bindFlags;
            desc.CPUAccessFlags = 0;
            desc.MiscFlags = 0;

            ID3D11Texture3D* pD3dTexture3d;
            hResult = NN_GFX_CALL_D3D_FUNCTION( pD3dDevice->CreateTexture3D(
                &desc, hasInitData ? initData.get() : NULL, &pD3dTexture3d ) );
            this->pTexture = pD3dTexture3d;

            NN_SDK_ASSERT( SUCCEEDED( hResult ) && IsD3dHandleValid( this->pTexture ) );

            desc.Usage = D3D11_USAGE_STAGING;
            desc.BindFlags = 0;
            desc.CPUAccessFlags = D3D11_CPU_ACCESS_READ | D3D11_CPU_ACCESS_WRITE;

            hResult = NN_GFX_CALL_D3D_FUNCTION( pD3dDevice->CreateTexture3D(
                &desc, NULL, &pD3dTexture3d ) );
            this->pStagingTexture = pD3dTexture3d;

            NN_SDK_ASSERT( SUCCEEDED( hResult ) && IsD3dHandleValid( this->pStagingTexture ) );
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }

    this->flags.SetBit( Flag_Shared, false );
    this->state = State_Initialized;
} //NOLINT(impl/function_size)

void TextureImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_SDK_ASSERT( !this->flags.GetBit( Flag_Shared ) );
    NN_UNUSED( pDevice );

    if( this->pTexture )
    {
        switch( this->imageStorageDimension )
        {
        case ImageStorageDimension_1d:
            {
                ID3D11Texture1D* pD3dTexture1d = static_cast< ID3D11Texture1D* >( this->pTexture );
                NN_GFX_CALL_D3D_FUNCTION( pD3dTexture1d->Release() );
                pD3dTexture1d = static_cast< ID3D11Texture1D* >( this->pStagingTexture );
                NN_GFX_CALL_D3D_FUNCTION( pD3dTexture1d->Release() );            }
            break;
        case ImageStorageDimension_2d:
            {
                ID3D11Texture2D* pD3dTexture2d = static_cast< ID3D11Texture2D* >( this->pTexture );
                NN_GFX_CALL_D3D_FUNCTION( pD3dTexture2d->Release() );
                if ( this->pStagingTexture )
                {
                    pD3dTexture2d = static_cast< ID3D11Texture2D* >( this->pStagingTexture );
                    NN_GFX_CALL_D3D_FUNCTION( pD3dTexture2d->Release() );
                }
            }
            break;
        case ImageStorageDimension_3d:
            {
                ID3D11Texture3D* pD3dTexture3d = static_cast< ID3D11Texture3D* >( this->pTexture );
                NN_GFX_CALL_D3D_FUNCTION( pD3dTexture3d->Release() );
                pD3dTexture3d = static_cast< ID3D11Texture3D* >( this->pStagingTexture );
                NN_GFX_CALL_D3D_FUNCTION( pD3dTexture3d->Release() );
            }
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }

        this->pTexture = NULL;
        this->pStagingTexture = NULL;
    }

    this->state = State_NotInitialized;
}

TextureViewImpl< Target >::TextureViewImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

TextureViewImpl< Target >::~TextureViewImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized || this->flags.GetBit( Flag_Shared ) );
}

void TextureViewImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized );

    NN_SDK_ASSERT( !IsD3dHandleValid( this->pShaderResourceView ) );

    const TextureImpl< Target >* pOriginalTexture = info.GetTexturePtr();
    NN_SDK_ASSERT_NOT_NULL( pOriginalTexture );
    NN_SDK_ASSERT( IsInitialized( *pOriginalTexture ) );

    D3D11_SHADER_RESOURCE_VIEW_DESC desc;

    D3dTextureFormat format = D3d::GetShaderResourceTextureViewFormat( info.GetImageFormat(), info.GetDepthStencilTextureMode() );
    desc.Format = format.dxgiFormat;
    switch ( info.GetImageDimension() )
    {
    case ImageDimension_1d:
        {
            desc.ViewDimension = D3D11_SRV_DIMENSION_TEXTURE1D;
            desc.Texture1D.MostDetailedMip = info.GetSubresourceRange().GetMipRange().GetMinMipLevel();
            desc.Texture1D.MipLevels = info.GetSubresourceRange().GetMipRange().GetMinMipLevel()
                + info.GetSubresourceRange().GetMipRange().GetMipCount();
        }
        break;
    case ImageDimension_1dArray:
        {
            desc.ViewDimension = D3D11_SRV_DIMENSION_TEXTURE1DARRAY;
            desc.Texture1DArray.ArraySize = info.GetSubresourceRange().GetArrayRange().GetArrayLength();
            desc.Texture1DArray.FirstArraySlice = info.GetSubresourceRange().GetArrayRange().GetBaseArrayIndex();
            desc.Texture1DArray.MostDetailedMip = info.GetSubresourceRange().GetMipRange().GetMinMipLevel();
            desc.Texture1DArray.MipLevels = info.GetSubresourceRange().GetMipRange().GetMinMipLevel()
                + info.GetSubresourceRange().GetMipRange().GetMipCount();
        }
        break;
    case ImageDimension_2d:
        {
            desc.ViewDimension = D3D11_SRV_DIMENSION_TEXTURE2D;
            desc.Texture2D.MostDetailedMip = info.GetSubresourceRange().GetMipRange().GetMinMipLevel();
            desc.Texture2D.MipLevels = info.GetSubresourceRange().GetMipRange().GetMinMipLevel()
                + info.GetSubresourceRange().GetMipRange().GetMipCount();
        }
        break;
    case ImageDimension_2dArray:
        {
            desc.ViewDimension = D3D11_SRV_DIMENSION_TEXTURE2DARRAY;
            desc.Texture2DArray.ArraySize = info.GetSubresourceRange().GetArrayRange().GetArrayLength();
            desc.Texture2DArray.FirstArraySlice = info.GetSubresourceRange().GetArrayRange().GetBaseArrayIndex();
            desc.Texture2DArray.MostDetailedMip = info.GetSubresourceRange().GetMipRange().GetMinMipLevel();
            desc.Texture2DArray.MipLevels = info.GetSubresourceRange().GetMipRange().GetMinMipLevel()
                + info.GetSubresourceRange().GetMipRange().GetMipCount();
        }
        break;
    case ImageDimension_2dMultisample:
        {
            desc.ViewDimension = D3D11_SRV_DIMENSION_TEXTURE2DMS;
        }
        break;
    case ImageDimension_2dMultisampleArray:
        {
            desc.ViewDimension = D3D11_SRV_DIMENSION_TEXTURE2DMSARRAY;
            desc.Texture2DMSArray.ArraySize = info.GetSubresourceRange().GetArrayRange().GetArrayLength();
            desc.Texture2DMSArray.FirstArraySlice = info.GetSubresourceRange().GetArrayRange().GetBaseArrayIndex();
        }
        break;
    case ImageDimension_3d:
        {
            desc.ViewDimension = D3D11_SRV_DIMENSION_TEXTURE3D;
            desc.Texture3D.MostDetailedMip = info.GetSubresourceRange().GetMipRange().GetMinMipLevel();
            desc.Texture3D.MipLevels = info.GetSubresourceRange().GetMipRange().GetMinMipLevel()
                + info.GetSubresourceRange().GetMipRange().GetMipCount();
        }
        break;
    case ImageDimension_CubeMap:
        {
            desc.ViewDimension = D3D11_SRV_DIMENSION_TEXTURECUBE;
            desc.TextureCube.MostDetailedMip = info.GetSubresourceRange().GetMipRange().GetMinMipLevel();
            desc.TextureCube.MipLevels = info.GetSubresourceRange().GetMipRange().GetMinMipLevel()
                + info.GetSubresourceRange().GetMipRange().GetMipCount();
        }
        break;
    case ImageDimension_CubeMapArray:
        {
            desc.ViewDimension = D3D11_SRV_DIMENSION_TEXTURECUBEARRAY;
            desc.TextureCubeArray.MostDetailedMip = info.GetSubresourceRange().GetMipRange().GetMinMipLevel();
            desc.TextureCubeArray.MipLevels = info.GetSubresourceRange().GetMipRange().GetMinMipLevel()
                + info.GetSubresourceRange().GetMipRange().GetMipCount();
            desc.TextureCubeArray.First2DArrayFace = info.GetSubresourceRange().GetArrayRange().GetBaseArrayIndex();
            desc.TextureCubeArray.NumCubes = info.GetSubresourceRange().GetArrayRange().GetArrayLength();
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }

    ID3D11Device* pD3dDevice = static_cast< ID3D11Device* >(
        pDevice->ToData()->renderingContext.hD3dDevice );
    NN_SDK_ASSERT( IsD3dHandleValid( pD3dDevice ) );

    ID3D11Resource* pResource = static_cast< ID3D11Resource* >(
        pOriginalTexture->ToData()->pTexture );
    NN_SDK_ASSERT( IsD3dHandleValid( pResource ) );

    ID3D11ShaderResourceView* pD3dShaderResourceView;
    HRESULT hResult = NN_GFX_CALL_D3D_FUNCTION( pD3dDevice->CreateShaderResourceView(
        pResource, &desc, &pD3dShaderResourceView ) );
    NN_SDK_ASSERT( SUCCEEDED( hResult ) && IsD3dHandleValid( pD3dShaderResourceView ) );
    NN_UNUSED( hResult );

    this->pShaderResourceView = pD3dShaderResourceView;
    this->flags.SetBit( Flag_Shared, false );
    this->state = State_Initialized;
} // NOLINT(impl/function_size)

void TextureViewImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_SDK_ASSERT( !this->flags.GetBit( Flag_Shared ) );
    NN_UNUSED( pDevice );

    if( this->pShaderResourceView )
    {
        ID3D11ShaderResourceView* pD3dShaderResourceView = static_cast< ID3D11ShaderResourceView* >( this->pShaderResourceView );
        NN_GFX_CALL_D3D_FUNCTION( pD3dShaderResourceView->Release() );
        this->pShaderResourceView = NULL;
    }

    this->state = State_NotInitialized;
}

ColorTargetViewImpl< Target >::ColorTargetViewImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

ColorTargetViewImpl< Target >::~ColorTargetViewImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized || this->flags.GetBit( Flag_Shared ) );
}

void ColorTargetViewImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_SDK_ASSERT( !IsD3dHandleValid( this->pRenderTargetView ) );

    const TextureImpl< Target >* pOriginalTexture = info.GetTexturePtr();
    NN_SDK_ASSERT_NOT_NULL( pOriginalTexture );
    NN_SDK_ASSERT( IsInitialized( *pOriginalTexture ) );
    const TextureImpl< Target >::DataType &textureData = pOriginalTexture->ToData();

    D3D11_RENDER_TARGET_VIEW_DESC desc;

    D3dTextureFormat format = D3d::GetTextureFormat( static_cast< ImageFormat >( textureData.imageFormat ) );
    desc.Format = format.dxgiFormat;
    switch ( static_cast< ImageStorageDimension >( textureData.imageStorageDimension ) )
    {
    case ImageStorageDimension_1d:
        if ( textureData.arrayLength > 1 )
        {
            desc.Texture1DArray.ArraySize = info.GetArrayRange().GetArrayLength();
            desc.Texture1DArray.FirstArraySlice = info.GetArrayRange().GetBaseArrayIndex();
            desc.Texture1DArray.MipSlice = info.GetMipLevel();
            desc.ViewDimension = D3D11_RTV_DIMENSION_TEXTURE1DARRAY;
        }
        else
        {
            desc.Texture1D.MipSlice = info.GetMipLevel();
            desc.ViewDimension = D3D11_RTV_DIMENSION_TEXTURE1D;
        }
        break;
    case ImageStorageDimension_2d:
        if ( textureData.sampleCount > 1 )
        {
            if ( textureData.arrayLength > 1 )
            {
                desc.Texture2DMSArray.ArraySize = info.GetArrayRange().GetArrayLength();
                desc.Texture2DMSArray.FirstArraySlice = info.GetArrayRange().GetBaseArrayIndex();
                desc.ViewDimension = D3D11_RTV_DIMENSION_TEXTURE2DMSARRAY;
            }
            else
            {
                desc.ViewDimension = D3D11_RTV_DIMENSION_TEXTURE2DMS;
            }
        }
        else
        {
            if ( textureData.arrayLength > 1 )
            {
                desc.Texture2DArray.ArraySize = info.GetArrayRange().GetArrayLength();
                desc.Texture2DArray.FirstArraySlice = info.GetArrayRange().GetBaseArrayIndex();
                desc.Texture2DArray.MipSlice = info.GetMipLevel();
                desc.ViewDimension = D3D11_RTV_DIMENSION_TEXTURE2DARRAY;
            }
            else
            {
                desc.Texture2D.MipSlice = info.GetMipLevel();
                desc.ViewDimension = D3D11_RTV_DIMENSION_TEXTURE2D;
            }
        }
        break;
    case ImageStorageDimension_3d:
        {
            desc.Texture3D.FirstWSlice = 0;
            desc.Texture3D.MipSlice = textureData.mipCount;
            desc.Texture3D.WSize = static_cast< UINT >( -1 );
            desc.ViewDimension = D3D11_RTV_DIMENSION_TEXTURE3D;
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }

    ID3D11Device* pD3dDevice = static_cast< ID3D11Device* >( pDevice->ToData()->renderingContext.hD3dDevice );
    NN_SDK_ASSERT( IsD3dHandleValid( pD3dDevice ) );

    ID3D11Resource* pResource = static_cast< ID3D11Resource* >( pOriginalTexture->ToData()->pTexture );
    NN_SDK_ASSERT( IsD3dHandleValid( pResource ) );

    ID3D11RenderTargetView* pD3dRenderTargetView;
    HRESULT hResult = NN_GFX_CALL_D3D_FUNCTION( pD3dDevice->CreateRenderTargetView( pResource, &desc, &pD3dRenderTargetView ) );
    NN_SDK_ASSERT( SUCCEEDED( hResult ) && IsD3dHandleValid( pD3dRenderTargetView ) );
    NN_UNUSED( hResult );

    this->pRenderTargetView = pD3dRenderTargetView;
    this->flags.SetBit( Flag_Shared, false );
    this->state = State_Initialized;
}

void ColorTargetViewImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_SDK_ASSERT( !this->flags.GetBit( Flag_Shared ) );
    NN_UNUSED( pDevice );

    if( this->pRenderTargetView )
    {
        ID3D11RenderTargetView* pD3dRenderTargetView = static_cast< ID3D11RenderTargetView* >( this->pRenderTargetView );
        NN_GFX_CALL_D3D_FUNCTION( pD3dRenderTargetView->Release() );
        this->pRenderTargetView = NULL;
    }

    this->state = State_NotInitialized;
}

DepthStencilViewImpl< Target >::DepthStencilViewImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

DepthStencilViewImpl< Target >::~DepthStencilViewImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized || this->flags.GetBit( Flag_Shared ) );
}

void DepthStencilViewImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized );

    NN_SDK_ASSERT( !IsD3dHandleValid( this->pDepthStencilView ) );

    const TextureImpl< Target >* pOriginalTexture = info.GetTexturePtr();
    NN_SDK_ASSERT_NOT_NULL( pOriginalTexture );
    NN_SDK_ASSERT( IsInitialized( *pOriginalTexture ) );
    const TextureImpl< Target >::DataType &textureData = pOriginalTexture->ToData();

    D3D11_DEPTH_STENCIL_VIEW_DESC desc;

    D3dTextureFormat format = D3d::GetTextureFormat( static_cast< ImageFormat >( textureData.imageFormat ) );
    desc.Format = format.dxgiFormat;
    desc.Flags = 0;
    switch ( static_cast< ImageStorageDimension >( textureData.imageStorageDimension ) )
    {
    case ImageStorageDimension_1d:
        if ( textureData.arrayLength > 1 )
        {
            desc.Texture1DArray.MipSlice = info.GetMipLevel();
            desc.Texture1DArray.FirstArraySlice = info.GetArrayRange().GetBaseArrayIndex();
            desc.Texture1DArray.ArraySize = info.GetArrayRange().GetArrayLength();
            desc.ViewDimension = D3D11_DSV_DIMENSION_TEXTURE1DARRAY;
        }
        else
        {
            desc.Texture1D.MipSlice = info.GetMipLevel();
            desc.ViewDimension = D3D11_DSV_DIMENSION_TEXTURE1D;
        }
        break;
    case ImageStorageDimension_2d:
        if ( textureData.sampleCount > 1 )
        {
            if ( textureData.arrayLength > 1 )
            {
                desc.Texture2DMSArray.ArraySize = info.GetArrayRange().GetArrayLength();
                desc.Texture2DMSArray.FirstArraySlice = info.GetArrayRange().GetBaseArrayIndex();
                desc.ViewDimension = D3D11_DSV_DIMENSION_TEXTURE2DMSARRAY;
            }
            else
            {
                desc.ViewDimension = D3D11_DSV_DIMENSION_TEXTURE2DMS;
            }
        }
        else
        {
            if ( textureData.arrayLength > 1 )
            {
                desc.Texture2DArray.ArraySize = info.GetArrayRange().GetArrayLength();
                desc.Texture2DArray.FirstArraySlice = info.GetArrayRange().GetBaseArrayIndex();
                desc.Texture2DArray.MipSlice = info.GetMipLevel();
                desc.ViewDimension = D3D11_DSV_DIMENSION_TEXTURE2DARRAY;
            }
            else
            {
                desc.Texture2D.MipSlice = info.GetMipLevel();
                desc.ViewDimension = D3D11_DSV_DIMENSION_TEXTURE2D;
            }
        }
        break;
    case ImageStorageDimension_3d:
        // DepthStencil 3D texture is not supported in D3d11.
    default: NN_UNEXPECTED_DEFAULT;
    }

    ID3D11Device* pD3dDevice = static_cast< ID3D11Device* >( pDevice->ToData()->renderingContext.hD3dDevice );
    NN_SDK_ASSERT( IsD3dHandleValid( pD3dDevice ) );

    ID3D11Resource* pResource = static_cast< ID3D11Resource* >( pOriginalTexture->ToData()->pTexture );
    NN_SDK_ASSERT( IsD3dHandleValid( pResource ) );

    ID3D11DepthStencilView* pD3dDepthStencilView;
    HRESULT hResult = NN_GFX_CALL_D3D_FUNCTION( pD3dDevice->CreateDepthStencilView( pResource, &desc, &pD3dDepthStencilView ) );
    NN_SDK_ASSERT( SUCCEEDED( hResult ) && IsD3dHandleValid( pD3dDepthStencilView ) );
    NN_UNUSED( hResult );

    this->pDepthStencilView = pD3dDepthStencilView;
    this->flags.SetBit( Flag_Shared, false );
    this->state = State_Initialized;
}

void DepthStencilViewImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_SDK_ASSERT( !this->flags.GetBit( Flag_Shared ) );
    NN_UNUSED( pDevice );

    if( this->pDepthStencilView )
    {
        ID3D11DepthStencilView* pD3dDepthStencilView = static_cast< ID3D11DepthStencilView* >( this->ToData()->pDepthStencilView );
        NN_GFX_CALL_D3D_FUNCTION( pD3dDepthStencilView->Release() );
        this->pDepthStencilView = NULL;
    }

    this->state = State_NotInitialized;
}

template<>
void GetImageFormatProperty< Target >( ImageFormatProperty* pOutImageFormatProperty,
    DeviceImpl< Target >* pDevice, ImageFormat imageFormat ) NN_NOEXCEPT
{
    NN_UNUSED( pOutImageFormatProperty );
    NN_UNUSED( pDevice );
    NN_UNUSED( imageFormat );

    // 本機能は未実装です。
    // 実装可能性は未調査です。
    NN_SDK_ASSERT( 0 );}

}
}
}
