﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/gfx/gfx_SyncInfo.h>
#include <nn/gfx/detail/gfx_Sync-api.gx.2.h>
#include <nn/gfx/detail/gfx_Misc.h>

#include "gfx_GxHelper.h"

namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationGx2 Target;

FenceImpl< Target >::FenceImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

FenceImpl< Target >::~FenceImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

void FenceImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( pDevice );
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_UNUSED( info );
    NN_UNUSED( pDevice );

    this->timeStamp = ~0ull;

    this->state = State_Initialized;
}

void FenceImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( pDevice );
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_UNUSED( pDevice );

    this->state = State_NotInitialized;
}

bool FenceImpl< Target >::IsSignaled() const NN_NOEXCEPT
{
    return this->timeStamp < NN_GFX_CALL_GX_FUNCTION( GX2GetRetiredTimeStamp() );
}

SyncResult FenceImpl< Target >::Sync( TimeSpan timeout ) const NN_NOEXCEPT
{
    SyncResult result = SyncResult_TimeoutExpired;
    Bit32 originalTimeout = NN_GFX_CALL_GX_FUNCTION( GX2GetGPUTimeout() );
    Bit32 timeoutMilliSeconds = timeout.GetMilliSeconds();

    // GX2 recommonds not having a timeout less than 17ms
    timeoutMilliSeconds = timeoutMilliSeconds > 17 ? timeoutMilliSeconds : 17;

    NN_GFX_CALL_GX_FUNCTION( GX2SetGPUTimeout( timeoutMilliSeconds ) );
    if( NN_GFX_CALL_GX_FUNCTION( GX2WaitTimeStamp( this->timeStamp ) ) )
    {
        result = SyncResult_Success;
    }

    // Restore the original timeout (default is 10 seconds)
    NN_GFX_CALL_GX_FUNCTION( GX2SetGPUTimeout( originalTimeout ) );
    return result;
}

SemaphoreImpl< Target >::SemaphoreImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

SemaphoreImpl< Target >::~SemaphoreImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

void SemaphoreImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_UNUSED( pDevice );
    NN_UNUSED( info );
    NN_SDK_ASSERT( NULL ); // This will be used for DMAE queue synchronization

    this->state = State_Initialized;
}

void SemaphoreImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_SDK_ASSERT( NULL ); // This will be used for DMAE queue synchronization

    this->state = State_NotInitialized;
}

}
}
}
