﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/gfx/gfx_SyncInfo.h>

#include <nn/gfx/detail/gfx_Sync-api.gl.4.h>
#include <nn/gfx/detail/gfx_Misc.h>
#include <nn/gfx/detail/gfx_Device-api.gl.4.h>
#include <nn/gfx/detail/gfx_Queue-api.gl.4.h>

#include "gfx_GlHelper.h"

namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationGl4 Target;

FenceImpl< Target >::FenceImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

FenceImpl< Target >::~FenceImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

void FenceImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_UNUSED( info );
    NN_UNUSED( pDevice );

    this->pQueue = NULL;
    this->flags.Clear();
    this->state = State_Initialized;
}

void FenceImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_UNUSED( pDevice );

    if( this->glSync )
    {
        // キューのコンテキストから破棄するとキューを先に破棄できないのでデバイスのコンテキストを使用
        GlDeviceActivator activator( pDevice );

        NN_GFX_CALL_GL_FUNCTION( ::glDeleteSync( static_cast< GLsync >( this->glSync ) ) );
        NN_GFX_GL_ASSERT();
        this->glSync = NULL;
    }

    this->pQueue = NULL;
    this->state = State_NotInitialized;
}

bool FenceImpl< Target >::IsSignaled() const NN_NOEXCEPT
{
    NN_SDK_ASSERT( IsInitialized( *this ) );

    if( this->flags.GetBit( Flag_Signaled ) )
    {
        return true;
    }

    GlDeviceActivator activator( this->pQueue.ptr );

    int value;
    NN_GFX_CALL_GL_FUNCTION( ::glGetSynciv( static_cast< GLsync >( this->glSync.ptr ),
        GL_SYNC_STATUS, sizeof( value ), NULL, &value ) );
    NN_GFX_GL_ASSERT();

    return value == GL_SIGNALED;
}

SyncResult FenceImpl< Target >::Sync( TimeSpan timeout ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT( IsInitialized( *this ) );

    if( this->flags.GetBit( Flag_Signaled ) )
    {
        return SyncResult_Success;
    }

    GlDeviceActivator activator( this->pQueue.ptr );

    GLenum result;
    result = NN_GFX_CALL_GL_FUNCTION( ::glClientWaitSync(
        static_cast< GLsync >( this->glSync.ptr ), 0, timeout.GetNanoSeconds() ) );
    NN_GFX_GL_ASSERT();

    return result == GL_TIMEOUT_EXPIRED ? SyncResult_TimeoutExpired : SyncResult_Success;
}

SemaphoreImpl< Target >::SemaphoreImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

SemaphoreImpl< Target >::~SemaphoreImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

void SemaphoreImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_UNUSED( pDevice );
    NN_UNUSED( info );

    this->pQueue = NULL;
    this->flags.Clear();
    this->state = State_Initialized;
}

void SemaphoreImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_UNUSED( pDevice );

    if( this->glSync )
    {
        GlDeviceActivator activator( this->pQueue.ptr );

        NN_GFX_CALL_GL_FUNCTION( ::glDeleteSync( static_cast< GLsync >( this->glSync ) ) );
        NN_GFX_GL_ASSERT();
        this->glSync = NULL;
    }

    this->state = State_NotInitialized;
}

}
}
}
