﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>

#if !defined( NN_BUILD_CONFIG_OS_COS )
#   error
#endif

#include <nn/util/util_BytePtr.h>

#include <nn/gfx/gfx_SwapChainInfo.h>
#include <nn/gfx/gfx_DataAccessorConverter.h>

#include <nn/gfx/detail/gfx_Device-api.gx.2-os.cos.h>
#include <nn/gfx/detail/gfx_SwapChain-api.gx.2-os.cos.h>
#include <nn/gfx/detail/gfx_MemoryPool-api.gx.2.h>
#include <nn/gfx/detail/gfx_Texture-api.gx.2.h>

#include "gfx_GxHelper.h"

namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationGx2 Target;

//
// Helper function to calculate GX2SurfaceFormat and some other variables
//
static void GetGX2Format( const SwapChainImpl< Target >::InfoType& info,
    GX2SurfaceFormat* pScanOutColorBufferFormat, GX2TVRenderMode* pRenderMode, bool* pIsTV ) NN_NOEXCEPT
{
    static const u32 s_TvRenderModeTable[] = {
         640,  480, GX2_TV_RENDER_480_NARROW,   // 4:3 ratio (640x480)
         854,  480, GX2_TV_RENDER_480_WIDE,     // 16:9 ratio (854x480)
        1280,  720, GX2_TV_RENDER_720,          // 16:9 for all the rest...
        1920, 1080, GX2_TV_RENDER_1080,
    };

    const int renderModeTableSize = 4;

    nn::gfx::ImageFormat format = info.GetFormat();

    int renderWidth = info.GetWidth();
    int renderHeight = info.GetHeight();

    // GX2 Supports a limited number of scan buffer formats

    switch( format )
    {
    case ImageFormat_R5_G6_B5_Unorm:
        {
            *pScanOutColorBufferFormat = GX2_SURFACE_FORMAT_TCS_R5_G6_B5_UNORM;
        }
        break;
    case ImageFormat_R10_G10_B10_A2_Unorm:
        {
            *pScanOutColorBufferFormat = GX2_SURFACE_FORMAT_TCS_R10_G10_B10_A2_UNORM;
        }
        break;
    case ImageFormat_R8_G8_B8_A8_Unorm:
        {
            *pScanOutColorBufferFormat = GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_UNORM;
        }
        break;
    case ImageFormat_R8_G8_B8_A8_UnormSrgb:
        {
            *pScanOutColorBufferFormat = GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_SRGB;
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }

    // Choose an appropriate TV render mode based on desired dimensions
    for( int idx = 0; idx < renderModeTableSize; idx++ )
    {
        if( ( ( s_TvRenderModeTable[ idx * 3 ] >= renderWidth ) &&
              ( s_TvRenderModeTable[ idx * 3 + 1 ] >= renderHeight ) ) || idx == ( renderModeTableSize - 1 ) )
        {
            *pRenderMode = static_cast< GX2TVRenderMode >( s_TvRenderModeTable[ idx * 3 + 2 ] );
            break;
        }
    }

#if 0
    // this is an ugly attempt to determine whether the display is a TV or DRC
    // removed for now because it is not strictly necessary
    DeviceImplData< ApiVariationGx2 >::DisplayInfo *display =
        reinterpret_cast< DeviceImplData< ApiVariationGx2 >::DisplayInfo* >( hDisplay.GetHandle() );
    *pIsTV = display->isTV;
#else
    // assume TV for now; this will work for DRC, but use a bit more memory than if we determined it accurately
    *pIsTV = true;
#endif
}

// find the required size

size_t SwapChainImpl< Target >::CalculateScanBufferSize(
    DeviceImpl< Target >*, const InfoType& info ) NN_NOEXCEPT
{
    GX2TVRenderMode renderMode;
    GX2SurfaceFormat scanOutColorBufferFormat;
    u32 scanSize;
    GX2Boolean scaleNeeded;
    bool isTV;

    NN_SDK_REQUIRES( info.GetFormat() != nn::gfx::ImageFormat_Undefined );
    GetGX2Format( info, &scanOutColorBufferFormat, &renderMode, &isTV );

    GX2BufferingMode bufferingMode = Gx::GetBufferingMode( info.GetBufferCount() );
    if( isTV )
    {
        NN_GFX_CALL_GX_FUNCTION( GX2CalcTVSize( renderMode,
            scanOutColorBufferFormat, bufferingMode, &scanSize, &scaleNeeded ) );
    }
    else
    {
        // Setup Scan Buffers
        NN_GFX_CALL_GX_FUNCTION( GX2CalcDRCSize( GX2_DRC_SINGLE,
            scanOutColorBufferFormat, bufferingMode, &scanSize, &scaleNeeded ) );
    }

    return static_cast< size_t >( scanSize );
}

size_t SwapChainImpl< Target >::GetScanBufferAlignment( DeviceImpl< Target >*, const InfoType& ) NN_NOEXCEPT
{
    return GX2_SCAN_BUFFER_ALIGNMENT;
}

SwapChainImpl< Target >::SwapChainImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

SwapChainImpl< Target >::~SwapChainImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

void SwapChainImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info,
    MemoryPoolImpl< Target >* pMemoryPool, ptrdiff_t memoryPoolOffset, size_t memoryPoolSize ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_SDK_REQUIRES( pDevice );
    NN_SDK_REQUIRES( pMemoryPool );
    NN_SDK_REQUIRES( IsInitialized( *pMemoryPool ) );
    NN_SDK_ASSERT( pMemoryPool->ToData()->pMemory );
    NN_SDK_REQUIRES( memoryPoolSize >= CalculateScanBufferSize( pDevice, info ) );

    GX2TVRenderMode renderMode;
    GX2SurfaceFormat scanOutCBFormat;
    bool isTV;

    NN_SDK_ASSERT( info.GetBufferCount() < GX2_BUFFERING_LAST );

    // Save the format and display handle for use in the QueueImpl::CopyColorBufferToScanBuffer routine.
    this->pDisplayInfo = &displays[ 0 ];

    GetGX2Format( info, &scanOutCBFormat, &renderMode, &isTV );
    this->scanBufferSize = memoryPoolSize;
    this->pScanBuffer = nn::util::BytePtr( pMemoryPool->ToData()->pMemory, memoryPoolOffset ).Get();

    GX2BufferingMode bufferingMode = Gx::GetBufferingMode( info.GetBufferCount() );
    if( isTV )
    {
        NN_GFX_CALL_GX_FUNCTION( GX2Invalidate(
            GX2_INVALIDATE_CPU, this->pScanBuffer, this->scanBufferSize ) );
        NN_GFX_CALL_GX_FUNCTION( GX2SetTVBuffer( this->pScanBuffer,
            this->scanBufferSize, renderMode, scanOutCBFormat, bufferingMode ) );
    }
    else
    {
        NN_GFX_CALL_GX_FUNCTION( GX2Invalidate(
            GX2_INVALIDATE_CPU, this->pScanBuffer, this->scanBufferSize ) );
        NN_GFX_CALL_GX_FUNCTION( GX2SetDRCBuffer( this->pScanBuffer,
            this->scanBufferSize, GX2_DRC_SINGLE, scanOutCBFormat, bufferingMode ) );
    }

    // Create some ColorTargetViews for GetCurrentScanBuffer. They can be left uninitialized since
    // GX2GetCurrentScanBuffer will setup the internal gx2ColorBuffer. For now simply set the state
    // to initialized.
    this->scanBufferView.state = ColorTargetViewImpl< Target >::DataType::State_Initialized;
    this->scanBufferView.pGx2ColorBuffer = &this->scanBufferView.gx2ColorBuffer;
    this->scanBufferTexture.state = TextureImpl< Target >::DataType::State_Initialized;

    this->state = State_Initialized;
}

void SwapChainImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );

    this->scanBufferView.state = ColorTargetViewImpl< Target >::DataType::State_NotInitialized;
    this->scanBufferTexture.state = TextureImpl< Target >::DataType::State_NotInitialized;

    this->state = State_NotInitialized;
}

int SwapChainImpl< Target >::GetScanBufferViews( TColorTargetView< Target >**, int  ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( 0 );
    return 0;
}

int SwapChainImpl< Target >::GetScanBuffers( TTexture< Target >**, int ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( 0 );
    return 0;
}

AcquireScanBufferResult SwapChainImpl< Target >::AcquireNextScanBufferIndex(
    int*, SemaphoreImpl< Target >*, FenceImpl< Target >* ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( 0 );
    return AcquireScanBufferResult_Failed;
}

int SwapChainImpl< Target >::AcquireNextScanBufferIndex() NN_NOEXCEPT
{
    NN_SDK_ASSERT( 0 );
    return 0;
}

ColorTargetViewImpl< Target >* SwapChainImpl< Target >::AcquireNextScanBufferView() NN_NOEXCEPT
{
    DisplayInfo* pDisplay = static_cast< DisplayInfo* >( this->pDisplayInfo );

    GX2ScanTarget target = pDisplay->isTV ? GX2_SCAN_TARGET_TV : GX2_SCAN_TARGET_DRC_FIRST;

    GX2ColorBuffer* pGx2ColorBuffer = reinterpret_cast< GX2ColorBuffer* >( &this->scanBufferView.gx2ColorBuffer );

    NN_GFX_CALL_GX_FUNCTION( GX2WaitForFreeScanBuffer( target ) );
    NN_GFX_CALL_GX_FUNCTION( GX2GetCurrentScanBuffer( target, pGx2ColorBuffer ) );

    return nn::gfx::DataToAccessor( this->scanBufferView );
}

TextureImpl< Target >* SwapChainImpl< Target >::AcquireNextScanBuffer() NN_NOEXCEPT
{
    // TODO 確認
    DisplayInfo* pDisplay = static_cast< DisplayInfo* >( this->pDisplayInfo );

    GX2ScanTarget target = pDisplay->isTV ? GX2_SCAN_TARGET_TV : GX2_SCAN_TARGET_DRC_FIRST;

    GX2ColorBuffer* pGx2ColorBuffer = reinterpret_cast< GX2ColorBuffer* >( &this->scanBufferView.gx2ColorBuffer );

    NN_GFX_CALL_GX_FUNCTION( GX2WaitForFreeScanBuffer( target ) );
    NN_GFX_CALL_GX_FUNCTION( GX2GetCurrentScanBuffer( target, pGx2ColorBuffer ) );

    this->scanBufferTexture.pGx2Surface = &pGx2ColorBuffer->surface;

    return nn::gfx::DataToAccessor( this->scanBufferTexture );
}

}
}
}
