﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstring>

#include <nn/nn_StaticAssert.h>

#include <nn/util/util_BytePtr.h>
#include <nn/util/util_BitArray.h>

#include <nn/gfx/gfx_StateInfo.h>

#include <nn/gfx/detail/gfx_State-api.nvn.8.h>
#include <nn/gfx/detail/gfx_Shader-api.nvn.8.h>
#include <nn/gfx/detail/gfx_Device-api.nvn.8.h>
#include <nn/gfx/detail/gfx_Misc.h>

#include "gfx_CommonHelper.h"
#include "gfx_NvnHelper.h"


namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationNvn8 Target;

RasterizerStateImpl< Target >::RasterizerStateImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

RasterizerStateImpl< Target >::~RasterizerStateImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

void RasterizerStateImpl< Target >::Initialize( DeviceImpl< Target >* pDevice,
    const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_UNUSED( pDevice );
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_UNUSED( pDevice );

    NN_STATIC_ASSERT( sizeof( this->nvnPolygonState ) == sizeof( NVNpolygonState ) );
    NN_STATIC_ASSERT( sizeof( this->nvnMultisampleState ) == sizeof( NVNmultisampleState ) );

    NVNpolygonState* pPolygonState =
        reinterpret_cast< NVNpolygonState* >( &this->nvnPolygonState );
    NVNmultisampleState* pMultisampleState =
        reinterpret_cast< NVNmultisampleState* >( &this->nvnMultisampleState );

    NN_GFX_CALL_NVN_FUNCTION( nvnPolygonStateSetDefaults( pPolygonState ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnPolygonStateSetCullFace(
        pPolygonState, Nvn::GetCullMode( info.GetCullMode() ) ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnPolygonStateSetFrontFace(
        pPolygonState, Nvn::GetFrontFace( info.GetFrontFace() ) ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnPolygonStateSetPolygonMode(
        pPolygonState, Nvn::GetFillMode( info.GetFillMode() ) ) );

    // Save for nvnCommandBufferSetPolygonOffsetClamp
    this->nvnDepthBias = static_cast< float >( info.GetDepthBias() );
    this->nvnDepthBiasClamp = info.GetDepthBiasClamp();
    this->nvnSlopeScaledDepthBias = info.GetSlopeScaledDepthBias();
    bool depthBiasEnabled = ( this->nvnDepthBias != 0 ||
        this->nvnDepthBiasClamp != 0.0f || this->nvnSlopeScaledDepthBias != 0.0f );

    // Set the offset enable flags based on if the bias is set at all.
    NN_GFX_CALL_NVN_FUNCTION( nvnPolygonStateSetPolygonOffsetEnables( pPolygonState,
        ( depthBiasEnabled ? NVN_POLYGON_OFFSET_ENABLE_POINT_BIT |
        NVN_POLYGON_OFFSET_ENABLE_LINE_BIT | NVN_POLYGON_OFFSET_ENABLE_FILL_BIT
        : NVN_POLYGON_OFFSET_ENABLE_NONE ) ) );

    this->nvnSampleMask = info.GetMultisampleStateInfo().GetSampleMask();

    NN_GFX_CALL_NVN_FUNCTION( nvnMultisampleStateSetDefaults( pMultisampleState ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnMultisampleStateSetMultisampleEnable(
        pMultisampleState, info.IsMultisampleEnabled() ) );
    int sampleCount = info.GetMultisampleStateInfo().GetSampleCount();
    NN_GFX_CALL_NVN_FUNCTION( nvnMultisampleStateSetSamples(
        pMultisampleState, sampleCount > 1 ? sampleCount : 0) );
    NN_GFX_CALL_NVN_FUNCTION( nvnMultisampleStateSetAlphaToCoverageEnable(
        pMultisampleState, info.GetMultisampleStateInfo().IsAlphaToCoverageEnabled() ) );

    this->flags.SetBit( Flag_MultisampleEnabled, info.IsMultisampleEnabled() );
    this->flags.SetBit( Flag_DepthClipEnabled, info.IsDepthClipEnabled() );
    this->flags.SetBit( Flag_RasterEnabled, info.IsRasterEnabled() );
    this->flags.SetBit( Flag_ConservativeRasterEnabled,
        info.GetConservativeRasterizationMode() == ConservativeRasterizationMode_Enable );

    this->state = State_Initialized;
}

void RasterizerStateImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_UNUSED( pDevice );
    NN_SDK_REQUIRES( IsInitialized( *this ) );
    NN_UNUSED( pDevice );

    this->state = State_NotInitialized;
}

size_t BlendStateImpl< Target >::GetRequiredMemorySize( const InfoType& info ) NN_NOEXCEPT
{
    return info.GetBlendTargetCount() * sizeof( NVNblendState );
}

BlendStateImpl< Target >::BlendStateImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

BlendStateImpl< Target >::~BlendStateImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

void BlendStateImpl< Target >::SetMemory( void* pMemory, size_t size ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( pMemory || size == 0 );
    NN_SDK_REQUIRES( this->state == State_NotInitialized );

    this->pNvnBlendStateData = pMemory;
    this->memorySize = static_cast< Bit32 >( size );
}

void* BlendStateImpl< Target >::GetMemory() NN_NOEXCEPT
{
    return this->pNvnBlendStateData;
}

const void* BlendStateImpl< Target >::GetMemory() const NN_NOEXCEPT
{
    return this->pNvnBlendStateData;
}

void BlendStateImpl< Target >::Initialize(
    DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_UNUSED( pDevice );
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_SDK_REQUIRES_NOT_NULL( this->pNvnBlendStateData.ptr );
    NN_SDK_REQUIRES( info.GetBlendTargetCount() <= 8 );
    NN_SDK_REQUIRES( this->memorySize >= GetRequiredMemorySize( info ) );
    NN_UNUSED( pDevice );

    NN_STATIC_ASSERT( sizeof( this->nvnColorState ) == sizeof( NVNcolorState ) );
    NN_STATIC_ASSERT( sizeof( this->nvnChannelMaskState ) == sizeof( NVNchannelMaskState ) );

    NVNcolorState* pColorState = reinterpret_cast< NVNcolorState* >( &this->nvnColorState );
    NVNchannelMaskState* pChannelMaskState =
        reinterpret_cast< NVNchannelMaskState* >( &this->nvnChannelMaskState );
    NVNblendState* pBlendStateData = this->pNvnBlendStateData;
    const BlendTargetStateInfo* pBlendInfo = info.GetBlendTargetStateInfoArray();

    this->nvnBlendConstant[ 0 ] = info.GetBlendConstant( ColorChannel_Red );
    this->nvnBlendConstant[ 1 ] = info.GetBlendConstant( ColorChannel_Green );
    this->nvnBlendConstant[ 2 ] = info.GetBlendConstant( ColorChannel_Blue );
    this->nvnBlendConstant[ 3 ] = info.GetBlendConstant( ColorChannel_Alpha );

    // Set the default state
    NN_GFX_CALL_NVN_FUNCTION( nvnColorStateSetDefaults( pColorState ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnChannelMaskStateSetDefaults( pChannelMaskState ) );

    this->targetCount = static_cast< Bit8 >( info.GetBlendTargetCount() );

    for ( int index = 0; index < info.GetBlendTargetCount(); index++ )
    {
        int infoIndex = info.IsIndependentBlendEnabled() ? index : 0;

        NN_GFX_CALL_NVN_FUNCTION( nvnColorStateSetBlendEnable(
            pColorState, index, pBlendInfo[ infoIndex ].IsBlendEnabled() ) );

        NN_GFX_CALL_NVN_FUNCTION( nvnBlendStateSetDefaults( &pBlendStateData[ index ] ) );
        NN_GFX_CALL_NVN_FUNCTION( nvnBlendStateSetBlendTarget( &pBlendStateData[ index ], index ) );
        NN_GFX_CALL_NVN_FUNCTION( nvnBlendStateSetBlendFunc( &pBlendStateData[ index ],
            Nvn::GetBlendFunction( pBlendInfo[ infoIndex ].GetSourceColorBlendFactor() ),
            Nvn::GetBlendFunction( pBlendInfo[ infoIndex ].GetDestinationColorBlendFactor() ),
            Nvn::GetBlendFunction( pBlendInfo[ infoIndex ].GetSourceAlphaBlendFactor() ),
            Nvn::GetBlendFunction( pBlendInfo[ infoIndex ].GetDestinationAlphaBlendFactor() ) ) );

        NN_GFX_CALL_NVN_FUNCTION( nvnBlendStateSetBlendEquation( &pBlendStateData[ index ],
            Nvn::GetBlendEquation( pBlendInfo[ infoIndex ].GetColorBlendFunction() ),
            Nvn::GetBlendEquation( pBlendInfo[ infoIndex ].GetAlphaBlendFunction() ) ) );

        ChannelMask channelMask = static_cast< ChannelMask >( pBlendInfo[ infoIndex ].GetChannelMask() );
        NN_GFX_CALL_NVN_FUNCTION( nvnChannelMaskStateSetChannelMask( pChannelMaskState, index,
            ( (channelMask & ChannelMask_Red) == ChannelMask_Red ) ? NVN_TRUE : NVN_FALSE,
            ( (channelMask & ChannelMask_Green) == ChannelMask_Green ) ? NVN_TRUE : NVN_FALSE,
            ( (channelMask & ChannelMask_Blue) == ChannelMask_Blue ) ? NVN_TRUE : NVN_FALSE,
            ( (channelMask & ChannelMask_Alpha) == ChannelMask_Alpha ) ? NVN_TRUE : NVN_FALSE ) );
    }

    if( info.IsLogicOperationEnabled() )
    {
        NN_GFX_CALL_NVN_FUNCTION( nvnColorStateSetLogicOp(
            pColorState, Nvn::GetLogicOperation( info.GetLogicOperation() ) ) );
    }

    // NOTE: NVN doesn't require any explicit enables for dual-source blending.

    this->state = State_Initialized;
}

void BlendStateImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_UNUSED( pDevice );
    NN_SDK_REQUIRES( IsInitialized( *this ) );
    NN_UNUSED( pDevice );

    this->state = State_NotInitialized;
}

DepthStencilStateImpl< Target >::DepthStencilStateImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

DepthStencilStateImpl< Target >::~DepthStencilStateImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

void DepthStencilStateImpl< Target >::Initialize(
    DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_UNUSED( pDevice );
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_UNUSED( pDevice );

    NN_STATIC_ASSERT( sizeof( NVNdepthStencilState ) == sizeof( this->nvnDepthStencilState ) );

    NVNdepthStencilState* pDepthStencilState = reinterpret_cast< NVNdepthStencilState* >( &this->nvnDepthStencilState );

    this->flag = info.ToData()->flag;

    NN_GFX_CALL_NVN_FUNCTION( nvnDepthStencilStateSetDefaults( pDepthStencilState ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnDepthStencilStateSetDepthTestEnable(
        pDepthStencilState, info.IsDepthTestEnabled() ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnDepthStencilStateSetDepthWriteEnable(
        pDepthStencilState, info.IsDepthWriteEnabled() ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnDepthStencilStateSetStencilTestEnable(
        pDepthStencilState, info.IsStencilTestEnabled() ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnDepthStencilStateSetDepthFunc(
        pDepthStencilState, Nvn::GetDepthFunction( info.GetDepthComparisonFunction() ) ) );

    // Back Face
    NN_GFX_CALL_NVN_FUNCTION( nvnDepthStencilStateSetStencilFunc(
        pDepthStencilState, NVN_FACE_BACK, Nvn::GetStencilFunction(
        info.GetBackStencilStateInfo().GetComparisonFunction() ) ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnDepthStencilStateSetStencilOp(
        pDepthStencilState, NVN_FACE_BACK,
        Nvn::GetStencilOperation( info.GetBackStencilStateInfo().GetStencilFailOperation() ),
        Nvn::GetStencilOperation( info.GetBackStencilStateInfo().GetDepthFailOperation() ),
        Nvn::GetStencilOperation( info.GetBackStencilStateInfo().GetDepthPassOperation() ) ) );

    // Front Face
    NN_GFX_CALL_NVN_FUNCTION( nvnDepthStencilStateSetStencilFunc(
        pDepthStencilState, NVN_FACE_FRONT, Nvn::GetStencilFunction(
        info.GetFrontStencilStateInfo().GetComparisonFunction() ) ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnDepthStencilStateSetStencilOp(
        pDepthStencilState, NVN_FACE_FRONT,
        Nvn::GetStencilOperation( info.GetFrontStencilStateInfo().GetStencilFailOperation() ),
        Nvn::GetStencilOperation( info.GetFrontStencilStateInfo().GetDepthFailOperation() ),
        Nvn::GetStencilOperation( info.GetFrontStencilStateInfo().GetDepthPassOperation() ) ) );

    // Save the back/front stencil reference values for later
    this->nvnStencilBackRef = info.GetBackStencilStateInfo().GetStencilRef();
    this->nvnStencilFrontRef = info.GetFrontStencilStateInfo().GetStencilRef();

    // Value mask is used to mask the stencil values read from the buffer and the primitive
    this->nvnStencilValueMask = info.GetStencilReadMask();

    // Mask is used to mask the written stencil value.
    this->nvnStencilMask = info.GetStencilWriteMask();

    this->state = State_Initialized;
}

void DepthStencilStateImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_UNUSED( pDevice );
    NN_SDK_REQUIRES( IsInitialized( *this ) );
    NN_UNUSED( pDevice );

    this->state = State_NotInitialized;
}

size_t VertexStateImpl< Target >::GetRequiredMemorySize( const InfoType& info ) NN_NOEXCEPT
{
    int maxSlot = -1;
    const VertexAttributeStateInfo* pAttribStates = info.GetVertexAttributeStateInfoArray();
    for( int idxAttribute = 0, attributeCount = info.GetVertexAttributeCount();
        idxAttribute < attributeCount; ++idxAttribute )
    {
        const VertexAttributeStateInfo& attribState = pAttribStates[ idxAttribute ];
        if( attribState.GetShaderSlot() < 0 || attribState.GetNamePtr() )
        {
            maxSlot = 15; // TODO: 名前による関連付けの場合はとりあえず最大値（64 バイト）
        }
        else
        {
            maxSlot = std::max NN_PREVENT_MACRO_FUNC (
                maxSlot, pAttribStates[ idxAttribute ].GetShaderSlot() );
        }
    }
    return sizeof( NVNvertexStreamState ) * info.GetVertexBufferCount()
        + sizeof( NVNvertexAttribState ) * ( maxSlot + 1 );
}

VertexStateImpl< Target >::VertexStateImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

VertexStateImpl< Target >::~VertexStateImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

void VertexStateImpl< Target >::SetMemory( void* pMemory, size_t size ) NN_NOEXCEPT
{
    NN_UNUSED( pMemory );
    NN_UNUSED( size );

    NN_SDK_ASSERT( nn::util::BytePtr( pMemory ).IsAligned( RequiredMemoryInfo_Alignment ) );

    this->pNvnVertexStreamState = pMemory;
    this->memorySize = static_cast< uint32_t >( size );
}

void* VertexStateImpl< Target >::GetMemory() NN_NOEXCEPT
{
    return this->pNvnVertexStreamState;
}

const void* VertexStateImpl< Target >::GetMemory() const NN_NOEXCEPT
{
    return this->pNvnVertexStreamState;
}

void VertexStateImpl< Target >::Initialize( DeviceImpl< Target >* pDevice,
    const InfoType& info, const ShaderImpl< Target >* pVertexShader ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_SDK_REQUIRES( pVertexShader == NULL || IsInitialized( *pVertexShader ) );
    NN_SDK_ASSERT( GetRequiredMemorySize( info ) == 0 ||
        ( this->pNvnVertexStreamState.ptr &&
        this->memorySize >= GetRequiredMemorySize( info ) ) );
    NN_UNUSED( pDevice );

    this->vertexStreamStateCount = info.GetVertexBufferCount();
    this->pNvnVertexAttribState = nn::util::BytePtr( this->pNvnVertexStreamState.ptr ).Advance(
        sizeof( NVNvertexStreamState ) * info.GetVertexBufferCount() ).Get();

    for( int idxVertexBufferState = 0; idxVertexBufferState
        < info.GetVertexBufferCount(); ++idxVertexBufferState )
    {
        const VertexBufferStateInfo& src = info.GetVertexBufferStateInfoArray()[ idxVertexBufferState ];
        NVNvertexStreamState* pDst = static_cast<
            NVNvertexStreamState* >( this->pNvnVertexStreamState ) + idxVertexBufferState;
        NN_GFX_CALL_NVN_FUNCTION( nvnVertexStreamStateSetDefaults( pDst ) );
        NN_GFX_CALL_NVN_FUNCTION( nvnVertexStreamStateSetStride( pDst, src.GetStride() ) );
        NN_GFX_CALL_NVN_FUNCTION( nvnVertexStreamStateSetDivisor( pDst, src.GetDivisor() ) );
    }

    const int maxAttribs = 256;
    Bit32 bitArrayMemory[ ( maxAttribs + 31 ) >> 5 ] = {};
    nn::util::BitArray setAttribs( bitArrayMemory, sizeof( bitArrayMemory ), maxAttribs );
    int maxSlot = -1;
    for( int idxVertexAttribState = 0; idxVertexAttribState
        < info.GetVertexAttributeCount(); ++idxVertexAttribState )
    {
        const VertexAttributeStateInfo& src =
            info.GetVertexAttributeStateInfoArray()[ idxVertexAttribState ];
        int idxDst = src.GetShaderSlot();
        if( pVertexShader && src.GetNamePtr() )
        {
            idxDst = pVertexShader->GetInterfaceSlot(
                ShaderStage_Vertex, ShaderInterfaceType_Input, src.GetNamePtr() );
        }
        if( idxDst < 0 )
        {
            continue;
        }

        NVNvertexAttribState* pDst = static_cast<
            NVNvertexAttribState* >( this->pNvnVertexAttribState ) + idxDst;

        NN_GFX_CALL_NVN_FUNCTION( nvnVertexAttribStateSetDefaults( pDst ) );
        NN_GFX_CALL_NVN_FUNCTION( nvnVertexAttribStateSetFormat( pDst,
            Nvn::GetAttributeFormat( src.GetFormat() ), src.GetOffset() ) );
        NN_GFX_CALL_NVN_FUNCTION( nvnVertexAttribStateSetStreamIndex( pDst, src.GetBufferIndex() ) );

        NN_SDK_ASSERT( !setAttribs.test( idxDst ) );
        setAttribs.set( idxDst, true );
        maxSlot = std::max NN_PREVENT_MACRO_FUNC ( maxSlot, idxDst );
    }
    this->vertexAttributeStateCount = info.GetVertexAttributeCount() ? maxSlot + 1 : 0;

    // 未使用スロットは SetDefaults で埋める
    if( maxSlot >= info.GetVertexAttributeCount() )
    {
        for( int idxSlot = 0; idxSlot <= maxSlot; ++idxSlot )
        {
            if( !setAttribs.test( idxSlot ) )
            {
                NN_GFX_CALL_NVN_FUNCTION( nvnVertexAttribStateSetDefaults(
                    static_cast< NVNvertexAttribState* >( this->pNvnVertexAttribState ) + idxSlot ) );
            }
        }
    }

    this->state = State_Initialized;
}

void VertexStateImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_UNUSED( pDevice );
    NN_SDK_REQUIRES( IsInitialized( *this ) );

    this->state = State_NotInitialized;
}

TessellationStateImpl< Target >::TessellationStateImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

TessellationStateImpl< Target >::~TessellationStateImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

void TessellationStateImpl< Target >::Initialize(
    DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pDevice );
    NN_UNUSED( pDevice );

    int maxPatchSize = int();
    NN_SDK_ASSERT( ( NN_GFX_CALL_NVN_FUNCTION( nvnDeviceGetInteger(
        pDevice->ToData()->pNvnDevice, NVN_DEVICE_INFO_MAX_PATCH_SIZE, &maxPatchSize ) ),
        info.GetPatchControlPointCount() <= maxPatchSize ) );
    NN_UNUSED( maxPatchSize );

    this->patchSize = info.GetPatchControlPointCount();

    this->state = State_Initialized;
}

void TessellationStateImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pDevice );
    NN_UNUSED( pDevice );
    NN_SDK_REQUIRES( IsInitialized( *this ) );

    this->state = State_NotInitialized;
}

size_t ViewportScissorStateImpl< Target >::GetRequiredMemorySize( const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( info.GetViewportCount() > 0 );
    NN_SDK_ASSERT( !info.IsScissorEnabled() || info.GetViewportCount() == info.GetScissorCount() );
    int extraViewportCount = info.GetViewportCount() - 1;
    return sizeof( float ) * 4 * extraViewportCount +
        sizeof( float ) * 2 * extraViewportCount + sizeof( int32_t ) * 4 * extraViewportCount;
}

ViewportScissorStateImpl< Target >::ViewportScissorStateImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

ViewportScissorStateImpl< Target >::~ViewportScissorStateImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

void ViewportScissorStateImpl< Target >::SetMemory( void* pMemory, size_t size ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( pMemory || size == 0 );

    this->pWorkMemory = pMemory;
    this->memorySize = static_cast< uint32_t >( size );
}

void* ViewportScissorStateImpl< Target >::GetMemory() NN_NOEXCEPT
{
    return this->pWorkMemory;
}

const void* ViewportScissorStateImpl< Target >::GetMemory() const NN_NOEXCEPT
{
    return this->pWorkMemory;
}

void ViewportScissorStateImpl< Target >::Initialize(
    DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_UNUSED( pDevice );
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_UNUSED( pDevice );

    NN_SDK_ASSERT( info.GetViewportCount() > 0 );
    NN_SDK_ASSERT( !info.IsScissorEnabled() || info.GetViewportCount() == info.GetScissorCount() );

    NN_SDK_REQUIRES( info.GetViewportCount() < 2 || this->pWorkMemory );
    NN_SDK_REQUIRES( info.GetViewportCount() < 2 ||
        this->memorySize >= GetRequiredMemorySize( info ) );

    this->flag = info.ToData()->flag;

    this->viewportCount = info.GetViewportCount();

    const ViewportStateInfo* pViewports = info.GetViewportStateInfoArray();
    const ScissorStateInfo* pScissor = info.GetScissorStateInfoArray();
    NN_SDK_ASSERT_NOT_NULL( pViewports );
    NN_SDK_ASSERT( !info.IsScissorEnabled() || pScissor );

    this->viewport[ 0 ] = pViewports[ 0 ].GetOriginX();
    this->viewport[ 1 ] = pViewports[ 0 ].GetOriginY();
    this->viewport[ 2 ] = pViewports[ 0 ].GetWidth();
    this->viewport[ 3 ] = pViewports[ 0 ].GetHeight();
    this->depthRange[ 0 ] = pViewports[ 0 ].GetMinDepth();
    this->depthRange[ 1 ] = pViewports[ 0 ].GetMaxDepth();
    // nvn ではシザーの有効・無効の設定がない
    if( info.IsScissorEnabled() )
    {
        this->scissor[ 0 ] = pScissor[ 0 ].GetOriginX();
        this->scissor[ 1 ] = pScissor[ 0 ].GetOriginY();
        this->scissor[ 2 ] = pScissor[ 0 ].GetWidth();
        this->scissor[ 3 ] = pScissor[ 0 ].GetHeight();
    }
    else
    {
        this->scissor[ 0 ] = 0;
        this->scissor[ 1 ] = 0;
        this->scissor[ 2 ] = 0x7FFFFFFF;
        this->scissor[ 3 ] = 0x7FFFFFFF;
    }

    int extraViewportCount = info.GetViewportCount() - 1;

    nn::util::BytePtr ptr( this->pWorkMemory.ptr );
    float* viewportArray = ptr.Get< float >();
    float* depthRangeArray = ptr.Advance( sizeof( float ) * 4 * extraViewportCount ).Get< float >();
    int32_t* scissorArray = ptr.Advance( sizeof( float ) * 2 * extraViewportCount ).Get< int32_t >();

    for( int idx = 1; idx < this->viewportCount; ++idx )
    {
        int idxExtra = idx - 1;
        const ViewportStateInfo& viewportInfo = pViewports[ idx ];
        viewportArray[ idxExtra * 4 + 0 ] = viewportInfo.GetOriginX();
        viewportArray[ idxExtra * 4 + 1 ] = viewportInfo.GetOriginY();
        viewportArray[ idxExtra * 4 + 2 ] = viewportInfo.GetWidth();
        viewportArray[ idxExtra * 4 + 3 ] = viewportInfo.GetHeight();
        depthRangeArray[ idxExtra * 2 + 0 ] = viewportInfo.GetMinDepth();
        depthRangeArray[ idxExtra * 2 + 1 ] = viewportInfo.GetMaxDepth();
    }

    for( int idx = 1; idx < this->viewportCount; ++idx )
    {
        int idxExtra = idx - 1;
        const ScissorStateInfo& scissorInfo = pScissor[ idx ];
        // nvn ではシザーの有効・無効の設定がない
        if( info.IsScissorEnabled() )
        {
            scissorArray[ idxExtra * 4 + 0 ] = scissorInfo.GetOriginX();
            scissorArray[ idxExtra * 4 + 1 ] = scissorInfo.GetOriginY();
            scissorArray[ idxExtra * 4 + 2 ] = scissorInfo.GetWidth();
            scissorArray[ idxExtra * 4 + 3 ] = scissorInfo.GetHeight();
        }
        else
        {
            scissorArray[ idxExtra * 4 + 0 ] = 0;
            scissorArray[ idxExtra * 4 + 1 ] = 0;
            scissorArray[ idxExtra * 4 + 2 ] = 0x7FFFFFFF;
            scissorArray[ idxExtra * 4 + 3 ] = 0x7FFFFFFF;
        }
    }

    this->state = State_Initialized;
}

void ViewportScissorStateImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_UNUSED( pDevice );
    NN_SDK_REQUIRES( IsInitialized( *this ) );
    NN_UNUSED( pDevice );

    this->state = State_NotInitialized;
}

}
}
}
