﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <string>
#include <vector>

#include <nn/nn_StaticAssert.h>

#include <nn/util/util_BitPack.h>
#include <nn/util/util_BytePtr.h>

#include <nn/gfx/gfx_StateInfo.h>

#include <nn/gfx/detail/gfx_State-api.d3d.11.h>
#include <nn/gfx/detail/gfx_Shader-api.d3d.11.h>
#include <nn/gfx/detail/gfx_Device-api.d3d.11.h>

#include <nn/gfx/detail/gfx_Misc.h>

#include "gfx_CommonHelper.h"
#include "gfx_D3dHelper.h"

namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationD3d11 Target;

namespace {

template< typename T >
struct FreeDeleter
{
    void operator()( T* ptr )
    {
        if( ptr )
        {
            free( ptr );
        }
    }
};

}

RasterizerStateImpl< Target >::RasterizerStateImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

RasterizerStateImpl< Target >::~RasterizerStateImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

void RasterizerStateImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( state == State_NotInitialized );

    D3D11_RASTERIZER_DESC desc;

    desc.FillMode = D3d::GetFillMode( info.GetFillMode() );
    desc.CullMode = D3d::GetCullMode( info.GetCullMode() );
    desc.FrontCounterClockwise = info.GetFrontFace() == FrontFace_Ccw ? TRUE : FALSE;
    desc.DepthBias = info.GetDepthBias();
    desc.DepthBiasClamp = info.GetDepthBiasClamp();
    desc.SlopeScaledDepthBias = info.GetSlopeScaledDepthBias();
    desc.DepthClipEnable = info.IsDepthClipEnabled() ? TRUE : FALSE;
    desc.ScissorEnable = info.IsScissorEnabled() ? TRUE : FALSE;
    desc.MultisampleEnable = info.IsMultisampleEnabled() ? TRUE : FALSE;
    desc.AntialiasedLineEnable = FALSE; // not supported

    // Conservative rasterization is supported since D3D11.3 on Windows10.
    //desc.ConservativeRaster = info.GetConservativeRasterizationMode() == ConservativeRasterizationMode_Enable ? TRUE : FALSE;

    ID3D11Device* pD3dDevice = static_cast< ID3D11Device* >( pDevice->ToData()->renderingContext.hD3dDevice );
    NN_SDK_ASSERT_NOT_NULL( pD3dDevice );

    ID3D11RasterizerState* pD3dRasterizerState;
    NN_GFX_CALL_D3D_FUNCTION( pD3dDevice->CreateRasterizerState( &desc, &pD3dRasterizerState ) );
    NN_SDK_ASSERT( IsD3dHandleValid( pD3dRasterizerState ) );

    this->pRasterizerState = pD3dRasterizerState;
    this->sampleMask = static_cast< uint32_t >( info.GetMultisampleStateInfo().GetSampleMask() );
    this->state = State_Initialized;
}

void RasterizerStateImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_UNUSED( pDevice );

    if ( this->pRasterizerState )
    {
        ID3D11RasterizerState* pD3dRasterizerState = static_cast< ID3D11RasterizerState* >( this->pRasterizerState );
        NN_GFX_CALL_D3D_FUNCTION( pD3dRasterizerState->Release() );
        this->pRasterizerState = NULL;
    }


    this->state = State_NotInitialized;
}

size_t BlendStateImpl< Target >::GetRequiredMemorySize( const InfoType& info ) NN_NOEXCEPT
{
    NN_UNUSED( info );
    return 16; // 一時的措置としてダミーで16バイト加算します。（サイズ0を許容しないTestsアプリがあるため。）
}

BlendStateImpl< Target >::BlendStateImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

BlendStateImpl< Target >::~BlendStateImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

void BlendStateImpl< Target >::SetMemory( void* pMemory, size_t size ) NN_NOEXCEPT
{
    NN_UNUSED( pMemory );
    NN_UNUSED( size );
}

void* BlendStateImpl< Target >::GetMemory() NN_NOEXCEPT
{
    return NULL;
}

const void* BlendStateImpl< Target >::GetMemory() const NN_NOEXCEPT
{
    return NULL;
}

void BlendStateImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_NotInitialized );

    D3D11_BLEND_DESC desc;

    desc.AlphaToCoverageEnable = info.IsAlphaToCoverageEnabled() ? TRUE : FALSE;
    desc.IndependentBlendEnable = info.IsIndependentBlendEnabled() ? TRUE : FALSE;

    const BlendTargetStateInfo* pTargetStates = info.GetBlendTargetStateInfoArray();
    const int maxRenderTarget = sizeof( desc.RenderTarget ) / sizeof( D3D11_RENDER_TARGET_BLEND_DESC );
    for ( int rtIndex = 0; rtIndex < maxRenderTarget; ++rtIndex )
    {
        if ( rtIndex < info.GetBlendTargetCount() )
        {
            const BlendTargetStateInfo& targetState = pTargetStates[ rtIndex ];
            desc.RenderTarget[ rtIndex ].BlendEnable = targetState.IsBlendEnabled() ? TRUE : FALSE;
            desc.RenderTarget[ rtIndex ].SrcBlend = D3d::GetBlendFactor( targetState.GetSourceColorBlendFactor() );
            desc.RenderTarget[ rtIndex ].DestBlend = D3d::GetBlendFactor( targetState.GetDestinationColorBlendFactor() );
            desc.RenderTarget[ rtIndex ].BlendOp = D3d::GetBlendOperation( targetState.GetColorBlendFunction() );
            desc.RenderTarget[ rtIndex ].SrcBlendAlpha = D3d::GetBlendFactor( targetState.GetSourceAlphaBlendFactor() );
            desc.RenderTarget[ rtIndex ].DestBlendAlpha = D3d::GetBlendFactor( targetState.GetDestinationAlphaBlendFactor() );
            desc.RenderTarget[ rtIndex ].BlendOpAlpha = D3d::GetBlendOperation( targetState.GetAlphaBlendFunction() );
            desc.RenderTarget[ rtIndex ].RenderTargetWriteMask = static_cast< UINT8 >( D3d::GetChannelMask( targetState.GetChannelMask() ) );
        }
        else
        {
            desc.RenderTarget[ rtIndex ].BlendEnable = FALSE;
        }
    }

    ID3D11Device* pD3dDevice = static_cast< ID3D11Device* >( pDevice->ToData()->renderingContext.hD3dDevice );
    NN_SDK_ASSERT_NOT_NULL( pD3dDevice );
    ID3D11BlendState* pD3dBlendState;
    NN_GFX_CALL_D3D_FUNCTION( pD3dDevice->CreateBlendState( &desc, &pD3dBlendState ) );
    NN_SDK_ASSERT( IsD3dHandleValid( pD3dBlendState ) );

    this->pBlendState = pD3dBlendState;
    this->blendColor[ 0 ] = info.GetBlendConstant( ColorChannel_Red );
    this->blendColor[ 1 ] = info.GetBlendConstant( ColorChannel_Green );
    this->blendColor[ 2 ] = info.GetBlendConstant( ColorChannel_Blue );
    this->blendColor[ 3 ] = info.GetBlendConstant( ColorChannel_Alpha );
    this->blendTargetCount = info.GetBlendTargetCount();
    this->state = State_Initialized;
}

void BlendStateImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_UNUSED( pDevice );

    if ( this->pBlendState )
    {
        ID3D11BlendState* pD3dBlendState = static_cast< ID3D11BlendState* >( this->pBlendState );
        NN_GFX_CALL_D3D_FUNCTION( pD3dBlendState->Release() );
        this->pBlendState = NULL;
    }

    this->state = State_NotInitialized;
}

DepthStencilStateImpl< Target >::DepthStencilStateImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

DepthStencilStateImpl< Target >::~DepthStencilStateImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

void DepthStencilStateImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_NotInitialized );

    D3D11_DEPTH_STENCIL_DESC desc;

    desc.DepthEnable = info.IsDepthTestEnabled() ? TRUE : FALSE;
    desc.DepthWriteMask = info.IsDepthWriteEnabled() ? D3D11_DEPTH_WRITE_MASK_ALL : D3D11_DEPTH_WRITE_MASK_ZERO;
    desc.DepthFunc = D3d::GetComparisonFunction( info.GetDepthComparisonFunction() );
    desc.StencilEnable = info.IsStencilTestEnabled() ? TRUE : FALSE;
    desc.StencilReadMask = static_cast< UINT8 >( info.GetStencilReadMask() );
    desc.StencilWriteMask = static_cast< UINT8 >( info.GetStencilWriteMask() );
    desc.FrontFace.StencilFailOp = D3d::GetStencilOperation( info.GetFrontStencilStateInfo().GetStencilFailOperation() );
    desc.FrontFace.StencilDepthFailOp = D3d::GetStencilOperation( info.GetFrontStencilStateInfo().GetDepthFailOperation() );
    desc.FrontFace.StencilPassOp = D3d::GetStencilOperation( info.GetFrontStencilStateInfo().GetDepthPassOperation() );
    desc.FrontFace.StencilFunc = D3d::GetComparisonFunction( info.GetFrontStencilStateInfo().GetComparisonFunction() );
    desc.BackFace.StencilFailOp = D3d::GetStencilOperation( info.GetBackStencilStateInfo().GetStencilFailOperation() );
    desc.BackFace.StencilDepthFailOp = D3d::GetStencilOperation( info.GetBackStencilStateInfo().GetDepthFailOperation() );
    desc.BackFace.StencilPassOp = D3d::GetStencilOperation( info.GetBackStencilStateInfo().GetDepthPassOperation() );
    desc.BackFace.StencilFunc = D3d::GetComparisonFunction( info.GetBackStencilStateInfo().GetComparisonFunction() );

    NN_SDK_ASSERT( info.GetFrontStencilStateInfo().GetStencilRef() == info.GetBackStencilStateInfo().GetStencilRef() );

    ID3D11Device* pD3dDevice = static_cast< ID3D11Device* >( pDevice->ToData()->renderingContext.hD3dDevice );
    NN_SDK_ASSERT_NOT_NULL( pD3dDevice );

    ID3D11DepthStencilState* pD3dDepthStencilState;
    NN_GFX_CALL_D3D_FUNCTION( pD3dDevice->CreateDepthStencilState( &desc, &pD3dDepthStencilState ) );
    NN_SDK_ASSERT( IsD3dHandleValid( pD3dDepthStencilState ) );

    this->pDepthStencilState = pD3dDepthStencilState;
    this->stencilRef = info.GetFrontStencilStateInfo().GetStencilRef();
    this->state = State_Initialized;
}

void DepthStencilStateImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_UNUSED( pDevice );

    if ( this->pDepthStencilState )
    {
        ID3D11DepthStencilState* pD3dDepthStencilState = static_cast< ID3D11DepthStencilState* >( this->pDepthStencilState );
        NN_GFX_CALL_D3D_FUNCTION( pD3dDepthStencilState->Release() );
        this->pDepthStencilState = NULL;
    }

    this->state = State_NotInitialized;
}

size_t VertexStateImpl< Target >::GetRequiredMemorySize( const InfoType& info ) NN_NOEXCEPT
{
    NN_UNUSED( info );
    return 16; // 一時的措置としてダミーで16バイト加算します。（サイズ0を許容しないTestsアプリがあるため。）
}

VertexStateImpl< Target >::VertexStateImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

VertexStateImpl< Target >::~VertexStateImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

void VertexStateImpl< Target >::SetMemory( void* value, size_t size ) NN_NOEXCEPT
{
    NN_UNUSED( value );
    NN_UNUSED( size );
}

void* VertexStateImpl< Target >::GetMemory() NN_NOEXCEPT
{
    return NULL;
}

const void* VertexStateImpl< Target >::GetMemory() const NN_NOEXCEPT
{
    return NULL;
}

void VertexStateImpl< Target >::Initialize( DeviceImpl< Target >* pDevice,
    const InfoType& info, const ShaderImpl< Target >* pVertexShader ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_SDK_REQUIRES( pVertexShader && IsInitialized( *pVertexShader ) );

    this->attributeCount = info.GetVertexAttributeCount();
    NN_SDK_ASSERT( this->attributeCount <= D3D11_STANDARD_VERTEX_ELEMENT_COUNT );

    this->bufferCount = info.GetVertexBufferCount();

    D3D11_INPUT_ELEMENT_DESC inputElementDescs[ D3D11_STANDARD_VERTEX_ELEMENT_COUNT ];
    D3D11_SIGNATURE_PARAMETER_DESC shaderInputParameterDescs[ D3D11_STANDARD_VERTEX_ELEMENT_COUNT ];

    ID3D11ShaderReflection* pReflector = static_cast< ID3D11ShaderReflection* >(
        pVertexShader->ToData()->pShaderReflector[ ShaderStage_Vertex ] );

    std::vector< std::string > tempNames;
    tempNames.reserve( this->attributeCount );

    const VertexAttributeStateInfo* pAttributes = info.GetVertexAttributeStateInfoArray();
    for ( int idxAttribute = 0; idxAttribute < this->attributeCount; ++idxAttribute )
    {
        const VertexAttributeStateInfo& src = pAttributes[ idxAttribute ];

        if ( src.GetNamePtr() == NULL )
        {
            pReflector->GetInputParameterDesc( src.GetShaderSlot(), &shaderInputParameterDescs[ idxAttribute ] );
            inputElementDescs[ idxAttribute ].SemanticName = shaderInputParameterDescs[ idxAttribute ].SemanticName;
            inputElementDescs[ idxAttribute ].SemanticIndex = src.GetSemanticIndex();
        }
        else
        {
            // SemanticNameの終端が数字の場合、その値をSemanticIndexに設定し、SemanticNameからは切り離す必要があります。
            int digitCount = 0;
            const char* pName = src.GetNamePtr();
            int nameLength = static_cast< int >( strlen( pName ) );
            for ( int idx = nameLength - 1; idx >= 0; --idx )
            {
                if ( pName[ idx ] >= '0' && pName[ idx ] <= '9' )
                {
                    digitCount++;
                }
                else
                {
                    break;
                }
            }
            if ( digitCount > 0 )
            {
                tempNames.push_back( std::string( pName, nameLength - digitCount ) );
                inputElementDescs[ idxAttribute ].SemanticName = tempNames.back().c_str();
                inputElementDescs[ idxAttribute ].SemanticIndex = std::atoi( &pName[ nameLength - digitCount ] );
            }
            else
            {
                inputElementDescs[ idxAttribute ].SemanticName = src.GetNamePtr();
                inputElementDescs[ idxAttribute ].SemanticIndex = src.GetSemanticIndex();
            }
        }
        inputElementDescs[ idxAttribute ].Format = D3d::GetAttributeFormat( src.GetFormat() ).dxgiFormat;
        inputElementDescs[ idxAttribute ].InputSlot = src.GetBufferIndex();
        inputElementDescs[ idxAttribute ].AlignedByteOffset = static_cast< UINT >( src.GetOffset() );
        NN_SDK_ASSERT( src.GetBufferIndex() < this->bufferCount );
        inputElementDescs[ idxAttribute ].InstanceDataStepRate = this->bufferCount < 1 ? 0 :
            info.GetVertexBufferStateInfoArray()[ src.GetBufferIndex() ].GetDivisor();
        inputElementDescs[ idxAttribute ].InputSlotClass = inputElementDescs[ idxAttribute ].InstanceDataStepRate == 0 ?
            D3D11_INPUT_PER_VERTEX_DATA : D3D11_INPUT_PER_INSTANCE_DATA;
    }

    ID3D11Device* pD3dDevice = static_cast< ID3D11Device* >( pDevice->ToData()->renderingContext.hD3dDevice );
    NN_SDK_ASSERT_NOT_NULL( pD3dDevice );

    ID3DBlob* pShaderBlob = static_cast< ID3DBlob* >(
        pVertexShader->ToData()->pShaderBlob[ ShaderStage_Vertex ] );

    ID3D11InputLayout* pD3dInputLayout;
    NN_GFX_CALL_D3D_FUNCTION( pD3dDevice->CreateInputLayout( inputElementDescs,
        this->attributeCount, pVertexShader ? pShaderBlob->GetBufferPointer() : NULL,
        pVertexShader ? pShaderBlob->GetBufferSize() : 0, &pD3dInputLayout) );
    NN_SDK_ASSERT( IsD3dHandleValid( pD3dInputLayout ) );

    this->pInputLayout = pD3dInputLayout;
    this->state = State_Initialized;
}

void VertexStateImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_UNUSED( pDevice );

    if ( this->pInputLayout )
    {
        ID3D11InputLayout* pD3dInputLayout = static_cast< ID3D11InputLayout* >( this->pInputLayout );
        NN_GFX_CALL_D3D_FUNCTION( pD3dInputLayout->Release() );
        this->pInputLayout = NULL;
    }

    this->state = State_NotInitialized;
}

TessellationStateImpl< Target >::TessellationStateImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

TessellationStateImpl< Target >::~TessellationStateImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

void TessellationStateImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_UNUSED( pDevice );

    this->patchControlPointCount = info.GetPatchControlPointCount();

    this->state = State_Initialized;
}

void TessellationStateImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_UNUSED( pDevice );

    this->state = State_NotInitialized;
}

size_t ViewportScissorStateImpl< Target >::GetRequiredMemorySize( const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( info.GetViewportCount() > 0 );
    NN_SDK_ASSERT( !info.IsScissorEnabled() || info.GetViewportCount() == info.GetScissorCount() );
    int extraViewportCount = info.GetViewportCount() - 1;
    return sizeof( D3D11_VIEWPORT ) * extraViewportCount +
        ( info.IsScissorEnabled() ? sizeof( D3D11_RECT ) * extraViewportCount : 0 );
}

ViewportScissorStateImpl< Target >::ViewportScissorStateImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

ViewportScissorStateImpl< Target >::~ViewportScissorStateImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

void ViewportScissorStateImpl< Target >::SetMemory( void* value, size_t size ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( value || size == 0 );

    this->pWorkMemory = value;
    this->memorySize = static_cast< uint32_t >( size );
}

void* ViewportScissorStateImpl< Target >::GetMemory() NN_NOEXCEPT
{
    return this->pWorkMemory;
}

const void* ViewportScissorStateImpl< Target >::GetMemory() const NN_NOEXCEPT
{
    return this->pWorkMemory;
}

void ViewportScissorStateImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_UNUSED( pDevice );

    NN_SDK_ASSERT( info.GetViewportCount() > 0 );
    NN_SDK_ASSERT( !info.IsScissorEnabled() || info.GetViewportCount() == info.GetScissorCount() );

    NN_SDK_REQUIRES( info.GetViewportCount() < 2 || this->pWorkMemory );
    NN_SDK_REQUIRES( info.GetViewportCount() < 2 || this->memorySize >= GetRequiredMemorySize( info ) );

    this->flag = info.ToData()->flag;
    this->viewportCount = info.GetViewportCount();

    const ViewportStateInfo* pViewports = info.GetViewportStateInfoArray();
    const ScissorStateInfo* pScissor = info.GetScissorStateInfoArray();
    NN_SDK_ASSERT_NOT_NULL( pViewports );
    NN_SDK_ASSERT( !info.IsScissorEnabled() || pScissor );

    this->viewport[ 0 ] = pViewports[ 0 ].GetOriginX();
    this->viewport[ 1 ] = pViewports[ 0 ].GetOriginY();
    this->viewport[ 2 ] = pViewports[ 0 ].GetWidth();
    this->viewport[ 3 ] = pViewports[ 0 ].GetHeight();
    this->viewport[ 4 ] = static_cast< float >( pViewports[ 0 ].GetMinDepth() );
    this->viewport[ 5 ] = static_cast< float >( pViewports[ 0 ].GetMaxDepth() );
    if( info.IsScissorEnabled() )
    {
        this->scissor[ 0 ] = pScissor[ 0 ].GetOriginX();
        this->scissor[ 1 ] = pScissor[ 0 ].GetOriginY();
        this->scissor[ 2 ] = pScissor[ 0 ].GetOriginX() + pScissor[ 0 ].GetWidth();
        this->scissor[ 3 ] = pScissor[ 0 ].GetOriginY() + pScissor[ 0 ].GetHeight();
    }

    int extraViewportCount = info.GetViewportCount() - 1;

    nn::util::BytePtr ptr( this->pWorkMemory.ptr );
    D3D11_VIEWPORT* viewportArray = ptr.Get< D3D11_VIEWPORT >();
    D3D11_RECT* scissorArray = ptr.Advance( sizeof( D3D11_VIEWPORT ) * extraViewportCount ).Get< D3D11_RECT >();

    for( int idx = 1; idx < this->viewportCount; ++idx )
    {
        int idxExtra = idx - 1;
        const ViewportStateInfo& viewportInfo = pViewports[ idx ];
        viewportArray[ idxExtra ].TopLeftX = viewportInfo.GetOriginX();
        viewportArray[ idxExtra ].TopLeftY = viewportInfo.GetOriginY();
        viewportArray[ idxExtra ].Width = viewportInfo.GetWidth();
        viewportArray[ idxExtra ].Height = viewportInfo.GetHeight();
        viewportArray[ idxExtra ].MinDepth = viewportInfo.GetMinDepth();
        viewportArray[ idxExtra ].MaxDepth = viewportInfo.GetMaxDepth();
    }

    if( info.IsScissorEnabled() )
    {
        for( int idx = 1; idx < this->viewportCount; ++idx )
        {
            int idxExtra = idx - 1;
            const ScissorStateInfo& scissorInfo = pScissor[ idx ];
            scissorArray[ idxExtra ].left = scissorInfo.GetOriginX();
            scissorArray[ idxExtra ].top = scissorInfo.GetOriginY();
            scissorArray[ idxExtra ].right = scissorInfo.GetOriginX() + scissorInfo.GetWidth();
            scissorArray[ idxExtra ].bottom = scissorInfo.GetOriginY() + scissorInfo.GetHeight();
        }
    }

    this->state = State_Initialized;
}

void ViewportScissorStateImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_UNUSED( pDevice );

    this->state = State_NotInitialized;
}

}
}
}
