﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/gfx/gfx_SamplerInfo.h>

#include <nn/gfx/detail/gfx_Sampler-api.gx.2.h>
#include <nn/gfx/detail/gfx_Misc.h>

#include "gfx_GxHelper.h"

namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationGx2 Target;

SamplerImpl< Target >::SamplerImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

SamplerImpl< Target >::~SamplerImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized || this->flags.GetBit( Flag_Shared ) );
}

void SamplerImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    ASSERT( pDevice );
    ASSERT( this->state == State_NotInitialized );
    NN_UNUSED( pDevice );

    GX2Sampler* pGx2Sampler = reinterpret_cast< GX2Sampler* >( &this->gx2Sampler );

    NN_GFX_CALL_GX_FUNCTION( GX2InitSampler( pGx2Sampler,
        GX2_TEX_CLAMP_CLAMP, GX2_TEX_XY_FILTER_POINT ) );

    GX2TexAnisoRatio anisoRatio = GX2_TEX_ANISO_1_TO_1;
    if( info.GetFilterMode() & 0x40 )
    {
        if( info.GetMaxAnisotropy() >= 16 )
        {
            anisoRatio = GX2_TEX_ANISO_16_TO_1;
        }
        else if( info.GetMaxAnisotropy() >= 8 )
        {
            anisoRatio = GX2_TEX_ANISO_8_TO_1;
        }
        else if( info.GetMaxAnisotropy() >= 4 )
        {
            anisoRatio = GX2_TEX_ANISO_4_TO_1;
        }
        else if( info.GetMaxAnisotropy() >= 2 )
        {
            anisoRatio = GX2_TEX_ANISO_2_TO_1;
        }
    }
    NN_GFX_CALL_GX_FUNCTION( GX2InitSamplerXYFilter( pGx2Sampler,
        static_cast< GX2TexXYFilterType >( ( ( info.GetFilterMode() >> 2 ) & 0x03 ) - 1 ),
        static_cast< GX2TexXYFilterType >( ( ( info.GetFilterMode() >> 4 ) & 0x03 ) - 1 ),
        anisoRatio ) );
    NN_GFX_CALL_GX_FUNCTION( GX2InitSamplerZMFilter( pGx2Sampler,
        GX2_TEX_Z_FILTER_USE_XY, static_cast< GX2TexMipFilterType >( info.GetFilterMode() & 0x03 ) ) );

    NN_GFX_CALL_GX_FUNCTION( GX2InitSamplerClamping( pGx2Sampler,
        Gx::GetTextureClamp( info.GetAddressU() ), Gx::GetTextureClamp(
        info.GetAddressV() ), Gx::GetTextureClamp( info.GetAddressW() ) ) );

    NN_GFX_CALL_GX_FUNCTION( GX2InitSamplerDepthCompare( pGx2Sampler,
        Gx::GetCompareFunction( info.GetComparisonFunction() ) ) );

    NN_GFX_CALL_GX_FUNCTION( GX2InitSamplerBorderType( pGx2Sampler,
        Gx::GetTextureBorderType( info.GetBorderColorType() ) ) );

    float minLOD = info.GetMinLod();
    float maxLOD = info.GetMaxLod();
    float biasLOD = info.GetLodBias();
    minLOD = minLOD < 0.0 ? 0.0 : minLOD; // GX2 doesn't allow a negative LOD but the default is -1000

    static const float s_Gx2MaxLod = 15.999;

    // Clamp the LOD to the maximum allowed for GX2
    minLOD = minLOD <= s_Gx2MaxLod ? minLOD : s_Gx2MaxLod;
    maxLOD = maxLOD <= s_Gx2MaxLod ? maxLOD : s_Gx2MaxLod;
    biasLOD = biasLOD >= -32.0 ? biasLOD : -32.0; // HW limit
    biasLOD = biasLOD < 32 ? biasLOD : 31.984375; // Internal format is S5.6

    NN_GFX_CALL_GX_FUNCTION( GX2InitSamplerLOD( pGx2Sampler,
        minLOD, maxLOD, biasLOD ) );

    this->pGx2Sampler = &this->gx2Sampler;

    this->flags.SetBit( Flag_Shared, false );
    this->state = State_Initialized;
}

void SamplerImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( pDevice );
    NN_SDK_ASSERT( this->state == State_Initialized );
    NN_SDK_ASSERT( !this->flags.GetBit( Flag_Shared ) );
    NN_UNUSED( pDevice );

    this->state = State_NotInitialized;
}


}
}
}
