﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>

#include <nn/os.h>
#include <nn/nn_SdkLog.h>

#include <nn/gfx/detail/gfx_Queue-api.vk.1.h>
#include <nn/gfx/detail/gfx_Device-api.vk.1.h>
#include <nn/gfx/detail/gfx_CommandBuffer-api.vk.1.h>
#include <nn/gfx/detail/gfx_Sync-api.vk.1.h>
#include <nn/gfx/detail/gfx_SwapChain-api.vk.1.h>
#include <nn/gfx/detail/gfx_Texture-api.vk.1.h>

#include "gfx_CommonHelper.h"
#include "gfx_VkHelper.h"
#include "gfx_VkCommand.h"

namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationVk1 Target;

QueueImpl< Target >::QueueImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

QueueImpl< Target >::~QueueImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL( this->state, State_NotInitialized );
    NN_SDK_ASSERT( this->state == State_NotInitialized || this->flags.GetBit( Flag_Shared ) );
}

void QueueImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_UNUSED( info );

    this->pGfxDevice = pDevice;

    uint32_t queueFamilyPropertyCount = 0;
    VkPhysicalDevice physicalDevice = CastToVkDispatchableObject< VkPhysicalDevice >( pDevice->ToData()->hPhysicalDevice );
    NN_GFX_CALL_VK_FUNCTION( vkGetPhysicalDeviceQueueFamilyProperties( physicalDevice,
        &queueFamilyPropertyCount, NULL ) );
    NN_SDK_ASSERT_NOT_EQUAL( queueFamilyPropertyCount, 0u );
    NN_SDK_ASSERT_LESS_EQUAL( queueFamilyPropertyCount, 4u );

    if ( queueFamilyPropertyCount > 0 )
    {
        VkQueueFamilyProperties *pQueueFamilyProperties =
            reinterpret_cast< VkQueueFamilyProperties * >( &this->queueFamilyProperties );
        NN_GFX_CALL_VK_FUNCTION( vkGetPhysicalDeviceQueueFamilyProperties( physicalDevice,
            &queueFamilyPropertyCount, pQueueFamilyProperties ) );
    }

    VkQueue queue;
    VkDevice device = CastToVkDispatchableObject< VkDevice >( pDevice->ToData()->hDevice );
    NN_GFX_CALL_VK_FUNCTION( vkGetDeviceQueue( device, pDevice->ToData()->queueFamilyIndex, 0, &queue ) );
    NN_SDK_ASSERT( queue != 0 );
    this->hQueue = CastFromVkDispatchableObject< VkQueue >( queue );

    this->flags.SetBit( Flag_Shared, false );
    this->state = State_Initialized;
}

void QueueImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( IsInitialized( *this ) );
    NN_SDK_ASSERT( !this->flags.GetBit( Flag_Shared ) );
    NN_UNUSED( pDevice );

    this->state = State_NotInitialized;
}

void QueueImpl< Target >::Present( SwapChainImpl< Target >* pSwapChain,
    int presentInterval ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pSwapChain );
    NN_SDK_REQUIRES( IsInitialized( *pSwapChain ) );
    NN_SDK_REQUIRES( IsInitialized( *this ) );
    NN_UNUSED( presentInterval );

    VkResult result;
    VkPresentInfoKHR presentInfo;
    VkSwapchainKHR swapChainKHR = CastToVkNonDispatchableObject< VkSwapchainKHR >(
        pSwapChain->ToData()->hSwapChainKHR );
    uint32_t currentBuffer = pSwapChain->ToData()->currentScanBufferIndex;
    memset( &presentInfo, 0, sizeof( presentInfo ) );
    presentInfo.sType = VK_STRUCTURE_TYPE_PRESENT_INFO_KHR;
    presentInfo.swapchainCount = 1;
    presentInfo.pSwapchains = &swapChainKHR;
    presentInfo.pImageIndices = &currentBuffer;

    // FIXME: The application may query the next scan buffer index before calling
    // AcquireNextScanBufferIndex() before using one of the new APIs. This is a
    // temporary work-around.
    pSwapChain->ToData()->currentScanBufferIndex = ( pSwapChain->ToData()->currentScanBufferIndex + 1 ) %
        pSwapChain->ToData()->totalScanBuffers;

    VkQueue vkQueue = CastToVkDispatchableObject< VkQueue >( this->hQueue );

    NN_GFX_CALL_NNOS_FUNCTION( nn::os::LockMutex( &this->pGfxDevice->ToData()->mutex ) );

    NN_GFX_CALL_VK_FUNCTION( result = vkQueuePresentKHR( vkQueue, &presentInfo ) );
    NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );
    NN_UNUSED( result );

    NN_GFX_CALL_NNOS_FUNCTION( nn::os::UnlockMutex( &this->pGfxDevice->ToData()->mutex ) );
}

void QueueImpl< Target >::Flush() NN_NOEXCEPT
{
    // No-op: Vulkan doesn't have a queue flush concept
}

void QueueImpl< Target >::Sync() NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsInitialized( *this ) );
    VkQueue vkQueue = CastToVkDispatchableObject< VkQueue >( this->ToData()->hQueue );
    VkResult result;

    NN_GFX_CALL_NNOS_FUNCTION( nn::os::LockMutex( &this->pGfxDevice->ToData()->mutex ) );

    NN_GFX_CALL_VK_FUNCTION( result = vkQueueWaitIdle( vkQueue ) );
    NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );
    NN_UNUSED( result );

    NN_GFX_CALL_NNOS_FUNCTION( nn::os::UnlockMutex( &this->pGfxDevice->ToData()->mutex ) );
}

void QueueImpl< Target >::ExecuteCommand( const CommandBufferImpl< Target >* pCommandBuffer,
    FenceImpl< Target >* pFence ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pCommandBuffer );
    NN_SDK_REQUIRES( IsInitialized( *pCommandBuffer ) );
    NN_SDK_REQUIRES( pFence == NULL || IsInitialized( *pFence ) );
    NN_SDK_REQUIRES( IsInitialized( *this ) );

    // Ensure that Timestamps are supported before submitting
    VkQueueFamilyProperties* pQueueFamilyProperties =
        reinterpret_cast< VkQueueFamilyProperties* >( this->queueFamilyProperties );
    NN_SDK_REQUIRES( !pCommandBuffer->ToData()->flags.GetBit( CommandBufferImplData< Target >::Flag_TimestampUsed )
        || this->pGfxDevice->ToData()->timestampComputeAndGraphics == VK_TRUE
        || pQueueFamilyProperties->timestampValidBits > 0 );
    NN_UNUSED( pQueueFamilyProperties );

    VkFence vkFence = VK_NULL_HANDLE;
    if ( pFence != NULL )
    {
        vkFence = CastToVkNonDispatchableObject< VkFence >( pFence->ToData()->hFence );
        NN_GFX_CALL_VK_FUNCTION( vkResetFences(
            CastToVkDispatchableObject< VkDevice >( this->pGfxDevice->ToData()->hDevice ), 1, &vkFence ) );
    }
    VkResult result;
    VkQueue vkQueue = CastToVkDispatchableObject< VkQueue >( this->hQueue );
    VkSubmitInfo info = { };
    VkCommandBuffer vkCommandBuffers[ CommandBufferImpl< Target >::DataType::VkCommandThreadCount
        * CommandBufferImpl< Target >::DataType::CommandBufferUsage_End ];
    NN_SDK_ASSERT( pCommandBuffer->ToData()->recordedCommandBufferCount <= CommandBufferImpl< Target >::DataType::VkCommandThreadCount );
    int idxCommandBuffer = 0;

    for ( uint32_t idxRecordCount = 0; idxRecordCount < pCommandBuffer->ToData()->recordedCommandBufferCount; ++idxRecordCount )
    {
        vkCommandBuffers[ idxCommandBuffer++ ] =
            CastToVkDispatchableObject< VkCommandBuffer >( pCommandBuffer->ToData()->vkCommandBufferHandles[ idxRecordCount ].hCommandBuffer[
            CommandBufferImpl< Target >::DataType::CommandBufferUsage_Prepare ] );
        vkCommandBuffers[ idxCommandBuffer++ ] =
            CastToVkDispatchableObject< VkCommandBuffer >( pCommandBuffer->ToData()->vkCommandBufferHandles[ idxRecordCount ].hCommandBuffer[
                CommandBufferImpl< Target >::DataType::CommandBufferUsage_Draw ] );
        vkCommandBuffers[ idxCommandBuffer++ ] =
            CastToVkDispatchableObject< VkCommandBuffer >( pCommandBuffer->ToData()->vkCommandBufferHandles[ idxRecordCount ].hCommandBuffer[
                CommandBufferImpl< Target >::DataType::CommandBufferUsage_Post ] );
    }

    info.sType = VK_STRUCTURE_TYPE_SUBMIT_INFO;
    info.pNext = NULL;
    info.commandBufferCount = idxCommandBuffer;
    info.pCommandBuffers = vkCommandBuffers;
    info.signalSemaphoreCount = 0;
    info.waitSemaphoreCount = 0;

    NN_GFX_CALL_NNOS_FUNCTION( nn::os::LockMutex( &this->pGfxDevice->ToData()->mutex ) );

    NN_GFX_CALL_VK_FUNCTION( result = vkQueueSubmit( vkQueue, 1, &info, vkFence ) );
    NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );

    NN_GFX_CALL_NNOS_FUNCTION( nn::os::UnlockMutex( &this->pGfxDevice->ToData()->mutex ) );
}

void QueueImpl< Target >::SetSemaphore( SemaphoreImpl< ApiVariationVk1 >* pSemaphore ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pSemaphore );
    NN_SDK_REQUIRES( IsInitialized( *pSemaphore ) );
    NN_SDK_REQUIRES( IsInitialized( *this ) );

    VkSubmitInfo info;
    VkSemaphore semaphore = CastToVkNonDispatchableObject< VkSemaphore >( pSemaphore->ToData()->hSemaphore );
    info.sType = VK_STRUCTURE_TYPE_SUBMIT_INFO;
    info.pNext = NULL;
    info.commandBufferCount = 0;
    info.signalSemaphoreCount = 1;
    info.pSignalSemaphores = &semaphore;
    info.waitSemaphoreCount = 0;
    VkResult result;

    NN_GFX_CALL_NNOS_FUNCTION( nn::os::LockMutex( &this->pGfxDevice->ToData()->mutex ) );

    NN_GFX_CALL_VK_FUNCTION( result = vkQueueSubmit(
        CastToVkDispatchableObject< VkQueue >( this->hQueue ), 1, &info, NULL ) );
    NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );
    NN_UNUSED( result );

    NN_GFX_CALL_NNOS_FUNCTION( nn::os::UnlockMutex( &this->pGfxDevice->ToData()->mutex ) );
}

void QueueImpl< Target >::SyncSemaphore( const SemaphoreImpl< ApiVariationVk1 >* pSemaphore ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pSemaphore );
    NN_SDK_REQUIRES( IsInitialized( *pSemaphore ) );
    NN_SDK_REQUIRES( IsInitialized( *this ) );

    VkSubmitInfo info;
    VkSemaphore semaphore = CastToVkNonDispatchableObject< VkSemaphore >( pSemaphore->ToData()->hSemaphore );
    VkPipelineStageFlags mask = VK_PIPELINE_STAGE_ALL_COMMANDS_BIT;
    info.sType = VK_STRUCTURE_TYPE_SUBMIT_INFO;
    info.pNext = NULL;
    info.commandBufferCount = 0;
    info.signalSemaphoreCount = 0;
    info.waitSemaphoreCount = 1;
    info.pWaitSemaphores = &semaphore;
    info.pWaitDstStageMask = &mask;
    VkResult result;
    NN_GFX_CALL_NNOS_FUNCTION( nn::os::LockMutex( &this->pGfxDevice->ToData()->mutex ) );

    NN_GFX_CALL_VK_FUNCTION( result = vkQueueSubmit(
        CastToVkDispatchableObject< VkQueue >( this->hQueue ), 1, &info, NULL ) );
    NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );
    NN_UNUSED( result );

    NN_GFX_CALL_NNOS_FUNCTION( nn::os::UnlockMutex( &this->pGfxDevice->ToData()->mutex ) );
}

void QueueImpl< Target >::CopyToScanBuffer( SwapChainImpl< Target >*,
    const ColorTargetViewImpl< Target >* ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( 0 );  // deprecated
}

}
}
}
