﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>

#include <nn/gfx/detail/gfx_Core-api.gx.2.h>
#include <nn/gfx/detail/gfx_Texture-api.gx.2.h>
#include <nn/gfx/detail/gfx_Queue-api.gx.2.h>
#include <nn/gfx/detail/gfx_SwapChain-api.gx.2.h>
#include <nn/gfx/detail/gfx_Device-api.gx.2.h>
#include <nn/gfx/detail/gfx_CommandBuffer-api.gx.2.h>
#include <nn/gfx/detail/gfx_Sync-api.gx.2.h>

#include "gfx_GxHelper.h"

namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationGx2 Target;

QueueImpl< Target >::QueueImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

QueueImpl< Target >::~QueueImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

void QueueImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized );

    this->state = State_Initialized;
}

void QueueImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );

    this->state = State_NotInitialized;
}

void QueueImpl< Target >::Flush() NN_NOEXCEPT
{
    NN_GFX_CALL_GX_FUNCTION( GX2Flush() );
}

void QueueImpl< Target >::Sync() NN_NOEXCEPT
{
    NN_GFX_CALL_GX_FUNCTION( GX2DrawDone() );
}

void QueueImpl< Target >::ExecuteCommand( const CommandBufferImpl< Target >* pCommandBuffer,
    FenceImpl< Target >* pFence ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( pCommandBuffer );

    if( pCommandBuffer->ToData()->pCommandList )
    {
        for( const DisplayList* pDisplayList = static_cast< const DisplayList* >(
            pCommandBuffer->ToData()->pCommandList ); pDisplayList; )
        {
            if( pDisplayList->bufferSize == 0 )
            {
                pDisplayList = static_cast< const DisplayList* >( pDisplayList->pNextDisplayList );
            }
            else
            {
                if ( pDisplayList->usedSize )
                {
                    NN_GFX_CALL_GX_FUNCTION( GX2DirectCallDisplayList( pDisplayList->pBuffer, pDisplayList->usedSize ) );
                }
                ++pDisplayList;
            }
        }
    }

    if( pFence )
    {
        // Get the last submitted timestamp
        pFence->ToData()->timeStamp = NN_GFX_CALL_GX_FUNCTION( GX2GetLastSubmittedTimeStamp() );
    }

}

void QueueImpl< Target >::SetSemaphore( SemaphoreImpl< ApiVariationGx2 >* pSemaphore ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( pSemaphore );
    NN_UNUSED( pSemaphore );
    NN_SDK_ASSERT( NULL ); // This will be used for DMAE queue synchronization
}

void QueueImpl< Target >::SyncSemaphore( const SemaphoreImpl< ApiVariationGx2 >* pSemaphore ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( pSemaphore );
    NN_UNUSED( pSemaphore );
    NN_SDK_ASSERT( NULL ); // This will be used for DMAE queue synchronization
}

void QueueImpl< Target >::CopyToScanBuffer( SwapChainImpl< Target >* pSwapChain,
    const ColorTargetViewImpl< Target >* pColorTarget ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( pSwapChain && pColorTarget);

    const GX2ColorBuffer* sourceColorBuffer = reinterpret_cast<
        const GX2ColorBuffer* >( &pColorTarget->ToData()->gx2ColorBuffer );
    GX2ColorBuffer destinationBuffer;
    GX2ScanTarget target = GX2_SCAN_TARGET_TV;
    DisplayInfo *pDisplay = static_cast< DisplayInfo* >( pSwapChain->ToData()->pDisplayInfo );

    if( !pDisplay->isTV )
    {
        target = GX2_SCAN_TARGET_DRC_FIRST;
    }

    // DEMO library mentions that this is necessary for GX2R resource tracking.
    Flush();

    NN_GFX_CALL_GX_FUNCTION( GX2WaitForFreeScanBuffer( target ) );
    NN_GFX_CALL_GX_FUNCTION( GX2GetCurrentScanBuffer( target, &destinationBuffer ) );
    if( sourceColorBuffer->surface.aa == GX2_AA_MODE_1X )
    {
        NN_GFX_CALL_GX_FUNCTION( GX2UTSetCopyState( GX2_TRUE ) );
        NN_GFX_CALL_GX_FUNCTION( GX2UTCopySurfaceOp(
            &sourceColorBuffer->surface, 0, 0, &destinationBuffer.surface, 0, 0, NULL, 0 ) );
        NN_GFX_CALL_GX_FUNCTION( GX2UTSetCopyState( GX2_FALSE ) );
    }
    else
    {
        NN_GFX_CALL_GX_FUNCTION( GX2UTSetResolveAAColorState( GX2_TRUE ) );
        NN_GFX_CALL_GX_FUNCTION( GX2UTResolveAAColorBufferOp(
            sourceColorBuffer, &destinationBuffer.surface, 0, 0 ) );
        NN_GFX_CALL_GX_FUNCTION( GX2UTSetResolveAAColorState( GX2_FALSE ) );
    }
    NN_GFX_CALL_GX_FUNCTION( GX2MarkScanBufferCopied( target ) );
}

void QueueImpl< Target >::Present( SwapChainImpl< Target >* pSwapChain,
    int presentInterval ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( pSwapChain );
    DisplayInfo *pDisplay = static_cast< DisplayInfo* >( pSwapChain->ToData()->pDisplayInfo );

    GX2ScanTarget target = pDisplay->isTV ? GX2_SCAN_TARGET_TV : GX2_SCAN_TARGET_DRC_FIRST;
    NN_GFX_CALL_GX_FUNCTION( GX2MarkScanBufferCopied( target ) );

    // This makes sure to change the swap interval when necessary
    Gx::SetSwapInterval( presentInterval );

    NN_GFX_CALL_GX_FUNCTION( GX2SwapScanBuffers() );

    // Need to flush to make sure the swap goes into the pipeline
    Flush();

    if ( !pDisplay->isEnabled )
    {
        NN_GFX_CALL_GX_FUNCTION( GX2SetTVEnable( GX2_TRUE ) );
        pDisplay->isEnabled = true;
    }
    else
    {
        if( presentInterval != 0 )
        {
            u32 swapCount = 0;
            u32 flipCount = 0;
            u32 waitCount = 0;
            OSTime tLastFlip;
            OSTime tLastVsync;

            // For now don't let the CPU get ahead of the GPU
            while( NN_STATIC_CONDITION( 1 ) )
            {
                GX2GetSwapStatus( &swapCount, &flipCount, &tLastFlip, &tLastVsync );

                if( flipCount >= swapCount )
                {
                    break;
                }

                // If we've waited over 10 seconds for a flip, consider the GPU hung
                // and stop running.
                if( waitCount++ > 60 * GX2GetGPUTimeout() / 1000 )
                {
                    OSReport("WaitForSwap timed out. Potential GPU hang detected?\n");
                    GX2SetMiscParam( GX2_MISC_HANG_STATE, GX2_HANG_STATE_ETC );
                    if( GX2GetMiscParam( GX2_MISC_HANG_RESPONSE ) == GX2_HANG_RESPONSE_DEBUG )
                    {
                        GX2PrintGPUStatus();
                        OSDebugStrIfConnected( 0 );
                    }
                    break;
                }

                // Call WaitForVsync instead of WaitForFlip due to possible
                // race condition of flip happening right after above test.
                // (There will always be more vsyncs, but not always more flips.)
                GX2WaitForVsync();
            }
        }
    }
}

}
}
}
