﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_BytePtr.h>

#include <nn/gfx/gfx_PipelineInfo.h>

#include <nn/gfx/detail/gfx_Misc.h>
#include <nn/gfx/detail/gfx_Pipeline-api.nvn.8.h>
#include <nn/gfx/detail/gfx_Device-api.nvn.8.h>
#include <nn/gfx/detail/gfx_Shader-api.nvn.8.h>
#include <nn/gfx/detail/gfx_State-api.nvn.8.h>

#include "gfx_NvnHelper.h"

namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationNvn8 Target;

namespace {

};

size_t PipelineImpl< Target >::GetRequiredMemorySize( const GraphicsInfoType& info ) NN_NOEXCEPT
{
    return BlendStateImpl< Target >::GetRequiredMemorySize( *info.GetBlendStateInfo() )
        + VertexStateImpl< Target >::GetRequiredMemorySize( *info.GetVertexStateInfo() );
}

size_t PipelineImpl< Target >::GetRequiredMemorySize( const ComputeInfoType& info ) NN_NOEXCEPT
{
    NN_UNUSED( info );
    return 0;
}

PipelineImpl< Target >::PipelineImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

PipelineImpl< Target >::~PipelineImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

void PipelineImpl< Target >::SetMemory( void* pMemory, size_t size ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( pMemory || size == 0 );

    this->pWorkMemory = pMemory;
    this->memorySize = static_cast< Bit32 >( size );
}

void* PipelineImpl< Target >::GetMemory() NN_NOEXCEPT
{
    return this->pWorkMemory;
}

const void* PipelineImpl< Target >::GetMemory() const NN_NOEXCEPT
{
    return this->pWorkMemory;
}

void PipelineImpl< Target >::Initialize( DeviceImpl< Target >*pDevice,
    const GraphicsInfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized );

    this->flags.Clear();

    this->nnPipelineType = PipelineType_Graphics;

    NN_SDK_ASSERT_NOT_NULL( this->pWorkMemory.ptr );

    NN_SDK_ASSERT_NOT_NULL( info.GetRasterizerStateInfo() );
    NN_SDK_ASSERT_NOT_NULL( info.GetBlendStateInfo() );
    NN_SDK_ASSERT_NOT_NULL( info.GetDepthStencilStateInfo() );
    NN_SDK_ASSERT_NOT_NULL( info.GetRenderTargetStateInfo() );
    NN_SDK_ASSERT_NOT_NULL( info.GetVertexStateInfo() );

    this->pShader = static_cast< const ShaderImpl< Target >* >( info.GetShaderPtr() );
    NN_SDK_ASSERT_NOT_NULL( this->pShader.ptr );

    RasterizerStateImpl< Target >& rasterizerState = nn::gfx::DataToAccessor( this->nnRasterizerState );
    BlendStateImpl< Target >& blendState = nn::gfx::DataToAccessor( this->nnBlendState );
    DepthStencilStateImpl< Target >& depthStencilState = nn::gfx::DataToAccessor( this->nnDepthStencilState );
    VertexStateImpl< Target >& vertexState = nn::gfx::DataToAccessor( this->nnVertexState );
    TessellationStateImpl< Target >& tessellationState = nn::gfx::DataToAccessor( this->nnTessellationState );

    blendState.SetMemory( this->pWorkMemory,
        BlendStateImpl< Target >::GetRequiredMemorySize( *info.GetBlendStateInfo() ) );
    vertexState.SetMemory( nn::util::BytePtr( this->pWorkMemory,
        BlendStateImpl< Target >::GetRequiredMemorySize( *info.GetBlendStateInfo() ) ).Get(),
        VertexStateImpl< Target >::GetRequiredMemorySize( *info.GetVertexStateInfo() ) );

    rasterizerState.Initialize( pDevice, *info.GetRasterizerStateInfo() );
    blendState.Initialize( pDevice, *info.GetBlendStateInfo() );
    depthStencilState.Initialize( pDevice, *info.GetDepthStencilStateInfo() );
    vertexState.Initialize( pDevice, *info.GetVertexStateInfo(),
        static_cast< const ShaderImpl< Target >* >( info.GetShaderPtr() ) );
    if( info.GetTessellationStateInfo() )
    {
        tessellationState.Initialize( pDevice, *info.GetTessellationStateInfo() );
        this->flags.SetBit( Flag_HasTessellationState, true );
    }

    this->state = State_Initialized;
}

void PipelineImpl< Target >::Initialize( DeviceImpl< Target >*, const ComputeInfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized );

    this->flags.Clear();

    this->nnPipelineType = PipelineType_Compute;

    this->pShader = static_cast< const ShaderImpl< Target >* >( info.GetShaderPtr() );
    NN_SDK_ASSERT_NOT_NULL( this->pShader.ptr );

    this->state = State_Initialized;
}

void PipelineImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsInitialized( *this ) );

    if ( this->nnPipelineType == PipelineType_Graphics )
    {
        static_cast< RasterizerStateImpl< Target >& >(
            nn::gfx::DataToAccessor( this->nnRasterizerState ) ).Finalize( pDevice );
        static_cast< BlendStateImpl< Target >& >(
            nn::gfx::DataToAccessor( this->nnBlendState ) ).Finalize( pDevice );
        static_cast< DepthStencilStateImpl< Target >& >(
            nn::gfx::DataToAccessor( this->nnDepthStencilState ) ).Finalize( pDevice );
        static_cast< VertexStateImpl< Target >& >(
            nn::gfx::DataToAccessor( this->nnVertexState ) ).Finalize( pDevice );
        if( this->flags.GetBit( Flag_HasTessellationState ) )
        {
            static_cast< TessellationStateImpl< Target >& >(
                nn::gfx::DataToAccessor( this->nnTessellationState ) ).Finalize( pDevice );
        }
    }

    this->state = State_NotInitialized;
}

}
}
}
