﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_BytePtr.h>

#include <nn/gfx/gfx_PipelineInfo.h>
#include <nn/gfx/detail/gfx_Pipeline-api.gx.2.h>
#include <nn/gfx/detail/gfx_Shader-api.gx.2.h>
#include <nn/gfx/detail/gfx_State-api.gx.2.h>
#include <nn/gfx/detail/gfx_Misc.h>

#include "gfx_GxHelper.h"

namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationGx2 Target;

size_t PipelineImpl< Target >::GetRequiredMemorySize( const GraphicsInfoType& info ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( info.GetBlendStateInfo() );
    NN_SDK_ASSERT( info.GetVertexStateInfo() );
    return BlendStateImpl< Target >::GetRequiredMemorySize( *info.GetBlendStateInfo() )
        + VertexStateImpl< Target >::GetRequiredMemorySize( *info.GetVertexStateInfo() );
}

size_t PipelineImpl< Target >::GetRequiredMemorySize( const ComputeInfoType& info ) NN_NOEXCEPT
{
    NN_UNUSED( info );
    return 0;
}

PipelineImpl< Target >::PipelineImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

PipelineImpl< Target >::~PipelineImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

void PipelineImpl< Target >::SetMemory( void* pMemory, size_t size ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( pMemory || size == 0 );

    this->pMemory = pMemory;
    this->memorySize = size;
}

void* PipelineImpl< Target >::GetMemory( ) NN_NOEXCEPT
{
    return this->pMemory;
}

void PipelineImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const GraphicsInfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_SDK_ASSERT( this->memorySize >= GetRequiredMemorySize( info ) );
    NN_SDK_ASSERT( this->pMemory );

    this->pipelineType = PipelineType_Graphics;

    NN_SDK_ASSERT( info.GetRasterizerStateInfo() );
    NN_SDK_ASSERT( info.GetBlendStateInfo() );
    NN_SDK_ASSERT( info.GetDepthStencilStateInfo() );
    NN_SDK_ASSERT( info.GetVertexStateInfo() );

    NN_SDK_ASSERT( this->pMemory );
    nn::util::BytePtr ptr( this->pMemory.ptr );
    size_t blendStateMemorySize = BlendStateImpl< Target >::GetRequiredMemorySize( *info.GetBlendStateInfo() );
    this->blendState.SetMemory( ptr.Get(), blendStateMemorySize );
    this->vertexState.SetMemory( ptr.Advance( blendStateMemorySize ).Get(),
        VertexStateImpl< Target >::GetRequiredMemorySize( *info.GetVertexStateInfo() ) );

    this->pShader = static_cast< const ShaderImpl< Target >* >( info.GetShaderPtr() );
    NN_SDK_ASSERT( this->pShader );
    this->rasterizerState.Initialize( pDevice, *info.GetRasterizerStateInfo() );
    this->blendState.Initialize( pDevice, *info.GetBlendStateInfo() );
    this->depthStencilState.Initialize( pDevice, *info.GetDepthStencilStateInfo() );
    this->vertexState.Initialize( pDevice, *info.GetVertexStateInfo(), this->pShader );

    this->state = State_Initialized;
}

void PipelineImpl< Target >::Initialize( DeviceImpl< Target >*pDevice, const ComputeInfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized );

    this->pipelineType = PipelineType_Compute;

    this->pShader = static_cast< const ShaderImpl< Target >* >( info.GetShaderPtr() );
    NN_SDK_ASSERT( this->pShader );

    this->state = State_Initialized;
}

void PipelineImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_UNUSED( pDevice );

    if( this->pipelineType == PipelineType_Graphics )
    {
        this->rasterizerState.Finalize( pDevice );
        this->blendState.Finalize( pDevice );
        this->depthStencilState.Finalize( pDevice );
        this->vertexState.Finalize( pDevice );
    }

    this->pShader = NULL;

    this->state = State_NotInitialized;
}

}
}
}
