﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>

#include <nn/os.h>
#include <nn/nn_SdkLog.h>
#include <nn/util/util_BytePtr.h>

#include <nn/gfx/gfx_Variation-api.gl.h>
#include <nn/gfx/gfx_BufferInfo.h>
#include <nn/gfx/gfx_GpuAddress.h>

#include <nn/gfx/detail/gfx_Misc.h>
#include <nn/gfx/detail/gfx_Buffer-api.vk.1.h>
#include <nn/gfx/detail/gfx_MemoryPool-api.vk.1.h>

#include "gfx_CommonHelper.h"
#include "gfx_VkHelper.h"

namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationVk1 Target;

#define REPLACE_ALLOCATION

size_t BufferImpl< Target >::GetBufferAlignment(
    DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    size_t alignment = sizeof(uint64_t);
    nn::util::BitPack16 flags;
    flags.storage = static_cast< uint16_t >( info.GetGpuAccessFlags() );

    if ( flags.IsAnyBitOn( GpuAccess_ConstantBuffer ) )
    {
        alignment = std::max NN_PREVENT_MACRO_FUNC ( alignment,
            static_cast< size_t >( pDevice->ToData()->minUniformBufferOffsetAlignment ) );
    }

    if ( flags.IsAnyBitOn( GpuAccess_UnorderedAccessBuffer ) )
    {
        alignment = std::max NN_PREVENT_MACRO_FUNC ( alignment,
            static_cast< size_t >( pDevice->ToData()->minStorageBufferOffsetAlignment ) );
    }

    if ( flags.IsAnyBitOn( GpuAccess_Texture | GpuAccess_ColorBuffer
        | GpuAccess_DepthStencil | GpuAccess_Image ) )
    {
        alignment = std::max NN_PREVENT_MACRO_FUNC ( alignment,
            static_cast< size_t >( pDevice->ToData()->minTexelBufferOffsetAlignment ) );
    }

    return alignment;
}

BufferImpl< Target >::BufferImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

BufferImpl< Target >::~BufferImpl() NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized || this->flags.GetBit( Flag_Shared ) );
}

void BufferImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info,
    MemoryPoolImpl< Target >* pMemoryPool, ptrdiff_t memoryPoolOffset, size_t memoryPoolSize ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_SDK_REQUIRES( !pMemoryPool || memoryPoolOffset >= 0 );
    NN_SDK_REQUIRES( !pMemoryPool || memoryPoolSize >= info.GetSize() );
    NN_SDK_ASSERT( !pMemoryPool || pMemoryPool->ToData()->pMemory );
    NN_UNUSED( memoryPoolSize );

    this->pGfxDevice = pDevice;
    VkDevice device = CastToVkDispatchableObject< VkDevice >( pDevice->ToData()->hDevice );

    VkBufferCreateInfo vkInfo = {};
    vkInfo.sType = VK_STRUCTURE_TYPE_BUFFER_CREATE_INFO;
    vkInfo.pNext = NULL;
    vkInfo.usage = Vk::GetBufferUsageFlags( info.GetGpuAccessFlags() );
    vkInfo.queueFamilyIndexCount = 0; // FIXME should not be hardcoding these 3 fields
    vkInfo.pQueueFamilyIndices = NULL;
    vkInfo.sharingMode = VK_SHARING_MODE_EXCLUSIVE;
    vkInfo.flags = 0;   // FIXME, need to leverage VkBufferCreateFlags somewhere
    vkInfo.size = info.GetSize( );

    VkResult result;
    VkBuffer vkBuffer;
    NN_GFX_CALL_VK_FUNCTION( result = vkCreateBuffer(
        device, &vkInfo, pDevice->ToData()->pAllocationCallback, &vkBuffer ) );
    NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );
    this->hBuffer = CastFromVkNonDispatchableObject< VkBuffer >( vkBuffer );

    MemoryPoolImpl< Target >::DataType* pMemoryPoolData = pMemoryPool->ToData();
    NN_SDK_ASSERT( !pMemoryPoolData || pMemoryPoolData->memorySize >= memoryPoolOffset + info.GetSize() );
    int memoryPoolProperty = pMemoryPoolData ? pMemoryPoolData->memoryPoolProperty
        : ( MemoryPoolProperty_CpuUncached | MemoryPoolProperty_GpuCached );

    // Required size may be larger than our buffer
    VkMemoryRequirements memReqs;
    NN_GFX_CALL_VK_FUNCTION( vkGetBufferMemoryRequirements( device, vkBuffer, &memReqs ) );

    VkMemoryAllocateInfo allocInfo;
    allocInfo.sType = VK_STRUCTURE_TYPE_MEMORY_ALLOCATE_INFO;
    allocInfo.pNext = NULL;
    allocInfo.memoryTypeIndex = 0;
    allocInfo.allocationSize = memReqs.size;

    uint32_t memFlags = 0;
    memFlags |= ( memoryPoolProperty & MemoryPoolProperty_CpuInvisible )
        ? VK_MEMORY_PROPERTY_DEVICE_LOCAL_BIT : VK_MEMORY_PROPERTY_HOST_VISIBLE_BIT;
    memFlags |= ( memoryPoolProperty & MemoryPoolProperty_CpuUncached )
        ? VK_MEMORY_PROPERTY_HOST_COHERENT_BIT : 0;

    result = Vk::SelectMemoryType( &allocInfo.memoryTypeIndex, pDevice->ToData()->memoryTypeCount,
        pDevice->ToData()->memoryPropertyFlags, memReqs.memoryTypeBits, memFlags );
    if ( result != VK_SUCCESS )
    {
        memFlags = memFlags & ( ~VK_MEMORY_PROPERTY_DEVICE_LOCAL_BIT );
        result = Vk::SelectMemoryType( &allocInfo.memoryTypeIndex, pDevice->ToData()->memoryTypeCount,
            pDevice->ToData()->memoryPropertyFlags, memReqs.memoryTypeBits, memFlags );
    }
    NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );

    VkDeviceMemory vkBufferMemory;
    VkDeviceSize memoryOffset = 0;
    #ifdef REPLACE_ALLOCATION
    NN_GFX_CALL_NNOS_FUNCTION( nn::os::LockMutex( &this->pGfxDevice->ToData()->mutex ) );
    static_cast< BufferMemoryHeap* >( pDevice->ToData()->pBufferHeap )->AllocateMemory( &vkBufferMemory, &memoryOffset, allocInfo );
    NN_GFX_CALL_NNOS_FUNCTION( nn::os::UnlockMutex( &this->pGfxDevice->ToData()->mutex ) );
    #else
    NN_GFX_CALL_VK_FUNCTION( result = vkAllocateMemory( device, &allocInfo,
        pDevice->ToData()->pAllocationCallback, &vkBufferMemory ) );
    #endif
    if ( result != VK_SUCCESS )
    {
        memFlags = memFlags & ( ~VK_MEMORY_PROPERTY_DEVICE_LOCAL_BIT );
        result = Vk::SelectMemoryType( &allocInfo.memoryTypeIndex, pDevice->ToData()->memoryTypeCount,
            pDevice->ToData()->memoryPropertyFlags, memReqs.memoryTypeBits, memFlags );
        NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );

        #ifdef REPLACE_ALLOCATION
        NN_GFX_CALL_NNOS_FUNCTION( nn::os::LockMutex( &this->pGfxDevice->ToData()->mutex ) );
        static_cast< BufferMemoryHeap* >( pDevice->ToData()->pBufferHeap )->AllocateMemory( &vkBufferMemory, &memoryOffset, allocInfo );
        NN_GFX_CALL_NNOS_FUNCTION( nn::os::UnlockMutex( &this->pGfxDevice->ToData()->mutex ) );
        #else
        NN_GFX_CALL_VK_FUNCTION(result =  vkAllocateMemory( device, &allocInfo,
            pDevice->ToData()->pAllocationCallback, &vkBufferMemory ) );
        NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );
        #endif
    }
    this->hBufferMemory = CastFromVkNonDispatchableObject< VkDeviceMemory >( vkBufferMemory );
    this->bufferOffset = static_cast< uint64_t >( memoryOffset );
    this->bufferSize = static_cast< uint64_t >( allocInfo.allocationSize );

    void* pInitData = pMemoryPoolData ? nn::util::BytePtr(
        pMemoryPoolData->pMemory, memoryPoolOffset ).Get() : NULL;
    if ( pInitData )
    {
        #ifdef REPLACE_ALLOCATION

        void* pData = static_cast< BufferMemoryHeap* >( pDevice->ToData()->pBufferHeap )->MapMemory(
            vkBufferMemory, memoryOffset, memReqs.size, 0 );

        memcpy( pData, pInitData, info.GetSize() );

        static_cast< BufferMemoryHeap* >( pDevice->ToData()->pBufferHeap )->UnmapMemory( vkBufferMemory );

        #else

        void* pData;
        NN_GFX_CALL_VK_FUNCTION( result = vkMapMemory( device, vkBufferMemory, memoryOffset, memReqs.size, 0, ( void** )&pData ) );
        NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );

        memcpy( pData, pInitData, info.GetSize() );

        vkUnmapMemory( device, vkBufferMemory );

        #endif
    }

    NN_GFX_CALL_VK_FUNCTION( result = vkBindBufferMemory( device, vkBuffer, vkBufferMemory, memoryOffset ) );
    NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );

    if ( ( memoryPoolProperty & MemoryPoolProperty_CpuInvisible ) == 0 )
    {
        this->pMapping = static_cast< BufferMemoryHeap* >( pDevice->ToData()->pBufferHeap )->MapMemory(
                vkBufferMemory, memoryOffset, memReqs.size, 0 );
    }
    else
    {
        this->pMapping = NULL;
    }

    this->flags.SetBit( Flag_Shared, false );
    this->flags.SetBit( Flag_FlushExplicit, ( memoryPoolProperty & MemoryPoolProperty_CpuCached ) != 0 );
    this->state = State_Initialized;
} // NOLINT(impl/function_size)

void BufferImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsInitialized( *this ) );
    NN_SDK_ASSERT( !this->flags.GetBit( Flag_Shared ) );

    VkDevice device = CastToVkDispatchableObject< VkDevice >( pDevice->ToData()->hDevice );
    VkBuffer vkBuffer = CastToVkNonDispatchableObject< VkBuffer >( this->hBuffer );
    NN_GFX_CALL_VK_FUNCTION( vkDestroyBuffer( device, vkBuffer, pDevice->ToData()->pAllocationCallback ) );

    VkDeviceMemory vkMemory = CastToVkNonDispatchableObject< VkDeviceMemory >( this->hBufferMemory );

    if ( this->pMapping.ptr != nullptr )
    {
        static_cast< BufferMemoryHeap* >( pDevice->ToData()->pBufferHeap )->UnmapMemory( vkMemory );
    }

    #ifdef REPLACE_ALLOCATION
    NN_GFX_CALL_NNOS_FUNCTION( nn::os::LockMutex( &this->pGfxDevice->ToData()->mutex ) );
    static_cast< BufferMemoryHeap* >( pDevice->ToData()->pBufferHeap )->FreeMemory( vkMemory, this->bufferOffset, this->bufferSize );
    NN_GFX_CALL_NNOS_FUNCTION( nn::os::UnlockMutex( &this->pGfxDevice->ToData()->mutex ) );
    #else
    NN_GFX_CALL_VK_FUNCTION( vkFreeMemory( device, vkMemory, pDevice->ToData()->pAllocationCallback ) );
    #endif

    this->hBuffer = 0;
    this->hBufferMemory = 0;

    this->state = State_NotInitialized;
}

void* BufferImpl< Target >::Map() const NN_NOEXCEPT
{
    void *ptr = 0;

    #ifdef REPLACE_ALLOCATION

    // This avoids duplicate map/unmap calls that are not allowed in Vulkan.
    ptr = pMapping;

    #else

    VkDeviceMemory vkBufferMemory = CastToVkNonDispatchableObject< VkDeviceMemory >( this->hBufferMemory );

    VkDevice device = CastToVkDispatchableObject< VkDevice >( this->pGfxDevice->ToData()->hDevice );
    VkResult result;
    NN_GFX_CALL_VK_FUNCTION( result = vkMapMemory(
        device, vkBufferMemory, this->bufferOffset, this->bufferSize, 0, &ptr ) );
    NN_SDK_ASSERT_EQUAL(result, VK_SUCCESS);

    #endif

    NN_SDK_ASSERT(ptr);
    return ptr;
}

void BufferImpl< Target >::Unmap() const NN_NOEXCEPT
{
    #ifndef REPLACE_ALLOCATION

    VkDeviceMemory vkBufferMemory = CastToVkNonDispatchableObject< VkDeviceMemory >( this->hBufferMemory );

    VkDevice device = CastToVkDispatchableObject< VkDevice >( this->pGfxDevice->ToData()->hDevice );
    NN_GFX_CALL_VK_FUNCTION( vkUnmapMemory( device, vkBufferMemory ) );

    #endif
}

void BufferImpl< Target >::FlushMappedRange( ptrdiff_t offset, size_t flushSize ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_Initialized );

    if ( this->flags.IsAllBitOn( Flag_FlushExplicit ) )
    {
        VkDevice device = CastToVkDispatchableObject< VkDevice >( this->pGfxDevice->ToData()->hDevice );
        VkMappedMemoryRange range;
        range.sType = VK_STRUCTURE_TYPE_MAPPED_MEMORY_RANGE;
        range.pNext = NULL;
        range.memory = CastToVkNonDispatchableObject< VkDeviceMemory >( this->hBufferMemory );
        range.offset = offset + this->bufferOffset;
        range.size = nn::util::align_up( flushSize, static_cast< size_t >( this->pGfxDevice->ToData()->nonCoherentAtomSize ) );
        NN_GFX_CALL_VK_FUNCTION( vkFlushMappedMemoryRanges( device, 1, &range ) );
    }
}

void BufferImpl< Target >::InvalidateMappedRange(
    ptrdiff_t offset, size_t invalidateSize ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_Initialized );

    if ( this->flags.IsAllBitOn( Flag_FlushExplicit ) )
    {
        VkDevice device = CastToVkDispatchableObject< VkDevice >( this->pGfxDevice->ToData()->hDevice );
        VkMappedMemoryRange range;
        range.sType = VK_STRUCTURE_TYPE_MAPPED_MEMORY_RANGE;
        range.pNext = NULL;
        range.memory = CastToVkNonDispatchableObject< VkDeviceMemory >( this->hBufferMemory );
        range.offset = offset + this->bufferOffset;
        range.size = invalidateSize;
        NN_GFX_CALL_VK_FUNCTION( vkInvalidateMappedMemoryRanges( device, 1, &range ) );
    }
}

void BufferImpl< Target >::GetGpuAddress( GpuAddress* pOutGpuAddress ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsInitialized( *this ) );

    pOutGpuAddress->ToData()->value = 0;
    pOutGpuAddress->ToData()->impl = static_cast< uint64_t >( this->hBuffer );
}

size_t BufferTextureViewImpl< Target >::GetOffsetAlignment(
    DeviceImpl< Target >* pDevice, const InfoType& ) NN_NOEXCEPT
{
    size_t alignment = sizeof(uint64_t);

    alignment = std::max NN_PREVENT_MACRO_FUNC ( alignment,
        static_cast< size_t >( pDevice->ToData()->minTexelBufferOffsetAlignment ) );

    return alignment;
}

BufferTextureViewImpl< Target >::BufferTextureViewImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

BufferTextureViewImpl< Target >::~BufferTextureViewImpl() NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized || this->flags.GetBit( Flag_Shared ) );
}

void BufferTextureViewImpl< Target >::Initialize(
    DeviceImpl< Target >* pDevice, const BufferTextureViewInfo& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( IsInitialized( *pDevice ) );
    NN_SDK_ASSERT( pDevice->ToData()->hDevice != 0 );
    NN_SDK_ASSERT( nn::util::is_aligned( info.GetOffset(), GetOffsetAlignment( pDevice, info ) ) );

    const BufferImpl< Target >* pBuffer = info.GetBufferPtr();
    NN_SDK_ASSERT_NOT_NULL( pBuffer );
    NN_SDK_ASSERT( IsInitialized( *pBuffer ) );
    VkBufferViewCreateInfo viewInfo = {};
    viewInfo.sType = VK_STRUCTURE_TYPE_BUFFER_VIEW_CREATE_INFO;
    viewInfo.pNext = NULL;
    viewInfo.buffer = CastToVkNonDispatchableObject< VkBuffer >( pBuffer->ToData()->hBuffer );
    viewInfo.format = Vk::GetImageFormat( info.GetImageFormat() );
    viewInfo.offset = info.GetOffset();
    viewInfo.range = info.GetSize();

    VkResult result;
    VkBufferView vkBufferView;
    NN_GFX_CALL_VK_FUNCTION( result = vkCreateBufferView(
        CastToVkDispatchableObject< VkDevice >( pDevice->ToData()->hDevice ), &viewInfo,
        pDevice->ToData()->pAllocationCallback, &vkBufferView ) );
    NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );
    NN_UNUSED( result );

    this->hBufferView = CastFromVkNonDispatchableObject< VkBufferView >( vkBufferView );
    this->flags.SetBit( Flag_Shared, false );
    this->state = State_Initialized;
}

void BufferTextureViewImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsInitialized( *this ) );
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( IsInitialized( *pDevice ) );
    NN_SDK_ASSERT( pDevice->ToData()->hDevice != 0 );
    NN_SDK_ASSERT( !this->flags.GetBit( Flag_Shared ) );

    VkBufferView vkBufferView = CastToVkNonDispatchableObject< VkBufferView >( this->hBufferView );

    NN_GFX_CALL_VK_FUNCTION( vkDestroyBufferView(
        CastToVkDispatchableObject< VkDevice >( pDevice->ToData()->hDevice ),
        vkBufferView, pDevice->ToData()->pAllocationCallback ) );

    this->hBufferView = 0;
    this->state = State_NotInitialized;
}

}
}
}
