﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ens/detail/util/ens_HttpUtility.h>
#include <nn/nn_SdkAssert.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_Utf8StringUtil.h>

namespace nn { namespace ens { namespace detail { namespace util {

namespace
{
    // RFC3986 で定義される unreserved に該当する文字かどうかを判定する。
    bool IsUnreserved(char c) NN_NOEXCEPT
    {
        // unreserved  = ALPHA / DIGIT / "-" / "." / "_" / "~"
        return (c >= 'a' && c <= 'z') || (c >= 'A' && c <= 'Z') || (c >= '0' && c <= '9') ||
            (c == '-') || (c == '.') || (c == '_') || (c == '~');
    }

    // RFC3986 で定義される sub-delims に該当する文字かどうかを判定する。
    bool IsSubDelimiters(char c) NN_NOEXCEPT
    {
        // sub-delims  = "!" / "$" / "&" / "'" / "(" / ")" / "*" / "+" / "," / ";" / "="
        return (c == '!') || (c == '$') || (c == '&') || (c == '\'') || (c == '(') || (c == ')') ||
            (c == '*') || (c == '+') || (c == ',') || (c == ';') || (c == '=');
    }

    // RFC3986 で定義される pchar に該当する文字かどうかを判定する。
    bool IsPchar(char c) NN_NOEXCEPT
    {
        // pchar = unreserved / pct-encoded / sub-delims / ":" / "@"
        return IsUnreserved(c) || (c == '%') || IsSubDelimiters(c) || (c == ':') || (c == '@');
    }

    // RFC3986 で定義される query に該当する文字かどうかを判定する。
    bool IsQuery(char c) NN_NOEXCEPT
    {
        // query = pchar / "/" / "?"
        return IsPchar(c) || (c == '/') || (c == '?');
    }
}

bool EscapeUriDataString(size_t* pOutLength, char* pOutBuffer, size_t bufferSize,
    const char* pString, size_t length) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutLength);
    NN_SDK_REQUIRES_NOT_NULL(pOutBuffer);
    NN_SDK_REQUIRES_GREATER(bufferSize, 0u);
    NN_SDK_REQUIRES_NOT_NULL(pString);
    NN_SDK_REQUIRES_GREATER(length, 0u);

    NN_SDK_REQUIRES(nn::util::VerifyUtf8String(pString, length));

    size_t stringLength = 0;

    for (size_t i = 0; i < length; i++)
    {
        if (IsUnreserved(pString[i]))
        {
            stringLength++;
        }
        else
        {
            stringLength += 3;
        }
    }

    if (stringLength >= bufferSize)
    {
        return false;
    }

    size_t written = 0;

    for (size_t i = 0; i < length; i++)
    {
        if (IsUnreserved(pString[i]))
        {
            pOutBuffer[written++] = pString[i];
        }
        else
        {
            nn::util::SNPrintf(&pOutBuffer[written], 4, "%%%02X", static_cast<nn::Bit8>(pString[i]));
            written += 3;
        }
    }

    pOutBuffer[written] = '\0';

    *pOutLength = written;

    return true;
}

bool VerifyUriQueryString(const char* pString, size_t length) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pString);
    NN_SDK_REQUIRES_GREATER(length, 0u);

    for (size_t i = 0; i < length; i++)
    {
        if (!IsQuery(pString[i]))
        {
            return false;
        }
    }

    return true;
}

}}}}
