﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ens/detail/core/ens_Allocator.h>

namespace nn { namespace ens { namespace detail { namespace core {

namespace
{
    nn::os::SdkMutex g_Mutex;
    nn::os::LightEvent g_AllocatorCreatedEvent(nn::os::EventClearMode_ManualClear);

    nn::mem::StandardAllocator g_Allocator;
}

void CreateAllocator(void* pBuffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_SDK_REQUIRES_GREATER(bufferSize, 0u);

    NN_UTIL_LOCK_GUARD(g_Mutex);

    g_Allocator.Initialize(pBuffer, bufferSize);
    g_AllocatorCreatedEvent.Signal();
}

void DestroyAllocator() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(g_Mutex);

    g_Allocator.Finalize();
    g_AllocatorCreatedEvent.Clear();
}

void WaitUntilAllocatorCreation() NN_NOEXCEPT
{
    g_AllocatorCreatedEvent.Wait();
}

void* AllocateMemory(size_t size) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(g_Mutex);

    if (!g_AllocatorCreatedEvent.TryWait())
    {
        return nullptr;
    }

    return g_Allocator.Allocate(size);
}

void FreeMemory(void* pMemory) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(g_Mutex);

    if (!g_AllocatorCreatedEvent.TryWait())
    {
        return;
    }

    g_Allocator.Free(pMemory);
}

}}}}
