﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nw/eft/eftvw2_Preview.h>


//------------------------------------------------------------------------------
namespace nw {
namespace eftvw2 {


//------------------------------------------------------------------------------
//      コンストラクタ
//------------------------------------------------------------------------------
Preview::Preview(){}


//------------------------------------------------------------------------------
//      初期化処理
//------------------------------------------------------------------------------
void Preview::Initialize( nw::eft2::Heap* heap, EftPreviewType type, eftcom::Guid guid )
{
    mGuId           = guid;
    mPreviewType    = type;
    mNext           = NULL;
    mPrev           = NULL;
    mHeap           = heap;
    mTime           = 0.0f;
    mAutoMoveMtx.SetIdentity();
    mResMatrix.SetIdentity();
    mDrawMatrix.SetIdentity();
    mIsSetResPreview = false;

    //mFunction.Initialize( eftHeap, this );
    //mPreviewName.Copy( name );
    //mPreviewId      = id;
    //mFlag           = 0;
    //mAnimIndex      = 0;
    //mAnimSpeed      = 1.0f;
    //mDestroyCallback = NULL;
    //SetFunctionEnable( true );
}

//---------------------------------------------------------------------------
//! @brief        終了処理です。
//---------------------------------------------------------------------------
void Preview::Finalize()
{
    //mFunction.Finalize();
}

//------------------------------------------------------------------------------
//      計算処理
//------------------------------------------------------------------------------
void Preview::Calc( bool bPause, f32 frameRate, const nw::math::MTX34& centerMatrix, const nw::math::MTX34& viewMatrix )
{
    if ( !bPause )
    {
        mTime += frameRate;
    }
    else
    {
        return;
    }

    // プレビューリソースからリソースマトリクスを生成する
    if ( mIsSetResPreview )
    {
        // Bone Matrix
        nw::math::MTX34 boneMatrix;
        boneMatrix.SetIdentity();

        Preview* preview = GetPreview( mResPreview.constrain.modelGuid );
        if ( preview )
        {
            if ( preview->GetPreviewType() == EFT_VWR_PRV_MODEL ||
                 preview->GetPreviewType() == EFT_VWR_PRV_USR_MODEL )
            {
                preview->GetMatrix( &boneMatrix, mResPreview.constrain.boneIndex );

                if ( mResPreview.constrain.matrixApplyType == EFT_VWR_MATRIX_APPLY_T )
                {
                    f32 x = boneMatrix.m[0][3];
                    f32 y = boneMatrix.m[1][3];
                    f32 z = boneMatrix.m[2][3];
                    boneMatrix.SetIdentity();
                    boneMatrix.m[0][3] = x;
                    boneMatrix.m[1][3] = y;
                    boneMatrix.m[2][3] = z;
                }
            }
        }

        // SRT制御 Matrix
        nw::math::VEC3 rot;
        rot.x = mResPreview.constrain.offsetRotateX;
        rot.y = mResPreview.constrain.offsetRotateY;
        rot.z = mResPreview.constrain.offsetRotateZ;
        nw::math::MTX34MakeSRT( &mResMatrix, &mResPreview.constrain.offsetScale,
                                             &rot,
                                             &mResPreview.constrain.offsetPosition );

        // 自動移動 mAutoMoveMtx
        CalcAutoMove();

        // エミッタビルボード
        if ( mResPreview.constrain.isEmitterBillboard )
        {
            MTX34Mult( &mDrawMatrix, &centerMatrix, &boneMatrix );
            MTX34Mult( &mDrawMatrix, &mDrawMatrix,  &mResMatrix );
            MTX34Mult( &mDrawMatrix, &mDrawMatrix,  &mAutoMoveMtx );

            nw::math::MTX34 dst( viewMatrix );
            dst.Inverse();
            dst.m[0][3] = mDrawMatrix.m[0][3];
            dst.m[1][3] = mDrawMatrix.m[1][3];
            dst.m[2][3] = mDrawMatrix.m[2][3];
            mDrawMatrix = dst;
        }
        else
        {
            if( preview &&
              ( preview->GetPreviewType() == EFT_VWR_PRV_MODEL || preview->GetPreviewType() == EFT_VWR_PRV_USR_MODEL ) )
            {
                MTX34Copy( &mDrawMatrix, &boneMatrix );
            }
            else
            {
                MTX34Mult( &mDrawMatrix, &centerMatrix, &boneMatrix );
            }

            MTX34Mult( &mDrawMatrix, &mDrawMatrix,  &mResMatrix );
            MTX34Mult( &mDrawMatrix, &mDrawMatrix,  &mAutoMoveMtx );
        }
    }
    else
    {
        MTX34Copy( &mDrawMatrix, &mResMatrix );
    }
}


//---------------------------------------------------------------------------
// プレビューをリストに追加します。
//---------------------------------------------------------------------------
void Preview::AddPreview( Preview* preview )
{
    if ( !mNext )
    {
        mNext                   = preview;
        preview->mPrev          = this;
        preview->mNext          = NULL;
    }
    else
    {
        Preview* previewTail    = GetTailPreview();
        previewTail->mNext      = preview;
        preview->mPrev          = previewTail;
        preview->mNext          = NULL;
    }
}

//---------------------------------------------------------------------------
// このプレビューをリストから削除します。
//---------------------------------------------------------------------------
void Preview::RemovePreview()
{
    if ( !mNext )
    {
        if( mPrev )mPrev->mNext = NULL;
    }
    else
    {
        mPrev->mNext    = mNext;
        mNext->mPrev    = mPrev;
    }

    mPrev   = NULL;
    mNext   = NULL;
}

//---------------------------------------------------------------------------
// guidを指定してプレビューを取得します。
//---------------------------------------------------------------------------
Preview* Preview::GetPreview( eftcom::Guid guid )
{
    Preview* preview = GetHeadPreview();
    while( preview )
    {
        if ( preview->GetGuID() == guid )
        {
            return preview;
        }

        preview = preview->GetNextPreview();
    }

    return NULL;
}

//---------------------------------------------------------------------------
//  最後尾のPreviewを取得します。
//---------------------------------------------------------------------------
Preview* Preview::GetTailPreview()
{
    if ( !mNext )
    {
        return this;
    }
    else
    {
        Preview* pret = this->GetNextPreview();
        while( pret )
        {
            if ( !pret->GetNextPreview() ) break;
            pret = pret->GetNextPreview();
        }
        return pret;
    }
}

//---------------------------------------------------------------------------
//  先頭のPreviewを取得します。
//---------------------------------------------------------------------------
Preview* Preview::GetHeadPreview()
{
    Preview* head = this;
    while( head->mPrev )
    {
        head = head->mPrev;
    }
    return head;
}


//---------------------------------------------------------------------------
//! @brief        自動移動を計算します。
//---------------------------------------------------------------------------
void Preview::CalcAutoMove()
{
    mAutoMoveMtx.SetIdentity();

    if ( mResPreview.autoMove.moveType == 0 ) return;

    // パラメータを取得
    f32 autoMoveRadius     = mResPreview.autoMove.radiusXZ;                         //automtic->autoMoveRadius;
    f32 autoMoveRotSpeed   = NW_MATH_DEG_TO_RAD( mResPreview.autoMove.speedXZ );    //DegToIdx( automtic->autoMoveRotVel );
    f32 autoYSwing         = mResPreview.autoMove.amplitudeY;                       //automtic->autoYSwing;
    f32 autoYSwingSpeed    = NW_MATH_DEG_TO_RAD( mResPreview.autoMove.speedY );     //DegToIdx( automtic->autoYSwingSpeed );
    f32 autoZRollSpeed     = NW_MATH_DEG_TO_RAD( mResPreview.autoMove.rollZ );      //DegToIdx( automtic->autoZRollSpeed );

    // 使用するマトリクスを初期化
    nw::math::MTX34 matAutoMove;
    nw::math::MTX34 matRot;
    nw::math::MTX34 matZRoll;

    matAutoMove.SetIdentity();
    matRot.SetIdentity();
    matZRoll.SetIdentity();

    f32 autoMoveRot     = autoMoveRotSpeed * mTime;
    f32 autoYSwingRot   = autoYSwingSpeed  * mTime;
    f32 autoZRoll       = autoZRollSpeed   * mTime;

    nw::math::VEC3 pos;
    nw::math::VEC3 pre_pos;

    // カレント
    {
        // XZ位置
        f32 sinV, cosV;
        nw::math::SinCosRad( &sinV, &cosV, autoMoveRot );
        pos.x = sinV * autoMoveRadius;
        pos.z = cosV * autoMoveRadius;

        // Y位置
        pos.y = nw::math::SinRad( autoYSwingRot ) * autoYSwing;
    }

    // １フレーム前
    {
        f32 preTime     = mTime - 1.0f;

        autoMoveRot     = autoMoveRotSpeed * preTime;
        autoYSwingRot   = autoYSwingSpeed  * preTime;
        autoZRoll       = autoZRollSpeed   * preTime;

        // XZ位置
        f32 sinV, cosV;
        nw::math::SinCosRad( &sinV, &cosV, autoMoveRot );
        pre_pos.x = sinV * autoMoveRadius;
        pre_pos.z = cosV * autoMoveRadius;

        // Y位置
        pre_pos.y = nw::math::SinRad( autoYSwingRot ) * autoYSwing;
    }

    // 回転行列作成
    nw::math::VEC3 basisZ = pos - pre_pos;

    if( basisZ.Length() > 0.00001f )
    {
        basisZ.Normalize();

        nw::math::VEC3 up( 0,1,0 );
        nw::math::VEC3 basisX = up.Cross( basisZ );

        if( basisX.Length() > 0.00001f )
        {

            basisX.Normalize();

            nw::math::VEC3 basisY;
            basisY.SetCross( basisZ, basisX );

            matAutoMove.m[0][0] = basisX.x;
            matAutoMove.m[1][0] = basisX.y;
            matAutoMove.m[2][0] = basisX.z;

            matAutoMove.m[0][1] = basisY.x;
            matAutoMove.m[1][1] = basisY.y;
            matAutoMove.m[2][1] = basisY.z;

            matAutoMove.m[0][2] = basisZ.x;
            matAutoMove.m[1][2] = basisZ.y;
            matAutoMove.m[2][2] = basisZ.z;

            // ZRoll
            matZRoll.SetRotate( nw::math::VEC3( 0.f, 0.f, 1.f ),  autoZRoll );

            // 合成
            MTX34Mult( &mAutoMoveMtx, &matAutoMove, &matZRoll );
            mAutoMoveMtx.m[0][3] = pos.x;
            mAutoMoveMtx.m[1][3] = pos.y;
            mAutoMoveMtx.m[2][3] = pos.z;
        }
    }
}




//------------------------------------------------------------------------------
} // namespace eftvw2
} // namespace nw

