﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/eft/eft2_Misc.h>
#include <nw/eft/eftvw2_FileSystem.h>
#include <nw/eft/eftvw2_CmdSender.h>
#include <nw/eft/eftvw2_CmdReceiver.h>
#include <nw/eft/ProtocolEM4F.h>
#include <nw/ut/os/ut_Thread.h>
#include <string>

#if EFT_IS_CAFE
#include <cafe/fs.h>
#endif

namespace nw     {
namespace eftvw2 {

// Releaseでは、devライブラリを使用しない。
#if defined( NW_RELEASE )


//------------------------------------------------------------------------------
//  ファイル読み込み処理です。
//------------------------------------------------------------------------------
bool FileSystem::Load(  nw::eft2::Heap* eftHeap,
                        const char*     fileName,
                        void**          retBuffer,
                        u32*            retBufferSize,
                        u32             alignment )
{
    EFT_UNUSED_VARIABLE( eftHeap );
    EFT_UNUSED_VARIABLE( fileName );
    EFT_UNUSED_VARIABLE( retBuffer );
    EFT_UNUSED_VARIABLE( retBufferSize );
    EFT_UNUSED_VARIABLE( alignment );
    return false;
}

//---------------------------------------------------------------------------
//  ファイル書き出し処理です。
//---------------------------------------------------------------------------
void FileSystem::Write(const char* fileName, void *buffer, u32 size, u32 count, bool isBinary)
{
    EFT_UNUSED_VARIABLE( fileName );
    EFT_UNUSED_VARIABLE( buffer );
    EFT_UNUSED_VARIABLE( size );
    EFT_UNUSED_VARIABLE( count );
    EFT_UNUSED_VARIABLE( isBinary );
    return;
}


#else


//------------------------------------------------------------------------------
//  ファイル読み込み処理です。
//------------------------------------------------------------------------------
bool FileSystem::Load(  nw::eft2::Heap* eftHeap,
                        const char*     fileName,
                        void**          retBuffer,
                        u32*            retBufferSize,
                        u32             alignment )
{

#if EFT_IS_WIN
    FILE* fp = NULL;
    errno_t err;

    err = fopen_s( &fp, fileName, "rb" );
    if ( err != 0 )
    {
        return false;
    }
    else
    {
        fseek( fp, 0, SEEK_END );
        *retBufferSize = (u32)ftell( fp );
        fseek( fp, 0, SEEK_SET );
        if ( *retBufferSize == 0 )
        {
            fclose( fp );
            return false;
        }

        *retBuffer = eftHeap->Alloc( *retBufferSize + 4, alignment );
        EFT_ASSERT( *retBuffer );
        if ( !(*retBuffer) )
        {
            fclose( fp );
            return false;
        }

        memset(*retBuffer, 0, *retBufferSize + 4);
        fread( *retBuffer, *retBufferSize, 1, fp );
        fclose( fp );

        return true;
    }
#endif

#if EFT_IS_CAFE
    if (fileName[1] == ':')
    {
        FSMountSource  mountSrc;
        char           mountPath[FS_MAX_MOUNTPATH_SIZE];
        char           path[FS_MAX_ARGPATH_SIZE];
        FSStatus       status;
        FSStat         stat;
        FSFileHandle   fh;

        memset( path, 0, FS_MAX_ARGPATH_SIZE );
        FSClient* pClient = (FSClient *)eftHeap->Alloc( sizeof( FSClient ) );
        EFT_NULL_ASSERT( pClient );
        FSCmdBlock* pCmd  = (FSCmdBlock *)eftHeap->Alloc( sizeof( FSCmdBlock ) );
        EFT_NULL_ASSERT( pCmd );
        FSAddClient( pClient, FS_RET_NO_ERROR );
        FSInitCmdBlock( pCmd );

        FSGetMountSource( pClient, pCmd, FS_SOURCETYPE_HFIO, &mountSrc, FS_RET_NO_ERROR );
        FSMount( pClient, pCmd, &mountSrc, mountPath, sizeof(mountPath), FS_RET_NO_ERROR );

        snprintf( path, sizeof(path), "%s/%s", mountPath, fileName );
        for ( int i = 0; i < FS_MAX_ARGPATH_SIZE; i++ )
        {
            if ( path[i] == '\\' || path[i] == ':' )
            {
                path[i] = '/';
            }
        }

        status = FSGetStat( pClient, pCmd, path, &stat, FS_RET_NOT_FOUND );
        if ( status == FS_STATUS_OK )
        {
            *retBuffer = eftHeap->Alloc( stat.size + 4, alignment );
            EFT_NULL_ASSERT( retBuffer );
            memset( *retBuffer, 0, stat.size + 4 );

            FSOpenFile( pClient, pCmd, path, "r", &fh, FS_RET_NO_ERROR );
            status = FSReadFile( pClient, pCmd, *retBuffer, 1, stat.size, fh, 0, FS_RET_NO_ERROR );
            *retBufferSize = stat.size;
            FSCloseFile( pClient, pCmd, fh, FS_RET_NO_ERROR );
        }

        FSUnmount(pClient, pCmd, mountPath, FS_RET_NO_ERROR);
        FSDelClient(pClient, FS_RET_NO_ERROR);
        eftHeap->Free( pClient );
        eftHeap->Free( pCmd );
        return true;
    }

    return false;
#endif
}


//---------------------------------------------------------------------------
//! @brief        ファイル書き出し処理です。
//---------------------------------------------------------------------------
void FileSystem::Write(const char* fileName, void *buffer, u32 size, u32 count, bool isBinary)
{
#if EFT_IS_WIN
    FILE*   fp = NULL;
    errno_t err;

    err = fopen_s( &fp, fileName, isBinary ? "wb" : "w");
    if ( err != 0 )
    {
        return;
    }
    else
    {
        fwrite(buffer, size, count, fp);
    }

    fclose(fp);

    return;
#else
    // WiiUの場合は書き出さない
    return;
#endif
}

#endif

} // namespace eftvw2
} // namespace nw
