﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/eft/eft2_Enum.h>
#include <nw/eft/eft2_Misc.h>
#include <nw/eft/eft2_MemUtil.h>
#include <nw/eft/eft2_EndianUtil.h>
#include <nw/eft/eft2_VertexBuffer.h>

namespace nw   {
namespace eft2 {


//------------------------------------------------------------------------------
//  初期化
//------------------------------------------------------------------------------
void VertexBuffer::Initialize()
{
    m_VertexElement      = 0;
    m_VertexBufferSize   = 0;
    m_VertexBuffer       = NULL;
    m_IsValidate         = false;
}

//---------------------------------------------------------------------------
// 終了処理
//---------------------------------------------------------------------------
void VertexBuffer::Finalize( Heap* heap )
{
    if ( m_VertexBuffer )
    {
        heap->Free( m_VertexBuffer );
        m_VertexBuffer = NULL;
    }
}

//------------------------------------------------------------------------------
//  バッファを確保します。
//------------------------------------------------------------------------------
void* VertexBuffer::AllocateVertexBuffer( Heap* heap, u32 size, u32 element )
{
    EFT_ASSERT( (!m_VertexBuffer) );

    m_VertexBufferSize   = nw::ut::RoundUp( size, 32 );
    m_VertexElement      = element;

#ifdef EFT_OGL
    m_VertexBuffer       = heap->Alloc( m_VertexBufferSize );
#endif
#if EFT_GX2
    // GX2_INDEX_BUFFER_ALIGNMENT(0x20)のアラインもGX2_VERTEX_BUFFER_ALIGNMENTで兼ねる。
    m_VertexBuffer       = heap->Alloc( m_VertexBufferSize, GX2_VERTEX_BUFFER_ALIGNMENT );
#endif
    return m_VertexBuffer;
}

//------------------------------------------------------------------------------
//  バッファを設定します。
//------------------------------------------------------------------------------
void VertexBuffer::SetVertexBuffer( void* buffer, u32 size, u32 element )
{
    EFT_NULL_ASSERT( buffer );
    EFT_ASSERT( (!m_VertexBuffer) );
    m_VertexBuffer       = buffer;
    m_VertexBufferSize   = size;
    m_VertexElement      = element;
}

//------------------------------------------------------------------------------
//  頂点バッファを有効化する
//------------------------------------------------------------------------------
void VertexBuffer::Validate()
{
    EFT_NULL_ASSERT( m_VertexBuffer );

#if EFT_GX2
    MemUtil::FlushCache( m_VertexBuffer, m_VertexBufferSize );
#endif

    m_IsValidate = true;
}

//------------------------------------------------------------------------------
//  バッファをバインドします。
//------------------------------------------------------------------------------
void VertexBuffer::BindBuffer( u32 index, u32 size, u32 stride ) const
{
    if ( index == EFT_INVALID_SHADER_ATTRIBUTE ) return;

#ifdef EFT_OGL
    EFT_UNUSED_VARIABLE( size );
#if EFT_IS_WIN
    glBindVertexArray( 0 );
#endif
    glVertexAttribPointer( index, m_VertexElement, GL_FLOAT, GL_FALSE, stride, m_VertexBuffer );
    glEnableVertexAttribArray( index );
    EFT_GLERR_CHECK();
#endif

#if EFT_GX2
    GX2SetAttribBuffer( index, size, stride, m_VertexBuffer );
#endif
}

//------------------------------------------------------------------------------
//  バッファをバインドします。
//------------------------------------------------------------------------------
void VertexBuffer::BindBuffer( u32 index, u32 start, u32 size, u32 stride )
{
    if ( index == EFT_INVALID_SHADER_ATTRIBUTE ) return;

#ifdef EFT_OGL
    EFT_UNUSED_VARIABLE( size );
    f32* top = static_cast<f32*>(m_VertexBuffer);
    top += start * m_VertexElement;
#if EFT_IS_WIN
    glBindVertexArray( 0 );
#endif
    glVertexAttribPointer( index, m_VertexElement, GL_FLOAT, GL_FALSE, stride, top );
    glEnableVertexAttribArray( index );
    EFT_GLERR_CHECK();
#endif

#if EFT_GX2
    f32* top = static_cast<f32*>(m_VertexBuffer);
    top += start * m_VertexElement;
    GX2SetAttribBuffer( index, size, stride, top );
#endif
}

//------------------------------------------------------------------------------
//  外部からのバッファをバインドする
//------------------------------------------------------------------------------
void VertexBuffer::BindExtBuffer( u32 index, u32 size, u32 element, u32 stride, void* buffer )
{
    if ( index == EFT_INVALID_SHADER_ATTRIBUTE ) return;

#ifdef EFT_OGL
    EFT_UNUSED_VARIABLE( size );
#if EFT_IS_WIN
    glBindVertexArray( 0 );
#endif
    glEnableVertexAttribArray( index );
    glVertexAttribPointer( index, element, GL_FLOAT, GL_FALSE, stride, buffer );
    EFT_GLERR_CHECK();
#endif

#if EFT_GX2
    GX2SetAttribBuffer( index, size, stride, buffer );
#endif
}


} // namespace eft2
} // namespace nw

