﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/eft/eft2_Texture.h>

#define TEX_GET_COMPONENT_X_R(v)  (((v)>>24)&0xff)
#define TEX_GET_COMPONENT_Y_G(v)  (((v)>>16)&0xff)
#define TEX_GET_COMPONENT_Z_B(v)  (((v)>> 8)&0xff)
#define TEX_GET_COMPONENT_W_A(v)  (((v)>> 0)&0xff)

#ifdef NW_PLATFORM_WIN32

#pragma warning(push)
#pragma warning(disable:4100)
#pragma warning(disable:4063)
#pragma warning(disable:4005)
#include <cafe/gx2/gx2Enum.h>
#include <texUtils.h>
#include <sdk_ver.h>
#include <windows.h>
#pragma warning(pop)

namespace nw   {
namespace eft2 {
namespace detail {

namespace
{
    typedef BOOL (*InitializeFunction)(TC2Config* pConfig);
    typedef BOOL (*ConvertTilingFunction)(GX2Surface *pInSurface, GX2TileMode dstTileMode, u32 initialSwizzle, GX2Surface* pOutSurface);
    typedef BOOL (*DestroyGX2SurfaceFunction)(GX2Surface* pGX2Surface);
    typedef BOOL (*DestroyFunction)(void);
    typedef BOOL (*GetSourceSurfaceSizeFunction)(GX2Surface* pGX2Surface);

}

class TextureTilingConverterImpl
{
public:

    //-----------------------------------------------------------------------------
    TextureTilingConverterImpl()
        : m_hDll(NULL)
        , Initialize(NULL)
        , ConvertTiling(NULL)
        , DestroyGX2Surface(NULL)
        , Destroy(NULL)
        , GetSourceSurfaceSize(NULL)
    {
    }

    //-----------------------------------------------------------------------------
    BOOL InitializeFunctions()
    {
        if(Initialize == nullptr || ConvertTiling == nullptr || DestroyGX2Surface == nullptr || Destroy == nullptr)
        {
            m_hDll = LoadLibraryEx(L"texUtils", NULL, LOAD_WITH_ALTERED_SEARCH_PATH);
            if (m_hDll == nullptr)
            {
                return FALSE;
            }

            // 関数アドレスを取得します。
            *reinterpret_cast<void**>(&Initialize) = GetProcAddress(m_hDll, "TC2Initialize");
            *reinterpret_cast<void**>(&ConvertTiling) = GetProcAddress(m_hDll, "TC2ConvertTiling");
            *reinterpret_cast<void**>(&DestroyGX2Surface) = GetProcAddress(m_hDll, "TC2DestroyGX2Surface");
            *reinterpret_cast<void**>(&Destroy) = GetProcAddress(m_hDll, "TC2Destroy");
            *reinterpret_cast<void**>(&GetSourceSurfaceSize) = GetProcAddress(m_hDll, "TC2GetSourceSurfaceSize");

            if (Initialize == nullptr || ConvertTiling == nullptr || DestroyGX2Surface == nullptr || Destroy == nullptr || GetSourceSurfaceSize == nullptr)
            {
                return FALSE;
            }
        }

        return TRUE;
    }

    //-----------------------------------------------------------------------------
    BOOL FinalizeFunctions()
    {
        FreeLibrary(m_hDll);
        m_hDll              = nullptr;

        Initialize          = nullptr;
        ConvertTiling       = nullptr;
        DestroyGX2Surface   = nullptr;
        Destroy             = nullptr;
    }

    //-----------------------------------------------------------------------------
    // DLL から取得する関数のポインタ
    InitializeFunction              Initialize;
    ConvertTilingFunction           ConvertTiling;
    DestroyGX2SurfaceFunction       DestroyGX2Surface;
    DestroyFunction                 Destroy;
    GetSourceSurfaceSizeFunction    GetSourceSurfaceSize;

private:
    HMODULE          m_hDll; // DLL のハンドラ
};


class TextureTilingConverter
{
    static TextureTilingConverterImpl  m_ImplEntity;
    static TextureTilingConverterImpl* m_pImpl;
public:
    static BOOL Initialize(TC2Config& config)
    {
        if(m_pImpl == nullptr)
        {
            TextureTilingConverterImpl* newImpl = &m_ImplEntity;
            if(newImpl->InitializeFunctions() == FALSE)
            {
                return FALSE;
            }

            m_pImpl = newImpl;
        }

        return m_pImpl->Initialize(&config);
    }

    static BOOL ConvertTiling(GX2Surface& src, GX2Surface& dst)
    {
        return m_pImpl != nullptr && m_pImpl->ConvertTiling(&src, GX2_TILE_MODE_LINEAR_SPECIAL, 0, &dst);
    }

    static BOOL DestorySurface(GX2Surface& surface)
    {
        return m_pImpl != nullptr && m_pImpl->DestroyGX2Surface(&surface);
    }

    static BOOL GetSourceSurfaceSize(GX2Surface& surface)
    {
        return m_pImpl != nullptr && m_pImpl->GetSourceSurfaceSize(&surface);
    }

    static BOOL Finalize()
    {
        return m_pImpl != nullptr && m_pImpl->Destroy();
    }
};

TextureTilingConverterImpl TextureTilingConverter::m_ImplEntity;
TextureTilingConverterImpl* TextureTilingConverter::m_pImpl = nullptr;

struct GLSurfaceFormat
{
    GX2SurfaceFormat gx2Format;
    u32 internalFormat;
    u32 format;
    u32 type;
};
GLSurfaceFormat s_GLSurfaceFormat[] =
{
    { GX2_SURFACE_FORMAT_INVALID,                  GL_INVALID_ENUM,      GL_INVALID_ENUM,          GL_INVALID_ENUM                 },
    { GX2_SURFACE_FORMAT_TC_R8_UNORM,              GL_R8,                GL_RED,                   GL_UNSIGNED_BYTE                },
    { GX2_SURFACE_FORMAT_TC_R8_UINT,               GL_R8UI,              GL_RED_INTEGER,           GL_UNSIGNED_BYTE                },
    { GX2_SURFACE_FORMAT_TC_R8_SNORM,              GL_R8_SNORM,          GL_RED,                   GL_BYTE                         },
    { GX2_SURFACE_FORMAT_TC_R8_SINT,               GL_R8I,               GL_RED_INTEGER,           GL_BYTE                         },
    { GX2_SURFACE_FORMAT_TCD_R16_UNORM,            GL_R16,               GL_RED,                   GL_UNSIGNED_SHORT               },
    { GX2_SURFACE_FORMAT_TC_R16_UINT,              GL_R16UI,             GL_RED_INTEGER,           GL_UNSIGNED_SHORT               },
    { GX2_SURFACE_FORMAT_TC_R16_SNORM,             GL_R16_SNORM,         GL_RED,                   GL_SHORT                        },
    { GX2_SURFACE_FORMAT_TC_R16_SINT,              GL_R16I,              GL_RED_INTEGER,           GL_SHORT                        },
    { GX2_SURFACE_FORMAT_TC_R16_FLOAT,             GL_R16F,              GL_RED,                   GL_HALF_FLOAT                   },
    { GX2_SURFACE_FORMAT_TC_R8_G8_UNORM,           GL_RG8,               GL_RG,                    GL_UNSIGNED_BYTE                },
    { GX2_SURFACE_FORMAT_TC_R8_G8_UINT,            GL_RG8UI,             GL_RG_INTEGER,            GL_UNSIGNED_BYTE                },
    { GX2_SURFACE_FORMAT_TC_R8_G8_SNORM,           GL_RG8_SNORM,         GL_RG,                    GL_BYTE                         },
    { GX2_SURFACE_FORMAT_TC_R8_G8_SINT,            GL_RG8I,              GL_RG_INTEGER,            GL_BYTE                         },
    { GX2_SURFACE_FORMAT_TCS_R5_G6_B5_UNORM,       GL_RGB,               GL_RGB,                   GL_UNSIGNED_SHORT_5_6_5_REV     },
    { GX2_SURFACE_FORMAT_TC_R5_G5_B5_A1_UNORM,     GL_RGB5_A1,           GL_RGBA,                  GL_UNSIGNED_SHORT_1_5_5_5_REV   },
    { GX2_SURFACE_FORMAT_TC_R4_G4_B4_A4_UNORM,     GL_RGBA4,             GL_RGBA,                  GL_UNSIGNED_SHORT_4_4_4_4_REV   },
    { GX2_SURFACE_FORMAT_TC_A1_B5_G5_R5_UNORM,     GL_RGBA8,             GL_ABGR_EXT,              GL_UNSIGNED_SHORT_5_5_5_1       },
    { GX2_SURFACE_FORMAT_TC_R32_UINT,              GL_R32UI,             GL_RED_INTEGER,           GL_UNSIGNED_INT                 },
    { GX2_SURFACE_FORMAT_TC_R32_SINT,              GL_R32I,              GL_RED_INTEGER,           GL_UNSIGNED_INT                 },
    { GX2_SURFACE_FORMAT_TCD_R32_FLOAT,            GL_R32F,              GL_RED,                   GL_FLOAT                        },
    { GX2_SURFACE_FORMAT_TC_R16_G16_UNORM,         GL_RG16,              GL_RG,                    GL_UNSIGNED_SHORT               },
    { GX2_SURFACE_FORMAT_TC_R16_G16_UINT,          GL_RG16UI,            GL_RG_INTEGER,            GL_UNSIGNED_SHORT               },
    { GX2_SURFACE_FORMAT_TC_R16_G16_SNORM,         GL_RG16_SNORM,        GL_RG,                    GL_SHORT                        },
    { GX2_SURFACE_FORMAT_TC_R16_G16_SINT,          GL_RG16I,             GL_RG_INTEGER,            GL_SHORT                        },
    { GX2_SURFACE_FORMAT_TC_R16_G16_FLOAT,         GL_RG16F,             GL_RG,                    GL_HALF_FLOAT                   },
    { GX2_SURFACE_FORMAT_TC_R11_G11_B10_FLOAT,     GL_RGB16F,            GL_RGB,                   GL_UNSIGNED_INT_10F_11F_11F_REV },
    { GX2_SURFACE_FORMAT_TCS_R10_G10_B10_A2_UNORM, GL_RGB10_A2,          GL_RGBA,                  GL_UNSIGNED_INT_2_10_10_10_REV  },
    { GX2_SURFACE_FORMAT_TC_R10_G10_B10_A2_UINT,   GL_RGB10_A2UI,        GL_RGBA_INTEGER,          GL_UNSIGNED_INT_2_10_10_10_REV  },
    { GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_UNORM,    GL_RGBA8,             GL_RGBA,                  GL_UNSIGNED_BYTE                },
    { GX2_SURFACE_FORMAT_TC_R8_G8_B8_A8_UINT,      GL_RGBA8UI,           GL_RGBA_INTEGER,          GL_UNSIGNED_BYTE                },
    { GX2_SURFACE_FORMAT_TC_R8_G8_B8_A8_SNORM,     GL_RGBA8_SNORM,       GL_RGBA,                  GL_BYTE                         },
    { GX2_SURFACE_FORMAT_TC_R8_G8_B8_A8_SINT,      GL_RGBA8I,            GL_RGBA_INTEGER,          GL_BYTE                         },
    { GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_SRGB,     GL_SRGB8_ALPHA8,      GL_RGBA,                  GL_UNSIGNED_BYTE                },
    { GX2_SURFACE_FORMAT_TCS_A2_B10_G10_R10_UNORM, GL_RGBA16,            GL_ABGR_EXT,              GL_UNSIGNED_INT_10_10_10_2      },
    { GX2_SURFACE_FORMAT_TC_A2_B10_G10_R10_UINT,   GL_RGBA16UI,          GL_ABGR_EXT,              GL_UNSIGNED_INT_10_10_10_2      },
    { GX2_SURFACE_FORMAT_TC_R32_G32_UINT,          GL_RG32UI,            GL_RG_INTEGER,            GL_UNSIGNED_INT                 },
    { GX2_SURFACE_FORMAT_TC_R32_G32_SINT,          GL_RG32I,             GL_RG_INTEGER,            GL_INT                          },
    { GX2_SURFACE_FORMAT_TC_R32_G32_FLOAT,         GL_RG32F,             GL_RG,                    GL_FLOAT                        },
    { GX2_SURFACE_FORMAT_TC_R16_G16_B16_A16_UNORM, GL_RGBA16,            GL_RGBA,                  GL_UNSIGNED_SHORT               },
    { GX2_SURFACE_FORMAT_TC_R16_G16_B16_A16_UINT,  GL_RGBA16UI,          GL_RGBA_INTEGER,          GL_UNSIGNED_SHORT               },
    { GX2_SURFACE_FORMAT_TC_R16_G16_B16_A16_SNORM, GL_RGBA16_SNORM,      GL_RGBA,                  GL_SHORT                        },
    { GX2_SURFACE_FORMAT_TC_R16_G16_B16_A16_SINT,  GL_RGBA16I,           GL_RGBA_INTEGER,          GL_SHORT                        },
    { GX2_SURFACE_FORMAT_TC_R16_G16_B16_A16_FLOAT, GL_RGBA16F,           GL_RGBA,                  GL_HALF_FLOAT                   },
    { GX2_SURFACE_FORMAT_TC_R32_G32_B32_A32_UINT,  GL_RGBA32UI,          GL_RGBA_INTEGER,          GL_UNSIGNED_INT                 },
    { GX2_SURFACE_FORMAT_TC_R32_G32_B32_A32_SINT,  GL_RGBA32I,           GL_RGBA_INTEGER,          GL_INT                          },
    { GX2_SURFACE_FORMAT_TC_R32_G32_B32_A32_FLOAT, GL_RGBA32F,           GL_RGBA,                  GL_FLOAT                        },
    { GX2_SURFACE_FORMAT_T_BC1_UNORM,              GL_COMPRESSED_RGBA_S3TC_DXT1_EXT,         GL_INVALID_ENUM, GL_INVALID_ENUM      },
    { GX2_SURFACE_FORMAT_T_BC1_SRGB,               GL_COMPRESSED_SRGB_ALPHA_S3TC_DXT1_EXT,   GL_INVALID_ENUM, GL_INVALID_ENUM      },
    { GX2_SURFACE_FORMAT_T_BC2_UNORM,              GL_COMPRESSED_RGBA_S3TC_DXT3_EXT,         GL_INVALID_ENUM, GL_INVALID_ENUM      },
    { GX2_SURFACE_FORMAT_T_BC2_SRGB,               GL_COMPRESSED_SRGB_ALPHA_S3TC_DXT3_EXT,   GL_INVALID_ENUM, GL_INVALID_ENUM      },
    { GX2_SURFACE_FORMAT_T_BC3_UNORM,              GL_COMPRESSED_RGBA_S3TC_DXT5_EXT,         GL_INVALID_ENUM, GL_INVALID_ENUM      },
    { GX2_SURFACE_FORMAT_T_BC3_SRGB,               GL_COMPRESSED_SRGB_ALPHA_S3TC_DXT5_EXT,   GL_INVALID_ENUM, GL_INVALID_ENUM      },
    { GX2_SURFACE_FORMAT_T_BC4_UNORM,              GL_COMPRESSED_RED_RGTC1,                  GL_INVALID_ENUM, GL_INVALID_ENUM      },
    { GX2_SURFACE_FORMAT_T_BC4_SNORM,              GL_COMPRESSED_SIGNED_RED_RGTC1,           GL_INVALID_ENUM, GL_INVALID_ENUM      },
    { GX2_SURFACE_FORMAT_T_BC5_UNORM,              GL_COMPRESSED_RG_RGTC2,                   GL_INVALID_ENUM, GL_INVALID_ENUM      },
    { GX2_SURFACE_FORMAT_T_BC5_SNORM,              GL_COMPRESSED_SIGNED_RG_RGTC2,            GL_INVALID_ENUM, GL_INVALID_ENUM      },
    { GX2_SURFACE_FORMAT_LAST,                     GL_INVALID_ENUM,                          GL_INVALID_ENUM, GL_INVALID_ENUM      },
};

//---------------------------------------------------------------------------
//  圧縮データ？
//---------------------------------------------------------------------------
bool IsCompressed(GX2SurfaceFormat format)
{
    return (format & 0xFF) >= (GX2_SURFACE_FORMAT_T_BC1_UNORM & 0xFF);
}

//---------------------------------------------------------------------------
//  イメージサイズを計算する。
//---------------------------------------------------------------------------
u32 CalcImageSize(GX2SurfaceFormat format, u32 width, u32 height, u32 depth)
{
    u32 size = 0;
    u32 type = format & 0xFF;
    if (IsCompressed(format))
    {
        width  = ( width + 3 ) >> 2;
        height = ( height + 3 ) >> 2;
        size = (type == 0x31 || type == 0x34) ? 8 : 16; // (BC1 BC4) or (BC2 BC3 BC5)
    }
    else
    {
        size = type == 0 ? 0 :      // GX2_SURFACE_FORMAT_INVALID
            type <= 0x02 ? 1 :      // GX2_SURFACE_FORMAT_T_4_4_UNORM
            type <= 0x0C ? 2 :      // GX2_SURFACE_FORMAT_TC_5_5_5_1_UNORM
            type <= 0x1C ? 4 :      // GX2_SURFACE_FORMAT_T_X24_8_X32_UINT
            type <= 0x20 ? 8 :      // GX2_SURFACE_FORMAT_TC_16_16_16_16_FLOAT
            type <= 0x23 ? 16 :     // GX2_SURFACE_FORMAT_TC_32_32_32_32_FLOAT
            type >= 0x2F ? 12 :     // GX2_SURFACE_FORMAT_T_32_32_32_UINT
            type == 0x2B ? 4 : 0;   // GX2_SURFACE_FORMAT_T_5_9_9_9_SHAREDEXP
    }
    return width * height * depth * size;
}

//---------------------------------------------------------------------------
//  GL formatを取得する。
//---------------------------------------------------------------------------
const GLSurfaceFormat& FindGLFormat(GX2SurfaceFormat format)
{
    u32 numTble = sizeof( s_GLSurfaceFormat ) / sizeof( GLSurfaceFormat );

    for ( u32 i = 0; i < numTble; i++ )
    {
        if ( s_GLSurfaceFormat[i].gx2Format == format) return s_GLSurfaceFormat[i];
    }

    return s_GLSurfaceFormat[0];
}

//---------------------------------------------------------------------------
//  アドレスを計算する。
//---------------------------------------------------------------------------
void* AddOffset(void* ptr, size_t offset)
{
    return static_cast<char*>(ptr) + offset;
}

//---------------------------------------------------------------------------
//  ミップマップレベルのテクスチャデータへのポインタを獲得する。
//---------------------------------------------------------------------------
void* GetImagePtr(GX2Surface& surface, u32 mipLevel)
{
    if ( mipLevel == 0 ) return (void *)surface.imagePtr;

    if ( mipLevel == 1 )
    {
        return (void *)surface.mipPtr;
    }
    else
    {
        if ( surface.mipPtr )
        {
            if ( mipLevel >= surface.numMips )
            {
                return NULL;
            }
            else
            {
                return AddOffset( (void*)surface.mipPtr, surface.mipOffset[mipLevel - 1]);
            }
        }
    }

    return NULL;
}

//---------------------------------------------------------------------------
//  大きい方の値を返す。
//---------------------------------------------------------------------------
u32 maxFunc(s32 a, s32 b )
{
    return (u32)( ( a > b ) ? a : b );
}

//---------------------------------------------------------------------------
//  最大ミップマップレベルの計算する。
//---------------------------------------------------------------------------
u32 calculateMaxMipLevel( u16 width )
{
    u32 widthSize =  (u32)width;
    u32 numMips = 0;
    while ( widthSize >= 1 )
    {
        numMips++;
        widthSize = widthSize >> 1;
    }
    return numMips;
}

//---------------------------------------------------------------------------
//  GL surfaceに変換する。
//---------------------------------------------------------------------------
void _ConvertToGLSurface(GX2Surface& surface, int arrayLength, void* texture_data )
{
    // NOTE: バグ回避のために GX2_SURFACE_DIM_2D_ARRAY に一旦変更してタイリング解除します。
    bool changeDim = false;
    if (surface.dim == GX2_SURFACE_DIM_CUBE &&
        arrayLength > 1)
    {
        surface.dim = GX2_SURFACE_DIM_2D_ARRAY;
        changeDim = true;
    }

    // TC2 APIの初期化
    TC2Config tc2_config = {};
    tc2_config.gbTilingConfig = 0;
    tc2_config.gpu = GPU_Cafe;
    nw::eft2::detail::TextureTilingConverter::Initialize( tc2_config );

    // surface情報を獲得する。
    nw::eft2::detail::TextureTilingConverter::GetSourceSurfaceSize( surface );

    surface.imagePtr = texture_data;
    surface.mipPtr   = (void *)((u32)surface.imagePtr + surface.imageSize);

    GX2Surface linear = surface;

    // リニアイメージに変換する。
    nw::eft2::detail::TextureTilingConverter::ConvertTiling( surface, linear );

    if (changeDim)
    {
        linear.dim = surface.dim = GX2_SURFACE_DIM_CUBE;
    }

  //  const int heightFactor = surface.dim == GX2_SURFACE_DIM_1D ||
  //      surface.dim == GX2_SURFACE_DIM_1D_ARRAY ? 0 : 1;
    const int depthFactor = surface.dim == GX2_SURFACE_DIM_3D ? 1 : 0;

    for (u32 mipLevel = 0; mipLevel < surface.numMips; ++mipLevel)
    {
        void* dstImage = GetImagePtr(surface, mipLevel);
        void* srcImage = GetImagePtr(linear, mipLevel);

        u32 width  = maxFunc(1, surface.width  >> mipLevel);
        u32 height = maxFunc(1, surface.height >> mipLevel ); //* heightFactor));
        u32 depth  = maxFunc(1, surface.depth  >> (mipLevel * depthFactor));
        u32 size   = CalcImageSize(surface.format, width, height, depth);
        memcpy(dstImage, srcImage, size);
    }

    surface.tileMode = linear.tileMode;
    surface.imageSize = linear.imageSize;
    surface.mipSize = linear.mipSize;

    nw::eft2::detail::TextureTilingConverter::DestorySurface( linear );
    nw::eft2::detail::TextureTilingConverter::Finalize();
}

}}}

#endif      // NW_PLATFORM_WIN32



namespace nw   {
namespace eft2 {

//---------------------------------------------------------------------------
// テクスチャを有効にします。
//---------------------------------------------------------------------------
bool BindTexture( TextureInterface texture, TextureType textureType, u32 fragLoc, u32 vertLoc, s32 slot )
{
    if ( fragLoc == EFT_INVALID_TEXTURE_LOCATION &&
        vertLoc == EFT_INVALID_TEXTURE_LOCATION )
    {
        return false;
    }

#ifdef EFT_OGL
    u32 loc = (u32)EFT_INVALID_TEXTURE_LOCATION;
    if ( fragLoc != EFT_INVALID_TEXTURE_LOCATION )
    {
        loc = fragLoc;
    }
    else if ( vertLoc != EFT_INVALID_TEXTURE_LOCATION )
    {
        loc = vertLoc;
    }

    glActiveTexture( GL_TEXTURE0 + slot );

    glBindTexture( GL_TEXTURE_2D, 0 );
#if EFT_OPENGL
    glBindTexture( GL_TEXTURE_2D_ARRAY, 0 );
    glBindTexture( GL_TEXTURE_3D, 0 );
    glBindTexture( GL_TEXTURE_CUBE_MAP, 0 );
#endif
    switch (textureType)
    {
    case EFT_TEXTURE_TYPE_2D:
    case EFT_TEXTURE_TYPE_SHADOW:
        glBindTexture( GL_TEXTURE_2D, texture );
        break;
#if EFT_OPENGL
    case EFT_TEXTURE_TYPE_2D_ARRAY:
    case EFT_TEXTURE_TYPE_SHADOW_ARRAY:
        glBindTexture( GL_TEXTURE_2D_ARRAY, texture );
        break;
    case EFT_TEXTURE_TYPE_3D:
        glBindTexture( GL_TEXTURE_3D, texture );
        break;
    case EFT_TEXTURE_TYPE_CUBE_MAP:
        glBindTexture( GL_TEXTURE_CUBE_MAP, texture );
        break;
    case EFT_TEXTURE_TYPE_CUBE_MAP_ARRAY:
        glBindTexture( GL_TEXTURE_CUBE_MAP_ARRAY, texture );
        break;
#endif
    default:
        break;
    }
    EFT_GLERR_CHECK();
    glUniform1i( loc, slot );
    EFT_GLERR_CHECK();
#endif

#if EFT_GX2
    if ( texture )
    {
        if ( fragLoc != EFT_INVALID_TEXTURE_LOCATION ) GX2SetPixelTexture(  texture, fragLoc );
        if ( vertLoc != EFT_INVALID_TEXTURE_LOCATION ) GX2SetVertexTexture( texture, vertLoc );
    }
#endif

    return true;
}


//---------------------------------------------------------------------------
// テクスチャを有効にします。
//---------------------------------------------------------------------------
bool TextureExt::Bind( u32 fragLoc, u32 vertLoc, TextureSlot slot )
{
    return BindTexture( m_Texture, m_TextureType, fragLoc, vertLoc, slot );
}

//---------------------------------------------------------------------------
//! @brief  TextureResourceの初期化
//---------------------------------------------------------------------------
void TextureResource::Initialize( TextureInterface texture, u64 globalId, ResTexture* texRes )
{
    m_Texture = texture;
    m_TextureGlobalId = globalId;
    m_TextureRes = texRes;
}

//---------------------------------------------------------------------------
// GX2イメージの初期化
//---------------------------------------------------------------------------
void TextureResource::InitializeGx2bImage( u64 id, ResTexture* texRes, void* resource, u32 resourceSize )
{
    m_TextureGlobalId = id;
    m_TextureRes = texRes;
    /*bool ret = */InitializeNativeTexture( resource, resourceSize );
}


void TextureResource::InitializeOrgbImage( u64 id, ResTexture* texRes, void* resource, u32 resourceSize )
{
    m_TextureGlobalId = id;
    m_TextureRes = texRes;

#ifdef EFT_OGL
    EFT_UNUSED_VARIABLE( resourceSize );

    // RGBチャンネルが同じだったら、GL_LUMINANCE_ALPHAとして扱う
    bool is2ch = TEX_GET_COMPONENT_X_R( m_TextureRes->compSel) == TEX_GET_COMPONENT_Y_G( m_TextureRes->compSel) &&
        TEX_GET_COMPONENT_Y_G( m_TextureRes->compSel) == TEX_GET_COMPONENT_Z_B( m_TextureRes->compSel);
    GLuint texFormat = is2ch ? GL_LUMINANCE_ALPHA : GL_RGBA;
    GLuint pixelStore = is2ch ? 2 : 4;

    glGenTextures( 1, &m_Texture );
    glBindTexture( GL_TEXTURE_2D, m_Texture );
    glPixelStorei( GL_UNPACK_ALIGNMENT, pixelStore );
    glTexImage2D( GL_TEXTURE_2D, 0, texFormat, m_TextureRes->width, m_TextureRes->height, 0, texFormat, GL_UNSIGNED_BYTE, resource );
#endif

#if EFT_GX2
    EFT_ERR( "Not Implemented." );
#endif
}

//---------------------------------------------------------------------------
//      終了処理
//---------------------------------------------------------------------------
void TextureResource::Finalize()
{
#ifdef EFT_OGL
    if ( m_Texture != EFT_INVALID_TEXTURE_ID ) glDeleteTextures( 1, &m_Texture );
#endif
}

//---------------------------------------------------------------------------
//      テクスチャを有効にします。
//---------------------------------------------------------------------------
bool TextureResource::Bind( u32 fragLoc, u32 vertLoc, TextureSlot slot )
{
    TextureType type = EFT_TEXTURE_TYPE_2D;
    if ( m_TextureRes->depth > 1 ) type = EFT_TEXTURE_TYPE_2D_ARRAY;

    return BindTexture( m_Texture, type, fragLoc, vertLoc, slot );
}

//---------------------------------------------------------------------------
//      ネイティブ テクスチャの初期化
//---------------------------------------------------------------------------
void TextureResource::InitializeNativeTexture( void* resource, u32 resourceSize )
{
#ifdef NW_PLATFORM_WIN32

    EFT_NULL_ASSERT( resourceSize );
    GX2Surface surface;
    memset( (void *)&surface, 0, sizeof(GX2Surface) );

    surface.depth     = m_TextureRes->depth;
    surface.dim       = ( ( m_TextureRes->depth == 1 ) || ( m_TextureRes->depth == 0 ) ) ? GX2_SURFACE_DIM_2D : GX2_SURFACE_DIM_2D_ARRAY;
    surface.numMips   = m_TextureRes->mipLevel;
    surface.width     = m_TextureRes->width;
    surface.height    = m_TextureRes->height;
    surface.tileMode  = (::GX2TileMode)m_TextureRes->nativeTileMode;
    surface.use       = GX2_SURFACE_USE_TEXTURE;
    surface.format    = (::GX2SurfaceFormat)m_TextureRes->nativeFormat;
    surface.swizzle   = m_TextureRes->swizzle;
    surface.alignment = 1;

    // textureをリニアモードに変換する。
    detail::_ConvertToGLSurface( surface, m_TextureRes->mipLevel, resource );

    GLuint texTarget = GL_TEXTURE_2D;

    if (m_TextureRes->depth > 1 )  texTarget = GL_TEXTURE_2D_ARRAY;

    glGenTextures( 1, &m_Texture );
    glBindTexture( texTarget, m_Texture );
    glPixelStorei( GL_UNPACK_ALIGNMENT, 1 );

    GLint tblSwizzle[] = {
        GL_RED, GL_GREEN, GL_BLUE, GL_ALPHA, GL_ZERO, GL_ONE
    };

    GLint swizzle[] = {
        tblSwizzle[GX2_GET_COMPONENT_X_R( m_TextureRes->compSel )],
        tblSwizzle[GX2_GET_COMPONENT_Y_G( m_TextureRes->compSel )],
        tblSwizzle[GX2_GET_COMPONENT_Z_B( m_TextureRes->compSel )],
        tblSwizzle[GX2_GET_COMPONENT_W_A( m_TextureRes->compSel )]
    };

    glTexParameteriv( texTarget, GL_TEXTURE_SWIZZLE_RGBA, swizzle );

    detail::GLSurfaceFormat glFormat = detail::FindGLFormat( (GX2SurfaceFormat)surface.format );
    for ( u32 mipLevel = 0; mipLevel < m_TextureRes->mipLevel; ++mipLevel )
    {
        GLsizei width  = detail::maxFunc( 1, m_TextureRes->width  >> mipLevel );
        GLsizei height = detail::maxFunc( 1, m_TextureRes->height >> mipLevel );
        GLsizei imageSize = detail::CalcImageSize( (GX2SurfaceFormat)m_TextureRes->nativeFormat, width, height, m_TextureRes->depth );
        void*   imagePtr  = detail::GetImagePtr( surface, mipLevel );

        if ( m_TextureRes->depth == 1 )
        {
            if ( detail::IsCompressed( (GX2SurfaceFormat)m_TextureRes->nativeFormat ) )
            {
                glCompressedTexImage2D(texTarget, mipLevel, glFormat.internalFormat,
                    width, height, 0, imageSize, imagePtr );
            }
            else
            {
               glTexImage2D(texTarget, mipLevel, glFormat.internalFormat, width, height, 0,
                   glFormat.format, glFormat.type, imagePtr );
            }
        }
        else
        {
            if ( detail::IsCompressed( (GX2SurfaceFormat)m_TextureRes->nativeFormat ) )
            {
                glCompressedTexImage3D(texTarget, mipLevel, glFormat.internalFormat,
                    width, height, m_TextureRes->depth, 0, imageSize, imagePtr);
            }
            else
            {
                glTexImage3D( texTarget, mipLevel, glFormat.internalFormat, width, height,
                    m_TextureRes->depth, 0, glFormat.format, glFormat.type, imagePtr );
            }
        }

    }
    EFT_GLERR_CHECK();
#elif EFT_GX2
    GX2SurfaceFormat texformat = (GX2SurfaceFormat)m_TextureRes->nativeFormat;

    if ( m_TextureRes->depth == 1 )
    {
        GX2InitTexture( &m_GxTexture,
            m_TextureRes->width,              // width
            m_TextureRes->height,             // height
            m_TextureRes->depth,              // depth
            m_TextureRes->mipLevel,           // num mips
            texformat,                      // format
            GX2_SURFACE_DIM_2D );           // dim
    }
    else
    {
        GX2InitTexture( &m_GxTexture,
            m_TextureRes->width,              // width
            m_TextureRes->height,             // height
            m_TextureRes->depth,              // depth
            m_TextureRes->mipLevel,           // num mips
            texformat,                      // format
            GX2_SURFACE_DIM_2D_ARRAY );     // dim
    }

    m_GxTexture.surface.tileMode = (GX2TileMode)m_TextureRes->nativeTileMode;
    GX2CalcSurfaceSizeAndAlignment( &m_GxTexture.surface );

    // Swizzle設定
    GX2SetSurfaceSwizzle( &m_GxTexture.surface, m_TextureRes->nativeSwizzle );

    // Nativeデータを設定
    GX2InitTexturePtrs( &m_GxTexture, resource, 0 );

    // CompSelを設定
    GX2InitTextureCompSel( &m_GxTexture, m_TextureRes->compSel );

    // レジスタを初期化
    GX2InitTextureRegs( &m_GxTexture );

    // キャッシュのフラッシュ
    DCFlushRange( m_GxTexture.surface.imagePtr,
                  m_GxTexture.surface.imageSize + m_GxTexture.surface.mipSize );

    // テクスチャのアライメントチェックを行う
    NW_ASSERT_ALIGN( (u32)m_GxTexture.surface.imagePtr, m_GxTexture.surface.alignment );
    if ( m_GxTexture.surface.mipSize > 0 ) NW_ASSERT_ALIGN( (u32)m_GxTexture.surface.mipPtr, m_GxTexture.surface.alignment );

    m_Texture = &m_GxTexture;
#else
    EFT_UNUSED_VARIABLE( resource );
    EFT_UNUSED_VARIABLE( resourceSize );
#endif
}

} // namespace eft2
} // namespace nw

