﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/eft/eft2_Misc.h>
#include <nw/eft/eft2_Stripe.h>
#include <nw/eft/eft2_StripeConnection.h>
#include <nw/eft/eft2_RenderContext.h>
#include <nw/eft/eft2_System.h>

#ifndef EFT_DEGRADATION_SPEC

//---------------------------------------------------------------------------
// エルミート補間で曲線上の位置を求める
//---------------------------------------------------------------------------
inline void CalcHermiteInterpolatedCurveVec(
    nw::math::VEC3* curvePos,
    nw::math::VEC3* curveDir,
    const nw::eft2::StripeHistory* hist,
    const u32 histPos,
    const u32 numHist,
    f32 innerPos )
{
    nw::math::VEC3 startPos, endPos;    // エルミート補間:始点・終点
    nw::math::VEC3 startDir, endDir;    // エルミート補間:開始ベクトル、終了ベクトル（点を通過する向き）
    f32 innerPosInv = 1.0f - innerPos;

    if( histPos == 0 )
    {
        // 先頭部分（3点で近似補間）
        const u32 pStart = 0;
        const u32 pStartNext = 1;
        const u32 pStartNext2 = 2;

        EFT_F32_VEC3_COPY( &startPos, &hist[pStart].pos );
        EFT_F32_VEC3_COPY( &endPos,   &hist[pStartNext].pos );

        nw::math::VEC4 end = 0.5f * ( hist[pStartNext2].pos - hist[pStart].pos );
        EFT_F32_VEC3_COPY( &endDir,   &end );
        nw::math::VEC4 start = 0.25f * (( hist[pStartNext].pos - hist[pStart].pos ) + ( hist[pStartNext].pos - hist[pStartNext2].pos ));
        EFT_F32_VEC3_COPY( &startDir, &start );

        // 進行方向
        curveDir->x = hist[ pStart ].dir.x * innerPosInv + hist[ pStartNext ].dir.x * innerPos;
        curveDir->y = hist[ pStart ].dir.y * innerPosInv + hist[ pStartNext ].dir.y * innerPos;
        curveDir->z = hist[ pStart ].dir.z * innerPosInv + hist[ pStartNext ].dir.z * innerPos;
    }
    else if( histPos >= numHist - 2 )
    {
        // ストライプの末端（3点で近似補間）
        const u32 pEnd = numHist - 1;
        const u32 pEndPrev = pEnd - 1;
        const u32 pEndPrev2 = pEnd - 2;
        if( histPos >= numHist - 1 )
        {
            innerPos = 1.0f;
            innerPosInv = 0.0f;
        }

        // 末端部分
        EFT_F32_VEC3_COPY( &startPos, &hist[pEndPrev].pos );
        EFT_F32_VEC3_COPY( &endPos,   &hist[pEnd].pos );
        nw::math::VEC4 start = 0.5f * ( hist[pEnd].pos - hist[pEndPrev2].pos );
        EFT_F32_VEC3_COPY( &startDir, &start );
        nw::math::VEC4 end = -0.25f * (( hist[pEndPrev].pos - hist[pEnd].pos ) + ( hist[pEndPrev].pos - hist[pEndPrev2].pos ));
        EFT_F32_VEC3_COPY( &endDir, &end );

        // 進行方向
        curveDir->x = hist[ pEndPrev ].dir.x * innerPosInv + hist[ pEnd ].dir.x * innerPos;
        curveDir->y = hist[ pEndPrev ].dir.y * innerPosInv + hist[ pEnd ].dir.y * innerPos;
        curveDir->z = hist[ pEndPrev ].dir.z * innerPosInv + hist[ pEnd ].dir.z * innerPos;
    }
    else
    {
        // 中間部（4点で補間）
        const u32 pNow = histPos;
        const u32 pNext = pNow + 1;
        const u32 pPrev = pNow - 1;
        const u32 pNext2 = pNow + 2;

        EFT_F32_VEC3_COPY( &startPos, &hist[pNow].pos );
        EFT_F32_VEC3_COPY( &endPos,   &hist[pNext].pos );

        nw::math::VEC4 start = ( hist[pNext].pos - hist[pPrev].pos ) * 0.5f;
        EFT_F32_VEC3_COPY( &startDir,   &start );
        nw::math::VEC4 end = ( hist[pNext2].pos - hist[pNow].pos ) * 0.5f;
        EFT_F32_VEC3_COPY( &endDir,   &end );

        // 進行方向
        curveDir->x = hist[ pNow ].dir.x * innerPosInv + hist[ pNext ].dir.x * innerPos;
        curveDir->y = hist[ pNow ].dir.y * innerPosInv + hist[ pNext ].dir.y * innerPos;
        curveDir->z = hist[ pNow ].dir.z * innerPosInv + hist[ pNext ].dir.z * innerPos;
    }
    nw::eft2::HermiteInterpolationOnCubic( curvePos, startPos, startDir, endPos, endDir, innerPos );

    return;
}

namespace nw   {
namespace eft2 {

StripeSystem* StripeSystem::sStripeSystem = NULL;

//------------------------------------------------------------------------------
//  ストライプシステムの初期化処理を行います。
//------------------------------------------------------------------------------
void StripeSystem::Initialize( nw::eft2::Heap* heap, nw::eft2::System* system, BufferMode bufferMode, u32 stripeNum )
{
    if ( sStripeSystem )
    {
        OutputError( "StripeSystem is Initialized.\n" );
    }

    // プレビュー生成用のメモリを確保します。
    void* buffer = heap->Alloc( sizeof(StripeSystem) );
    if( !buffer )
    {
        nw::eft2::OutputWarning( "Memory Allocate Error!! : %d\n", sizeof(StripeSystem) );
        return;
    }

    sStripeSystem = new (buffer) StripeSystem( heap, system, bufferMode, stripeNum );
}

//------------------------------------------------------------------------------
//  ストライプシステムの終了処理を行います。
//------------------------------------------------------------------------------
void StripeSystem::Finalize( nw::eft2::Heap* heap )
{
    sStripeSystem->~StripeSystem();
    heap->Free( sStripeSystem );
    sStripeSystem = NULL;
}

//------------------------------------------------------------------------------
//  ストライプシステムで確保したワークサイズを取得します。
//------------------------------------------------------------------------------
u32 StripeSystem::GetWorkSize()
{
    if ( !sStripeSystem ) return 0;
    return sStripeSystem->_GetWorkSize();
}

//------------------------------------------------------------------------------
//  パーティクル生成コールバック
//------------------------------------------------------------------------------
bool StripeSystem::_ParticleEmitCallback( ParticleEmitArg& arg )
{
    if ( arg.emitter->GetCalcType() == EFT_EMITTER_CALC_TYPE_CPU )
    {
        nw::eft2::Stripe* stripe = sStripeSystem->Alloc();
        arg.emPluginData = stripe;
        if ( stripe )
        {
            nw::math::VEC4 random;
            arg.GetRandom( &random );
            stripe->random.x = random.x;
            stripe->random.y = random.y;
            stripe->random.z = random.z;
            stripe->random.w = random.w;
        }
        return true;
    }

    return false;
}

//------------------------------------------------------------------------------
//  パーティクル削除コールバック
//------------------------------------------------------------------------------
bool StripeSystem::_ParticleRemoveCallback( ParticleRemoveArg& arg )
{
    if ( arg.emitter->GetCalcType() == EFT_EMITTER_CALC_TYPE_CPU )
    {
        if ( arg.emPluginData )
        {
            Stripe* stripe = reinterpret_cast<Stripe*>( arg.emPluginData );

            arg.CalcLocalColor0( &stripe->color0 );
            arg.CalcLocalColor1( &stripe->color1 );

            sStripeSystem->AddDelayList( arg.emitter, stripe );
        }
        return true;
    }

    return false;
}

//------------------------------------------------------------------------------
//  エミッタ生成後コールバック
//------------------------------------------------------------------------------
bool StripeSystem::_EmitterInitializeCallback( EmitterInitializeArg& arg )
{
    if ( arg.emitter->GetCalcType() == EFT_EMITTER_CALC_TYPE_CPU )
    {
        bool result = sStripeSystem->AllocStripeVertexBuffer( arg.emitter );
        return result;
    }
    return false;
}

//------------------------------------------------------------------------------
//  エミッタ破棄後コールバック
//------------------------------------------------------------------------------
void StripeSystem::_EmitterFinalizeCallback( EmitterFinalizeArg& arg )
{
    if ( arg.emitter->GetCalcType() == EFT_EMITTER_CALC_TYPE_CPU )
    {
        // 遅延ストライプの解放
        sStripeSystem->FreeDelayStripeList( arg.emitter );
    }
}


//---------------------------------------------------------------------------
//  コンストラクタ
//---------------------------------------------------------------------------
StripeSystem::StripeSystem( nw::eft2::Heap* heap, nw::eft2::System* system, BufferMode bufferMode, u32 stripeNum )
{
    mHeap   = heap;
    mSystem = system;

    mStripeArrayNum     = stripeNum;
    mStripeWorkSize     = sizeof( Stripe ) * mStripeArrayNum;
    mStripeArray        = reinterpret_cast< Stripe* >( mHeap->Alloc( sizeof( Stripe ) * mStripeArrayNum ) );
    mStripeNum          = 0;
    mBufferMode         = bufferMode;

    for( u32 i = 0; i < mStripeArrayNum; i++ )
    {
        mStripeArray[i].used        = false;
        mStripeArray[i].numHistory  = 0;
        mStripeArray[i].next        = NULL;
    }

    // ストライプコールバックを設定する
    CallbackSet set;
    set.emitterPreCalc      = StripeSystem::_EmitterPreCalcCallback;
    set.emitterInitialize   = StripeSystem::_EmitterInitializeCallback;
    set.particleEmit        = StripeSystem::_ParticleEmitCallback;
    set.particleRemove      = StripeSystem::_ParticleRemoveCallback;
    set.particleCalc        = StripeSystem::_ParticleCalcCallback;
    set.emitterPostCalc     = StripeSystem::_EmitterPostCalcCallback;
    set.emitterDraw         = StripeSystem::_EmitterDrawCallback;
    set.emitterFinalize     = StripeSystem::_EmitterFinalizeCallback;
    set.renderStateSet      = _StripeRenderStateSetCallback;
    mSystem->SetEmitterPluginCallbackSet( EFT_EMITTER_PLUGIN_CALLBACK_ID_2, set );
}

//---------------------------------------------------------------------------
//  デストラクタ
//---------------------------------------------------------------------------
StripeSystem::~StripeSystem()
{
    mHeap->Free( mStripeArray );
}

//---------------------------------------------------------------------------
//  空きストライプインスタンスを確保します。
//---------------------------------------------------------------------------
Stripe* StripeSystem::Alloc()
{
    u32 i=0;

    do
    {
        mStripeArrayIdx++;
        if ( mStripeArrayIdx >= mStripeArrayNum )
        {
            mStripeArrayIdx = 0;
        }

        if( mStripeArray[mStripeArrayIdx].used == false )
        {
            mStripeArray[mStripeArrayIdx].used                 = true;
            mStripeArray[mStripeArrayIdx].numHistory           = 0;
            mStripeArray[mStripeArrayIdx].next                 = NULL;
            mStripeArray[mStripeArrayIdx].vertexBuffer.pos     = NULL;
            mStripeArray[mStripeArrayIdx].vertexBuffer.dir     = NULL;
            mStripeArray[mStripeArrayIdx].vertexBuffer.emat    = NULL;
            mStripeNum++;

            return &mStripeArray[mStripeArrayIdx];
        }
    }
    while( (++i) < mStripeArrayNum );

    OutputWarning( "Stripe is Empty.\n" );
    return NULL;
}

//---------------------------------------------------------------------------
//  ストライプインスタンスを返却します。
//---------------------------------------------------------------------------
void StripeSystem::Free( Stripe* stripe )
{
    stripe->used = false;
    mStripeNum--;
}

//---------------------------------------------------------------------------
//  ストライプインスタンスを返却します。
//---------------------------------------------------------------------------
void StripeSystem::AddDelayList( Emitter* emitter, Stripe* stripe )
{
    if ( !stripe ) return;
    EFT_ASSERT( mStripeNum != 0 );

    if ( emitter->emitterCalc )
    {
        // 遅延リストにストライプを繋ぐ
        if ( !emitter->stripeDelayList )
        {
            emitter->stripeDelayList = stripe;
        }
        else
        {
            Stripe* temp = reinterpret_cast<Stripe*>( emitter->stripeDelayList );
            Stripe* tail = NULL;

            while( temp )
            {
                tail = temp;
                temp = temp->next;
            }

            if ( tail )
            {
                tail->next = stripe;
            }
        }
    }
    else
    {
        this->Free( stripe );
    }
}


//---------------------------------------------------------------------------
//! @brief        ポリゴン用ワークを確保します。
//---------------------------------------------------------------------------
bool StripeSystem::AllocStripeVertexBuffer( Emitter* emitter )
{
    if ( !emitter ) { return false; }

    ResStripeHistory* stripeData = reinterpret_cast<ResStripeHistory*>( emitter->emitterRes->emitterPluginData );

    // ストライプが遅延(尾が縮む時間)で破棄される為、多目にバッファを確保する
    const u32 particleNum = emitter->ptclAttrFillMax;
    const u32 element     = GetVertexBufferElementNum();

    // 最大頂点数
    const u32 numDiv = static_cast<u32>( stripeData->numDivide );
    const u32 numHist = static_cast<u32>( stripeData->numHistory );
    emitter->stripeVertexBufferNum = ( numHist + ( ( numHist - 1 ) * numDiv ) ) * 2 * particleNum * mBufferMode;

    // 頂点バッファサイズ
    const u32 bufferSize = emitter->stripeVertexBufferNum * element * sizeof( nw::math::VEC4 );

    // ポリゴン用バッファを確保
    if ( bufferSize > 0 )
    {
        nw::math::VEC4* vec4Buffer = reinterpret_cast<nw::math::VEC4*>( AllocFromDynamicHeap( bufferSize ) );
        if ( !vec4Buffer )
        {
            emitter->stripeVertexBuffer = NULL;
            return false;
        }

        emitter->stripeVertexBuffer = vec4Buffer;

        // バッファを切り分ける
        if ( mBufferMode == EFT_BUFFER_MODE_DOUBLE )
        {
            emitter->stripeVertexBufferStartIdx[EFT_BUFFER_ID_0] = 0;
            emitter->stripeVertexBufferEndIdx[EFT_BUFFER_ID_0]   = emitter->stripeVertexBufferNum/2;
            emitter->stripeVertexBufferStartIdx[EFT_BUFFER_ID_1] = emitter->stripeVertexBufferNum/2;
            emitter->stripeVertexBufferEndIdx[EFT_BUFFER_ID_1]   = emitter->stripeVertexBufferNum;
            emitter->stripeVertexBufferStartIdx[EFT_BUFFER_ID_2] = 0;
            emitter->stripeVertexBufferEndIdx[EFT_BUFFER_ID_2]   = emitter->stripeVertexBufferNum/2;
        }
        else
        {
            emitter->stripeVertexBufferStartIdx[EFT_BUFFER_ID_0] = 0;
            emitter->stripeVertexBufferEndIdx[EFT_BUFFER_ID_0]   = emitter->stripeVertexBufferNum/3;
            emitter->stripeVertexBufferStartIdx[EFT_BUFFER_ID_1] = emitter->stripeVertexBufferNum/3;
            emitter->stripeVertexBufferEndIdx[EFT_BUFFER_ID_1]   = emitter->stripeVertexBufferNum/3*2;
            emitter->stripeVertexBufferStartIdx[EFT_BUFFER_ID_2] = emitter->stripeVertexBufferNum/3*2;
            emitter->stripeVertexBufferEndIdx[EFT_BUFFER_ID_2]   = emitter->stripeVertexBufferNum;
        }
    }
    else
    {
        emitter->stripeVertexBuffer = NULL;
    }
    return true;
}

//---------------------------------------------------------------------------
//  遅延解放ストライプを破棄します。
//---------------------------------------------------------------------------
void StripeSystem::FreeDelayStripeList( Emitter* emitter )
{
    if ( emitter->stripeVertexBuffer )
    {
        FreeFromDynamicHeap( emitter->stripeVertexBuffer, false );
        emitter->stripeVertexBuffer = NULL;
    }

    if ( !emitter->stripeDelayList ) return;
    Stripe* temp = reinterpret_cast<Stripe*>( emitter->stripeDelayList );

    while( temp )
    {
        this->Free( temp );
        temp = temp->next;
    }

    emitter->stripeDelayList = NULL;
}

//---------------------------------------------------------------------------
//  遅延ストライプ計算処理を行います。
//---------------------------------------------------------------------------
inline void StripeSystem::CalcDelayStripe( Emitter* emitter )
{
    if ( !emitter->stripeDelayList ){ return; }

    Stripe* stripe = reinterpret_cast< Stripe* >( emitter->stripeDelayList );
    if( !stripe ){ return; }

    ResStripeHistory* stripeData = reinterpret_cast< ResStripeHistory* >( emitter->emitterRes->emitterPluginData );

    nw::math::VEC4* vec4Buffer = reinterpret_cast< nw::math::VEC4* >(emitter->stripeVertexBuffer );

    if ( emitter->stripeVertexBuffer )
    {

        Stripe* preStripe = NULL;
        while( stripe )
        {
            Stripe* next = stripe->next;

            // 死亡判定
            if( stripe->numHistory > 1 )
            {
                stripe->numHistory--;

                const u32 posIndex = emitter->stripeVertexBufferCurIdx;
                const u32 dirIndex = ( emitter->stripeVertexBufferNum ) + emitter->stripeVertexBufferCurIdx;
                const u32 ematIndex = ( emitter->stripeVertexBufferNum * 2 ) + emitter->stripeVertexBufferCurIdx;

                stripe->vertexBuffer.pos  = &vec4Buffer[posIndex];
                stripe->vertexBuffer.dir  = &vec4Buffer[dirIndex];
                stripe->vertexBuffer.emat = &vec4Buffer[ematIndex];

                // 遅延中はストライプタイムを進める
                stripe->time += emitter->frameRate;

                if ( (stripe->numHistory-1)*2 < ( emitter->stripeVertexBufferEndIdx[emitter->bufferID] - emitter->stripeVertexBufferCurIdx ) )
                {
                    // スケールフェードイン／アウト
                    const f32 scaleFadeValue = emitter->CalculateCurrentScaleFadeValue();

                    const f32 textureBaseValue = static_cast<f32>( stripe->numHistory ); // 今あるストライプの数
                    if( stripeData->numDivide == 0 )
                    {
                        // 分割無し
                        MakeStripeVertexBuffer( stripe, stripeData, &stripe->vertexBuffer, textureBaseValue, scaleFadeValue );
                        emitter->stripeVertexBufferCurIdx += stripe->vertexBufferNum;
                    }
                    else
                    {
                        // 分割あり
                        MakeStripeVertexBufferForStripeDivision( stripe, stripeData, &stripe->vertexBuffer, textureBaseValue, scaleFadeValue );
                        emitter->stripeVertexBufferCurIdx += stripe->vertexBufferNum;
                    }

                    EFT_ASSERT( emitter->stripeVertexBufferNum >= emitter->stripeVertexBufferCurIdx );
                }
                else
                {
                    // これ以上書き込めない場合は、判定用の頂点バッファを空にしておく。
                    stripe->vertexBuffer.pos = NULL;
                    stripe->vertexBufferNum = 0;
                }

                preStripe = stripe;
                stripe    = next;
            }
            else
            {
                if( preStripe )
                {
                    preStripe->next = next;
                }
                else
                {
                    emitter->stripeDelayList = next;
                }
                this->Free( stripe );

                stripe    = next;
            }
        }
    }
}

//---------------------------------------------------------------------------
//  ストライプ計算処理を行います。
//---------------------------------------------------------------------------
inline void StripeSystem::CalcStripe( ParticleCalcArg& arg, Stripe* stripe )
{
    if ( !stripe ) return;
    if ( !stripe->used ) return;

    // 生成時間をコピー
    stripe->time = arg.GetTime();
    stripe->life = arg.GetLife();

    ResStripeHistory* stripeData = reinterpret_cast<ResStripeHistory*>( arg.emitter->emitterRes->emitterPluginData );

    // ストライプのカラーを更新（ストライプ一本で一つのカラーを共有）
    arg.CalcLocalColor0( &stripe->color0 );
    arg.CalcLocalColor1( &stripe->color1 );

    // 履歴をずらす
    if ( stripe->numHistory > 0 )
    {
        for ( u32 i = stripe->numHistory; i > 0 ; i-- )
        {
            // 高速に代入
            {
                EFT_F32_VEC4_COPY( &stripe->hist[i].pos,          &stripe->hist[i-1].pos );
                EFT_F32_VEC3_COPY( &stripe->hist[i].dir,          &stripe->hist[i-1].dir );
                EFT_F32_VEC3_COPY( &stripe->hist[i].emitterSRT_Y, &stripe->hist[i-1].emitterSRT_Y );
            }
        }
    }

    // 履歴を積む
    {
        nw::math::VEC3 scale;
        nw::math::VEC3 stripePos, stripeVec;

        if( stripeData->emitterFollow )
        {
            // エミッタに完全追従：ローカル系で位置を覚える
            arg.CalcWorldScale( &scale );
            arg.GetLocalPos( &stripePos );
            arg.GetLocalVec( &stripeVec );
        }
        else
        {
            // エミッタに完全追従しない：ワールド系で位置を覚える
            arg.CalcWorldScale( &scale );
            arg.GetWorldPos( &stripePos );
            arg.GetWorldVec( &stripeVec );
        }

        EFT_F32_VEC3_COPY( &stripe->hist[0].pos, &stripePos );
        stripe->hist[0].pos.w = scale.x;
        EFT_F32_VEC3_COPY( &stripe->hist[0].dir, &stripeVec );   // 履歴ポイントの向き

        if( stripeData->emitterFollow )
        {
            // エミッタに完全追従：ローカル系で位置を覚えるので上方向はあくまでY軸方向
            stripe->hist[0].emitterSRT_Y = nw::math::VEC3( 0, 1, 0 );
        }
        else
        {
            // エミッタに完全追従しない：ワールド系で位置を覚えるので上方向はエミッタ行列のY軸方向
            stripe->hist[0].emitterSRT_Y = nw::math::VEC3(
                arg.emitter->matrixSRT.m[0][1],
                arg.emitter->matrixSRT.m[1][1],
                arg.emitter->matrixSRT.m[2][1] );
        }
    }
    if( stripe->numHistory < stripeData->numHistory ){ stripe->numHistory++; }

    if( stripe->numHistory >= 2 )
    {
        // 方向補間率による補正
        // ずらす方式になっているので常にインデックス0が最新。よって次の1が直前。
        StripeHistory* histPrev = &stripe->hist[ 1 ];
        StripeHistory* histNow = &stripe->hist[ 0 ];
        const nw::math::VEC4 tempDir = histNow->pos - histPrev->pos;
        nw::math::VEC3 direction;
        EFT_F32_VEC3_COPY( &direction, &tempDir );
        if( stripeData->dirInterpolate > 0 )
        {
            // 方向補間処理（テスト）
            if( stripe->time < 2  )
            {
                EFT_F32_VEC3_COPY( &stripe->interpolateNextDir, &direction );
                if ( stripe->interpolateNextDir.Length() > 0.0f ){ stripe->interpolateNextDir.Normalize(); }
                // ストライプの進行方向
                EFT_F32_VEC3_COPY( &histNow->dir, &stripe->interpolateNextDir );
            }
            else
            {
                if ( direction.Length() > 0.0f ){ direction.Normalize();}
                stripe->interpolateNextDir = stripe->interpolateNextDir + ( direction - stripe->interpolateNextDir ) * stripeData->dirInterpolate;
                if ( stripe->interpolateNextDir.Length() > 0.0f ){ stripe->interpolateNextDir.Normalize(); }
                // ストライプの進行方向
                EFT_F32_VEC3_COPY( &histNow->dir, &stripe->interpolateNextDir );
            }
        }
        else
        {
            EFT_F32_VEC3_COPY( &histNow->dir, &direction );
        }

        if( stripe->numHistory == 2 )
        {
            // ヒストリが2つの時、（≒一番最初に描画されるとき）
            // dirはhistNowの方向に合わせておく！
            // ※以降、同じ方向のまま後ろに移動していくので面倒を見るのはこのタイミングだけでOK。
            EFT_F32_VEC3_COPY( &histPrev->dir, &histNow->dir );
        }
    }

    //-------------------------------
    // ポリゴンバッファを生成
    //-------------------------------
    if ( arg.emitter->stripeVertexBuffer )
    {
        nw::math::VEC4* vec4Buffer = reinterpret_cast<nw::math::VEC4*>( arg.emitter->stripeVertexBuffer );

        if ( (stripe->numHistory-1)*2 < ( arg.emitter->stripeVertexBufferEndIdx[arg.emitter->bufferID] - arg.emitter->stripeVertexBufferCurIdx ) )
        {
            f32 textureBaseValue = 1;             //テクスチャ座標を決めるための分母
            if( stripeData->texturing == 0 )
            {
                // 「全て貼る」
                textureBaseValue = static_cast<f32>( stripe->numHistory ); // 今あるストライプの数
            }
            else if( stripeData->texturing == 1 )
            {
                // 「描画範囲に合わせて貼る」
                textureBaseValue = stripeData->numHistory;    // ストライプの総数
            }

            const u32 posIndex = arg.emitter->stripeVertexBufferCurIdx;
            const u32 dirIndex = ( arg.emitter->stripeVertexBufferNum ) + arg.emitter->stripeVertexBufferCurIdx;
            const u32 ematIndex = ( arg.emitter->stripeVertexBufferNum * 2 ) + arg.emitter->stripeVertexBufferCurIdx;

            stripe->vertexBuffer.pos  = &vec4Buffer[ posIndex ];
            stripe->vertexBuffer.dir  = &vec4Buffer[ dirIndex ];
            stripe->vertexBuffer.emat = &vec4Buffer[ ematIndex ];

            // スケールフェードイン／アウト
            const f32 scaleFadeValue = arg.emitter->CalculateCurrentScaleFadeValue();

            if( stripeData->numDivide == 0 )
            {
                // 分割無し
                const u32 bufferNum = stripe->numHistory * 2;
#if EFT_IS_CAFE
                // pos, dir
                const u32 dcbzSize = bufferNum * sizeof( nw::math::VEC4 );
                MemUtil::ZeroRange( stripe->vertexBuffer.pos, dcbzSize );
                MemUtil::ZeroRange( stripe->vertexBuffer.dir, dcbzSize );
                if( stripeData->calcType != EFT_STRIPE_POLYGON_TYPE_BILLBOARD )
                {
                    DCZeroRange( stripe->vertexBuffer.emat, dcbzSize );
                }
#endif
                MakeStripeVertexBuffer( stripe, stripeData, &stripe->vertexBuffer, textureBaseValue, scaleFadeValue );

                EFT_ASSERT( bufferNum == stripe->vertexBufferNum ); // バッファ追加数の予測＝実測のチェック
                arg.emitter->stripeVertexBufferCurIdx += stripe->vertexBufferNum;
            }
            else
            {
                // 分割あり
                if( stripe->numHistory > 2 )
                {
                    const u32 numDivPoint = ( ( stripe->numHistory - 1 ) * static_cast< u32 >( stripeData->numDivide ) );   // 分割により発生する履歴ポイントの数
                    const u32 numDrawHist = stripe->numHistory + numDivPoint;
                    const u32 bufferNum = numDrawHist * 2;
#if EFT_IS_CAFE
                    // pos, dir
                    const u32 dcbzSize = bufferNum * sizeof( nw::math::VEC4 );
                    MemUtil::ZeroRange( stripe->vertexBuffer.pos, dcbzSize );
                    MemUtil::ZeroRange( stripe->vertexBuffer.dir, dcbzSize );
                    if( stripeData->calcType != EFT_STRIPE_POLYGON_TYPE_BILLBOARD )
                    {
                        MemUtil::ZeroRange( stripe->vertexBuffer.emat, dcbzSize );
                    }
#endif
                    MakeStripeVertexBufferForStripeDivision( stripe, stripeData, &stripe->vertexBuffer, textureBaseValue, scaleFadeValue );

                    EFT_ASSERT( bufferNum == stripe->vertexBufferNum ); // バッファ追加数の予測＝実測のチェック
                }
                else
                {
                    stripe->vertexBuffer.pos = NULL;
                    stripe->vertexBufferNum = 0;
                }
                arg.emitter->stripeVertexBufferCurIdx += stripe->vertexBufferNum;
            }

            // 確保した領域からはみ出ていないことのチェック！
            // 加算後なので、Fullに埋めた場合＝になる事はあるのでそれは許容する。
            EFT_ASSERT( arg.emitter->stripeVertexBufferCurIdx <= arg.emitter->stripeVertexBufferCurIdx );
        }
        else
        {
            stripe->vertexBuffer.pos = NULL;
            stripe->vertexBufferNum = 0;
        }
    }
    else
    {
        stripe->vertexBuffer.pos = NULL;
        stripe->vertexBufferNum = 0;
    }
}

//---------------------------------------------------------------------------
//  ストライプのポリゴン生成を行います。
//---------------------------------------------------------------------------
void StripeSystem::MakeStripeVertexBuffer( Stripe* stripe, ResStripeHistory* stripeData, StripeVertexBuffer* vertexBuffer, f32 textureBaseValue, f32 scaleFadeValue )
{
    u32 idxL = 0;
    u32 idxR = 0;
    u32 idx  = 0;
    const u32 numDrawHist = stripe->numHistory;
    const f32 texBase = ( textureBaseValue - 1 );   // ストライプ末尾でv=1.0になってほしいので-1する

    if( stripeData->calcType == EFT_STRIPE_POLYGON_TYPE_BILLBOARD )
    {
        // ビルボードストライプの時はEmatが不要！
        for( u32 i = 0; i < numDrawHist; i++ )
        {
            idxL  = idx * 2;
            idxR  = idxL + 1;

            EFT_F32_VEC4_COPY( &vertexBuffer->pos[idxL], &stripe->hist[i].pos ); // 翼の長さのコピーも一緒に行う！
            //vertexBuffer->pos[idxL].w = stripe->hist[i].pos.w;                 // ↑の代入処理に含まれるので不要！
            EFT_F32_VEC3_COPY( &vertexBuffer->pos[idxR], &stripe->hist[i].pos );
            vertexBuffer->pos[idxR].w = -stripe->hist[i].pos.w;                  // 翼の長さ（※マイナスにするので別途行う）

            // スケールフェード
            vertexBuffer->pos[idxL].w *= scaleFadeValue;
            vertexBuffer->pos[idxR].w *= scaleFadeValue;

            EFT_F32_VEC3_COPY( &vertexBuffer->dir[idxL], &stripe->hist[i].dir );
            EFT_F32_VEC3_COPY( &vertexBuffer->dir[idxR], &stripe->hist[i].dir );

            // テクスチャ座標
            vertexBuffer->dir[idxR].w = vertexBuffer->dir[idxL].w = ( static_cast< f32 >(i) / texBase );

            idx++;
        }
        stripe->vertexBufferNum = idx * 2;
    }
    else
    {
        // Ematが必要なストライプタイプの場合。
        for( u32 i = 0; i < numDrawHist; i++ )
        {
            idxL  = idx * 2;
            idxR  = idxL + 1;

            EFT_F32_VEC4_COPY( &vertexBuffer->pos[idxL], &stripe->hist[i].pos ); // 翼の長さのコピーも一緒に行う！
            //vertexBuffer->pos[idxL].w = stripe->hist[i].pos.w;                 // ↑の代入処理に含まれるので不要！
            EFT_F32_VEC3_COPY( &vertexBuffer->pos[idxR], &stripe->hist[i].pos );
            vertexBuffer->pos[idxR].w = -stripe->hist[i].pos.w;                  // 翼の長さ（※マイナスにするので別途行う）

            // スケールフェード
            vertexBuffer->pos[idxL].w *= scaleFadeValue;
            vertexBuffer->pos[idxR].w *= scaleFadeValue;

            EFT_F32_VEC3_COPY( &vertexBuffer->dir[idxL], &stripe->hist[i].dir );
            EFT_F32_VEC3_COPY( &vertexBuffer->dir[idxR], &stripe->hist[i].dir );

            EFT_F32_VEC3_COPY( &vertexBuffer->emat[idxL], &stripe->hist[i].emitterSRT_Y );
            EFT_F32_VEC3_COPY( &vertexBuffer->emat[idxR], &stripe->hist[i].emitterSRT_Y );

            // テクスチャ座標
            vertexBuffer->dir[idxR].w = vertexBuffer->dir[idxL].w = ( static_cast< f32 >(i) / texBase );

            idx++;
        }
        stripe->vertexBufferNum = idx * 2;
    }

#if EFT_IS_CAFE
    MemUtil::FlushCache( vertexBuffer->pos,  sizeof( nw::math::VEC4 ) * stripe->vertexBufferNum );
    MemUtil::FlushCache( vertexBuffer->dir,  sizeof( nw::math::VEC4 ) * stripe->vertexBufferNum );
    if( stripeData->calcType != EFT_STRIPE_POLYGON_TYPE_BILLBOARD )
    {
        MemUtil::FlushCache( vertexBuffer->emat, sizeof( nw::math::VEC4 ) * stripe->vertexBufferNum );
    }
#endif

    return;
}

//---------------------------------------------------------------------------
//  ストライプのポリゴン生成を行います。（ストライプ分割版）
//---------------------------------------------------------------------------
void StripeSystem::MakeStripeVertexBufferForStripeDivision( Stripe* stripe, ResStripeHistory* stripeData, StripeVertexBuffer* vertexBuffer, f32 textureBaseValue, f32 scaleFadeValue )
{
    EFT_NULL_ASSERT( stripe );
    EFT_NULL_ASSERT( stripeData );
    EFT_NULL_ASSERT( vertexBuffer );

    u32 idxL = 0;
    u32 idxR = 0;
    u32 idx  = 0;
    stripe->vertexBufferNum = 0;

    const u32 numHist = static_cast< u32 >( stripe->numHistory );

    // 履歴ポイントが2つは無いとポリゴンにならないので処理を返す。
    if( numHist <= 2 ){ return; }

    const StripeHistory* hist = stripe->hist;

    const u32 numDiv = static_cast< u32 >( stripeData->numDivide ); // 分割数。
    const u32 numDivPoint = ( ( numHist - 1 ) * numDiv );           // 分割により発生する履歴ポイントの数
    const u32 numDrawHist = numHist + numDivPoint;                  // 分割分の頂点を追加。
    const f32 texBase = ( textureBaseValue - 1 ) + numDivPoint;     // ストライプ末尾でv=1.0になってほしいので-1する

    const f32 fIndexDiv = 1.0f / ( 1 + numDiv );

    if( stripeData->calcType == EFT_STRIPE_POLYGON_TYPE_BILLBOARD )
    {
        // ビルボードストライプの時はEmatが不要！
        for( u32 i = 0; i < numDrawHist; i++ )
        {
            idxL  = idx * 2;
            idxR  = idxL + 1;

            const f32 fIndex = i * fIndexDiv;                                       // ストライプ状の位置（履歴ポイント場が整数になる）
            const u32 histPos = static_cast<u32>( nw::math::FFloor( fIndex ) );     // 所属する履歴ポイント
            f32 innerPos = fIndex - histPos;                                        // 履歴ポイント内の位置 [0.0f, 1.0f]

            nw::math::VEC3 curvePos;            // 履歴ポイントの位置（エルミート補間後）
            nw::math::VEC3 curveDir;            // 履歴ポイントの向き（線形補間後）

            //-----------------------------------------------------------------
            // 間を曲線補間
            //-----------------------------------------------------------------
            CalcHermiteInterpolatedCurveVec( &curvePos, &curveDir, hist, histPos, numHist, innerPos );

            EFT_F32_VEC3_COPY( &vertexBuffer->pos[idxL], &curvePos );       // 翼の長さ
            EFT_F32_VEC3_COPY( &vertexBuffer->pos[idxR], &curvePos );
            vertexBuffer->pos[idxL].w =  hist[histPos].pos.w * scaleFadeValue;
            vertexBuffer->pos[idxR].w = -hist[histPos].pos.w * scaleFadeValue;               // 翼の長さ（※マイナスにするので別途行う）

            EFT_F32_VEC3_COPY( &vertexBuffer->dir[idxL], &curveDir );
            EFT_F32_VEC3_COPY( &vertexBuffer->dir[idxR], &curveDir );

            // テクスチャ座標
            vertexBuffer->dir[idxR].w = vertexBuffer->dir[idxL].w = ( static_cast< f32 >(i) / texBase );

            idx++;
        }
        stripe->vertexBufferNum = idx * 2;
    }
    else
    {
        // Ematが必要なストライプタイプの場合。
        for( u32 i = 0; i < numDrawHist; i++ )
        {
            idxL  = idx * 2;
            idxR  = idxL + 1;

            const f32 fIndex = i * fIndexDiv;                                       // ストライプ状の位置（履歴ポイント場が整数になる）
            const u32 histPos = static_cast<u32>( nw::math::FFloor( fIndex ) );     // 所属する履歴ポイント
            f32 innerPos = fIndex - histPos;                                        // 履歴ポイント内の位置 [0.0f, 1.0f]

            nw::math::VEC3 curvePos;            // 履歴ポイントの位置（エルミート補間後）
            nw::math::VEC3 curveDir;            // 履歴ポイントの向き（線形補間後）

            //-----------------------------------------------------------------
            // 間を曲線補間
            //-----------------------------------------------------------------
            CalcHermiteInterpolatedCurveVec( &curvePos, &curveDir, stripe->hist, histPos, numHist, innerPos );

            EFT_F32_VEC3_COPY( &vertexBuffer->pos[idxL], &curvePos );       // 翼の長さ
            EFT_F32_VEC3_COPY( &vertexBuffer->pos[idxR], &curvePos );
            vertexBuffer->pos[idxL].w =  hist[histPos].pos.w * scaleFadeValue;
            vertexBuffer->pos[idxR].w = -hist[histPos].pos.w * scaleFadeValue;               // 翼の長さ（※マイナスにするので別途行う）

            EFT_F32_VEC3_COPY( &vertexBuffer->dir[idxL], &curveDir );
            EFT_F32_VEC3_COPY( &vertexBuffer->dir[idxR], &curveDir );

            EFT_F32_VEC3_COPY( &vertexBuffer->emat[idxL], &hist[histPos].emitterSRT_Y );
            EFT_F32_VEC3_COPY( &vertexBuffer->emat[idxR], &hist[histPos].emitterSRT_Y );

            // テクスチャ座標
            vertexBuffer->dir[idxR].w = vertexBuffer->dir[idxL].w = ( static_cast< f32 >(i) / texBase );

            idx++;
        }
        stripe->vertexBufferNum = idx * 2;
    }

#if EFT_IS_CAFE
    MemUtil::FlushCache( vertexBuffer->pos,  sizeof( nw::math::VEC4 ) * stripe->vertexBufferNum );
    MemUtil::FlushCache( vertexBuffer->dir,  sizeof( nw::math::VEC4 ) * stripe->vertexBufferNum );
    if( stripeData->calcType != EFT_STRIPE_POLYGON_TYPE_BILLBOARD )
    {
        MemUtil::FlushCache( vertexBuffer->emat, sizeof( nw::math::VEC4 ) * stripe->vertexBufferNum );
    }
#endif

    return;
}

//---------------------------------------------------------------------------
//  ストライプ描画処理を行います。
//---------------------------------------------------------------------------
void StripeSystem::Draw( Emitter* emitter, ShaderType shaderType, void* userParam )
{
    System* system = emitter->emitterSet->GetSystem();

    // シェーダ切り替え
    Shader* shader = emitter->shader[shaderType];
    shader->Bind();

    // 履歴ストライプデータ
    ResStripeHistory* stripeData = reinterpret_cast<ResStripeHistory*>(
        emitter->emitterRes->emitterPluginData );

    // 描画設定
    ResRenderState* state = &emitter->emitterData->renderState;
    Rendercontext::SetRenderStateResource( state, system->GetColorBufferEnbaleFlag() );

    // テクスチャ設定
    for ( u32 i = 0; i < EFT_TEXTURE_SLOT_MAX; i++ )
    {
        TextureSlot slot = static_cast<TextureSlot>( i );
        TextureResource* texture = emitter->emitterRes->texture[i];

        if ( texture )
        {
            u32 fragLoc = shader->GetFragmentTextureSamplerLocation( slot );
            u32 vertLoc = shader->GetVertexTextureSamplerLocation( slot );

            if ( texture->Bind( fragLoc, vertLoc, slot ) )
            {
                emitter->emitterRes->textureSampler[i].Bind( fragLoc, vertLoc, slot );
            }
        }
    }

    /*
        TODO:サンプラの間借りを辞める。
    */

    // フレームバッファテクスチャの設定
    {
        u32 fragLoc = shader->GetFrameBufferTextureFragmentSamplerLocation();
        u32 vertLoc = shader->GetFrameBufferTextureVertexSamplerLocation();

        if ( system->GetFrameBufferTexture().Bind( fragLoc, vertLoc, EFT_TEXTURE_SLOT_EXT_FB ) )
        {
            emitter->emitterRes->textureSampler[0].Bind( fragLoc, vertLoc, EFT_TEXTURE_SLOT_EXT_FB );
        }
    }

    // デプスバッファテクスチャの設定
    {
        u32 fragLoc = shader->GetDepthBufferTextureFragmentSamplerLocation();
        u32 vertLoc = shader->GetDepthBufferTextureVertexSamplerLocation();

        if ( system->GetDepthBufferTexture().Bind( fragLoc, vertLoc, EFT_TEXTURE_SLOT_EXT_DB ) )
        {
            emitter->emitterRes->textureSampler[0].Bind( fragLoc, vertLoc, EFT_TEXTURE_SLOT_EXT_DB );
        }
    }

    // エミッタ動的UBOの設定
    shader->BindUniformBlock( Shader::EFT_UBO_TYPE_EMITTER_DYNAMIC, &emitter->dynamicUBO[emitter->bufferID] );

    // エミッタ静的UBOの設定
    shader->BindUniformBlock( Shader::EFT_UBO_TYPE_EMITTER_STATIC, &emitter->emitterRes->emitterStaticUbo );

    // カスタムシェーダコールバックの呼び出し
    {
        RenderStateSetArg rssArg;
        rssArg.emitter      = emitter;
        rssArg.userParam    = userParam;
        rssArg.shaderType   = shaderType;
        rssArg.isStreamOut  = false;

        mSystem->ResetCustomShaderSetting( shader );
        {
            // 描画設定後コールバックの呼び出し
            bool ret = InvokeRenderStateSetCallback( rssArg );
            if( !ret ){ return; }

            // DrawPath コールバックの呼び出し
            if ( emitter->drawPathCallback ) { emitter->drawPathCallback( rssArg ); }
        }
        mSystem->ResetCustomShaderSetting( NULL );
    }

    const u32 posAttr      = shader->GetAttribute( Shader::EFT_ATTR_IDX_POSITION );
    const u32 epParam0Attr = shader->GetEmitterPluginAttribute( Shader::EFT_EMTR_PLUGIN_ATTR_IDX_0 );
    const u32 epParam1Attr = shader->GetEmitterPluginAttribute( Shader::EFT_EMTR_PLUGIN_ATTR_IDX_1 );

    // 遅延リストの描画
    if ( emitter->stripeDelayList )
    {
        Stripe* stripe = reinterpret_cast<Stripe*>( emitter->stripeDelayList );
        while( stripe )
        {
            u32 vertexNum = stripe->vertexBufferNum;

            if ( vertexNum > 0 && stripe->vertexBuffer.pos )
            {
                // ストライプUBOを設定する
                TemporaryUniformBlockBuffer stripeTempUbo;
                StripeUniformBlockParam* ubo = reinterpret_cast< StripeUniformBlockParam* >( stripeTempUbo.Alloc( system, sizeof( StripeUniformBlockParam ) ) );
                if ( !ubo )
                {
                    stripe = stripe->next;
                    continue;
                }

                ubo->param0.x = stripeData->headAlpha;
                ubo->param0.y = stripeData->tailAlpha;
                EFT_F32_VEC4_COPY( &ubo->random, &stripe->random );
                EFT_F32_VEC4_COPY( &ubo->color0, &stripe->color0 );
                EFT_F32_VEC4_COPY( &ubo->color1, &stripe->color1 );
                ubo->param1.x = stripe->time;
                ubo->param1.y = stripe->numHistory + stripeData->numDivide * ( stripe->numHistory - 1 );
                ubo->param1.z = EFT_STRIPE_NORMAL_MODE;
                ubo->param1.w = stripe->life;

                stripeTempUbo.Validate();

                // ストライプデータのUBO設定
                shader->BindEmitterPluginUniformBlock( &stripeTempUbo );

                // 頂点を設定
                VertexBuffer::BindExtBuffer( posAttr,
                    sizeof(nw::math::VEC4) * vertexNum, 4, sizeof(nw::math::VEC4), stripe->vertexBuffer.pos );
                VertexBuffer::BindExtBuffer( epParam0Attr,
                    sizeof(nw::math::VEC4) * vertexNum, 4, sizeof(nw::math::VEC4), stripe->vertexBuffer.dir );
                VertexBuffer::BindExtBuffer( epParam1Attr,
                    sizeof(nw::math::VEC4) * vertexNum, 4, sizeof(nw::math::VEC4), stripe->vertexBuffer.emat );

                // 描画
                Render::DrawPrimitive( Render::PRIM_TYPE_TRIANGLE_STRIP, 0, vertexNum );
                if( stripeData->option == EFT_STRIPE_CROSS_MODE )
                {
                    // クロスの場合、再度描画する！
                    // ストライプUBOを設定する
                    TemporaryUniformBlockBuffer stripeCrossTempUbo;
                    StripeUniformBlockParam* ubo2 = reinterpret_cast< StripeUniformBlockParam* >( stripeCrossTempUbo.Alloc( system, sizeof( StripeUniformBlockParam ) ) );
                    if ( !ubo2 )
                    {
                        stripe = stripe->next;
                        continue;
                    }

                    EFT_F32_VEC4_COPY( &ubo2->random, &stripe->random );
                    ubo2->param0.x = stripeData->headAlpha;
                    ubo2->param0.y = stripeData->tailAlpha;
                    EFT_F32_VEC4_COPY( &ubo2->color0, &stripe->color0 );
                    EFT_F32_VEC4_COPY( &ubo2->color1, &stripe->color1 );
                    ubo2->param1.x = stripe->time;
                    ubo2->param1.y = stripe->numHistory + stripeData->numDivide * ( stripe->numHistory - 1 );
                    ubo2->param1.z = EFT_STRIPE_CROSS_MODE;  // クロス指定
                    ubo2->param1.w = stripe->life;

                    stripeCrossTempUbo.Validate();

                    // ストライプデータのUBO設定
                    shader->BindEmitterPluginUniformBlock( &stripeCrossTempUbo );

                    // 描画
                    Render::DrawPrimitive( Render::PRIM_TYPE_TRIANGLE_STRIP, 0, vertexNum );
                }
            }
            stripe = stripe->next;
        }
    }

    // 描画
    for ( u32 i = 0; i < emitter->ptclNum; i++ )
    {
        Stripe* stripe = reinterpret_cast<Stripe*>( emitter->particleData[i].emitterPluginData );
        if ( !stripe || !stripe->used ) continue;
        if ( !stripe->vertexBuffer.pos ) continue;

        u32 vertexNum = stripe->vertexBufferNum;

        if ( vertexNum > 0 && stripe->vertexBuffer.pos )
        {
            // ストライプUBOを設定する
            TemporaryUniformBlockBuffer stripeTempUbo;
            StripeUniformBlockParam* ubo = reinterpret_cast< StripeUniformBlockParam* >( stripeTempUbo.Alloc( system, sizeof( StripeUniformBlockParam ) ) );
            if ( !ubo )
            {
                stripe = stripe->next;
                continue;
            }

            ubo->param0.x = stripeData->headAlpha;
            ubo->param0.y = stripeData->tailAlpha;
            EFT_F32_VEC4_COPY( &ubo->random, &stripe->random );
            EFT_F32_VEC4_COPY( &ubo->color0, &stripe->color0 );
            EFT_F32_VEC4_COPY( &ubo->color1, &stripe->color1 );
            ubo->param1.x = stripe->time;
            ubo->param1.y = stripe->numHistory + stripeData->numDivide * ( stripe->numHistory - 1 );
            ubo->param1.z = EFT_STRIPE_NORMAL_MODE;  // クロス指定無し
            ubo->param1.w = stripe->life;

            stripeTempUbo.Validate();

            // ストライプデータのUBO設定
            shader->BindEmitterPluginUniformBlock( &stripeTempUbo );

            // 頂点を設定
            VertexBuffer::BindExtBuffer( posAttr,
                sizeof(nw::math::VEC4) * vertexNum, 4, sizeof(nw::math::VEC4), stripe->vertexBuffer.pos );
            VertexBuffer::BindExtBuffer( epParam0Attr,
                sizeof(nw::math::VEC4) * vertexNum, 4, sizeof(nw::math::VEC4), stripe->vertexBuffer.dir );
            VertexBuffer::BindExtBuffer( epParam1Attr,
                sizeof(nw::math::VEC4) * vertexNum, 4, sizeof(nw::math::VEC4), stripe->vertexBuffer.emat );

            // 描画
            Render::DrawPrimitive( Render::PRIM_TYPE_TRIANGLE_STRIP, 0, vertexNum );

            if( stripeData->option == EFT_STRIPE_CROSS_MODE )
            {
                // クロスの場合、再度描画する！
                // ストライプUBOを設定する
                TemporaryUniformBlockBuffer stripeCrossTempUbo;
                StripeUniformBlockParam* ubo2 = reinterpret_cast< StripeUniformBlockParam* >( stripeCrossTempUbo.Alloc( system, sizeof( StripeUniformBlockParam ) ) );
                if ( !ubo2 )
                {
                    stripe = stripe->next;
                    continue;
                }

                EFT_F32_VEC4_COPY( &ubo2->random, &stripe->random );
                ubo2->param0.x = stripeData->headAlpha;
                ubo2->param0.y = stripeData->tailAlpha;
                EFT_F32_VEC4_COPY( &ubo2->color0, &stripe->color0 );
                EFT_F32_VEC4_COPY( &ubo2->color1, &stripe->color1 );
                ubo2->param1.x = stripe->time;
                ubo2->param1.y = stripe->numHistory + stripeData->numDivide * ( stripe->numHistory - 1 );
                ubo2->param1.z = EFT_STRIPE_CROSS_MODE;  // クロス指定
                ubo2->param1.w = stripe->life;

                stripeCrossTempUbo.Validate();

                // ストライプデータのUBO設定
                shader->BindEmitterPluginUniformBlock( &stripeCrossTempUbo );

                // 描画
                Render::DrawPrimitive( Render::PRIM_TYPE_TRIANGLE_STRIP, 0, vertexNum );
            }
        }
    }
}

//------------------------------------------------------------------------------
//  ワンタイムエミッタ用の追加のエミッタ寿命を取得
//------------------------------------------------------------------------------
s32 StripeSystem::GetExtendedEndTimeForOneTimeEmitter( nw::eft2::Emitter* pEmitter )
{
    ResStripeHistory* pStripeData = reinterpret_cast<ResStripeHistory*>( pEmitter->emitterRes->emitterPluginData );
    f32 extendLife = pStripeData->numHistory * pEmitter->frameRate;
    return static_cast< s32 >( extendLife );
}

} // namespace eft2
} // namespace nw

#endif // EFT_DEGRADATION_SPEC

