﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/eft/eft2_EmitterCalc.h>
#include <nw/eft/eft2_Emitter.h>
#include <nw/eft/eft2_FieldRandom.h>
#include <nw/eft/eft2_MemUtil.h>

namespace nw  {
namespace eft2 {

void CalcFieldRandom( Emitter* emitter, nw::math::VEC4* pos, nw::math::VEC4* vec, const ParticleAttribute* attr, const nw::math::VEC3& randomVel  )
{
    // GPUノイズ: 積算可能な疑似乱数を用いたフィールド
    ResFieldRandom* resFieldRandom = emitter->emitterRes->fieldRandomData;

    const f32 time = emitter->time - vec->w;
    const f32 frameRate   = emitter->frameRate;
    const f32 invRate     = 1.0f - frameRate;
    const f32 ubAirRegist = emitter->emitterRes->resEmitterStaticUniformBlock->airRegist;
    const f32 airRegist = ubAirRegist + ( 1.0f - ubAirRegist ) * invRate;
    const f32 RandCycle = static_cast<f32>( resFieldRandom->blank );

    f32 velTime = time;
    if( resFieldRandom->enableAirRegist )
    {
        velTime = ( airRegist == 1.0 ) ? time : ( 1 - pow( airRegist, time ) )/( 1 - airRegist );
    }

    const f32 speed = ( resFieldRandom->unifiedPhaseSpeed / 100.0f ) * RandCycle;
    const f32 randWidth = ( resFieldRandom->unifiedPhaseDistribution / 100.0f ) * RandCycle;

    nw::math::VEC4 seed( 0, 0, 0, 0 );
    nw::math::VEC3 randomValue( 0, 0, 0 );
    f32 deltaSpeed = 0;
    const f32 deltaVelTime = frameRate * 1.0f;

    if( resFieldRandom->enableUnifiedPhase )
    {
        // 位相を統一
        deltaSpeed = frameRate * speed;

        static const nw::math::VEC4 SEED( 0.0f, 0.31415f, 0.92653f, 0.58979f );
        EFT_F32_VEC4_COPY( &seed, &SEED );
        seed *= RandCycle;

        const f32 eTime = emitter->time * speed;
        const f32 temp = eTime - randWidth;
        const f32 temp2 = randWidth * 2;
        seed.x += temp + ( temp2 * attr->random.x );
        seed.y += temp + ( temp2 * attr->random.y );
        seed.z += temp + ( temp2 * attr->random.z );
        seed.w += temp + ( temp2 * attr->random.w );

        if( resFieldRandom->enableDetailedOption )
        {
            const f32 interval = detail::_2PI / static_cast< f32 >( resFieldRandom->blank );               // 基本周期
            randomValue.x = GetRand( seed.x, velTime, deltaSpeed, deltaVelTime, interval, resFieldRandom );
            randomValue.y = GetRand( seed.y, velTime, deltaSpeed, deltaVelTime, interval, resFieldRandom );
            randomValue.z = GetRand( seed.z, velTime, deltaSpeed, deltaVelTime, interval, resFieldRandom );
        }
        else
        {
            const f32 interval = detail::_2PI / static_cast< f32 >( resFieldRandom->blank );
            randomValue.x = GetRandDefault( seed.x, velTime, deltaSpeed, deltaVelTime, interval );
            randomValue.y = GetRandDefault( seed.y, velTime, deltaSpeed, deltaVelTime, interval );
            randomValue.z = GetRandDefault( seed.z, velTime, deltaSpeed, deltaVelTime, interval );
        }
    }
    else
    {
        // 位相の統一なし
        EFT_F32_VEC4_COPY( &seed, &attr->random );
        seed *= RandCycle;

        if( resFieldRandom->enableDetailedOption )
        {
            const f32 interval = detail::_2PI / static_cast< f32 >( resFieldRandom->blank );               // 基本周期
            randomValue.x = GetRandWithNoPhaseShift( seed.x, velTime, deltaVelTime, interval, resFieldRandom );
            randomValue.y = GetRandWithNoPhaseShift( seed.y, velTime, deltaVelTime, interval, resFieldRandom );
            randomValue.z = GetRandWithNoPhaseShift( seed.z, velTime, deltaVelTime, interval, resFieldRandom );
        }
        else
        {

            const f32 interval = detail::_2PI / static_cast< f32 >( resFieldRandom->blank );
            randomValue.x = GetRandWithNoPhaseShiftDefault( seed.x, velTime, deltaVelTime, interval );
            randomValue.y = GetRandWithNoPhaseShiftDefault( seed.y, velTime, deltaVelTime, interval );
            randomValue.z = GetRandWithNoPhaseShiftDefault( seed.z, velTime, deltaVelTime, interval );
        }
    }

    pos->x += randomValue.x * randomVel.x;
    pos->y += randomValue.y * randomVel.y;
    pos->z += randomValue.z * randomVel.z;

    return;
}

}
}





