﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/eft/eft2_Emitter.h>
#include <nw/eft/eft2_EndianUtil.h>
#include <nw/eft/eft2_Resource.h>
#include <nw/eft/eft2_Random.h>
#include <nw/eft/eft2_System.h>
#include <nw/eft/eft2_MemUtil.h>
#include <nw/eft/eft2_StreamOutBuffer.h>

namespace nw   {
namespace eft2 {

//---------------------------------------------------------------------------
//  エミッタ初期化を行います。
//---------------------------------------------------------------------------
bool Emitter::Initialize( const EmitterResource* emitterResourceSet, u32 particleMax, Heap* debugHeap )
{
    // memset( 0 ) で変数は初期化済みなので必要な箇所のみ初期化
    heap                    = debugHeap;
    emitterData             = emitterResourceSet->emitterData;
    particleAttr            = NULL;
    particlePosAttr         = NULL;
    emitterRes              = emitterResourceSet;
    fadeOutAlpha            = 1.0f;
    fadeInAlpha             =
        (( emitterResourceSet->emitterData->emitter.isAlphaFadeIn ) || ( emitterResourceSet->emitterData->emitter.isScaleFadeIn ))
        ? 0.0f : 1.0f;     // MEMO: フェードインを使わない場合は 1.0f を先に埋めておく。
    ptclMaxAssignmentNum    = particleMax;
    manualEmitCommonOffset  = nw::math::VEC3( 0, 0, 0 );
    processEmitterFlag      = 1; // StreamOut用フラグ: 最下位ビットだけ初期値として立てておく
    matrixSRT = nw::math::MTX34::Identity();
    matrixRT = nw::math::MTX34::Identity();

    if ( emitterResourceSet->childEmitterResSet[0] )
    {
        for ( u32 i = 0; i < EFT_EMITTER_INSET_NUM; i++ )
        {
            childEmitterResSet[i]  = emitterResourceSet->childEmitterResSet[i];
        }
    }

    emitRatio               = 1.0f;
    emitIntervalRatio       = 1.0f;
    particleLifeScale       = 1.0f;
    groupID                 = emitterSet->GetGroupID();
    bufferID                = EFT_BUFFER_ID_0;
    isChild                 = false;
    isVisible               = true;
    drawViewFlag            = EFT_DRAW_VIEW_FLAG_ALL;
    isEnableEmit            = true;

    // 乱数を初期化
    RandomGenerator* rgenerator = Random::GetGlobalRandom();
    const u32 seed = static_cast<u32>( emitterData->emitter.randomSeed * 12345678901 );
    switch ( emitterData->emitter.randomSeedType )
    {
        case  0: random.SetSeed( rgenerator->GetU32() );                    break;
        case  1: random.SetSeed( emitterSet->GetEmitterSetRandomSeed() );   break;
        case  2: random.SetSeed( seed );                                    break;
    }

    // StreamOutクラス初期化
    streamOutFlip = 0;
    streamOutPos.Invalidate();
    streamOutVec.Invalidate();

    // リソース更新時処理
    if ( !ResourceUpdate() )
    {
        isEmitterInitializeFailed = true;
        return false;
    }

    // 初回放出の為、インターバルを例外設定
    emitInterval = 0.0f;

    // エミッタセット位置を前フレーム位置として扱う。
    emitterPrevPos.Set( emitterSet->GetSRTMatrix()._03,
                        emitterSet->GetSRTMatrix()._13,
                        emitterSet->GetSRTMatrix()._23 );

    // 初期化時の親エミッタセット生成IDを保持する
    emitterSetCreateID = emitterSet->GetCreateID();

    // エミッタカラーを白で初期化
    color0.x = 1.0f;
    color0.y = 1.0f;
    color0.z = 1.0f;
    color0.w = 1.0f;
    color1.x = 1.0f;
    color1.y = 1.0f;
    color1.z = 1.0f;
    color1.w = 1.0f;

    // エミッタ初期化後処理後コールバック
    EmitterInitializeArg arg;
    arg.emitter  = this;
    bool ret = InvokeEmitterInitializeCallback( arg );

    return ret;
}

//---------------------------------------------------------------------------
//  確保しているバッファの解放処理を行います。
//---------------------------------------------------------------------------
void Emitter::ReleaseBuffer()
{
    if ( allocedFromDynamicHeap )
    {
        FreeFromDynamicHeap( allocedFromDynamicHeap, false );
        allocedFromDynamicHeap = NULL;
    }

    streamOutPos.Finalize();
    streamOutVec.Finalize();
}

f32 GetAnimKeyMaxValueX( ResAnimEmitterKeyParamSet* anim, const f32 defaultValue )
{
    f32 maxRate = defaultValue;
    if( anim )
    {
        const s32 keyNum = anim->keyNum;

        for( int i=0; i < keyNum; i++ )
        {
            if( maxRate < anim->keyValue[i].x )
            {
                maxRate = anim->keyValue[i].x;
            }
        }
    }

    return maxRate;
}

//---------------------------------------------------------------------------
//  リソース更新時処理を行います。
//---------------------------------------------------------------------------
bool Emitter::ResourceUpdate()
{
    u32 particleMax = 0;

    // CPUエミッタかつ何かしらのParticleCalcコールバックが着いているかどうか
    m_IsParticleAttributeDoubleBuffer = emitterData->emitter.calcType == EFT_EMITTER_CALC_TYPE_CPU &&
           ( ( emitterData->action.customActionIndex > 0 &&
               callbackSet[ EFT_CALLBACK_SET_TYPE_CA ] != NULL && callbackSet[ EFT_CALLBACK_SET_TYPE_CA ]->particleCalc != NULL ) ||
             ( emitterData->shader.customShaderIndex > 0 &&
               callbackSet[ EFT_CALLBACK_SET_TYPE_CS ] != NULL && callbackSet[ EFT_CALLBACK_SET_TYPE_CS ]->particleCalc != NULL ) ||
             ( emitterRes->emitterPluginIndex > 0 &&
               callbackSet[ EFT_CALLBACK_SET_TYPE_EP ] != NULL && callbackSet[ EFT_CALLBACK_SET_TYPE_EP ]->particleCalc != NULL ) );

    // パーティクル運用バッファを確保する
    if ( ptclMaxAssignmentNum != 0 )
    {
        particleMax     = ptclMaxAssignmentNum;
    }
    else
    {
        // TODO:エミッタ形状でのパーティクル最大数の変化

        if ( emitterData->emission.isEmitDistEnabled )
        {
            // 距離放出
            particleMax = emitterData->emission.emitDistParticleMax;  // UIから取得
        }
        else
        {
            // エミッタアニメに依存してバッファサイズが変更される件への対処
            f32 emitRateMax = emitterData->emission.rate;
            f32 ptclLifeMax = static_cast<f32>( emitterData->ptcl.life );

            if ( emitterRes->emitterAnimArray[EFT_EMITTER_ANIM_EMISSION_RATE] )
            {
                emitRateMax = GetAnimKeyMaxValueX( emitterRes->emitterAnimArray[EFT_EMITTER_ANIM_EMISSION_RATE], emitterData->emission.rate );
                if ( emitRateMax < 1.0f ){ emitRateMax = 1.0f; }
            }

            if ( emitterRes->emitterAnimArray[EFT_EMITTER_ANIM_PTCL_LIFE] )
            {
                ptclLifeMax = GetAnimKeyMaxValueX( emitterRes->emitterAnimArray[EFT_EMITTER_ANIM_PTCL_LIFE], static_cast<f32>( emitterData->ptcl.life ) );
            }

            s32 emitCount = 0;
            if( emitterData->ptcl.isLifeInfinity && emitterData->emission.isOneTime )
            {
                // 寿命無限（ワンタイム時限定）
                // 放出回数：( 放出時間 / 放出間隔+1 )×放出レート
                const f32 interval = ( emitterData->emission.emitDuration == 0 ) ? 1.0f : static_cast<f32>( emitterData->emission.emitDuration );
                emitCount = static_cast<s32>( nw::math::FCeil(interval / ( emitterData->emission.interval + 1 ) ) );
            }
            else
            {
                // 寿命有限
                // 放出回数：寿命÷（放出間隔＋１）の切り上げ。
                emitCount = static_cast<s32>( nw::math::FCeil( ptclLifeMax / ( emitterData->emission.interval + 1 ) ) );

                if( emitterData->emission.isOneTime )
                {
                    // ワンタイムの時だけ、放出時間による放出回数も計算
                    const f32 emitterInterval = ( emitterData->emission.emitDuration == 0 ) ? 1.0f : static_cast<f32>( emitterData->emission.emitDuration );
                    const s32 emitCountByTime = static_cast<s32>( nw::math::FCeil(emitterInterval / ( emitterData->emission.interval + 1 ) ) );
                    // 小さい方を採用
                    if( emitCountByTime < emitCount ){ emitCount = emitCountByTime; }
                }
            }

            // 放出レートが小数の場合、多めに切り上げる
            particleMax = static_cast<s32>( nw::math::FCeil( emitCount * emitRateMax ) + nw::math::FCeil( emitRateMax ) );

            if( emitterRes->emitterData->volume.primEmitType == EFT_EMIT_FROM_PRIMITIVE_UNISON )
            {
                // 等分割系の一斉放出の場合は、放出レートを分割数倍にする
                if( emitterRes->emitterData->volume.volumeType == EFT_EMITTER_VOLUME_TYPE_CIRCLE_SAME_DIVIDE )
                {
                    particleMax *= emitterRes->emitterData->volume.numDivideCircle;
                }
                else if( emitterRes->emitterData->volume.volumeType == EFT_EMITTER_VOLUME_TYPE_LINE_SAME_DIVIDE )
                {
                    particleMax *= emitterRes->emitterData->volume.numDivideLine;
                }
            }

            // ループ、ストリームアウト時は、一回の放出分のバッファをさらにプラスする
            if ( !emitterData->emission.isOneTime &&
                ( emitterData->emitter.calcType == EFT_EMITTER_CALC_TYPE_GPU_SO ) )
            {
                int addBuffer = static_cast< int >( ::std::ceilf( emitRateMax ) );

                if( emitterRes->emitterData->volume.primEmitType == EFT_EMIT_FROM_PRIMITIVE_UNISON )
                {
                    // 等分割系の一斉放出の場合は、放出レートを分割数倍にする
                    if( emitterRes->emitterData->volume.volumeType == EFT_EMITTER_VOLUME_TYPE_CIRCLE_SAME_DIVIDE )
                    {
                        addBuffer *= emitterRes->emitterData->volume.numDivideCircle;
                    }
                    else if( emitterRes->emitterData->volume.volumeType == EFT_EMITTER_VOLUME_TYPE_LINE_SAME_DIVIDE )
                    {
                        addBuffer *= emitterRes->emitterData->volume.numDivideLine;
                    }
                }

                particleMax += addBuffer;
            }

            if ( particleMax == 0 )
            {
                Warning( reinterpret_cast< void* >( this ), EFT_WARNING_PARTICLE_MAX_NUM_IS_ZERO );
                return false;
            }
        }
    }

    if ( ptclAttrFillMax != particleMax )
    {
        // パーティクル削除コールバック
        InvokeParticleRemoveCallbackAll( this );
        ReleaseBuffer();

        // 動的ヒープからバッファを確保
        // TODO:ビューア経由の場合は、別枠からバッファを確保するようにする。
        {
            bool isCpuEmitter = false;
            if ( emitterData->emitter.calcType == EFT_EMITTER_CALC_TYPE_CPU ){ isCpuEmitter = true; }

            // CPUエミッタの場合は、ParticlePosAttributeをダブルバッファで確保する
            u32 bufferNum = 1;
            if ( isCpuEmitter ){ bufferNum = 2; }

            // 確保するメモリサイズを決定する
            s32 ptclNumTemp = particleMax;
            if ( ptclNumTemp % 2 == 1 ){ ptclNumTemp++; }

            u8* ptr  = NULL;
            u32 addr = 0;

            const s32 particleAttributeSize = ptclNumTemp * sizeof( ParticleAttribute );
            const s32 particlePosAttributeSize = ptclNumTemp * sizeof( ParticlePosAttribute );
            const s32 particleDataSize = ptclNumTemp * sizeof( ParticleData );
            const s32 parentParticleDataSize = ( emitterRes->childEmitterResCount ) ? ptclNumTemp * sizeof( ParentParticleData ) : 0;

            // CPUエミッタかつ何かしらのParticleCalcコールバックが着いている場合、ParticleAttrをダブルバッファ化する
            if( IsParticleAttributeDoubleBuffer() )
            {
                allocedFromDynamicHeapSize = particleAttributeSize * bufferNum +
                    particlePosAttributeSize * bufferNum +
                    particleDataSize +
                    parentParticleDataSize;

                // 動的バッファからメモリを確保
                allocedFromDynamicHeap  = AllocFromDynamicHeap( allocedFromDynamicHeapSize );
                if ( !allocedFromDynamicHeap ) return false;

                // 確保したメモリを切り分ける
                ptr  = reinterpret_cast<u8*>( allocedFromDynamicHeap );
                addr = 0;

                // ParticleAttr
                particleAttrOrg[EFT_BUFFER_ID_0] = reinterpret_cast<ParticleAttribute*>( ptr + addr );
                addr += particleAttributeSize;
                particleAttrOrg[EFT_BUFFER_ID_1] = reinterpret_cast<ParticleAttribute*>( ptr + addr );
                addr += particleAttributeSize;
            }
            else
            {
                // 確保するメモリサイズを決定する
                allocedFromDynamicHeapSize = particleAttributeSize +
                    particlePosAttributeSize * bufferNum +
                    particleDataSize +
                    parentParticleDataSize;

                // 動的バッファからメモリを確保
                allocedFromDynamicHeap  = AllocFromDynamicHeap( allocedFromDynamicHeapSize );
                if ( !allocedFromDynamicHeap ) return false;

                // 確保したメモリを切り分ける
                ptr  = reinterpret_cast<u8*>( allocedFromDynamicHeap );
                addr = 0;

                // シングルバッファの場合、同じアドレスを指しておく
                particleAttrOrg[EFT_BUFFER_ID_0] = reinterpret_cast<ParticleAttribute*>( ptr + addr );
                particleAttrOrg[EFT_BUFFER_ID_1] = particleAttrOrg[EFT_BUFFER_ID_0];
                addr += particleAttributeSize;
            }

            // ParticleData
            particleData = reinterpret_cast<ParticleData*>( ptr + addr );
            addr += particleDataSize;
            MemUtil::FillZero( particleData, particleDataSize );

            // ParentParticleData
            if( emitterRes->childEmitterResCount )
            {
                parentParticleData = reinterpret_cast<ParentParticleData*>( ptr + addr );
                addr += parentParticleDataSize;
                MemUtil::FillZero( parentParticleData, parentParticleDataSize );
            }
            else
            {
                parentParticleData = NULL;
            }

            // ParticlePosAttr
            particlePosAttrOrg[EFT_BUFFER_ID_0] = reinterpret_cast<ParticlePosAttribute*>( ptr + addr );
            addr += particlePosAttributeSize;
            if ( isCpuEmitter )
            {
                particlePosAttrOrg[EFT_BUFFER_ID_1] = reinterpret_cast<ParticlePosAttribute*>( ptr + addr );
                addr += particlePosAttributeSize;
            }
            else
            {
                // シングルバッファの場合、同じアドレスを指しておく
                particlePosAttrOrg[EFT_BUFFER_ID_1] = particlePosAttrOrg[EFT_BUFFER_ID_0];
            }

            // ダブルバッファするアトリビュートを指すポインタを初期化
            particlePosAttr = particlePosAttrOrg[EFT_BUFFER_ID_0];
            particleAttr = particleAttrOrg[EFT_BUFFER_ID_0];

            if( isCpuEmitter )
            {
                // ダブルバッファリングする場合、反対側のバッファを指す。
                particlePosAttrBack = particlePosAttrOrg[EFT_BUFFER_ID_1];
            }
            else
            {
                // ダブルバッファリングしない場合は、フロントバッファと同じアドレスを指す。
                particlePosAttrBack = particlePosAttr;
            }

            if( IsParticleAttributeDoubleBuffer() )
            {
                // ダブルバッファリングする場合、反対側のバッファを指す。
                particleAttrBack = particleAttrOrg[EFT_BUFFER_ID_1];
            }
            else
            {
                // ダブルバッファリングしない場合は、フロントバッファと同じアドレスを指す。
                particleAttrBack = particleAttr;
            }
        }

        // ストリームアウトクラス
        if ( emitterData->emitter.calcType == EFT_EMITTER_CALC_TYPE_GPU_SO )
        {
            streamOutPos.Initialize( particleMax * sizeof( nw::math::VEC4 ) );
            streamOutVec.Initialize( particleMax * sizeof( nw::math::VEC4 ) );

            allocedFromDynamicHeapSize += particleMax * sizeof( nw::math::VEC4 ) * 2;
        }

        ptclAttrFillMax     = particleMax;
        ptclAttrFillIndex   = 0;
        ptclNum             = 0;
    }

    // マトリクスを更新する
    CreateResMatrix();

    // エミッタアニメ計算用バッファを初期化
    EFT_F32_VEC3_COPY( &m_EmitterAnimV[EFT_EMITTER_ANIM_SCALE],        &emitterRes->emitterData->emitter.scale );
    EFT_F32_VEC3_COPY( &m_EmitterAnimV[EFT_EMITTER_ANIM_ROTATE],       &emitterRes->emitterData->emitter.rotate );
    EFT_F32_VEC3_COPY( &m_EmitterAnimV[EFT_EMITTER_ANIM_TRANS],        &emitterRes->emitterData->emitter.trans );
    EFT_F32_VEC3_COPY( &m_EmitterAnimV[EFT_EMITTER_ANIM_COLOR0],       &emitterRes->emitterData->emitter.color0 );
    EFT_F32_VEC3_COPY( &m_EmitterAnimV[EFT_EMITTER_ANIM_COLOR1],       &emitterRes->emitterData->emitter.color1 );
    EFT_F32_VEC3_COPY( &m_EmitterAnimV[EFT_EMITTER_ANIM_PTCL_SCALE],   &emitterRes->emitterData->ptclScale.base );
    EFT_F32_VEC3_COPY( &m_EmitterAnimV[EFT_EMITTER_ANIM_VOLUME_SCALE], &emitterRes->emitterData->volume.volumeFormScale );
    m_EmitterAnimV[EFT_EMITTER_ANIM_EMISSION_RATE].x                   = emitterRes->emitterData->emission.rate;
    m_EmitterAnimV[EFT_EMITTER_ANIM_PTCL_LIFE].x                       = static_cast<f32>( emitterRes->emitterData->ptcl.life );
    m_EmitterAnimV[EFT_EMITTER_ANIM_ALPHA0].x                          = emitterRes->emitterData->emitter.color0.w;
    m_EmitterAnimV[EFT_EMITTER_ANIM_ALPHA1].x                          = emitterRes->emitterData->emitter.color1.w;
    m_EmitterAnimV[EFT_EMITTER_ANIM_ALL_DIR_VEL].x                     = emitterRes->emitterData->ptclVel.allDirection;
    m_EmitterAnimV[EFT_EMITTER_ANIM_DESIGNATED_DIR_SCALE].x            = emitterRes->emitterData->ptclVel.designatedDirScale;
    m_EmitterAnimV[EFT_EMITTER_ANIM_GRAVITY_SCALE].x                   = emitterRes->emitterData->emission.gravityScale;

    return true;
}

//---------------------------------------------------------------------------
//  エミッタ終了処理を行います。
//---------------------------------------------------------------------------
void Emitter::Finalize( bool invokeCallback )
{
    EFT_NULL_ASSERT( emitterCalc );
    emitterCalc = NULL;

    shader[EFT_SHADER_TYPE_NORMAL]      = NULL;
    shader[EFT_SHADER_TYPE_PATH_DEF1]   = NULL;
    shader[EFT_SHADER_TYPE_PATH_DEF2]   = NULL;

    if ( invokeCallback )
    {
        // パーティクル削除コールバック
        {
            InvokeParticleRemoveCallbackAll( this );
        }

        // エミッタ終了処理後コールバック
        {
            EmitterFinalizeArg arg;
            arg.emitter  = this;
            InvokeEmitterFinalizeCallback( arg );
        }
    }

    ReleaseBuffer();

    return;
}

//---------------------------------------------------------------------------
//  放出毎に更新が必要な情報を更新します。
//---------------------------------------------------------------------------
void Emitter::UpdateByEmit()
{
    // 放出間隔の更新
    emitInterval = static_cast<f32>( emitterData->emission.interval + 1.0f ) +
        random.GetS32( emitterData->emission.intervalRandom );
    emitInterval *= emitIntervalRatio;
    emitInterval *= emitterSet->GetEmissionIntervalScale();

    // エミッタマトリクス更新
    if ( !emitterData->emitter.isUpdateMatrixByEmit ) return;
    CreateResMatrix();
}

//---------------------------------------------------------------------------
//  マトリクス情報を更新します。
//---------------------------------------------------------------------------
void Emitter::CreateResMatrix()
{
    nw::math::VEC3 rotate;
    nw::math::VEC3 trans;
    nw::math::VEC3 scale;

    rotate.x = emitterData->emitter.rotate.x + ( random.GetF32() * 2.0f - 1.0f ) * emitterData->emitter.rotateRand.x;
    rotate.y = emitterData->emitter.rotate.y + ( random.GetF32() * 2.0f - 1.0f ) * emitterData->emitter.rotateRand.y;
    rotate.z = emitterData->emitter.rotate.z + ( random.GetF32() * 2.0f - 1.0f ) * emitterData->emitter.rotateRand.z;

    trans.x  = emitterData->emitter.trans.x + ( random.GetF32() * 2.0f - 1.0f ) * emitterData->emitter.transRand.x;
    trans.y  = emitterData->emitter.trans.y + ( random.GetF32() * 2.0f - 1.0f ) * emitterData->emitter.transRand.y;
    trans.z  = emitterData->emitter.trans.z + ( random.GetF32() * 2.0f - 1.0f ) * emitterData->emitter.transRand.z;

    scale.x = emitterData->emitter.scale.x;
    scale.y = emitterData->emitter.scale.y;
    scale.z = emitterData->emitter.scale.z;

    nw::math::MTX34MakeSRT( &resMatrixSRT, &scale, &rotate, &trans );
    scale.Set( 1.0f, 1.0f, 1.0f );
    nw::math::MTX34MakeSRT( &resMatrixRT, &scale, &rotate, &trans );
}

//---------------------------------------------------------------------------
//  内部バッファをスワップします。
//---------------------------------------------------------------------------
void Emitter::Swap( BufferMode bufferMode )
{
    if ( bufferMode == EFT_BUFFER_MODE_DOUBLE )
    {
        bufferID = 1 - bufferID;
    }
    else
    {
        if ( bufferID != EFT_BUFFER_ID_2 )
        {
            bufferID++;
        }
        else
        {
            bufferID = EFT_BUFFER_ID_0;
        }
    }

    // 現状のFrontとBackを差し替え
    particlePosAttr         = particlePosAttrOrg[ bufferID ];
    particlePosAttrBack     = particlePosAttrOrg[ 1-bufferID ];

    if ( IsParticleAttributeDoubleBuffer() )
    {
        particleAttr        = particleAttrOrg[ bufferID ];
        particleAttrBack    = particleAttrOrg[ 1-bufferID ];
    }
}

//---------------------------------------------------------------------------
//  エミッタがフレームバッファテクスチャを要求するか。
//---------------------------------------------------------------------------
bool Emitter::IsRequestFrameBufferTexture() const
{
    bool req = false;

    for ( u32 i = 0; i < EFT_SHADER_TYPE_MAX; i++ )
    {
        if ( shader[i] &&
            ( shader[i]->GetFrameBufferTextureFragmentSamplerLocation() != EFT_INVALID_TEXTURE_LOCATION ||
              shader[i]->GetFrameBufferTextureVertexSamplerLocation() != EFT_INVALID_TEXTURE_LOCATION ) )
        {
            req = true;
        }
    }
    return req;
}

//---------------------------------------------------------------------------
//  エミッタがデプスバッファテクスチャを要求するか。
//---------------------------------------------------------------------------
bool Emitter::IsRequestDepthTexture() const
{
    bool req = false;

    for ( u32 i = 0; i < EFT_SHADER_TYPE_MAX; i++ )
    {
        if ( shader[i] &&
             ( shader[i]->GetDepthBufferTextureFragmentSamplerLocation() != EFT_INVALID_TEXTURE_LOCATION ||
               shader[i]->GetDepthBufferTextureVertexSamplerLocation() != EFT_INVALID_TEXTURE_LOCATION ) )
        {
            req = true;
        }
    }

    return req;
}

//---------------------------------------------------------------------------
//  描画パスを設定します。
//---------------------------------------------------------------------------
void Emitter::SetDrawPath( u32 path )
{
    drawPath = path;

    // 描画パスコールバックも差し替える
    drawPathCallback = emitterSet->GetSystem()->GetDrawPathRenderStateSetCallback( static_cast<DrawPathFlag>( 0x01 << drawPath ) );
}

//---------------------------------------------------------------------------
//  マニュアルエミッタが消えても大丈夫な状態かを取得します。
//---------------------------------------------------------------------------
bool Emitter::IsManualEmitterReadyToExit() const
{
    if( !emitterSet->IsManualParticleEmission() )
    {
        // マニュアルエミッタセットでない場合は false
        return false;
    }
    if( emitterSet->IsResidentManualEmitterSet() )
    {
        // 常駐型マニュアルエミッタセットなら false
        return false;
    }
    if( !emitterSet->IsResidentManualEmitterSet() &&
        ( time < emitLastFrame + emitterData->ptcl.life + emitterSet->GetResidentEmitterTime() ) )
    {
        // 非常駐型のマニュアルエミッタセットで、まだ待機時間が経過していない場合も false
        return false;
    }
    if( GetCalcType() == EFT_EMITTER_CALC_TYPE_CPU && ptclProcessingNum > 0 )
    {
        // まだ粒が生きている場合も false
        return false;
    }

    // 以上の条件をすべて通過している場合、もう消しても大丈夫なので true
    return true;
}

//---------------------------------------------------------------------------
//  カラー0 RGBA値(乗算値)を設定します。
//---------------------------------------------------------------------------
void Emitter::SetColor0( const nw::math::VEC4 &color0 )
{
    this->color0.x = color0.x;
    this->color0.y = color0.y;
    this->color0.z = color0.z;
    this->color0.w = color0.w;
}

//---------------------------------------------------------------------------
//  カラー1 RGBA値(乗算値)を設定します。
//---------------------------------------------------------------------------
void Emitter::SetColor1( const nw::math::VEC4 &color1 )
{
    this->color1.x = color1.x;
    this->color1.y = color1.y;
    this->color1.z = color1.z;
    this->color1.w = color1.w;
}

nw::math::VEC3 CalcIntegratedGravityAndAirRegistanceVector( float t, const nw::math::VEC3& gravity, float airRegist, float gravityScale, const Emitter* pEmitter, const ParticleAttribute* pParticleAttr )
{
    nw::math::VEC3 g = gravity;

    if( airRegist == 1.0f )
    {
        // 重力のみ
        g *= ( gravityScale * ( t * t * 0.5f ) );
    }
    else
    {
        // 重力 x 空気抵抗
        float klog = std::log( airRegist );
        float kpow = std::pow( airRegist, t );
        float tempK = 1.0f / ( 1.0f - airRegist );
        float tempKPow = ( t - ( ( kpow - 1.0f ) / klog ) );
        float integration = tempK * tempKPow;

        g *= ( gravityScale * integration );
    }

    // ワールド座標系で適用
    if( pEmitter->emitterData->emission.isWorldGravity )
    {
        if ( pEmitter->emitterData->emitter.followType == EFT_EMITTER_FOLLOW_TYPE_NONE )
        {
            g.x = ( g.x * pParticleAttr->emitterRTMat0.x + g.y * pParticleAttr->emitterRTMat1.x + g.z * pParticleAttr->emitterRTMat2.x );
            g.y = ( g.x * pParticleAttr->emitterRTMat0.y + g.y * pParticleAttr->emitterRTMat1.y + g.z * pParticleAttr->emitterRTMat2.y );
            g.z = ( g.x * pParticleAttr->emitterRTMat0.z + g.y * pParticleAttr->emitterRTMat1.z + g.z * pParticleAttr->emitterRTMat2.z );
        }
        else
        {
            g.x = ( g.x * pEmitter->matrixRT.m[0][0] + g.y * pEmitter->matrixRT.m[1][0] + g.z * pEmitter->matrixRT.m[2][0] );
            g.y = ( g.x * pEmitter->matrixRT.m[0][1] + g.y * pEmitter->matrixRT.m[1][1] + g.z * pEmitter->matrixRT.m[2][1] );
            g.z = ( g.x * pEmitter->matrixRT.m[0][2] + g.y * pEmitter->matrixRT.m[1][2] + g.z * pEmitter->matrixRT.m[2][2] );
        }
    }

    return g;
}

//---------------------------------------------------------------------------
//  パーティクルの位置と速度を指定した時間分強制的に進めます（誤差あり）
//---------------------------------------------------------------------------
void Emitter::ForceUpdateParticlePosAttribute( nw::math::VEC4* pOutPos, nw::math::VEC4* pOutVec, float deltaTime, const ParticleAttribute* pParticleAttr, const ParticleData* pParticleData )
{
    //---------------------------------------------------------------------------
    // nw::eft2::_CalcParticleBehavior_Integrating() の挙動をできる範囲で近似する。
    //---------------------------------------------------------------------------
    NW_UNUSED_VARIABLE( pParticleData );

    // 運動量ランダム x フレームレート
    const f32 dynamicsRand  = pParticleAttr->scale.w;
    const f32 gravScaleAnim = GetEmitterAnimValue().gravityScale;
    const f32 ubGravityX    = emitterData->emission.gravityDir.x;
    const f32 ubGravityY    = emitterData->emission.gravityDir.y;
    const f32 ubGravityZ    = emitterData->emission.gravityDir.z;
    const nw::math::VEC3 gravity( ubGravityX, ubGravityY, ubGravityZ );
    f32 airRegist           = emitterRes->resEmitterStaticUniformBlock->airRegist;
#if EFT_GX2
    EndianUtil::ForceFlip( &airRegist );    // Cafe ではフリップしないと使えない
#endif

    // deltaTime 間の合計の速度ベクトルを積分で計算
    const nw::math::VEC3 vNow = CalcIntegratedGravityAndAirRegistanceVector( deltaTime,        gravity, airRegist, gravScaleAnim, this, pParticleAttr );
    const nw::math::VEC3 vPre = CalcIntegratedGravityAndAirRegistanceVector( deltaTime - 1.0f, gravity, airRegist, gravScaleAnim, this, pParticleAttr );

    // 空気抵抗値を加味した時間位置
    float t = ( airRegist == 1.0f ) ? deltaTime : ( 1.0f - std::pow( airRegist, deltaTime ) )/( 1.0f - airRegist );

    // 運動量ランダムを含める。
    const nw::math::VEC3 initVec( pOutVec->x, pOutVec->y, pOutVec->z );
    const nw::math::VEC3 initPos( pOutPos->x, pOutPos->y, pOutPos->z );

    // 今の速度ベクトルは、今の積分値と、1F 前の積分値と等価。
    const nw::math::VEC3 retVec = vNow - vPre;
    EFT_F32_VEC3_COPY( pOutVec, &retVec );

    const nw::math::VEC3 integratedVec = ( t * initVec + vNow );
    const nw::math::VEC3 retPos = initPos + integratedVec * dynamicsRand;
    EFT_F32_VEC3_COPY( pOutPos, &retPos );
}

//---------------------------------------------------------------------------
//  StreamOut バッファを強制的に更新します。
//---------------------------------------------------------------------------
void Emitter::ForceUpdateForStreamOutBuffer()
{
    // MEMO: 現状、GX2 のみ対応。
    // MEMO: ForceCalcされるのはEmitterSet生成直後の想定
    //       途中からのForceCalcは想定しない
    // MEMO: OpenGLはひとまず対象外

    // 初期位置がアトリビュートに入っているので、それをSOバッファにコピー
#if EFT_GX2
    {
        // ParticleMax x VEC4 なので、
        // Pos/Vec アトリビュートの内容を両方のバッファにコピーする
        // MEMO: 今はナイーブな実装。
        nw::math::VEC4* pPosBuffer0 = reinterpret_cast< nw::math::VEC4* >( streamOutPos.streamOutBuf[ StreamOutAttributeBuffer::EFT_STREAM_OUT_ATTRB_0 ].dataPtr );
        nw::math::VEC4* pPosBuffer1 = reinterpret_cast< nw::math::VEC4* >( streamOutPos.streamOutBuf[ StreamOutAttributeBuffer::EFT_STREAM_OUT_ATTRB_1 ].dataPtr );
        nw::math::VEC4* pVecBuffer0 = reinterpret_cast< nw::math::VEC4* >( streamOutVec.streamOutBuf[ StreamOutAttributeBuffer::EFT_STREAM_OUT_ATTRB_0 ].dataPtr );
        nw::math::VEC4* pVecBuffer1 = reinterpret_cast< nw::math::VEC4* >( streamOutVec.streamOutBuf[ StreamOutAttributeBuffer::EFT_STREAM_OUT_ATTRB_1 ].dataPtr );
        if( !pPosBuffer0 || !pPosBuffer1 || !pVecBuffer0 || !pVecBuffer1 )
        {
            // 何らかの原因でポインタが取れてない場合は何もしないで処理を返す（不正アクセス防止）
            return;
        }

        const int size = streamOutPos.streamOutBuf[StreamOutAttributeBuffer::EFT_STREAM_OUT_ATTRB_0].size;
        const float emitterTime = GetTime();

        for( int i = 0; i < ptclAttrFillMax; ++i )
        {
            const float deltaTime = emitterTime - particlePosAttr[ i ].localVec.w; // パーティクル経過時間

            // 位置／速度を経過時間分進める（CPU処理）
            ForceUpdateParticlePosAttribute(
                &particlePosAttr[ i ].localPos,
                &particlePosAttr[ i ].localVec,
                deltaTime,
                &particleAttr[ i ],
                &particleData[ i ] );

            // SOバッファに書き込み
            pPosBuffer0[ i ] = particlePosAttr[ i ].localPos;
            pVecBuffer0[ i ] = particlePosAttr[ i ].localVec;

            // 粒の時刻を更新
            pVecBuffer0[ i ].w = deltaTime;
        }

        // CPU -> GPU への書き込みなので、エンディアンを強制的に裏返す
        EndianUtil::ForceFlipArray( ptclAttrFillMax, pPosBuffer0 );
        EndianUtil::ForceFlipArray( ptclAttrFillMax, pVecBuffer0 );

        // 裏バッファに丸ごとコピー
        MemUtil::Copy( pPosBuffer1, pPosBuffer0, size );
        MemUtil::Copy( pVecBuffer1, pVecBuffer0, size );

        // キャッシュを落とす
        const GX2InvalidateType type = static_cast< GX2InvalidateType >( GX2_INVALIDATE_CPU );
        GX2Invalidate( type, pPosBuffer0, size );
        GX2Invalidate( type, pVecBuffer0, size );
        GX2Invalidate( type, pPosBuffer1, size );
        GX2Invalidate( type, pVecBuffer1, size );
    }

    // グローバルカウンタを更新（ EmitterCalc::CalcStreamOut() で行っている分の帳尻合わせ ）
    soCounter = GetEmitterSet()->GetSystem()->GetGlobalCounter();
#endif
}

} // namespace eft2
} // namespace nw

