﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/eft/eft2_Callback.h>
#include <nw/eft/eft2_Shader.h>
#include <nw/eft/eft2_SuperStripe.h>

namespace nw   {
namespace eft2 {

//---------------------------------------------------
// 引数ビューマトリクスからエミッタビルボード用マトリクスを生成し設定します。
//---------------------------------------------------
void EmitterMatrixSetArg::SetEmitterBillboardMatrix( nw::math::Matrix34& viewMatrix )
{
    nw::math::MTX34 invView( viewMatrix );
    invView.Inverse();

    const nw::math::MTX34& esetRT  = emitter->emitterSet->GetRTMatrix();
    const nw::math::MTX34& esetSRT = emitter->emitterSet->GetSRTMatrix();

    nw::math::VEC3 pos = esetRT.GetColumn( 3 );
    invView.SetColumn( 3, pos );

    nw::math::VEC3 scale( 0, 0, 0 );
    if ( esetRT.m[0][0] != 0.f )
    {
        scale.x = esetSRT.m[0][0] / esetRT.m[0][0];
    }
    if ( esetRT.m[1][1] != 0.f )
    {
        scale.y = esetSRT.m[1][1] / esetRT.m[1][1];
    }
    if ( esetRT.m[2][2] != 0.f )
    {
        scale.z = esetSRT.m[2][2] / esetRT.m[2][2];
    }
    nw::math::MTX34 scaleMtx;
    scaleMtx.SetSMtx( scale );

    nw::math::MTX34 invEsetRT = esetRT;
    invEsetRT.Inverse();
    nw::math::MTX34 invEsetSRT = esetSRT;
    invEsetSRT.Inverse();

    {
        emitter->matrixRT.SetMult( invEsetRT, emitter->matrixRT );
        emitter->matrixRT.SetMult( invView, emitter->matrixRT );
    }
    {
        emitter->matrixSRT.SetMult( invEsetSRT, emitter->matrixSRT );
        emitter->matrixSRT.SetMult( scaleMtx, emitter->matrixSRT );
        emitter->matrixSRT.SetMult( invView, emitter->matrixSRT );
    }
}

//------------------------------------------------------------------------------
// シェーダ取得
//------------------------------------------------------------------------------
Shader* RenderStateSetArg::GetShader()
{
    return emitter->shader[shaderType];
}

//------------------------------------------------------------------------------
// システム取得
//------------------------------------------------------------------------------
System* RenderStateSetArg::GetSystem()
{
    return emitter->emitterSet->GetSystem();
}

System* EmitterDrawArg::GetSystem()
{
    return emitter->emitterSet->GetSystem();
}

//------------------------------------------------------------------------------
// UBO反転 コールバック
//------------------------------------------------------------------------------
bool _EndianFlipCallback( nw::eft2::EndianFlipArg& arg )
{
#if EFT_IS_WIN
    if ( arg.customShaderParam && arg.customShaderParamSize >= 4 )
    {
        EFT_ASSERT( arg.customShaderParamSize%4 == 0 );
        u32  loop  = arg.customShaderParamSize/4;
        f32* param = reinterpret_cast<f32*>( arg.customShaderParam );

        for ( u32 i = 0; i < loop; i++ )
        {
            nw::eft2::EndianUtil::Flip( reinterpret_cast<f32*>( &param[i] ) );
        }
    }
#endif
    return true;
}

//------------------------------------------------------------------------------
// 描画設定後コールバック:カスタムシェーダパラメータを自動的に設定する。
//------------------------------------------------------------------------------
bool _BindReservedCustomShaderUniformBlock( nw::eft2::RenderStateSetArg& arg )
{
#ifdef EFT_USE_UNIFORM_BLOCK
    Emitter*    emitter = arg.emitter;
    Shader*     shader  = arg.GetShader();
    shader->BindReservedCustomShaderUniformBlock( emitter );
#else
    EFT_UNUSED_VARIABLE( arg );
#endif
    return true;
}

bool BindReservedCustomShaderConstantBuffer( nw::eft2::RenderStateSetArg& arg )
{
    return _BindReservedCustomShaderUniformBlock( arg );
}

//------------------------------------------------------------------------------
// 描画設定後コールバック:エミッタプラグインパラメータを自動的に設定する。
//------------------------------------------------------------------------------
bool _BindEmitterPluginUniformBlockAuto( nw::eft2::RenderStateSetArg& arg )
{
#ifdef EFT_USE_UNIFORM_BLOCK
    Emitter*    emitter = arg.emitter;
    Shader*     shader  = arg.GetShader();
    shader->BindEmitterPluginUniformBlockAuto( emitter );
#else
    EFT_UNUSED_VARIABLE( arg );
#endif
    return true;
}

//------------------------------------------------------------------------------
// 描画設定後のコールバック呼び出し
//------------------------------------------------------------------------------
bool InvokeRenderStateSetCallback( nw::eft2::RenderStateSetArg& arg )
{
    if( arg.emitter->callbackSet[EFT_CALLBACK_SET_TYPE_CA] && arg.emitter->callbackSet[EFT_CALLBACK_SET_TYPE_CA]->renderStateSet )
    {
        bool ret = arg.emitter->callbackSet[EFT_CALLBACK_SET_TYPE_CA]->renderStateSet( arg );
        if( !ret ){ return false; }
    }

    if( arg.emitter->callbackSet[EFT_CALLBACK_SET_TYPE_CS] && arg.emitter->callbackSet[EFT_CALLBACK_SET_TYPE_CS]->renderStateSet )
    {
        bool ret = arg.emitter->callbackSet[EFT_CALLBACK_SET_TYPE_CS]->renderStateSet( arg );
        if( !ret ){ return false; }
    }
    else
    {
        if ( arg.emitter->emitterRes->customShaderParam ) _BindReservedCustomShaderUniformBlock( arg );
    }

    if( arg.emitter->callbackSet[EFT_CALLBACK_SET_TYPE_EP] && !arg.emitter->callbackSet[EFT_CALLBACK_SET_TYPE_EP]->renderStateSet )
    {
        _BindEmitterPluginUniformBlockAuto( arg );
    }
    return true;
}

//------------------------------------------------------------------------------
// エミッタ初期化後のコールバック呼び出し
//------------------------------------------------------------------------------
bool InvokeEmitterInitializeCallback( nw::eft2::EmitterInitializeArg& arg )
{
    const CallbackSet* callbackEmitterPlugin = arg.emitter->callbackSet[ EFT_CALLBACK_SET_TYPE_EP ];
    const CallbackSet* callbackCustomAction = arg.emitter->callbackSet[ EFT_CALLBACK_SET_TYPE_CA ];
    const CallbackSet* callbackCustomShader = arg.emitter->callbackSet[ EFT_CALLBACK_SET_TYPE_CS ];

    bool epResult = true;
    bool caResult = true;
    bool csResult = true;

    if( callbackEmitterPlugin && callbackEmitterPlugin->emitterInitialize )
    {
        epResult = callbackEmitterPlugin->emitterInitialize( arg );
    }
    if( callbackCustomAction && callbackCustomAction->emitterInitialize )
    {
        caResult = callbackCustomAction->emitterInitialize( arg );
    }
    if( callbackCustomShader && callbackCustomShader->emitterInitialize )
    {
        csResult = callbackCustomShader->emitterInitialize( arg );
    }

    // 一つでも失敗したコールバックがあれば全体として false を返す。
    // その後にエミッタ破棄コールバックが呼ばれるので後処理は問題なし。
    return ( epResult && caResult && csResult );
}

//------------------------------------------------------------------------------
// 全パーティクル削除時のコールバック呼び出し
//------------------------------------------------------------------------------
void InvokeParticleRemoveCallbackAll( Emitter* emitter )
{
    ParticleRemoveCallback ptclRmvCbEP = NULL;
    ParticleRemoveCallback ptclRmvCbCA = NULL;
    ParticleRemoveCallback ptclRmvCbCS = NULL;

    if( emitter->callbackSet[EFT_CALLBACK_SET_TYPE_EP] && emitter->callbackSet[EFT_CALLBACK_SET_TYPE_EP]->particleRemove )
    {
        ptclRmvCbEP = emitter->callbackSet[EFT_CALLBACK_SET_TYPE_EP]->particleRemove;
    }
    if( emitter->callbackSet[EFT_CALLBACK_SET_TYPE_CA] && emitter->callbackSet[EFT_CALLBACK_SET_TYPE_CA]->particleRemove )
    {
        ptclRmvCbCA = emitter->callbackSet[EFT_CALLBACK_SET_TYPE_CA]->particleRemove;
    }
    if( emitter->callbackSet[EFT_CALLBACK_SET_TYPE_CS] && emitter->callbackSet[EFT_CALLBACK_SET_TYPE_CS]->particleRemove )
    {
        ptclRmvCbCS = emitter->callbackSet[EFT_CALLBACK_SET_TYPE_CS]->particleRemove;
    }

    if( ptclRmvCbEP || ptclRmvCbCA || ptclRmvCbCS )
    {
        for ( u32 i = 0; i < emitter->ptclNum; i++ )
        {
            const f32 time = emitter->particlePosAttr[i].GetTime( emitter->time );
            const f32 life = emitter->particlePosAttr[i].GetLife();
            // MEMO: CalcParticle 同様、粒の生存判定をした方が今後良さそう
            InvokeParticleRemoveCallback( emitter, i, time, life, ptclRmvCbEP, ptclRmvCbCA, ptclRmvCbCS );
        }
    }
}

//------------------------------------------------------------------------------
// エミッタ終了時のコールバック呼び出し
//------------------------------------------------------------------------------
void InvokeEmitterFinalizeCallback( nw::eft2::EmitterFinalizeArg& arg )
{
    if( arg.emitter->callbackSet[EFT_CALLBACK_SET_TYPE_EP] && arg.emitter->callbackSet[EFT_CALLBACK_SET_TYPE_EP]->emitterFinalize )
    {
        arg.emitter->callbackSet[EFT_CALLBACK_SET_TYPE_EP]->emitterFinalize( arg );
    }
    if( arg.emitter->callbackSet[EFT_CALLBACK_SET_TYPE_CA] && arg.emitter->callbackSet[EFT_CALLBACK_SET_TYPE_CA]->emitterFinalize )
    {
        arg.emitter->callbackSet[EFT_CALLBACK_SET_TYPE_CA]->emitterFinalize( arg );
    }
    if( arg.emitter->callbackSet[EFT_CALLBACK_SET_TYPE_CS] && arg.emitter->callbackSet[EFT_CALLBACK_SET_TYPE_CS]->emitterFinalize )
    {
        arg.emitter->callbackSet[EFT_CALLBACK_SET_TYPE_CS]->emitterFinalize( arg );
    }
}

//---------------------------------------------------------------------------
//  履歴式ストライプ2 での履歴点の配列への参照を取得します。
//  この API は予告なく仕様が変更される可能性があります。
//---------------------------------------------------------------------------
void ParticleCalcArgImpl::GetSuperStripeHistoryBuffer( SuperStripeHistory** pOutHistoryHead, s32* pOutHistoryNum )
{
    NW_NULL_ASSERT( pOutHistoryHead );
    NW_NULL_ASSERT( pOutHistoryNum );

    if( !( emitter->GetCalcType() == EFT_EMITTER_CALC_TYPE_CPU &&
           emitter->emitterRes->emitterPluginIndex == SuperStripeSystem::PluginId ) )
    {
        // 履歴式ストライプ2 を持っていない場合は無効値を返す。
        *pOutHistoryHead = NULL;
        *pOutHistoryNum = 0;
        return;
    }

    SuperStripe* stripe = reinterpret_cast< SuperStripe* >( emPluginData );
    if( stripe == NULL )
    {
        // ストライプのデータが無い場合も無効値を返す。
        // ここに来る場合はエミッタが不正な状態。
        *pOutHistoryHead = NULL;
        *pOutHistoryNum = 0;
        return;
    }

    *pOutHistoryHead = &stripe->hist[ 0 ];
    *pOutHistoryNum = stripe->numHistory;
}

} // namespace eft2
} // namespace nw

