﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_ScreenShotUtility.h"

#include <nn/result/result_HandlingUtility.h>
#include "../capsrvServer_Config.h"
#include "../capsrvServer_ResultPrivate.h"

namespace nn{ namespace capsrv{ namespace server{ namespace screenshot{

    namespace {
        static const int QualityCandidateList[] = { 90, 80, 70, 60, 50, 40, 30, 20, 10, 0 };
        static const int QualityCandidateCount = sizeof(QualityCandidateList) / sizeof(QualityCandidateList[0]);

        template<typename OutputInfo, typename InputInfo, typename EncodeFunction>
        nn::Result EncodeWithSizeLimitation(
            OutputInfo& outputInfo,
            const InputInfo& inputInfo,
            nn::image::ExifBuilder* pExifBuilder,
            void* pWorkBuffer,
            size_t workBufferSize,
            EncodeFunction encodeFunction
        ) NN_NOEXCEPT
        {
            int q;
            for(q = 0; q < QualityCandidateCount; q++)
            {
                int quality = QualityCandidateList[q];
                NN_CAPSRV_LOG_DEV("Trying encoding #%d/%d (quality %d)\n", q + 1, QualityCandidateCount, quality);
                NN_RESULT_TRY(encodeFunction(outputInfo, inputInfo, quality, pExifBuilder, pWorkBuffer, workBufferSize))
                    NN_RESULT_CATCH(ResultInternalJpegOutBufferShortage)
                    {
                        NN_CAPSRV_LOG_WARN("Jpeg encoding out buffer shortage(quality=%d)\n", quality);
                        continue;
                    }
                    NN_RESULT_CATCH(ResultInternalJpegWorkMemoryShortage)
                    {
                        NN_CAPSRV_LOG_ERR("Jpeg encoding work buffer shortage(quality=%d)\n", quality);
                        NN_RESULT_THROW(ResultScreenShotEncodingFailed());
                    }
                    NN_RESULT_CATCH_ALL
                    {
                        NN_RESULT_THROW(ResultScreenShotEncodingFailed());
                    }
                NN_RESULT_END_TRY;
                break; // if success
            }
            NN_RESULT_THROW_UNLESS(q < QualityCandidateCount, ResultScreenShotEncodingFailed());
            NN_CAPSRV_LOG_DEV("Encode success: quality %d, size %d bytes\n", QualityCandidateList[q], static_cast<int>(outputInfo.GetEncodedSize()));
            NN_RESULT_SUCCESS;
        }
    }

    nn::Result ScreenShotUtility::EncodeRgba8BufferToBuffer(
        jpeg::SoftwareJpegEncoderBufferOutputInfo& outputInfo,
        const jpeg::SoftwareJpegEncoderBufferInputInfoRgba8& inputInfo,
        nn::image::ExifBuilder* pExifBuilder,
        void* pWorkBuffer,
        size_t workBufferSize
    ) NN_NOEXCEPT
    {
        return EncodeWithSizeLimitation(outputInfo, inputInfo, pExifBuilder, pWorkBuffer, workBufferSize, jpeg::SoftwareJpegEncoderRgba8::EncodeRgba8BufferToBuffer);
    }

    nn::Result ScreenShotUtility::EncodeYuvStreamToBuffer(
        jpeg::SoftwareJpegEncoderBufferOutputInfo& outputInfo,
        const jpeg::SoftwareJpegEncoderStreamInputInfoYuv& inputInfo,
        nn::image::ExifBuilder* pExifBuilder,
        void* pWorkBuffer,
        size_t workBufferSize
    ) NN_NOEXCEPT
    {
        return EncodeWithSizeLimitation(outputInfo, inputInfo, pExifBuilder, pWorkBuffer, workBufferSize, jpeg::SoftwareJpegEncoderYuv::EncodeYuvStreamToBuffer);
    }

}}}}
