﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_Optional.h>
#include <nn/capsrv/capsrv_Result.h>
#include "../../capsrvServer_ResultPrivate.h"
#include "../visrv_ScreenShotContext.h"
#include "visrv_ActionTimeMeasure.h"

#include <nn/ovln/format/ovln_CapsrvMessage.h>

namespace nn{ namespace capsrv{ namespace server{ namespace screenshot{ namespace action{

    // OverlayDisp に画面写真の撮影結果を通知します
    static nn::Result NotifyScreenShotReultToOverlayDisp(ScreenShotContext& context) NN_NOEXCEPT
    {
        NN_CAPSRV_SCREENSHOT_CONTEXT_SCOPE(context);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_IN(context, FileSize);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_IN(context, Result);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_IN(context, TakingScreenShotReason);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_IN(context, OverlayNotifier);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_IN(context, OverlayNotificationRequest);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_IN_OPTIONAL(context, AlbumFileId);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_IN_OPTIONAL(context, OverlayAlbumFileId);

        NN_CAPSRV_SCREENSHOT_ACTION_TIMEMEASURE("notify res");

        // 通知条件の確認
        NN_CAPSRV_SCREENSHOT_CONTEXT_GET(request, context, OverlayNotificationRequest);
        NN_CAPSRV_SCREENSHOT_CONTEXT_GET(result, context, Result);
        if(result.IsSuccess())
        {
            if((request & OverlayNotificationRequest_OnSuccess) == 0)
            {
                NN_RESULT_SUCCESS;
            }
        }
        else
        {
            if((request & OverlayNotificationRequest_OnFailure) == 0)
            {
                NN_RESULT_SUCCESS;
            }
        }

        nn::ovln::format::ScreenShotCapturedData data = {};

        // FileEntry を設定
        NN_CAPSRV_SCREENSHOT_CONTEXT_GET_OPTIONAL(albumFileId, context, AlbumFileId);
        NN_CAPSRV_SCREENSHOT_CONTEXT_GET(fileSize, context, FileSize);
        {
            AlbumEntry entry = {};
            if(albumFileId)
            {
                entry.fileId = *albumFileId;

            }
            entry.size   = fileSize;
            std::memcpy(&data._entry, &entry, sizeof(AlbumEntry));
        }

        // 撮影理由の設定
        NN_CAPSRV_SCREENSHOT_CONTEXT_GET(reason, context, TakingScreenShotReason);
        data.reason = reason;

        // サムネイル存否を設定
        NN_CAPSRV_SCREENSHOT_CONTEXT_GET_OPTIONAL(overlayFileId, context, OverlayAlbumFileId);
        data.flags.Set(nn::ovln::format::ScreenShotResultFlag::IsThumbnailAvailable::Index, !!overlayFileId);

        // 撮影結果の設定
        //   特定のエラーでは↑を上書きしてサムネイルがないことにする
        if(result.IsSuccess())
        {
            NN_CAPSRV_LOG_DEV("Taking ScreenShot Success\n");
            data.result = nn::ovln::format::ScreenShotResult::Success;
        }
        else
        {
            NN_RESULT_TRY(result)
                NN_RESULT_CATCH(ResultScreenShotNoTarget)
                {
                    // 撮影対象なしはエラー扱いにする
                    NN_CAPSRV_LOG_ERR("Taking ScreenShot Failed: No target\n");
                    data.result = nn::ovln::format::ScreenShotResult::ScreenCaptureError;
                    data.flags.Set(nn::ovln::format::ScreenShotResultFlag::IsThumbnailAvailable::Index, false); // 上書き
                }
                NN_RESULT_CATCH(ResultScreenShotError)
                {
                    NN_CAPSRV_LOG_ERR("Taking ScreenShot Failed: Screen capture error(0x%08X)\n", result.GetInnerValueForDebug());
                    data.result = nn::ovln::format::ScreenShotResult::ScreenCaptureError;
                    data.flags.Set(nn::ovln::format::ScreenShotResultFlag::IsThumbnailAvailable::Index, false); // 上書き
                }
                NN_RESULT_CATCH(ResultAlbumIsFull)
                {
                    NN_CAPSRV_LOG_ERR("Taking ScreenShot Failed: Album is full\n");
                    data.result = nn::ovln::format::ScreenShotResult::AlbumIsFull;
                }
                NN_RESULT_CATCH(ResultAlbumFileCountLimit)
                {
                    NN_CAPSRV_LOG_ERR("Taking ScreenShot Failed: Album file count limit\n");
                    data.result = nn::ovln::format::ScreenShotResult::AlbumFileCountLimit;
                }
                NN_RESULT_CATCH(ResultAlbumFileSystemError)
                {
                    NN_CAPSRV_LOG_ERR("Taking ScreenShot Failed: Album file system error(0x%08X)\n", result.GetInnerValueForDebug());
                    data.result = nn::ovln::format::ScreenShotResult::AlbumFileSystemError;
                }
                NN_RESULT_CATCH(ResultInternalScreenShotCanceledAsNoTarget)
                {
                    NN_CAPSRV_LOG_INFO("Taking ScreenShot Canceled: No target\n");
                    data.result = nn::ovln::format::ScreenShotResult::ScreenCaptureError; // NoTarget と同じ。
                    data.flags.Set(nn::ovln::format::ScreenShotResultFlag::IsThumbnailAvailable::Index, false); // 上書き
                }
                NN_RESULT_CATCH(ResultInternalScreenShotCanceledAsProhibited)
                {
                    NN_CAPSRV_LOG_INFO("Taking ScreenShot Canceled: Prohibited\n");
                    data.result = nn::ovln::format::ScreenShotResult::NotPermitted;
                    data.flags.Set(nn::ovln::format::ScreenShotResultFlag::IsThumbnailAvailable::Index, false); // 上書き
                }
                NN_RESULT_CATCH_ALL
                {
                    NN_CAPSRV_LOG_ERR("Taking ScreenShot Failure(0x%08X)\n", result.GetInnerValueForDebug());
                    data.result = nn::ovln::format::ScreenShotResult::Failure;
                }
            NN_RESULT_END_TRY;
        }

        NN_CAPSRV_SCREENSHOT_CONTEXT_GET(pOverlayNotifier, context, OverlayNotifier);
        NN_RESULT_DO(pOverlayNotifier->SendToOverlay(data));

        NN_RESULT_SUCCESS;
    }

}}}}}
