﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn{ namespace capsrv{ namespace server{ namespace detail{

    typedef uint32_t MakerNoteVersionType;
    typedef uint16_t MakerNoteEntryTagType;

    enum MakerNoteValueClass
    {
        MakerNoteValueClass_Bytes,
        MakerNoteValueClass_Integer,
    };

    class MakerNoteValuePropertyUtility
    {
    public:
        typedef MakerNoteValueClass ValueClass;

        //-------------------------

        template<MakerNoteEntryTagType TTag>
        struct TagDefinition
        {
            static const MakerNoteEntryTagType EntryTag = TTag;
        };

        //-------------------------

        template<typename TInfoType, typename TValueType, TValueType TInfoType::*TMemberVariablePointer, ValueClass TValueClass>
        struct ValueDefinition
        {
            typedef TValueType ValueType;
            typedef TInfoType InfoType;

            static const ValueClass ValueClass = TValueClass;

            static ValueType* GetPointer(InfoType& info) NN_NOEXCEPT
            {
                return &(info.*TMemberVariablePointer);
            }

            static const ValueType* GetPointer(const InfoType& info) NN_NOEXCEPT
            {
                return &(info.*TMemberVariablePointer);
            }
        };

        //-------------------------

        template<typename TValueType>
        struct DefaultValueUnexpected
        {
            static const MakerNoteVersionType FromVersion = 0;

            static void SetDefault(TValueType* pOutValue, MakerNoteVersionType srcVersion) NN_NOEXCEPT
            {
                NN_SDK_ASSERT(0, "unexpected version: %d", srcVersion);
                NN_UNUSED(srcVersion);
                std::memset(pOutValue, 0, sizeof(TValueType));
            }

            static bool IsDefaultValue(const TValueType& value, MakerNoteVersionType srcVersion) NN_NOEXCEPT
            {
                NN_SDK_ASSERT(0, "unexpected version: %d", srcVersion);
                NN_UNUSED(srcVersion);
                const char* p = reinterpret_cast<const char*>(&value);
                for(size_t i = 0; i < sizeof(TValueType); i++)
                {
                    if(p[i] != 0)
                    {
                        return false;
                    }
                }
                return true;
            }

        };

        template<MakerNoteVersionType TFromVersion, typename TValueType, typename LowerVersion>
        struct DefaultValueZeroClear
        {
            static const MakerNoteVersionType FromVersion = TFromVersion;

            static void SetDefault(TValueType* pOutValue, MakerNoteVersionType srcVersion) NN_NOEXCEPT
            {
                NN_STATIC_ASSERT(TFromVersion >= LowerVersion::FromVersion);
                if(srcVersion >= TFromVersion)
                {
                    std::memset(pOutValue, 0, sizeof(TValueType));
                }
                else
                {
                    LowerVersion::SetDefault(pOutValue, srcVersion);
                }
            }

            static bool IsDefaultValue(const TValueType& value, MakerNoteVersionType srcVersion) NN_NOEXCEPT
            {
                NN_STATIC_ASSERT(TFromVersion >= LowerVersion::FromVersion);
                if(srcVersion >= TFromVersion)
                {
                    const char* p = reinterpret_cast<const char*>(&value);
                    for(size_t i = 0; i < sizeof(TValueType); i++)
                    {
                        if(p[i] != 0)
                        {
                            return false;
                        }
                    }
                    return true;
                }
                else
                {
                    return LowerVersion::IsDefaultValue(value, srcVersion);
                }
            }
        };

        template<MakerNoteVersionType TFromVersion, typename TValueType>
        struct DefaultValueZeroClear<TFromVersion, TValueType, void>
            : DefaultValueZeroClear<TFromVersion, TValueType, DefaultValueUnexpected<TValueType>>
        {
        };

        template<MakerNoteVersionType TFromVersion, typename TValueType, TValueType TDefaultValue, typename LowerVersion>
        struct DefaultValueInteger
        {
            static const MakerNoteVersionType FromVersion = TFromVersion;

            static void SetDefault(TValueType* pOutValue, MakerNoteVersionType srcVersion) NN_NOEXCEPT
            {
                NN_STATIC_ASSERT(TFromVersion >= LowerVersion::FromVersion);
                if(srcVersion >= TFromVersion)
                {
                    *pOutValue = TDefaultValue;
                }
                else
                {
                    LowerVersion::SetDefault(pOutValue, srcVersion);
                }
            }

            static bool IsDefaultValue(const TValueType& value, MakerNoteVersionType srcVersion) NN_NOEXCEPT
            {
                NN_STATIC_ASSERT(TFromVersion >= LowerVersion::FromVersion);
                if(srcVersion >= TFromVersion)
                {
                    return value == TDefaultValue;
                }
                else
                {
                    return LowerVersion::IsDefaultValue(value, srcVersion);
                }
            }
        };

        template<MakerNoteVersionType TFromVersion, typename TValueType, TValueType TDefaultValue>
        struct DefaultValueInteger<TFromVersion, TValueType, TDefaultValue, void>
            : DefaultValueInteger<TFromVersion, TValueType, TDefaultValue, DefaultValueUnexpected<TValueType>>
        {
        };


        //-------------------------

        template<MakerNoteVersionType TFromVersion, int TCountMin, int TCountMax, typename LowerVersion>
        struct EntryCountMinMaxFrom
        {
            static const int FromVersion = FromVersion;

            static int GetEntryCountMin(MakerNoteVersionType version) NN_NOEXCEPT
            {
                NN_STATIC_ASSERT(TFromVersion > LowerVersion::FromVersion);
                NN_STATIC_ASSERT(TCountMin <= TCountMax);
                if(version >= TFromVersion)
                {
                    return TCountMin;
                }
                else
                {
                    return LowerVersion::GetEntryCountMin(version);
                }
            }

            static int GetEntryCountMax(MakerNoteVersionType version) NN_NOEXCEPT
            {
                NN_STATIC_ASSERT(TFromVersion > LowerVersion::FromVersion);
                NN_STATIC_ASSERT(TCountMin <= TCountMax);
                if(version >= TFromVersion)
                {
                    return TCountMax;
                }
                else
                {
                    return LowerVersion::GetEntryCountMax(version);
                }
            }
        };

        template<MakerNoteVersionType TFromVersion, int TCountMin, int TCountMax>
        struct EntryCountMinMaxFrom<TFromVersion, TCountMin, TCountMax, void>
        {
            static const int FromVersion = TFromVersion;

            static int GetEntryCountMin(MakerNoteVersionType version) NN_NOEXCEPT
            {
                NN_STATIC_ASSERT(TFromVersion >= 0);
                NN_STATIC_ASSERT(TCountMin <= TCountMax);
                if(version >= TFromVersion)
                {
                    return TCountMin;
                }
                else
                {
                    return 0;
                }
            }

            static int GetEntryCountMax(MakerNoteVersionType version) NN_NOEXCEPT
            {
                NN_STATIC_ASSERT(TFromVersion >= 0);
                NN_STATIC_ASSERT(TCountMin <= TCountMax);
                if(version >= TFromVersion)
                {
                    return TCountMax;
                }
                else
                {
                    return 0;
                }
            }
        };

    };

}}}}
