﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/os.h>
#include <nn/lmem/lmem_ExpHeap.h>

namespace nn{ namespace capsrv{ namespace server{ namespace detail{

    class FunctionInvokationRequestQueue1
    {
    public:

        void Initialize(uintptr_t dequeueWaitHolderTag, void* pWorkMemory, size_t workMemorySize) NN_NOEXCEPT;
        void Finalize() NN_NOEXCEPT;

        // @brief Dequeue 可能になるまで待つための MultiWaitHolder を取得します。
        nn::os::MultiWaitHolderType* GetDequeueMultiWaitHolder() NN_NOEXCEPT;

        // @brief 関数をキューに入れます
        // @param[in] pFunction      キューイングする関数ポインタです
        // @param[in] pParam         pFunction の呼出時に渡されるポインタです。
        // @param[in] pCompleteEvent pFunction の処理が完了した際にシグナルされるイベントのポインタです。
        // @retval ResultInternalRequestQueueIsFull
        // @details
        //   キューが満杯の場合、即座に返ります。
        //
        //   pParam の内容はこのインスタンスの内部にコピーされません。
        //   pCompleteEvent が nullptr でない場合、pFunction の処理が完了した際にシグナルされます。
        //
        //   pParam が指すオブジェクトは pFunction の処理が完了するまで有効でなければなりません。
        //   pCompleteEvent が指すオブジェクトは pFunction の処理が完了するまで有効でなければなりません。
        nn::Result TryEnqueueReference(
            void (*pFunction)(void* pParam),
            void* pParam,
            nn::os::EventType* pCompleteEvent
            ) NN_NOEXCEPT;

        // @brief 関数をキューに入れます
        // @param[in] pFunction      キューイングする関数ポインタです
        // @param[in] pParam         pFunction の呼出時に渡されるポインタです。
        // @param[in] pCompleteEvent pFunction の処理が完了した際にシグナルされるイベントのポインタです。
        // @details
        //   キューが満杯の場合、空きができるまでブロックします。
        //
        //   pParam の内容はこのインスタンスの内部にコピーされません。
        //   pCompleteEvent が nullptr でない場合、pFunction の処理が完了した際にシグナルされます。
        //
        //   pParam が指すオブジェクトは pFunction の処理が完了するまで有効でなければなりません。
        //   pCompleteEvent が指すオブジェクトは pFunction の処理が完了するまで有効でなければなりません。
        nn::Result EnqueueReference(
            void (*pFunction)(void* pParam),
            void* pParam,
            nn::os::EventType* pCompleteEvent
            ) NN_NOEXCEPT;

        // @brief 関数をキューに入れます
        // @param[in] pFunction      キューイングする関数ポインタです
        // @param[in] pParam         pFunction の呼出時に渡されるオブジェクトのポインタです。
        // @param[in] paramSize      pParam の指すオブジェクトの大きさ（バイト）です。
        // @param[in] paramAlignment pParam の指すオブジェクトのアライメント（バイト）です。
        // @param[in] pCompleteEvent pFunction の処理が完了した際にシグナルされるイベントのポインタです。
        // @retval ResultInternalRequestQueueIsFull
        // @retval ResultInternalRequestQueueAllocationFailed
        // @details
        //   キューが満杯の場合、即座に返ります。
        //
        //   pParam の内容はこのインスタンスの内部に単純コピーされます。
        //   pCompleteEvent が nullptr でない場合、pFunction の処理が完了した際にシグナルされます。
        //
        //   pParam が指すオブジェクトはこの関数の呼出後破棄することができます。
        //   pCompleteEvent が指すオブジェクトは pFunction の処理が完了するまで有効でなければなりません。
        nn::Result TryEnqueueCopy(
            void (*pFunction)(void* pParam),
            void* pParam,
            size_t paramSize,
            size_t paramAlignment,
            nn::os::EventType* pCompleteEvent
            ) NN_NOEXCEPT;

        // @brief 関数をキューに入れます
        // @param[in] pFunction      キューイングする関数ポインタです
        // @param[in] pParam         pFunction の呼出時に渡されるオブジェクトのポインタです。
        // @param[in] paramSize      pParam の指すオブジェクトの大きさ（バイト）です。
        // @param[in] paramAlignment pParam の指すオブジェクトのアライメント（バイト）です。
        // @param[in] pCompleteEvent pFunction の処理が完了した際にシグナルされるイベントのポインタです。
        // @retval ResultInternalRequestQueueAllocationFailed
        // @details
        //   キューが満杯の場合、空きができるまでブロックします。
        //
        //   pParam の内容はこのインスタンスの内部に単純コピーされます。
        //   pCompleteEvent が nullptr でない場合、pFunction の処理が完了した際にシグナルされます。
        //
        //   pParam が指すオブジェクトはこの関数の呼出後破棄することができます。
        //   pCompleteEvent が指すオブジェクトは pFunction の処理が完了するまで有効でなければなりません。
        nn::Result EnqueueCopy(
            void (*pFunction)(void* pParam),
            void* pParam,
            size_t paramSize,
            size_t paramAlignment,
            nn::os::EventType* pCompleteEvent
            ) NN_NOEXCEPT;

        // @brief キューの先頭にある関数を取得して処理を開始します。
        // @retval ResultInternalRequestQueueIsEmpty
        nn::Result BeginProcessingRequest(void (**pOutFunction)(void* pParam), void** ppParam) NN_NOEXCEPT;

        // @brief キューの先頭にある関数の処理を終了して関数を Dequeue します。
        // @details
        //   Dequeue の完了後 pCompleteEvent がシグナルされます。
        void EndProcessingRequest() NN_NOEXCEPT;

    private:
        nn::os::SemaphoreType m_RequesterSemaphore;
        nn::os::SemaphoreType m_RequesteeSemaphore;
        nn::os::MultiWaitHolderType m_RequesteeSemaphoreWaitHolder;

        void* m_pParameterHeapMemory;
        size_t m_ParameterHeapSize;
        nn::lmem::HeapHandle m_ParameterHeapHandle;

        // このクラスでは 1 つしかキューイングできない
        void (*m_pRequestedFunction)(void* pParam);
        void* m_pRequestedParameter;
        bool m_IsRequestParameterCopy;
        nn::os::EventType* m_pRequestedCompleteEvent;

        // BeginProcessingRequest() の呼出から EndProcessingRequest() の呼出までの間 true
        bool m_IsRequestProcessing;
    };

}}}}
