﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include "capsrvServer_AlbumWorkerManager.h"
#include "capsrvServer_ScreenShotWorkerManager.h"

namespace nn{ namespace capsrv{ namespace server{

    enum WorkRequester
    {
        WorkRequester_SystemServer,
        WorkRequester_ApplicationServer,
        WorkRequester_ScreenShotWorker,
    };

    enum WorkReceiver
    {
        WorkReceiver_ScreenShotWorker,
        WorkReceiver_AlbumWorker,
    };

    class WorkerDispatcher
    {
    public:
        typedef detail::LambdaFunctionInvokationRequestQueue<detail::FunctionInvokationRequestQueue1> QueueType;

    public:
        WorkerDispatcher() NN_NOEXCEPT
            : m_AlbumWorkerQueueTable()
            , m_ScreenShotWorkerQueueTable()
        {
        }

        void RegisterAlbumWorkerTable(const AlbumWorkerManager::QueueTable& table) NN_NOEXCEPT
        {
            m_AlbumWorkerQueueTable = table;
        }

        void RegisterScreenShotWorkerTable(const ScreenShotWorkerManager::QueueTable& table) NN_NOEXCEPT
        {
            m_ScreenShotWorkerQueueTable = table;
        }

        template<typename F>
        nn::Result EnqueueSync(WorkReceiver receiver, WorkRequester requester, F function) NN_NOEXCEPT
        {
            auto pQueue = GetQueue(receiver, requester);
            if(pQueue != nullptr)
            {
                return pQueue->EnqueueSync(function);
            }
            else
            {
                return function();
            }
        }

        template<typename F>
        nn::Result EnqueueAsync(WorkReceiver receiver, WorkRequester requester, F function) NN_NOEXCEPT
        {
            auto pQueue = GetQueue(receiver, requester);
            if(pQueue != nullptr)
            {
                return pQueue->EnqueueAsync(function);
            }
            else
            {
                return function();
            }
        }

        template<typename F>
        nn::Result TryEnqueueAsync(WorkReceiver receiver, WorkRequester requester, F function) NN_NOEXCEPT
        {
            auto pQueue = GetQueue(receiver, requester);
            if(pQueue != nullptr)
            {
                return pQueue->TryEnqueueAsync(function);
            }
            else
            {
                return function();
            }
        }

    private:
        QueueType* GetQueue(WorkReceiver receiver, WorkRequester requester) NN_NOEXCEPT
        {
            QueueType* pQueue = nullptr;
            switch(receiver)
            {
            case WorkReceiver_AlbumWorker:
                {
                    pQueue = GetAlbumWorkerQueue(requester);
                    break;
                }
            case WorkReceiver_ScreenShotWorker:
                {
                    pQueue = GetScreenShotWorkerQueue(requester);
                    break;
                }
            default: NN_UNEXPECTED_DEFAULT;
            }
            return pQueue;
        }

        QueueType* GetAlbumWorkerQueue(WorkRequester requester) NN_NOEXCEPT
        {
            switch(requester)
            {
            case WorkRequester_SystemServer:
                return m_AlbumWorkerQueueTable.pSystemQueue;
            case WorkRequester_ApplicationServer:
                return m_AlbumWorkerQueueTable.pApplicationQueue;
            case WorkRequester_ScreenShotWorker:
                return m_AlbumWorkerQueueTable.pScreenShotWorkerQueue;
            default: NN_UNEXPECTED_DEFAULT;
            }
        }

        QueueType* GetScreenShotWorkerQueue(WorkRequester requester) NN_NOEXCEPT
        {
            switch(requester)
            {
            case WorkRequester_SystemServer:
                return m_ScreenShotWorkerQueueTable.pSystemQueue;
            case WorkRequester_ApplicationServer:
                return m_ScreenShotWorkerQueueTable.pApplicationQueue;
            default: NN_UNEXPECTED_DEFAULT;
            }
        }

    private:
        AlbumWorkerManager::QueueTable      m_AlbumWorkerQueueTable;
        ScreenShotWorkerManager::QueueTable m_ScreenShotWorkerQueueTable;
    };

}}}

