﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "capsrvServer_ConvertResult.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include "capsrvServer_Config.h"
#include "capsrvServer_ResultPrivate.h"
#include "capsrvServer_ConvertFileSystemResult.h"

#define NN_CAPSRV_LOG_REPORT_RESULT_VALUE(result) \
    do {                                          \
        nn::Result tmpResult = result;            \
        NN_CAPSRV_LOG_RESULT("%d-%d(0x%08X)\n", tmpResult.GetModule(), tmpResult.GetDescription(), tmpResult.GetInnerValueForDebug());   \
        NN_UNUSED(tmpResult);                     \
    } while(NN_STATIC_CONDITION(false))

namespace nn{ namespace capsrv{ namespace server{

    namespace{
        nn::Result ConvertInternalAlbumLimitationError(nn::Result value) NN_NOEXCEPT
        {
            NN_RESULT_TRY(value)
                NN_RESULT_CATCH(ResultInternalAlbumLimitationFileCountLimit)
                {
                    NN_RESULT_THROW(ResultAlbumFileCountLimit());
                }
                NN_RESULT_CATCH_ALL
                {
                    NN_RESULT_THROW(ResultAlbumIsFull());
                }
            NN_RESULT_END_TRY;
            NN_RESULT_SUCCESS;
        }

        nn::Result ConvertInternalError(nn::Result value) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_REPORT_RESULT_VALUE(value);
            NN_RESULT_TRY(value)
                // ファイル内容の検証エラー
                NN_RESULT_CATCH(ResultInternalFileDataVerificationError)
                {
                    NN_RESULT_THROW(ResultAlbumInvalidFileData());
                }
                NN_RESULT_CATCH(ResultInternalSignatureError)
                {
                    NN_RESULT_THROW(ResultAlbumInvalidFileData());
                }
                // アルバムのファイル作成制限エラー
                NN_RESULT_CATCH(ResultInternalAlbumLimitationError)
                {
                    NN_RESULT_THROW(ConvertInternalAlbumLimitationError(value));
                }
                // caps:c のリクエスト発行エラー
                NN_RESULT_CATCH(ResultInternalRequestQueueIsFull)
                {
                    NN_RESULT_THROW(ResultControlServerIsBusy());
                }
                NN_RESULT_CATCH(ResultInternalRequestQueueAllocationFailed)
                {
                    NN_RESULT_THROW(ResultControlServerIsBusy());
                }
                // セッションエラー
                NN_RESULT_CATCH(ResultInternalAlbumSessionCountLimit)
                {
                    NN_RESULT_THROW(ResultAlbumResourceLimit());
                }
                // ストリームエラー
                NN_RESULT_CATCH(ResultInternalAlbumStreamCountLimit)
                {
                    NN_RESULT_THROW(ResultAlbumResourceLimit());
                }
                NN_RESULT_CATCH(ResultInternalAlbumStreamNotFound)
                {
                    NN_RESULT_THROW(ResultAlbumNotFound());
                }
                NN_RESULT_CATCH(ResultInternalAlbumStreamAlreadyOpened)
                {
                    NN_RESULT_THROW(ResultAlbumAlreadyOpened());
                }
                NN_RESULT_CATCH(ResultInternalAlbumStreamOutOfRange)
                {
                    NN_RESULT_THROW(ResultAlbumOutOfRange());
                }
                // その他の内部エラー
                NN_RESULT_CATCH_ALL
                {
                    NN_RESULT_THROW(ResultInternalError());
                }
            NN_RESULT_END_TRY;
            NN_RESULT_SUCCESS;
        }

        nn::Result ConvertFsError(nn::Result value) NN_NOEXCEPT
        {
            NN_UNUSED(value);
            NN_CAPSRV_LOG_REPORT_RESULT_VALUE(value);
            NN_RESULT_THROW(ConvertFileSystemResult(value, AlbumStorageDirection_Unknown));
        }
    }

    nn::Result ConvertResult(nn::Result value) NN_NOEXCEPT
    {
        NN_RESULT_TRY(value)
            NN_RESULT_CATCH(ResultInternalError)
            {
                NN_RESULT_THROW(ConvertInternalError(value));
            }
            NN_RESULT_CATCH(nn::fs::ResultHandledByAllProcess)
            {
                NN_RESULT_THROW(ConvertFsError(value));
            }
            NN_RESULT_CATCH(nn::fs::ResultHandledBySystemProcess)
            {
                NN_RESULT_THROW(ConvertFsError(value));
            }
            NN_RESULT_CATCH(nn::fs::ResultInternal)
            {
                NN_RESULT_THROW(ConvertFsError(value));
            }
        NN_RESULT_END_TRY;
        NN_RESULT_SUCCESS;
    }

}}}
