﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include "capsrvServer_TemporaryFileManipulator.h"
#include "capsrvServer_AlbumMovieUtility.h"
#include "../capsrvServer_Config.h"

namespace nn{ namespace capsrv{ namespace server{ namespace album{

    enum MovieWriteHashSignatureState
    {
        MovieWriteHashSignatureState_Unused = 0,
        MovieWriteHashSignatureState_Empty,
        MovieWriteHashSignatureState_WritingData,
        MovieWriteHashSignatureState_DataComplete,
        MovieWriteHashSignatureState_WritingMeta,
        MovieWriteHashSignatureState_MetaComplete,
        MovieWriteHashSignatureState_CalculatingSign,
        MovieWriteHashSignatureState_SignComplete,
        MovieWriteHashSignatureState_Broken,
    };

    class MovieWriteHashSignature
    {
    public:
        static const int64_t BlockSize = MovieDataBlockSize;
        static const size_t HashSize = MovieDataBlockHashSize;
        static const size_t SignSize = MovieSignatureSize;

    public:
        MovieWriteHashSignature() NN_NOEXCEPT;

        nn::Result Initialize() NN_NOEXCEPT;
        void Finalize() NN_NOEXCEPT;

        nn::Result StartDataSection() NN_NOEXCEPT;
        nn::Result EndDataSection() NN_NOEXCEPT;
        nn::Result StartMetaSection() NN_NOEXCEPT;
        nn::Result EndMetaSection() NN_NOEXCEPT;
        nn::Result StartSignCalculation(int64_t* pOutHashListSize) NN_NOEXCEPT;
        nn::Result EndSignCalculation() NN_NOEXCEPT;

        // @brief Data のハッシュを計算して登録する
        // @pre WritingData
        // @pre offset % BlockSize == 0
        // @pre size % BlockSize == 0
        // @details
        // 同じ領域に対して複数回登録した場合、上書きされる。
        nn::Result RegisterData(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT;

        // @brief 指定した Data ブロックのハッシュを取得する
        // @pre WritingData
        // @details
        // 署名計算する際は ProcessSignCalculationForNextHash でまとめて読み込みする
        nn::Result GetDataHash(AlbumMovieUtility::BlockHash* pOutHash, int64_t blockIndex) NN_NOEXCEPT;

        // @brief Data のサイズを変更する
        // @pre WritingData
        // @pre newSize % BlockSize == 0
        nn::Result ResizeData(int64_t newSize) NN_NOEXCEPT;

        // @brief Meta のハッシュを計算して登録する
        // @pre WritingMeta
        // @details
        // 同じ領域に対して複数回登録した場合、上書きされる。
        nn::Result RegisterMeta(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT;

        // @brief サイズの妥当性を検査する
        // @pre MetaComplete
        nn::Result VerifyHashListSize(int64_t dataSize, int64_t metaSize) const NN_NOEXCEPT;

        // @brief 未処理のハッシュを読込む。同時に署名の計算を進める
        // @pre CalculationSign
        nn::Result ProcessSignCalculationForNextHash(size_t* pOutSize, void* buffer, size_t bufferSize) NN_NOEXCEPT;

        // @brief 計算した署名を取得する
        // @pre SignComplete
        nn::Result GetSignature(size_t* pOutSize, void* buffer, size_t bufferSize) NN_NOEXCEPT;

    private:
        void MarkAsBroken(nn::Result brokenReason) NN_NOEXCEPT;
        nn::Result MarkAsBrokenIfFailure(nn::Result result) NN_NOEXCEPT;

        nn::Result ExpandFileImpl(int64_t requiredFileSize) NN_NOEXCEPT;

    private:
        MovieWriteHashSignatureState m_State;
        TemporaryFileHandle m_HashFileHandle;
        int64_t m_FileSize;

        int64_t m_DataHashListOffset;
        int64_t m_DataHashListSize;
        int64_t m_MetaHashListOffset;
        int64_t m_MetaHashListSize;
        int64_t m_SignCalculationCurrentPosition;
        int64_t m_SignCalculationEndPosition;

        AlbumMovieUtility::FileSignGenerationState m_FileSignGenerationState;

    };

}}}}
