﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "capsrvServer_MovieWriteHashSignature.h"

#include <algorithm>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/crypto/crypto_Sha256Generator.h>
#include "../../capsrv_Macro.h"
#include "../capsrvServer_ResultPrivate.h"

namespace nn{ namespace capsrv{ namespace server{ namespace album{
    NN_STATIC_ASSERT(MovieWriteHashSignature::BlockSize % nn::crypto::Sha256Generator::BlockSize == 0);
    NN_STATIC_ASSERT(MovieWriteHashSignature::HashSize == nn::crypto::Sha256Generator::HashSize);

    MovieWriteHashSignature::MovieWriteHashSignature() NN_NOEXCEPT
    {
        std::memset(this, 0, sizeof(*this));
    }

    nn::Result MovieWriteHashSignature::Initialize() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_EQUAL(m_State, MovieWriteHashSignatureState_Unused);
        NN_RESULT_THROW_UNLESS(m_State == MovieWriteHashSignatureState_Unused, ResultAlbumDenied());

        const int64_t initialFileSize = MovieHashFileInitialSize;

        NN_CAPSRV_PROCESS_START();

        TemporaryFileHandle hashFile = {};
        NN_RESULT_DO(TemporaryFileManipulator::OpenNewTemporaryFile(&hashFile, initialFileSize));
        NN_CAPSRV_PROCESS_ROLLBACK(TemporaryFileManipulator::DiscardTemporaryFile(hashFile));

        NN_RESULT_DO(AlbumMovieUtility::StartFileSignGeneration(&m_FileSignGenerationState));

        NN_CAPSRV_PROCESS_SUCCESS();
        m_State = MovieWriteHashSignatureState_Empty;
        m_HashFileHandle = hashFile;
        m_FileSize = initialFileSize;
        m_DataHashListOffset = 0;
        m_DataHashListSize = 0;
        m_MetaHashListOffset = 0;
        m_MetaHashListSize = 0;
        m_SignCalculationCurrentPosition = 0;
        m_SignCalculationEndPosition = 0;
        NN_RESULT_SUCCESS;
    }

    void MovieWriteHashSignature::Finalize() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_EQUAL(m_State, MovieWriteHashSignatureState_Unused);
        if(m_State == MovieWriteHashSignatureState_Unused)
        {
            return;
        }

        TemporaryFileManipulator::DiscardTemporaryFile(m_HashFileHandle);

        std::memset(this, 0, sizeof(*this));
        NN_SDK_ASSERT_EQUAL(m_State, MovieWriteHashSignatureState_Unused);
    }

    nn::Result MovieWriteHashSignature::StartDataSection() NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_State == MovieWriteHashSignatureState_Empty, ResultAlbumDenied());
        m_State = MovieWriteHashSignatureState_WritingData;
        m_DataHashListOffset = 0;
        m_DataHashListSize = 0;
        NN_RESULT_SUCCESS;
    }

    nn::Result MovieWriteHashSignature::EndDataSection() NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_State == MovieWriteHashSignatureState_WritingData, ResultAlbumDenied());
        m_State = MovieWriteHashSignatureState_DataComplete;
        NN_RESULT_SUCCESS;
    }

    nn::Result MovieWriteHashSignature::StartMetaSection() NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_State == MovieWriteHashSignatureState_DataComplete, ResultAlbumDenied());
        m_State = MovieWriteHashSignatureState_WritingMeta;
        m_MetaHashListOffset = m_DataHashListOffset + m_DataHashListSize;
        m_MetaHashListSize = 0;
        NN_RESULT_SUCCESS;
    }

    nn::Result MovieWriteHashSignature::EndMetaSection() NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_State == MovieWriteHashSignatureState_WritingMeta, ResultAlbumDenied());
        m_State = MovieWriteHashSignatureState_MetaComplete;
        NN_RESULT_SUCCESS;
    }

    nn::Result MovieWriteHashSignature::StartSignCalculation(int64_t* pOutHashListSize) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_State == MovieWriteHashSignatureState_MetaComplete, ResultAlbumDenied());
        m_State = MovieWriteHashSignatureState_CalculatingSign;
        m_SignCalculationCurrentPosition = 0;
        m_SignCalculationEndPosition = m_DataHashListSize + m_MetaHashListSize;
        *pOutHashListSize = m_SignCalculationEndPosition;
        NN_RESULT_SUCCESS;
    }

    nn::Result MovieWriteHashSignature::EndSignCalculation() NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_State == MovieWriteHashSignatureState_CalculatingSign, ResultAlbumDenied());
        NN_RESULT_THROW_UNLESS(m_SignCalculationCurrentPosition == m_SignCalculationEndPosition, ResultAlbumDenied());
        m_State = MovieWriteHashSignatureState_SignComplete;
        NN_RESULT_SUCCESS;
    }

    void MovieWriteHashSignature::MarkAsBroken(nn::Result brokenReason) NN_NOEXCEPT
    {
        NN_UNUSED(brokenReason);
        m_State = MovieWriteHashSignatureState_Broken;
    }

    nn::Result MovieWriteHashSignature::MarkAsBrokenIfFailure(nn::Result result) NN_NOEXCEPT
    {
        if(result.IsFailure())
        {
            MarkAsBroken(result);
        }
        return result;
    }

    nn::Result MovieWriteHashSignature::ExpandFileImpl(int64_t requiredFileSize) NN_NOEXCEPT
    {
        if(requiredFileSize > m_FileSize)
        {
            int64_t newFileSize = ((requiredFileSize + MovieHashFileUnitSize - 1) / MovieHashFileUnitSize) * MovieHashFileUnitSize;
            NN_RESULT_DO(MarkAsBrokenIfFailure(
                TemporaryFileManipulator::ResizeFile(m_HashFileHandle, newFileSize)
            ));
            m_FileSize = newFileSize;
        }
        NN_SDK_ASSERT_LESS_EQUAL(requiredFileSize, m_FileSize);
        NN_RESULT_SUCCESS;
    }

    //-------------------------------

    nn::Result MovieWriteHashSignature::RegisterData(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_State == MovieWriteHashSignatureState_WritingData, ResultAlbumDenied());
        NN_RESULT_THROW_UNLESS(buffer != nullptr, ResultAlbumOutOfRange());
        NN_RESULT_THROW_UNLESS(offset % BlockSize == 0, ResultAlbumOutOfRange());
        NN_RESULT_THROW_UNLESS(size == static_cast<size_t>(BlockSize), ResultAlbumOutOfRange());

        int64_t blockIndex = offset / BlockSize;
        NN_RESULT_THROW_UNLESS(blockIndex >= 0, ResultAlbumOutOfRange());
        NN_RESULT_THROW_UNLESS(blockIndex < MovieDataBlockCountLimit, ResultAlbumOutOfRange());

        char hash[HashSize] = {};

        nn::crypto::Sha256Generator hasher;
        hasher.Initialize();
        hasher.Update(buffer, size);
        hasher.GetHash(&hash, sizeof(hash));

        int64_t hashListOffset = blockIndex * HashSize;
        int64_t hashFileOffset = m_DataHashListOffset + hashListOffset;

        // 必要なサイズまで拡大
        int64_t requiredFileSize = hashFileOffset + sizeof(hash);
        NN_RESULT_DO(MarkAsBrokenIfFailure(
            ExpandFileImpl(requiredFileSize)
        ));

        NN_RESULT_DO(MarkAsBrokenIfFailure(
            TemporaryFileManipulator::WriteFile(m_HashFileHandle, hashFileOffset, hash, sizeof(hash))
        ));

        m_DataHashListSize = std::max(m_DataHashListSize, hashListOffset + static_cast<int64_t>(sizeof(hash)));
        NN_RESULT_SUCCESS;
    }

    nn::Result MovieWriteHashSignature::GetDataHash(AlbumMovieUtility::BlockHash* pOutHash, int64_t blockIndex) NN_NOEXCEPT
    {
        NN_STATIC_ASSERT(sizeof(*pOutHash) == HashSize);
        NN_RESULT_THROW_UNLESS(m_State == MovieWriteHashSignatureState_WritingData, ResultAlbumDenied());
        NN_RESULT_THROW_UNLESS(pOutHash != nullptr, ResultInternalError());

        NN_RESULT_THROW_UNLESS(blockIndex >= 0, ResultAlbumOutOfRange());
        NN_RESULT_THROW_UNLESS(blockIndex < MovieDataBlockCountLimit, ResultAlbumOutOfRange());

        int64_t offsetInDataHash = blockIndex * HashSize;
        int64_t offsetInHashFile = m_DataHashListOffset + offsetInDataHash;

        NN_RESULT_THROW_UNLESS(offsetInHashFile + HashSize <= m_DataHashListOffset + m_DataHashListSize, ResultAlbumOutOfRange());

        size_t readSize = 0;
        NN_RESULT_DO(MarkAsBrokenIfFailure(
            TemporaryFileManipulator::ReadFile(&readSize, m_HashFileHandle, offsetInHashFile, pOutHash, HashSize)
        ));
        NN_RESULT_THROW_UNLESS(readSize == HashSize, ResultInternalError());

        NN_RESULT_SUCCESS;
    }

    nn::Result MovieWriteHashSignature::ResizeData(int64_t newSize) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_State == MovieWriteHashSignatureState_WritingData, ResultAlbumDenied());
        NN_RESULT_THROW_UNLESS(newSize % BlockSize == 0, ResultAlbumOutOfRange());

        int64_t newBlockCount = newSize / BlockSize;
        NN_RESULT_THROW_UNLESS(newBlockCount >= 0, ResultAlbumOutOfRange());
        NN_RESULT_THROW_UNLESS(newBlockCount <= MovieDataBlockCountLimit, ResultAlbumOutOfRange());

        int64_t newHashListSize = newBlockCount * HashSize;
        int64_t newHashFileSize = m_DataHashListOffset + newHashListSize;

        // 必要なサイズまで拡大
        NN_RESULT_DO(MarkAsBrokenIfFailure(
            ExpandFileImpl(newHashFileSize)
        ));

        m_DataHashListSize = newHashListSize;
        NN_RESULT_SUCCESS;

    }

    nn::Result MovieWriteHashSignature::RegisterMeta(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_State == MovieWriteHashSignatureState_WritingMeta, ResultAlbumDenied());
        NN_RESULT_THROW_UNLESS(buffer != nullptr, ResultAlbumOutOfRange());
        NN_RESULT_THROW_UNLESS(offset % BlockSize == 0, ResultAlbumOutOfRange());
        NN_RESULT_THROW_UNLESS(size == static_cast<size_t>(BlockSize), ResultAlbumOutOfRange());

        char hash[HashSize] = {};

        nn::crypto::Sha256Generator hasher;
        hasher.Initialize();
        hasher.Update(buffer, size);
        hasher.GetHash(&hash, sizeof(hash));

        int64_t hashListOffset = offset / BlockSize * HashSize;
        int64_t hashFileOffset = m_MetaHashListOffset + hashListOffset;

        // 必要なサイズまで拡大
        int64_t requiredFileSize = hashFileOffset + sizeof(hash);
        NN_RESULT_DO(MarkAsBrokenIfFailure(
            ExpandFileImpl(requiredFileSize)
        ));

        NN_RESULT_DO(MarkAsBrokenIfFailure(
            TemporaryFileManipulator::WriteFile(m_HashFileHandle, hashFileOffset, hash, sizeof(hash))
        ));

        m_MetaHashListSize = std::max(m_MetaHashListSize, hashListOffset + static_cast<int64_t>(sizeof(hash)));
        NN_RESULT_SUCCESS;
    }

    //------------------------------------------------------------

    nn::Result MovieWriteHashSignature::VerifyHashListSize(int64_t dataSize, int64_t metaSize) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_EQUAL(m_State, MovieWriteHashSignatureState_MetaComplete);
        NN_SDK_REQUIRES_GREATER_EQUAL(dataSize, 0);
        NN_SDK_REQUIRES_GREATER_EQUAL(metaSize, 0);
        NN_RESULT_THROW_UNLESS(m_State == MovieWriteHashSignatureState_MetaComplete, ResultAlbumDenied());
        NN_RESULT_THROW_UNLESS(dataSize >= 0, ResultAlbumOutOfRange());
        NN_RESULT_THROW_UNLESS(metaSize >= 0, ResultAlbumOutOfRange());

        NN_SDK_ASSERT_EQUAL(m_DataHashListOffset + m_DataHashListSize, m_MetaHashListOffset);

        NN_SDK_ASSERT_EQUAL(dataSize % BlockSize, 0);
        NN_SDK_ASSERT_EQUAL(metaSize % BlockSize, 0);
        NN_SDK_ASSERT_EQUAL(dataSize / BlockSize * HashSize, m_DataHashListSize);
        NN_SDK_ASSERT_EQUAL(metaSize / BlockSize * HashSize, m_MetaHashListSize);
        NN_RESULT_THROW_UNLESS(dataSize % BlockSize == 0, ResultAlbumInvalidFileData());
        NN_RESULT_THROW_UNLESS(metaSize % BlockSize == 0, ResultAlbumInvalidFileData());
        NN_RESULT_THROW_UNLESS(dataSize / BlockSize * HashSize == m_DataHashListSize, ResultAlbumInvalidFileData());
        NN_RESULT_THROW_UNLESS(metaSize / BlockSize * HashSize == m_MetaHashListSize, ResultAlbumInvalidFileData());

        NN_RESULT_SUCCESS;
    }

    nn::Result MovieWriteHashSignature::ProcessSignCalculationForNextHash(size_t* pOutSize, void* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_State == MovieWriteHashSignatureState_CalculatingSign, ResultAlbumDenied());
        NN_RESULT_THROW_UNLESS(m_SignCalculationCurrentPosition < m_SignCalculationEndPosition, ResultAlbumDenied());
        NN_RESULT_THROW_UNLESS(bufferSize % HashSize == 0, ResultAlbumWorkMemoryError());

        size_t size = 0;
        NN_RESULT_DO(MarkAsBrokenIfFailure(
            TemporaryFileManipulator::ReadFile(&size, m_HashFileHandle, m_SignCalculationCurrentPosition, buffer, bufferSize)
        ));

        auto blockHashes = static_cast<AlbumMovieUtility::BlockHash*>(buffer);
        auto blockCount = size / sizeof(AlbumMovieUtility::BlockHash);
        AlbumMovieUtility::ProcessFileSignGeneration(m_FileSignGenerationState, blockHashes, blockCount);

        m_SignCalculationCurrentPosition += static_cast<int64_t>(size);
        NN_SDK_ASSERT_LESS_EQUAL(m_SignCalculationCurrentPosition, m_SignCalculationEndPosition);

        *pOutSize = size;
        NN_RESULT_SUCCESS;
    }

    nn::Result MovieWriteHashSignature::GetSignature(size_t* pOutSize, void* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_State == MovieWriteHashSignatureState_SignComplete, ResultAlbumDenied());
        NN_RESULT_THROW_UNLESS(bufferSize >= SignSize, ResultAlbumOutOfRange());

        AlbumMovieUtility::FileSign sign;
        NN_RESULT_DO(AlbumMovieUtility::FinishFileSignGeneration(&sign, m_FileSignGenerationState));

        static_assert(sizeof(sign.value) == SignSize, "");
        std::memcpy(buffer, &sign.value, SignSize);

        *pOutSize = SignSize;
        NN_RESULT_SUCCESS;
    }


}}}}
