﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/capsrv/capsrv_AlbumEntry.h>

#include "../capsrvServer_Config.h"
#include "../capsrvServer_EnvironmentInfo.h"

namespace nn{ namespace capsrv{ namespace server{ namespace album{

    // このクラスはファイルパス文字列の操作を行います。
    // ファイルシステムの操作は行いません。
    class AlbumPathUtility
    {
    public:
        // NOTE:
        //   Length は終端文字を含まない
        //   Size   は終端文字を含む
        //
        //   Path はフルパス
        //   ディレクトリパスは末尾の "/" を含む

    //--------
    // RootPath
    //--------

        // "ST"
        static const int MountNameLength = AlbumMountNameLength;

        // "ST:/"
        static const int RootPathLength = MountNameLength + 2;

    //--------
    // SubDirectory
    //--------
        // サブディレクトリは fileId のみから一意に計算できること。
        // 現在の実装では撮影日を使用している。

        // サブディレクトリの深さ
        static const int SubDirectoryDepth = 3;

        // サブディレクトリの名前の長さ
        static const int SubDirectoryNameLength0 = 4; // YYYY
        static const int SubDirectoryNameLength1 = 2; // MM
        static const int SubDirectoryNameLength2 = 2; // DD

        // サブディレクトリのパスの長さ
        // "ST:/YYYY/"
        static const int SubDirectoryPathLength0 = MountNameLength + 2 + SubDirectoryNameLength0 + 1;
        // "ST:/YYYY/MM/"
        static const int SubDirectoryPathLength1 = SubDirectoryPathLength0 + SubDirectoryNameLength1 + 1;
        // "ST:/YYYY/MM/DD/"
        static const int SubDirectoryPathLength2 = SubDirectoryPathLength1 + SubDirectoryNameLength2 + 1;

    //--------
    // Filename
    //--------

        // ファイル名の長さ
        // "YYYYMMDDhhmmssii-XXXXXXXXXXXXXXXXYYYYYYYYYYYYYYYY.jpg"
        //  12345678901234567890123456789012345678901234567890123
        //           1         2         3         4         5
        static const int FilenameLength = 53;

        // Filename '\0'
        static const int FilenameSize = FilenameLength + 1;

        // "ST:/YYYY/MM/DD/YYYYMMDDhhmmssii-XXXXXXXXXXXXXXXXYYYYYYYYYYYYYYYY.jpg"
        static const int PathLength = SubDirectoryPathLength2 + FilenameLength;

        // Path + '\0'
        static const int PathSize = PathLength + 1;

    //--------
    // Extra な SubDirectory
    //--------
        // サブディレクトリは fileId のみから一意に計算できること。
        // 現在の実装では撮影日を使用している。

        // サブディレクトリの深さ
        static const int ExtraSubDirectoryDepth = 5;

        // 0 階層目のディレクトリ名
        #define NN_CAPSRV_DIRECTORY_NAME_FOR_EXTRA  "Extra"

        // サブディレクトリの名前の長さ
        static const int ExtraSubDirectoryNameLength0 = 5;  // "Extra"
        static const int ExtraSubDirectoryNameLength1 = 32; // XXXX...XXXX
        static const int ExtraSubDirectoryNameLength2 = 4;  // YYYY
        static const int ExtraSubDirectoryNameLength3 = 2;  // MM
        static const int ExtraSubDirectoryNameLength4 = 2;  // DD

        // サブディレクトリのパスの長さ
        // "ST:/Extra/"
        static const int ExtraSubDirectoryPathLength0 = MountNameLength + 2 + ExtraSubDirectoryNameLength0 + 1;
        // "ST:/XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX/" (32文字)
        static const int ExtraSubDirectoryPathLength1 = ExtraSubDirectoryPathLength0 + ExtraSubDirectoryNameLength1 + 1;
        // "ST:/YYYY/"
        static const int ExtraSubDirectoryPathLength2 = ExtraSubDirectoryPathLength1 + ExtraSubDirectoryNameLength2 + 1;
        // "ST:/YYYY/MM/"
        static const int ExtraSubDirectoryPathLength3 = ExtraSubDirectoryPathLength2 + ExtraSubDirectoryNameLength3 + 1;
        // "ST:/YYYY/MM/DD/"
        static const int ExtraSubDirectoryPathLength4 = ExtraSubDirectoryPathLength3 + ExtraSubDirectoryNameLength4 + 1;

    //--------
    // Extra な Filename
    //--------

        // ファイル名の長さ
        // "YYYYMMDDhhmmssii-XXXXXXXXXXXXXXXXYYYYYYYYYYYYYYYYZ.jpg"
        //  123456789012345678901234567890123456789012345678901234
        //           1         2         3         4         5
        //
        // 最後の 'Z' は Extra 属性で、G～Z のアルファベットが入る。
        // 'X': AlbumFileContents_ExtraScreenShot もしくは AlbumFileContents_ExtraMovie
        //
        static const int ExtraFilenameLength = 54;

        // Filename '\0'
        static const int ExtraFilenameSize = ExtraFilenameLength + 1;

        // "ST:/Extra/XXXX...XXXX/YYYY/MM/DD/YYYYMMDDhhmmssii-XXXXXXXXXXXXXXXXYYYYYYYYYYYYYYYYZ.jpg"
        static const int ExtraPathLength = ExtraSubDirectoryPathLength4 + ExtraFilenameLength;

        // Path + '\0'
        static const int ExtraPathSize = ExtraPathLength + 1;

    //--------
    // サブディレクトリパラメータ
    //--------

    public:
        static int GetSubDirectoryNameLengthForRegular(int depth) NN_NOEXCEPT;
        static int GetSubDirectoryNameLengthForExtra(int depth) NN_NOEXCEPT;
        static int GetSubDirectoryNameLength(int depth, bool isExtra) NN_NOEXCEPT;
        static int GetSubDirectoryPathLength(int depth, bool isExtra) NN_NOEXCEPT;

    private:
        static int GetSubDirectoryPathLengthForRegular(int depth) NN_NOEXCEPT;
        static int GetSubDirectoryPathLengthForExtra(int depth) NN_NOEXCEPT;

    //--------
    // 検査関数
    //--------

    public:
        // @brief ファイル ID が正常か検査します。
        // @retval nn::ResultSuccess
        // @retval nn::capsrv::ResultAlbumInvalidFileId
        static nn::Result ValidateFileId(
            const AlbumFileId* pFileId,
            const EnvironmentInfo& env
            ) NN_NOEXCEPT;

        static nn::Result ValidateSubDirectoryId(
            const AlbumFileId* pDirectoryId,
            int depth,
            bool isExtra
            ) NN_NOEXCEPT;

        static nn::Result ValidateYear(uint16_t value) NN_NOEXCEPT;
        static nn::Result ValidateMonth(uint8_t value) NN_NOEXCEPT;
        static nn::Result ValidateDay(uint8_t value) NN_NOEXCEPT;

        // @brief 日時が正常か検査します。
        // @retval nn::ResultSuccess
        // @retval nn::capsrv::ResultAlbumInvalidTimestamp
        static nn::Result ValidateTime(
            const AlbumFileDateTime* pTime
            ) NN_NOEXCEPT;

        // @breif アプリケーション ID が正常か検査します。
        // @retval nn::ResultSuccess
        // @retval nn::capsrv::ResultAlbumInvalidApplicationId
        static nn::Result ValidateApplicationId(
            nn::ncm::ApplicationId applicationId
            ) NN_NOEXCEPT;

        // @brief ストレージの値が定義済の値か検査します。
        // @retval nn::ResultSuccess
        // @retval nn::capsrv::ResultAlbumInvalidStorage
        static nn::Result ValidateStorage(
            AlbumStorageType storage
            ) NN_NOEXCEPT;

        // @brief コンテンツの値が定義済の値か検査します。
        // @retval nn::ResultSuccess
        // @retval nn::capsrv::ResultAlbumInvalidContents
        static nn::Result ValidateFileContents(
            AlbumFileContentsType contents,
            const EnvironmentInfo& env
            ) NN_NOEXCEPT;

    //--------
    // パス生成関数
    //--------

        // @brief ストレージのルートパスを取得します。
        // @pre pOutPathLength != nullptr
        // @pre pOutPath != nullptr
        // @pre outPathBufferSize >= PathSize
        // @pre ValidateStorage(storage).IsSuccess()
        static void GetStorageRootPath(
            int* pOutPathLength,
            char* pOutPath,
            size_t outPathBufferSize,
            AlbumStorageType storage
            ) NN_NOEXCEPT;

        // @brief サブディレクトリのパスを取得します。
        // @pre pOutPathLength != nullptr
        // @pre pOutPath != nullptr
        // @pre ValidateFileId(fileId).IsSuccess()
        // @pre depth >= 0 && depth < SubDirectoryDepth
        static void GetSubDirectoryPath(
            int* pOutPathLength,
            char* pOutPath,
            size_t outPathBuffersize,
            const AlbumFileId& fileId,
            int depth,
            bool isExtra
            ) NN_NOEXCEPT;

        // @brief ファイル名を生成します。
        // @pre pOutPathLength != nullptr
        // @pre pOutPath != nullptr
        // @pre outPathBufferSize >= FilenameSize
        // @pre ValidateTime(pTime).IsSuccess()
        // @pre ValidateApplicationId(applicationId).IsSuccess()
        // @pre ValidateFileContents(contents).IsSuccess()
        static void GetFilename(
            int* pOutPathLength,
            char* pOutPath,
            size_t outPathBufferSize,
            const AlbumFileDateTime* pTime,
            nn::ncm::ApplicationId applicationId,
            AlbumFileContentsType contents,
            const EnvironmentInfo& env
            ) NN_NOEXCEPT;

        // @brief ファイルパスを取得します。
        // @pre pOutPathLength != nullptr
        // @pre pOutPath != nullptr
        // @pre outPathBufferSize >= PathSize
        // @pre ValidateFileId(&fileId).IsSuccess()
        static void GetFilepath(
            int* pOutPathLength,
            char* pOutPath,
            size_t outPathBufferSize,
            const AlbumFileId& fileId,
            const EnvironmentInfo& env
            ) NN_NOEXCEPT;

    //--------
    // パス解析関数
    //--------

        static nn::Result ParseNullOrEoi(
            const char** pOutNext,
            const char* pString,
            size_t length
            ) NN_NOEXCEPT;

        static nn::Result ParseAlbumStorageName(
            const char** pOutNext,
            AlbumStorageType* pOutValue,
            const char* pString,
            size_t length
            ) NN_NOEXCEPT;

        // pOutValue の該当部分だけが書き換えられる。その他の部分はそのまま。
        static nn::Result ParseSubDirectoryName(
            const char** pOutNext,
            AlbumFileId* pOutValue,
            const char* pString,
            size_t length,
            int depth,
            bool isExtra
            ) NN_NOEXCEPT;

        // pOutValue の該当部分だけが書き換えられる。その他の部分はそのまま。
        static nn::Result ParseAlbumFilename(
            const char** pOutNext,
            AlbumFileId* pOutValue,
            const char* pString,
            size_t length,
            const EnvironmentInfo& env
            ) NN_NOEXCEPT;

        static nn::Result CheckFileIdConsistency(
            const AlbumFileId& fileId,
            const AlbumFileId& directoryId
            ) NN_NOEXCEPT;

    private:
        static nn::Result ParseSubDirectoryNameForRegular(
            const char** pOutNext,
            AlbumFileId* pOutValue,
            const char* pString,
            size_t length,
            int depth
            ) NN_NOEXCEPT;

        static nn::Result ParseSubDirectoryNameForExtra(
            const char** pOutNext,
            AlbumFileId* pOutValue,
            const char* pString,
            size_t length,
            int depth
            ) NN_NOEXCEPT;

    };

}}}}
