﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "capsrvServer_AlbumMovieUtility.h"

#include <algorithm>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/capsrv/capsrv_Result.h>
#include <nn/capsrv/movie/capsrv_MovieMetaDataBuilder.h> // MovieMetaDataVersion
#include "../capsrvServer_Config.h"
#include "../capsrvServer_ResultPrivate.h"

namespace nn{ namespace capsrv{ namespace server{ namespace album{

    const int AlbumMovieUtility::MetaHashList::Length;

    //-------------------------------------------------------------------------------------------

    AlbumMovieMetaInfo::AlbumMovieMetaInfo() NN_NOEXCEPT
    {
        std::memset(this, 0, sizeof(*this));
    }

    AlbumMovieMetaInfo::AlbumMovieMetaInfo(
        int64_t movieDataSize,
        int64_t imageDataSize,
        uint64_t version
    ) NN_NOEXCEPT
        : m_MovieDataSize(movieDataSize)
        , m_ImageDataSize(imageDataSize)
        , m_Version(version)
    {
    }

    int64_t AlbumMovieMetaInfo::GetMovieDataSize() const NN_NOEXCEPT
    {
        return m_MovieDataSize;
    }

    int64_t AlbumMovieMetaInfo::GetImageDataSize() const NN_NOEXCEPT
    {
        return m_ImageDataSize;
    }

    uint64_t AlbumMovieMetaInfo::GetVersion() const NN_NOEXCEPT
    {
        return m_Version;
    }

    //-------------------------------------------------------------------------------------------

    nn::Result AlbumMovieUtility::CalculateSectionSize(
        int64_t* pOutDataBlockCount,
        int64_t* pOutDataSize,
        int64_t* pOutMetaSize,
        int64_t* pOutDataHashSize,
        int64_t* pOutMetaHashSize,
        int64_t* pOutSignSize,
        int64_t* pOutVersionSize,
        int64_t fileSize,
        uint64_t untrustedVersion
    ) NN_NOEXCEPT
    {
        NN_CAPSRV_LOG_MOVIE_DEV("calculating sizes of movie sections(filesize=%lld, ver=%llu)\n", fileSize, untrustedVersion);

        // バージョンを確認
        NN_RESULT_THROW_UNLESS(untrustedVersion == nn::capsrv::movie::MovieMetaDataVersion_1, ResultAlbumInvalidFileData());

        // 固定サイズ
        const int64_t MetaSize = MovieMetaDataSize;
        const int64_t MetaBlockCount = (MovieMetaDataSize + MovieDataBlockSize - 1) / MovieDataBlockSize;
        const int64_t SignSize = MovieSignatureSize;
        const int64_t VersionSize = sizeof(uint64_t);

        const int64_t FixedSize = MetaSize + (MetaBlockCount * MovieDataBlockHashSize) + SignSize + VersionSize;

        int64_t variableSize = fileSize - FixedSize;
        NN_RESULT_THROW_UNLESS(variableSize >= 0, ResultAlbumInvalidFileData());
        NN_CAPSRV_LOG_MOVIE_DEV("  variable size = %lld\n", variableSize);

        // ハッシュを無視してブロック数を推定（大きく推定される）
        int64_t estimatedBlockCountMax = (variableSize + MovieDataBlockSize - 1) / MovieDataBlockSize;
        NN_RESULT_THROW_UNLESS(estimatedBlockCountMax >= 0, ResultAlbumInvalidFileData());
        NN_CAPSRV_LOG_MOVIE_DEV("  data block count max = %lld\n", estimatedBlockCountMax);

        // ↑のブロック数を使って再計算（小さく推定される）
        int64_t estimatedBlockCountMin = (variableSize - MovieDataBlockHashSize * estimatedBlockCountMax + MovieDataBlockSize - 1) / MovieDataBlockSize;
        NN_RESULT_THROW_UNLESS(estimatedBlockCountMin >= 0, ResultAlbumInvalidFileData());
        NN_CAPSRV_LOG_MOVIE_DEV("  data block count min = %lld\n", estimatedBlockCountMin);

        // 真のブロック数は [min, max] の範囲にある
        int64_t dataSize = 0;
        int64_t blockCount = -1;
        for(int64_t c = estimatedBlockCountMin; c <= estimatedBlockCountMax; c++)
        {
            int64_t estimatedDataSize = variableSize - MovieDataBlockHashSize * c;
            int64_t estimatedBlockCount = (estimatedDataSize + MovieDataBlockSize - 1) / MovieDataBlockSize;
            if(estimatedBlockCount == c)
            {
                dataSize = estimatedDataSize;
                blockCount = c;
                break;
            }
        }

        // 丁度一致しなければファイルサイズが不正
        NN_RESULT_THROW_UNLESS(blockCount >= 0, ResultAlbumInvalidFileData());
        NN_CAPSRV_LOG_MOVIE_DEV("  data block count = %lld\n", blockCount);
        NN_CAPSRV_LOG_MOVIE_DEV("  -> success\n");

        *pOutDataBlockCount = blockCount;
        *pOutDataSize = dataSize;
        *pOutMetaSize = MetaSize;
        *pOutDataHashSize = MovieDataBlockHashSize * blockCount;
        *pOutMetaHashSize = MovieDataBlockHashSize * MetaBlockCount;
        *pOutSignSize = SignSize;
        *pOutVersionSize = VersionSize;
        NN_RESULT_SUCCESS;
    }

    //-------------------------------------------------------------------------------------------

    namespace {

        void CalculateFileSignForTest(AlbumMovieUtility::FileSign* pOut, const void* data, size_t dataSize) NN_NOEXCEPT
        {
            *pOut = {};
            std::memcpy(pOut, data, std::min(dataSize, sizeof(pOut)));
        }

        void (*g_CalculateFileSign)(AlbumMovieUtility::FileSign* pOut, const void* data, size_t dataSize) = CalculateFileSignForTest;

    }

    void AlbumMovieUtility::SetCalculateFileSign(void (*f)(FileSign* pOut, const void* data, size_t dataSize)) NN_NOEXCEPT
    {
        g_CalculateFileSign = f;
    }

    nn::Result AlbumMovieUtility::StartHashValidation(
        HashSectionValidationState* pOutState,
        uint64_t untrustedVersion
    ) NN_NOEXCEPT
    {
        NN_CAPSRV_LOG_MOVIE_DEV("starting hash validation(ver=%llu)\n", untrustedVersion);

        // バージョンを確認
        NN_RESULT_THROW_UNLESS(untrustedVersion == nn::capsrv::movie::MovieMetaDataVersion_1, ResultAlbumInvalidFileData());

        NN_CAPSRV_LOG_MOVIE_DEV("  -> success\n");
        pOutState->hasher.Initialize();
        pOutState->untrustedVersion = untrustedVersion;
        NN_RESULT_SUCCESS;
    }

    void AlbumMovieUtility::ProcessHashValidation(
        HashSectionValidationState& state,
        const BlockHash* pHashList,
        int64_t hashCount
    ) NN_NOEXCEPT
    {
        NN_CAPSRV_LOG_MOVIE_DEV("processing hash validation(count=%lld)\n", hashCount);
        NN_SDK_REQUIRES_NOT_NULL(pHashList);
        NN_SDK_REQUIRES_GREATER_EQUAL(hashCount, 0);
        state.hasher.Update(pHashList, hashCount * sizeof(BlockHash));
    }

    nn::Result AlbumMovieUtility::FinishHashValidation(
        HashSectionValidationState& state,
        const FileSign& sign
    ) NN_NOEXCEPT
    {
        NN_CAPSRV_LOG_MOVIE_DEV("finishing hash validation\n");

        char hash[decltype(state.hasher)::HashSize];
        state.hasher.GetHash(hash, sizeof(hash));

        NN_SDK_REQUIRES(g_CalculateFileSign);
        FileSign calculatedSign;
        g_CalculateFileSign(&calculatedSign, hash, sizeof(hash));

#ifdef NN_CAPSRV_DEBUG_ENABLE_FILE_SIGNATURE_FOR_UNIT_TEST
        NN_STATIC_ASSERT(sizeof(hash) == sizeof(sign));
        if(0 == std::memcmp(hash, &sign, sizeof(hash)))
        {
            NN_CAPSRV_LOG_WARN("movie signature for testing enabled!!\n");
            NN_RESULT_SUCCESS;
        }
#endif

        NN_RESULT_THROW_UNLESS(0 == std::memcmp(&calculatedSign, &sign, sizeof(FileSign)), ResultInternalFileDataVerificationInconsistentSignature());

        NN_CAPSRV_LOG_MOVIE_DEV("  -> success\n");
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumMovieUtility::StartFileSignGeneration(
        FileSignGenerationState* pState
    ) NN_NOEXCEPT
    {
        NN_CAPSRV_LOG_MOVIE_DEV("starting file sign generation\n");

        pState->hasher.Initialize();
        NN_RESULT_SUCCESS;
    }

    void AlbumMovieUtility::ProcessFileSignGeneration(
        FileSignGenerationState& state,
        const BlockHash* pHashList,
        int64_t hashCount
    )
    {
        NN_CAPSRV_LOG_MOVIE_DEV("processing file sign generation(count=%lld)\n", hashCount);
        NN_SDK_REQUIRES_NOT_NULL(pHashList);
        NN_SDK_REQUIRES_GREATER_EQUAL(hashCount, 0);
        state.hasher.Update(pHashList, hashCount * sizeof(BlockHash));
    }

    nn::Result AlbumMovieUtility::FinishFileSignGeneration(
        FileSign* pOut,
        FileSignGenerationState& state
    )
    {
        NN_CAPSRV_LOG_MOVIE_DEV("finishing hash validation\n");

        char hash[decltype(state.hasher)::HashSize];
        state.hasher.GetHash(hash, sizeof(hash));

        NN_SDK_REQUIRES(g_CalculateFileSign);
        FileSign sign;
        g_CalculateFileSign(&sign, hash, sizeof(hash));

        *pOut = sign;
        NN_RESULT_SUCCESS;
    }

    //-------------------------------------------------------------------------------------------

    nn::Result AlbumMovieUtility::ValidateBlockData(
        const void* pBlock,
        size_t blockSize,
        const BlockHash& expectedHash
    ) NN_NOEXCEPT
    {
        NN_CAPSRV_LOG_MOVIE_DEV("validating block data\n");
        NN_SDK_REQUIRES_NOT_NULL(pBlock);
        NN_SDK_REQUIRES_EQUAL(blockSize, MovieDataBlockSize);

        BlockHash calculatedHash = {};
        nn::crypto::GenerateSha256Hash(&calculatedHash, sizeof(calculatedHash), pBlock, blockSize);

        NN_CAPSRV_LOG_MOVIE_DEV("  comparing block hash\n");
        NN_CAPSRV_LOG_MOVIE_DEV("    calc: %02X%02X%02X%02X%02X%02X%02X%02X...\n",
            calculatedHash.value[0], calculatedHash.value[1], calculatedHash.value[2], calculatedHash.value[3],
            calculatedHash.value[4], calculatedHash.value[5], calculatedHash.value[6], calculatedHash.value[7]);
        NN_CAPSRV_LOG_MOVIE_DEV("    expt: %02X%02X%02X%02X%02X%02X%02X%02X...\n",
            expectedHash.value[0], expectedHash.value[1], expectedHash.value[2], expectedHash.value[3],
            expectedHash.value[4], expectedHash.value[5], expectedHash.value[6], expectedHash.value[7]);

        NN_RESULT_THROW_UNLESS(0 == std::memcmp(&calculatedHash, &expectedHash, sizeof(BlockHash)), ResultInternalFileDataVerificationInconsistentBlockHash());

        NN_CAPSRV_LOG_MOVIE_DEV("  -> success\n");
        NN_RESULT_SUCCESS;
    }

    //-------------------------------------------------------------------------------------------

    nn::Result AlbumMovieUtility::ExtractMetaInfo(
        AlbumMovieMetaInfo* pOutValue,
        const nn::capsrv::movie::MovieMetaInfoData* pMetaInfoData,
        uint64_t untrustedVersion,
        int64_t dataSectionSize
    ) NN_NOEXCEPT
    {
        NN_CAPSRV_LOG_MOVIE_DEV("extracting meta info\n");
        NN_SDK_REQUIRES_NOT_NULL(pMetaInfoData);

        // バージョンを確認
        NN_RESULT_THROW_UNLESS(untrustedVersion == nn::capsrv::movie::MovieMetaDataVersion_1, ResultAlbumInvalidFileData());

        // メタ中のバージョンを読み込み
        uint64_t version = {};
        std::memcpy(&version, &pMetaInfoData->version, sizeof(version));
        NN_CAPSRV_LOG_MOVIE_DEV("  checking version(meta=%llu, tail=%llu)\n", version, untrustedVersion);
        NN_RESULT_THROW_UNLESS(version == untrustedVersion, ResultAlbumInvalidFileData());

        // 予約領域は 0 埋めになっている
        NN_CAPSRV_LOG_MOVIE_DEV("  checking reserved region is zero\n");
        {
            bool isOk = true;
            NN_STATIC_ASSERT(sizeof(pMetaInfoData->_reserved) % sizeof(uint64_t) == 0);
            const uint64_t zero = 0;
            auto p = reinterpret_cast<const char*>(&pMetaInfoData->_reserved[0]);
            auto pEnd = p + sizeof(pMetaInfoData->_reserved);
            for(; p < pEnd; p += sizeof(zero))
            {
                if(std::memcmp(p, &zero, sizeof(zero)) != 0)
                {
                    isOk = false;
                    break;
                }
            }
            NN_RESULT_THROW_UNLESS(isOk, ResultAlbumInvalidFileData());
        }

        // その他の情報を取得
        uint64_t movieDataSize = 0;
        uint64_t imageDataSize = 0;
        std::memcpy(&movieDataSize, &pMetaInfoData->movieDataSize, sizeof(movieDataSize));
        std::memcpy(&imageDataSize, &pMetaInfoData->imageDataSize, sizeof(imageDataSize));

        int64_t dataBlockCount = (dataSectionSize + MovieDataBlockSize - 1) / MovieDataBlockSize;

        // 妥当性確認
        int64_t capableMovieDataSizeMin = MovieDataBlockSize * (dataBlockCount - 1) + 1;
        int64_t capableMovieDataSizeMax = dataSectionSize;
        NN_CAPSRV_LOG_MOVIE_DEV("  checking movie-data-size(%lld in [%lld-%lld]\n", movieDataSize, capableMovieDataSizeMin, capableMovieDataSizeMax);
        NN_RESULT_THROW_UNLESS(movieDataSize >= capableMovieDataSizeMin, ResultAlbumInvalidFileData());
        NN_RESULT_THROW_UNLESS(movieDataSize <= capableMovieDataSizeMax, ResultAlbumInvalidFileData());

        int64_t capableImageDataSizeMin = 0; // FIXME: 最小サイズ
        int64_t capableImageDataSizeMax = ScreenShotImageSizeLimit;
        NN_CAPSRV_LOG_MOVIE_DEV("  checking image-data-size(%lld in [%lld-%lld]\n", imageDataSize, capableImageDataSizeMin, capableImageDataSizeMax);
        NN_RESULT_THROW_UNLESS(imageDataSize >= capableImageDataSizeMin, ResultAlbumInvalidFileData());
        NN_RESULT_THROW_UNLESS(imageDataSize <= capableImageDataSizeMax, ResultAlbumInvalidFileData());

        NN_CAPSRV_LOG_MOVIE_DEV("  -> success\n");
        *pOutValue = AlbumMovieMetaInfo(movieDataSize, imageDataSize, version);
        NN_RESULT_SUCCESS;
    }

    //-------------------------------------------------------------------------------------------

    void AlbumMovieUtilityStaticObject::Initialize(void* workMemory, size_t workMemorySize) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(workMemory);
        NN_SDK_REQUIRES_GREATER_EQUAL(workMemorySize, MovieDataBlockSize);

        std::memset(workMemory, 0, workMemorySize);

        // ゼロ埋めされたデータに対するハッシュ
        nn::crypto::GenerateSha256Hash(&m_ZeroBlockHash, sizeof(m_ZeroBlockHash), workMemory, MovieDataBlockSize);
    };

    void AlbumMovieUtilityStaticObject::Finalize() NN_NOEXCEPT
    {
    }

    AlbumMovieUtility::BlockHash AlbumMovieUtilityStaticObject::GetZeroBlockHash() const NN_NOEXCEPT
    {
        return m_ZeroBlockHash;
    }

}}}}
