﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "capsrvServer_AlbumManager.h"

#include <algorithm>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include "../detail/capsrvServer_AlbumContentsAttribute.h"
#include "capsrvServer_AlbumPathUtility.h"
#include "capsrvServer_AlbumFileManipulator.h"
#include "capsrvServer_VerifyScreenShotFileData.h"
#include "capsrvServer_ExtractJpegThumbnail.h"

#include <nn/capsrv/capsrv_AlbumFileCountLimit.h>
#include <nn/capsrv/movie/capsrv_MovieMetaData.h>
#include "capsrvServer_AlbumManagerCommon.h"
#include "capsrvServer_AlbumMovieUtility.h"
#include "capsrvServer_VerifyMovieFileData.h"

namespace nn{ namespace capsrv{ namespace server{ namespace album{

    //-------------------------------------------------------------------------------
    // 単発
    //-------------------------------------------------------------------------------

    nn::Result AlbumManager::LoadMovieImageFileImpl(
        AlbumFileAttribute* pOutAttribute,
        AppletData* pOutAppletData,
        ApplicationData* pOutApplicationData,
        SystemReservedInfo* pOutSystemReservedInfo,
        size_t* pOutFileSize,
        void* pBuffer,
        size_t bufferSize,
        const AlbumFileId& fileId,
        void* workMemory,
        size_t workSize
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutFileSize);
        NN_SDK_REQUIRES(fileId.contents == AlbumFileContents_Movie ||
                        fileId.contents == AlbumFileContents_ExtraMovie);
        NN_SDK_REQUIRES_NOT_NULL(workMemory);
        NN_SDK_REQUIRES_GREATER_EQUAL(workSize, AlbumManagerMovieValidationWorkBufferSize);
        NN_RESULT_THROW_UNLESS(workSize >= AlbumManagerMovieValidationWorkBufferSize, nn::capsrv::ResultAlbumWorkMemoryError());

        NN_CAPSRV_PROCESS_START();

        NN_CAPSRV_CREATE_FILEPATH(filepath, fileId, AlbumStorageDirection_Source);
        NN_CAPSRV_OPEN_FILE_TO_READ(hFile, filepath);
        NN_CAPSRV_GET_FILESIZE_TO_READ(fileSize, fileId, hFile);

        // ファイルを検証しながら画像データを読み込み
        int64_t imageDataSize = 0;
        size_t readSize = 0;
        NN_RESULT_DO(VerifyMovieFileAndLoadImage(
            &readSize,
            &imageDataSize,
            pBuffer,
            bufferSize,
            hFile,
            fileSize,
            *m_pEnvironmentInfo,
            workMemory,
            workSize
        ));

        // 画像データがバッファに収まっていなければエラー
        NN_RESULT_THROW_UNLESS(imageDataSize == readSize, ResultAlbumReadBufferShortage());

        // 画像データとしての検査
        AlbumFileAttribute attribute = {};
        // 画像データの内容が正当か検査
        NN_RESULT_DO(VerifyScreenShotFileDataEx(
            &attribute.attribute.screenshot,
            pOutAppletData,
            pOutApplicationData,
            pOutSystemReservedInfo,
            &fileId,
            pBuffer,
            static_cast<size_t>(imageDataSize),
            *m_pEnvironmentInfo
        ));
        attribute.content = fileId.contents;

        NN_CAPSRV_PROCESS_SUCCESS();
        if(pOutAttribute)
        {
            *pOutAttribute = attribute;
        }
        *pOutFileSize = static_cast<size_t>(imageDataSize);
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumManager::LoadMovieThumbnailFileImpl(
        AlbumFileAttribute* pOutAttribute,
        AppletData* pOutAppletData,
        ApplicationData* pOutApplicationData,
        SystemReservedInfo* pOutSystemReservedInfo,
        size_t* pOutSize,
        void* pBuffer,
        size_t bufferSize,
        const AlbumFileId& fileId,
        void* workMemory,
        size_t workSize
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutSize);
        NN_SDK_REQUIRES(fileId.contents == AlbumFileContents_Movie ||
                        fileId.contents == AlbumFileContents_ExtraMovie);
        NN_SDK_REQUIRES_NOT_NULL(workMemory);
        NN_SDK_REQUIRES_GREATER_EQUAL(workSize, AlbumManagerMovieValidationWorkBufferSize + AlbumManagerThumbnailLoadSize);
        NN_RESULT_THROW_UNLESS(workSize >= AlbumManagerMovieValidationWorkBufferSize + AlbumManagerThumbnailLoadSize, nn::capsrv::ResultAlbumWorkMemoryError());

        NN_CAPSRV_PROCESS_START();

        NN_CAPSRV_CREATE_FILEPATH(filepath, fileId, AlbumStorageDirection_Source);
        NN_CAPSRV_OPEN_FILE_TO_READ(hFile, filepath);
        NN_CAPSRV_GET_FILESIZE_TO_READ(fileSize, fileId, hFile);


        // ファイルを検証しながら画像データの先頭を読み込み
        int64_t imageDataSize = 0;
        size_t readHeadSize = 0;
        void* pImageDataHead = workMemory;
        {
            auto pValidationWorkMemory = reinterpret_cast<char*>(workMemory) + AlbumManagerThumbnailLoadSize;
            NN_RESULT_DO(VerifyMovieFileAndLoadImage(
                &readHeadSize,
                &imageDataSize,
                pImageDataHead,
                AlbumManagerThumbnailLoadSize,
                hFile,
                fileSize,
                *m_pEnvironmentInfo,
                pValidationWorkMemory,
                AlbumManagerMovieValidationWorkBufferSize
            ));
            NN_SDK_ASSERT(readHeadSize == imageDataSize || readHeadSize == AlbumManagerThumbnailLoadSize);
        }


        AlbumFileAttribute attribute = {};
        size_t thumbSize = 0;

        // ファイルの検査と属性の取得
        NN_RESULT_DO(VerifyScreenShotFileHeaderEx(&attribute.attribute.screenshot, pOutAppletData, pOutApplicationData, pOutSystemReservedInfo, &fileId, pImageDataHead, readHeadSize, *m_pEnvironmentInfo));
        attribute.content = fileId.contents;

        // サムネイル部分を抽出
        NN_RESULT_DO(ExtractJpegThumbnail(&thumbSize, pBuffer, bufferSize, pImageDataHead, readHeadSize));

        NN_CAPSRV_PROCESS_SUCCESS();
        if(pOutAttribute)
        {
            *pOutAttribute = attribute;
        }
        *pOutSize = thumbSize;
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumManager::VerifyAndLoadMovieMakerNoteInfo(
        uint64_t* pOutFileSize,
        nn::capsrv::server::detail::MakerNoteInfo* pOutMakerNoteInfo,
        const AlbumFileId& fileId,
        void* pWorkBuffer,
        size_t workBufferSize
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(fileId.contents == AlbumFileContents_Movie ||
                        fileId.contents == AlbumFileContents_ExtraMovie);
        NN_SDK_REQUIRES_GREATER_EQUAL(workBufferSize, AlbumManagerMovieValidationWorkBufferSize);
        NN_RESULT_THROW_UNLESS(workBufferSize >= AlbumManagerMovieValidationWorkBufferSize, nn::capsrv::ResultAlbumWorkMemoryError());

        NN_CAPSRV_PROCESS_START();
        NN_CAPSRV_CREATE_FILEPATH(filepath, fileId, AlbumStorageDirection_Source);
        NN_CAPSRV_OPEN_FILE_TO_READ(hFile, filepath);
        NN_CAPSRV_GET_FILESIZE_TO_READ(fileSize, fileId, hFile);

        // ファイルを検証しながら画像データの先頭を読み込み
        int64_t imageDataSize = 0;
        {
            size_t readHeadSize = 0;
            void* pImageDataHead = pWorkBuffer;
            auto pValidationWorkMemory = reinterpret_cast<char*>(pWorkBuffer) + AlbumManagerThumbnailLoadSize;
            NN_RESULT_DO(VerifyMovieFileAndLoadImage(
                &readHeadSize,
                &imageDataSize,
                pImageDataHead,
                AlbumManagerThumbnailLoadSize,
                hFile,
                fileSize,
                *m_pEnvironmentInfo,
                pValidationWorkMemory,
                AlbumManagerMovieValidationWorkBufferSize
            ));
            NN_RESULT_THROW_UNLESS(readHeadSize == imageDataSize || readHeadSize == AlbumManagerThumbnailLoadSize, ResultAlbumReadBufferShortage());
        }

        // 画像データの内容が正当か検査
        NN_RESULT_DO(VerifyAndLoadScreenShotMakerNoteInfoImpl(
            pOutMakerNoteInfo,
            &fileId,
            pWorkBuffer,
            static_cast<size_t>(fileSize),
            *m_pEnvironmentInfo,
            FileDataRange_HeaderOnly
        ));

        NN_CAPSRV_PROCESS_SUCCESS();
        *pOutFileSize = static_cast<size_t>(imageDataSize);
        NN_RESULT_SUCCESS;
    }


    //-------------------------------------------------------------------------------
    // ReadStream
    //-------------------------------------------------------------------------------

    nn::Result AlbumManager::OpenMovieReadStream(MovieStreamId* pOutStreamId, const AlbumFileId& fileId, SessionId sessionId) NN_NOEXCEPT
    {
        NN_CAPSRV_ALBUMMANAGER_REQUIRES_AND_CLEAR_WORKMEMORY();
        NN_CAPSRV_CHECK_FILE_NOT_WRITE_LOCKED(fileId, ResultAlbumFileNotFound());
        NN_RESULT_DO(AlbumPathUtility::ValidateFileId(&fileId, *m_pEnvironmentInfo));
        NN_RESULT_DO(EnsureMounted(fileId.storage));
        return m_MovieStreamManager.OpenReadStream(pOutStreamId, fileId, sessionId, m_Memory, m_MemorySize);
    }

    nn::Result AlbumManager::CloseMovieReadStream(MovieStreamId streamId) NN_NOEXCEPT
    {
        return m_MovieStreamManager.CloseReadStream(streamId);
    }

    nn::Result AlbumManager::ReadMovieDataFromMovieReadStream(size_t* pOutReadSize, void* buffer, size_t size, MovieStreamId streamId, int64_t offset) NN_NOEXCEPT
    {
        return m_MovieStreamManager.ReadMovieDataFromReadStream(pOutReadSize, buffer, size, streamId, offset);
    }

    nn::Result AlbumManager::GetMovieReadStreamMovieDataSize(int64_t* pOutValue, MovieStreamId streamId) NN_NOEXCEPT
    {
        return m_MovieStreamManager.GetReadStreamMovieDataSize(pOutValue, streamId);
    }

    nn::Result AlbumManager::ReadImageDataFromMovieReadStream(size_t* pOutReadSize, void* buffer, size_t size, MovieStreamId streamId, int64_t offset) NN_NOEXCEPT
    {
        NN_CAPSRV_ALBUMMANAGER_REQUIRES_AND_CLEAR_WORKMEMORY();
        return m_MovieStreamManager.ReadImageDataFromReadStream(pOutReadSize, buffer, size, streamId, offset, m_Memory, m_MemorySize);
    }

    nn::Result AlbumManager::GetMovieReadStreamImageDataSize(int64_t* pOutValue, MovieStreamId streamId) NN_NOEXCEPT
    {
        return m_MovieStreamManager.GetReadStreamImageDataSize(pOutValue, streamId);
    }

    nn::Result AlbumManager::GetMovieReadStreamBrokenReason(nn::Result* pOutValue, MovieStreamId streamId) NN_NOEXCEPT
    {
        return m_MovieStreamManager.GetReadStreamBrokenReason(pOutValue, streamId);
    }

    nn::Result AlbumManager::ReadFileAttributeFromMovieReadStream(AlbumFileAttribute* pOutAttribute, AppletData* pOutAppletData, ApplicationData* pOutApplicationData, MovieStreamId streamId) NN_NOEXCEPT
    {
        NN_CAPSRV_ALBUMMANAGER_REQUIRES_AND_CLEAR_WORKMEMORY();
        NN_SDK_REQUIRES_GREATER_EQUAL(m_MemorySize, MovieDataBlockSize + AlbumManagerThumbnailLoadSize);
        NN_RESULT_THROW_UNLESS(m_MemorySize >= MovieDataBlockSize + AlbumManagerThumbnailLoadSize, ResultAlbumWorkMemoryError());

        NN_CAPSRV_PROCESS_START();

        char* pBlockBuffer = reinterpret_cast<char*>(m_Memory);
        size_t blockBufferSize = MovieDataBlockSize;
        char* pHeadBuffer = reinterpret_cast<char*>(m_Memory) + MovieDataBlockSize;
        size_t headBufferSize = AlbumManagerThumbnailLoadSize;

        AlbumFileId fileId = {};
        NN_RESULT_DO(m_MovieStreamManager.GetReadStreamFileId(&fileId, streamId));

        // 画像の先頭部分を読み込み
        size_t readSize = 0;
        NN_RESULT_DO(m_MovieStreamManager.ReadImageDataFromReadStream(&readSize, pHeadBuffer, headBufferSize, streamId, 0, pBlockBuffer, blockBufferSize));

        AlbumFileAttribute attribute = {};
        // ファイルの検査と属性の取得
        NN_RESULT_DO(VerifyScreenShotFileHeaderEx(&attribute.attribute.screenshot, pOutAppletData, pOutApplicationData, nullptr, &fileId, pHeadBuffer, readSize, *m_pEnvironmentInfo));
        attribute.content = fileId.contents;

        NN_CAPSRV_PROCESS_SUCCESS();
        if(pOutAttribute)
        {
            *pOutAttribute = attribute;
        }
        NN_RESULT_SUCCESS;
    }

    //-------------------------------------------------------------------------------
    // WriteStream
    //-------------------------------------------------------------------------------

    nn::Result AlbumManager::OpenMovieWriteStream(MovieStreamId* pOutStreamId, const AlbumFileId& fileId, SessionId sessionId) NN_NOEXCEPT
    {
        NN_CAPSRV_CHECK_FILE_NOT_READ_LOCKED(fileId, ResultAlbumDestinationAccessCorrupted());
        NN_CAPSRV_CHECK_FILE_NOT_WRITE_LOCKED(fileId, ResultAlbumDestinationAccessCorrupted());
        NN_RESULT_DO(AlbumPathUtility::ValidateFileId(&fileId, *m_pEnvironmentInfo));
        NN_RESULT_DO(EnsureMounted(fileId.storage));
        NN_RESULT_DO(CheckAlbumLimitation(fileId.storage, fileId.contents, 1));

        MovieStreamId streamId = {};
        AlbumCacheDelta delta = {};
        NN_RESULT_DO(m_MovieStreamManager.OpenWriteStream(&streamId, &delta, fileId, sessionId));
        m_AlbumStorageCache.UpdateEntry(delta);

        *pOutStreamId = streamId;
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumManager::StartMovieWriteStreamDataSection(MovieStreamId streamId) NN_NOEXCEPT
    {
        return m_MovieStreamManager.StartWriteStreamDataSection(streamId);
    }

    nn::Result AlbumManager::EndMovieWriteStreamDataSection(MovieStreamId streamId) NN_NOEXCEPT
    {
        return m_MovieStreamManager.EndWriteStreamDataSection(streamId);
    }

    nn::Result AlbumManager::StartMovieWriteStreamMetaSection(MovieStreamId streamId) NN_NOEXCEPT
    {
        return m_MovieStreamManager.StartWriteStreamMetaSection(streamId);
    }

    nn::Result AlbumManager::EndMovieWriteStreamMetaSection(MovieStreamId streamId) NN_NOEXCEPT
    {
        return m_MovieStreamManager.EndWriteStreamMetaSection(streamId);
    }

    nn::Result AlbumManager::FinishMovieWriteStream(MovieStreamId streamId) NN_NOEXCEPT
    {
        NN_CAPSRV_ALBUMMANAGER_REQUIRES_AND_CLEAR_WORKMEMORY();
        return m_MovieStreamManager.FinishWriteStream(streamId, m_Memory, m_MemorySize);
    }

    nn::Result AlbumManager::CommitMovieWriteStream(MovieStreamId streamId) NN_NOEXCEPT
    {
        return m_MovieStreamManager.CommitWriteStream(streamId);
    }

    nn::Result AlbumManager::DiscardMovieWriteStream(MovieStreamId streamId) NN_NOEXCEPT
    {
        AlbumCacheDelta delta = {};
        NN_RESULT_DO(m_MovieStreamManager.DiscardWriteStream(&delta, streamId, false /*isDeleteSuppressed*/));
        m_AlbumStorageCache.UpdateEntry(delta);
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumManager::DiscardMovieWriteStreamNoDelete(MovieStreamId streamId) NN_NOEXCEPT
    {
        AlbumCacheDelta delta = {};
        NN_RESULT_DO(m_MovieStreamManager.DiscardWriteStream(&delta, streamId, true /*isDeleteSuppressed*/));
        m_AlbumStorageCache.UpdateEntry(delta);
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumManager::ReadDataFromMovieWriteStream(size_t* pOutReadSize, void* buffer, size_t size, MovieStreamId streamId, int64_t offset) NN_NOEXCEPT
    {
        return m_MovieStreamManager.ReadDataFromWriteStream(pOutReadSize, buffer, size, streamId, offset);
    }

    nn::Result AlbumManager::WriteDataToMovieWriteStream(MovieStreamId streamId, int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_RESULT_DO(m_MovieStreamManager.WriteDataToWriteStream(streamId, offset, buffer, size));
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumManager::GetMovieWriteStreamDataSize(int64_t* pOutValue, MovieStreamId streamId) NN_NOEXCEPT
    {
        return m_MovieStreamManager.GetWriteStreamDataSize(pOutValue, streamId);
    }

    nn::Result AlbumManager::SetMovieWriteStreamDataSize(MovieStreamId streamId, int64_t size) NN_NOEXCEPT
    {
        NN_RESULT_DO(m_MovieStreamManager.SetWriteStreamDataSize(streamId, size));
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumManager::WriteMetaToMovieWriteStream(MovieStreamId streamId, const void* buffer, size_t size, uint64_t makerNoteVersion, int64_t makerNoteOffset, int64_t makerNoteSize) NN_NOEXCEPT
    {
        NN_CAPSRV_ALBUMMANAGER_REQUIRES_AND_CLEAR_WORKMEMORY();
        NN_RESULT_DO(m_MovieStreamManager.WriteMetaToWriteStream(streamId, buffer, size, makerNoteVersion, makerNoteOffset, makerNoteSize, m_Memory, m_MemorySize));
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumManager::GetMovieWriteStreamBrokenReason(nn::Result* pOutValue, MovieStreamId streamId) NN_NOEXCEPT
    {
        return m_MovieStreamManager.GetWriteStreamBrokenReason(pOutValue, streamId);
    }

}}}}
