﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/capsrv/movie/capsrv_MovieWriterFileSystem.h>

#include <algorithm>
#include <nn/nn_SdkLog.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_Optional.h>
#include <nn/fs/fsa/fs_Registrar.h>
#include <nn/capsrv/capsrv_AlbumControl.h>
#include <nn/capsrv/capsrv_AlbumFileSizeLimit.h>

//#define NN_CAPSRV_LOG_MOVIEFS_HEAD()    NN_SDK_LOG("[capsrv]")
//#define NN_CAPSRV_LOG_MOVIEFS_DATA(...) NN_SDK_LOG(__VA_ARGS__)

#ifndef NN_CAPSRV_LOG_MOVIEFS_HEAD
#define NN_CAPSRV_LOG_MOVIEFS_HEAD()
#endif
#ifndef NN_CAPSRV_LOG_MOVIEFS_DATA
#define NN_CAPSRV_LOG_MOVIEFS_DATA(...)
#endif

namespace {
    int g_CallDepth = 0;
}

#define NN_CAPSRV_LOG_MOVIEFS_I(...) \
    do{                                 \
        NN_CAPSRV_LOG_MOVIEFS_HEAD();   \
        for(int i = 0; i < g_CallDepth; i++)    \
        {                                       \
            NN_CAPSRV_LOG_MOVIEFS_DATA("  ");        \
        }                                       \
        NN_CAPSRV_LOG_MOVIEFS_DATA(__VA_ARGS__);     \
    }while(NN_STATIC_CONDITION(false));

#define NN_CAPSRV_LOG_MOVIEFS_II(...)       \
    NN_CAPSRV_LOG_MOVIEFS_I(__VA_ARGS__)    \
    g_CallDepth++;                          \
    NN_UTIL_SCOPE_EXIT{ g_CallDepth--; };

#define NN_CAPSRV_LOG_MOVIEFS_LINE(...)  NN_CAPSRV_LOG_MOVIEFS_I(__VA_ARGS__)

#define NN_CAPSRV_LOG_MOVIEFS_MWFS(...)  NN_CAPSRV_LOG_MOVIEFS_II("MWFS:" __VA_ARGS__)
#define NN_CAPSRV_LOG_MOVIEFS_MWFSI(...) NN_CAPSRV_LOG_MOVIEFS_II("MWFSI:" __VA_ARGS__)
#define NN_CAPSRV_LOG_MOVIEFS_MWFSP(...) NN_CAPSRV_LOG_MOVIEFS_II("MWFSP:" __VA_ARGS__)
#define NN_CAPSRV_LOG_MOVIEFS_MWFI(...)  NN_CAPSRV_LOG_MOVIEFS_II("MWFI:" __VA_ARGS__)
#define NN_CAPSRV_LOG_MOVIEFS_MWFP(...)  NN_CAPSRV_LOG_MOVIEFS_II("MWFP:" __VA_ARGS__)

namespace nn{ namespace capsrv{ namespace movie{

    namespace detail{

        class MovieWriterFileImpl::ScopedFsPriorityRawChanger
        {
        public:
            explicit ScopedFsPriorityRawChanger(const util::optional<fs::PriorityRaw>& pFsPriorityRaw) NN_NOEXCEPT
            {
                if (pFsPriorityRaw)
                {
                    this->m_pSavedValue = fs::GetPriorityRawOnCurrentThread();
                    fs::SetPriorityRawOnCurrentThread(*pFsPriorityRaw);
                }
            }
            ~ScopedFsPriorityRawChanger() NN_NOEXCEPT
            {
                if (m_pSavedValue)
                {
                    fs::SetPriorityRawOnCurrentThread(*m_pSavedValue);
                }
            }
        private:
            util::optional<fs::PriorityRaw> m_pSavedValue = util::nullopt;
        };

        size_t MovieWriterFileImpl::GetRequiredMemorySize(int64_t cacheChunkSize, int64_t cacheChunkCount) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_GREATER(cacheChunkSize, 0);
            NN_SDK_REQUIRES_EQUAL(cacheChunkSize % AlbumMovieDataUnitSize, 0);
            NN_SDK_REQUIRES_GREATER(cacheChunkCount, 0);

            return CachedMovieStream::GetRequiredWorkMemorySize(cacheChunkSize, cacheChunkCount);
        }

        size_t MovieWriterFileImpl::GetRequiredMemoryAlignment() NN_NOEXCEPT
        {
            return CachedMovieStream::GetRequiredWorkMemoryAlignment();
        }

        MovieWriterFileImpl::MovieWriterFileImpl() NN_NOEXCEPT
        {
            m_Accessor.handle = {};
        }

        bool MovieWriterFileImpl::IsInitialized() const NN_NOEXCEPT
        {
            return m_Accessor.handle != AlbumMovieWriteStreamHandle();
        }

        void MovieWriterFileImpl::Initialize(
            const FilePath& filename,
            AlbumMovieWriteStreamHandle handle,
            int64_t cacheChunkSize,
            int64_t cacheChunkCount,
            void* memory,
            size_t memorySize,
            util::optional<fs::PriorityRaw> pFsPriorityRaw
        ) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MWFI("Initialize()\n");
            NN_SDK_REQUIRES_NOT_EQUAL(handle, AlbumMovieWriteStreamHandle());
            NN_SDK_REQUIRES_GREATER(cacheChunkSize, 0);
            NN_SDK_REQUIRES_EQUAL(cacheChunkSize % AlbumMovieDataUnitSize, 0);
            NN_SDK_REQUIRES_GREATER(cacheChunkCount, 0);
            NN_SDK_REQUIRES_NOT_NULL(memory);
            NN_SDK_REQUIRES_ALIGNED(memory, GetRequiredMemoryAlignment());
            NN_SDK_REQUIRES_GREATER_EQUAL(memorySize, GetRequiredMemorySize(cacheChunkSize, cacheChunkCount));
            NN_UNUSED(memorySize);

            m_Accessor.handle = handle;
            m_Strategy.Initialize(cacheChunkCount);
            m_Stream.Initialize(
                cacheChunkSize,
                cacheChunkCount,
                m_Accessor.GetAccessor(),
                m_Strategy.GetStrategy(),
                memory,
                memorySize,
                0 /* initialChunkCount */
            );
            m_Filename = filename;
            m_FileSize = 0;
            m_pFsPriorityRaw = pFsPriorityRaw;
        }

        void MovieWriterFileImpl::Finalize() NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MWFI("Finalize()\n");
            m_Stream.Finalize();
            m_Strategy.Finalize();
            m_Accessor.handle = AlbumMovieWriteStreamHandle();

            m_Filename = {};
            m_FileSize = 0;
        }

        bool MovieWriterFileImpl::IsMatchFilename(const FilePath& filename) const NN_NOEXCEPT
        {
            return m_Filename == filename;
        }

        AlbumMovieWriteStreamHandle MovieWriterFileImpl::GetHandle() const NN_NOEXCEPT
        {
            return m_Accessor.handle;
        }

        nn::Result MovieWriterFileImpl::DoReadImpl(size_t* outValue, int64_t offset, void* buffer, size_t size, const nn::fs::ReadOption& option) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MWFI("DoReadImpl()\n");
            NN_UNUSED(option);
            ScopedFsPriorityRawChanger scopedFsPriorityRawChanger{this->m_pFsPriorityRaw};
            return m_Stream.Read(outValue, buffer, size, offset);
        }

        nn::Result MovieWriterFileImpl::DoWriteImpl(int64_t offset, const void* buffer, size_t size, const nn::fs::WriteOption& option) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MWFI("DoWriteImpl()\n");
            NN_UNUSED(option);
            m_FileSize = std::max(m_FileSize, offset + static_cast<int64_t>(size));
            ScopedFsPriorityRawChanger scopedFsPriorityRawChanger{this->m_pFsPriorityRaw};
            return m_Stream.Write(offset, buffer,size);
        }

        nn::Result MovieWriterFileImpl::DoFlushImpl() NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MWFI("DoFlushImpl()\n");
            ScopedFsPriorityRawChanger scopedFsPriorityRawChanger{this->m_pFsPriorityRaw};
            return m_Stream.Flush();
        }

        nn::Result MovieWriterFileImpl::DoSetSizeImpl(int64_t size) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MWFI("DoSetSizeImpl()\n");
            m_FileSize = size;
            ScopedFsPriorityRawChanger scopedFsPriorityRawChanger{this->m_pFsPriorityRaw};
            return m_Stream.Resize(size);
        }

        nn::Result MovieWriterFileImpl::DoGetSizeImpl(int64_t* outValue) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MWFI("DoGetSizeImpl()\n");
            *outValue = m_FileSize;
            NN_RESULT_SUCCESS;
        }

        nn::Result MovieWriterFileImpl::DoOperateRangeImpl(
            void* outBuffer,
            size_t outBufferSize,
            fs::OperationId operationId,
            int64_t offset,
            int64_t size,
            const void* inBuffer,
            size_t inBufferSize
        ) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MWFI("DoOperateRangeImpl()\n");
            NN_UNUSED(outBuffer);
            NN_UNUSED(outBufferSize);
            NN_UNUSED(operationId);
            NN_UNUSED(offset);
            NN_UNUSED(size);
            NN_UNUSED(inBuffer);
            NN_UNUSED(inBufferSize);
            NN_RESULT_SUCCESS;
        }

        //-------------------------------------------------------

        MovieWriterFileProxy::MovieWriterFileProxy(MovieWriterFileImpl* pImpl) NN_NOEXCEPT
            : m_pImpl(pImpl)
        {
            NN_CAPSRV_LOG_MOVIEFS_MWFP("ctor(p=%llX)\n", pImpl);
            NN_SDK_REQUIRES_NOT_NULL(pImpl);
            NN_SDK_REQUIRES(pImpl->IsInitialized());
        }

        MovieWriterFileProxy::~MovieWriterFileProxy() NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MWFP("dtor(p=%llX)\n", m_pImpl);
            // Close
        }

        nn::Result MovieWriterFileProxy::DoRead(size_t* outValue, int64_t offset, void* buffer, size_t size, const nn::fs::ReadOption& option) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MWFP("DoRead()\n");
            return m_pImpl->DoReadImpl(outValue, offset, buffer, size, option);
        }

        nn::Result MovieWriterFileProxy::DoWrite(int64_t offset, const void* buffer, size_t size, const nn::fs::WriteOption& option) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MWFP("DoWrite()\n");
            return m_pImpl->DoWriteImpl(offset, buffer, size, option);
        }

        nn::Result MovieWriterFileProxy::DoFlush() NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MWFP("DoFlush()\n");
            return m_pImpl->DoFlushImpl();
        }

        nn::Result MovieWriterFileProxy::DoSetSize(int64_t size) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MWFP("DoSetSize()\n");
            return m_pImpl->DoSetSizeImpl(size);
        }

        nn::Result MovieWriterFileProxy::DoGetSize(int64_t* outValue) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MWFP("DoGetSize()\n");
            return m_pImpl->DoGetSizeImpl(outValue);
        }

        nn::Result MovieWriterFileProxy::DoOperateRange(
            void* outBuffer,
            size_t outBufferSize,
            fs::OperationId operationId,
            int64_t offset,
            int64_t size,
            const void* inBuffer,
            size_t inBufferSize
        ) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MWFP("DoOperateRange()\n");
            return m_pImpl->DoOperateRangeImpl(outBuffer, outBufferSize, operationId, offset, size, inBuffer, inBufferSize);
        }

        //-------------------------------------------------------

        const int MovieWriterFileSystemImpl::FileCountMax;

        size_t MovieWriterFileSystemImpl::GetRequiredMemorySizeForFile(int64_t cacheChunkSize, int64_t cacheChunkCount) NN_NOEXCEPT
        {
            return MovieWriterFileImpl::GetRequiredMemorySize(cacheChunkSize, cacheChunkCount);
        }

        size_t MovieWriterFileSystemImpl::GetRequiredMemoryAlignmentForFile() NN_NOEXCEPT
        {
            return MovieWriterFileImpl::GetRequiredMemoryAlignment();
        }

        MovieWriterFileSystemImpl::MovieWriterFileSystemImpl() NN_NOEXCEPT
        {
        }

        void MovieWriterFileSystemImpl::Initialize() NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MWFSI("Initialize()\n");
        }

        void MovieWriterFileSystemImpl::Finalize() NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MWFSI("Finalize()\n");
            for(int i = 0; i < FileCountMax; i++)
            {
                NN_SDK_REQUIRES(!m_File[i].IsInitialized());
            }
        }


        MovieWriterFileImpl* MovieWriterFileSystemImpl::FindFile(const char* filename) NN_NOEXCEPT
        {
            auto path = FilePath::FromString(filename);
            for(int i = 0; i < FileCountMax; i++)
            {
                if(m_File[i].IsInitialized() && m_File[i].IsMatchFilename(path))
                {
                    return &m_File[i];
                }
            }
            return nullptr;
        }

        void MovieWriterFileSystemImpl::AttachWriteStream(
            int index,
            const FilePath& filename,
            AlbumMovieWriteStreamHandle handle,
            int64_t cacheChunkSize,
            int64_t cacheChunkCount,
            void* memory,
            size_t memorySize,
            util::optional<fs::PriorityRaw> pFsPriorityRaw
        ) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MWFSI("AttachWriteStream(i=%d,name=%s,h=llu)\n", index, filename.value, handle.GetInnerValue());
            NN_SDK_REQUIRES_RANGE(index, 0, FileCountMax);
            auto& file = m_File[index];
            NN_ABORT_UNLESS(!file.IsInitialized());

            // 初期化
            file.Initialize(
                filename,
                handle,
                cacheChunkSize,
                cacheChunkCount,
                memory,
                memorySize,
                pFsPriorityRaw
            );
        }

        void MovieWriterFileSystemImpl::DetachWriteStream(int index) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MWFSI("DetachWriteStream(i=%d)\n", index);
            NN_SDK_REQUIRES_RANGE(index, 0, FileCountMax);
            auto& file = m_File[index];
            NN_ABORT_UNLESS(file.IsInitialized());

            file.Finalize();
        }

        nn::Result MovieWriterFileSystemImpl::DoGetEntryTypeImpl(nn::fs::DirectoryEntryType* outValue, const char* path) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MWFSI("DoGetEntryTypeImpl(path=%s)\n", path);
            auto pFile = FindFile(path);
            NN_RESULT_THROW_UNLESS(pFile != nullptr, nn::fs::ResultNotFound());

            *outValue = nn::fs::DirectoryEntryType_File;
            NN_RESULT_SUCCESS;
        }

        nn::Result MovieWriterFileSystemImpl::DoOpenFileImpl(std::unique_ptr<nn::fs::fsa::IFile>* outValue, const char* path, nn::fs::OpenMode mode) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MWFSI("DoOpenFileImpl(path=%s)\n", path);
            NN_UNUSED(mode);
            auto pFile = FindFile(path);
            NN_RESULT_THROW_UNLESS(pFile != nullptr, nn::fs::ResultNotFound());

            *outValue = std::unique_ptr<nn::fs::fsa::IFile>(new MovieWriterFileProxy(pFile));
            NN_RESULT_SUCCESS;
        }

        //-------------------------------------------------------

        MovieWriterFileSystemProxy::MovieWriterFileSystemProxy(MovieWriterFileSystemImpl* pImpl) NN_NOEXCEPT
            : m_pImpl(pImpl)
        {
            NN_SDK_REQUIRES_NOT_NULL(pImpl);
            NN_CAPSRV_LOG_MOVIEFS_MWFSP("ctor(p=%llX)\n", pImpl);
        }

        MovieWriterFileSystemProxy::~MovieWriterFileSystemProxy() NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MWFSP("dtor(p=%llX)\n", m_pImpl);
        }

        nn::Result MovieWriterFileSystemProxy::DoCreateFile(const char* path, int64_t size, int option) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MWFSP("DoCreateFile(path=%s)\n", path);
            NN_UNUSED(path);
            NN_UNUSED(size);
            NN_UNUSED(option);
            // ファイルの作成は nn::fs のインタフェースを使わない
            NN_RESULT_SUCCESS;
        }

        nn::Result MovieWriterFileSystemProxy::DoDeleteFile(const char* path) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MWFSP("DoDeleteFile(path=%s)\n", path);
            NN_UNUSED(path);
            // ファイルの削除は nn::fs のインタフェースを使わない
            NN_RESULT_SUCCESS;
        }

        nn::Result MovieWriterFileSystemProxy::DoCreateDirectory(const char* path) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MWFSP("DoCreateDirectory(path=%s)\n", path);
            NN_UNUSED(path);
            NN_RESULT_THROW(nn::fs::ResultNotImplemented());
        }

        nn::Result MovieWriterFileSystemProxy::DoDeleteDirectory(const char* path) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MWFSP("DoDeleteDirectory(path=%s)\n", path);
            NN_UNUSED(path);
            NN_RESULT_THROW(nn::fs::ResultNotImplemented());
        }

        nn::Result MovieWriterFileSystemProxy::DoDeleteDirectoryRecursively(const char* path) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MWFSP("DoDeleteDirectoryRecursively(path=%s)\n", path);
            NN_UNUSED(path);
            NN_RESULT_THROW(nn::fs::ResultNotImplemented());
        }

        nn::Result MovieWriterFileSystemProxy::DoCleanDirectoryRecursively(const char* path) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MWFSP("DoCleanDirectoryRecursively(path=%s)\n", path);
            NN_UNUSED(path);
            NN_RESULT_THROW(nn::fs::ResultNotImplemented());
        }

        nn::Result MovieWriterFileSystemProxy::DoRenameFile(const char* currentPath, const char* newPath) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MWFSP("DoRenameFile(curPath=%s,newPath=%s)\n", currentPath, newPath);
            NN_UNUSED(currentPath);
            NN_UNUSED(newPath);
            NN_RESULT_THROW(nn::fs::ResultNotImplemented());
        }

        nn::Result MovieWriterFileSystemProxy::DoRenameDirectory(const char* currentPath, const char* newPath) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MWFSP("DoRenameDirectory(curPath=%s,newPath=%s)\n", currentPath, newPath);
            NN_UNUSED(currentPath);
            NN_UNUSED(newPath);
            NN_RESULT_THROW(nn::fs::ResultNotImplemented());
        }

        nn::Result MovieWriterFileSystemProxy::DoGetEntryType(nn::fs::DirectoryEntryType* outValue, const char* path) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MWFSP("DoGetEntryType(path=%s)\n", path);
            return m_pImpl->DoGetEntryTypeImpl(outValue, path);
        }

        nn::Result MovieWriterFileSystemProxy::DoOpenFile(std::unique_ptr<nn::fs::fsa::IFile>* outValue, const char* path, nn::fs::OpenMode mode) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MWFSP("DoOpenFile(path=%s)\n", path);
            return m_pImpl->DoOpenFileImpl(outValue, path, mode);
        }

        nn::Result MovieWriterFileSystemProxy::DoOpenDirectory(std::unique_ptr<nn::fs::fsa::IDirectory>* outValue, const char* path, nn::fs::OpenDirectoryMode mode) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MWFSP("DoOpenDirectory(path=%s)\n", path);
            NN_UNUSED(outValue);
            NN_UNUSED(path);
            NN_UNUSED(mode);
            NN_RESULT_THROW(nn::fs::ResultNotImplemented());
        }

        nn::Result MovieWriterFileSystemProxy::DoCommit() NN_NOEXCEPT
        {
            NN_RESULT_THROW(nn::fs::ResultNotImplemented());
        }

    }// namespace detail

    const int MovieWriterFileSystem::FileCountMax;

    size_t MovieWriterFileSystem::GetRequiredMemorySizeForFile(int64_t cacheChunkSize, int64_t cacheChunkCount) NN_NOEXCEPT
    {
        return detail::MovieWriterFileSystemImpl::GetRequiredMemorySizeForFile(cacheChunkSize, cacheChunkCount);
    }

    size_t MovieWriterFileSystem::GetRequiredMemoryAlignmentForFile() NN_NOEXCEPT
    {
        return detail::MovieWriterFileSystemImpl::GetRequiredMemoryAlignmentForFile();
    }

    MovieWriterFileSystem::MovieWriterFileSystem() NN_NOEXCEPT
    {
    }

    void MovieWriterFileSystem::Initialize(const char* mountName) NN_NOEXCEPT
    {
        NN_CAPSRV_LOG_MOVIEFS_MWFS("Initialize(m=%s)\n", mountName);
        m_Impl.Initialize();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::fsa::Register(mountName, std::unique_ptr<nn::fs::fsa::IFileSystem>(new detail::MovieWriterFileSystemProxy(&m_Impl))));
        m_MountName = FilePath::FromString(mountName);
        std::memset(m_FileStateList, 0, sizeof(m_FileStateList));
        std::memset(m_FileHandleList, 0, sizeof(m_FileHandleList));
    }

    void MovieWriterFileSystem::Finalize() NN_NOEXCEPT
    {
        NN_CAPSRV_LOG_MOVIEFS_MWFS("Finalize() m=%s\n", m_MountName.value);
        nn::fs::Unmount(m_MountName.value);
        //nn::fs::fsa::Unregister(m_MountName.value);
        m_Impl.Finalize();
        m_MountName = {};
        std::memset(m_FileStateList, 0, sizeof(m_FileStateList));
        std::memset(m_FileHandleList, 0, sizeof(m_FileHandleList));
    }

    MovieWriterFileSystem::FilePath MovieWriterFileSystem::GetFilename(int index) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(index, 0, FileCountMax);
        // FileCountMax は 1 桁程度なので 1 文字あれば十分。
        NN_STATIC_ASSERT(MovieWriterFileSystemFileCountMax <= 'Z' - 'A' + 1);
        char filename[3] = "/A";
        filename[1] += static_cast<char>(index);
        NN_SDK_REQUIRES_MINMAX(filename[1], 'A', 'Z');

        FilePath path = {};
        char* p = path.value;
        int pos = 0;
        pos += nn::util::Strlcpy(p + pos, filename, sizeof(FilePath) - pos);
        NN_SDK_ASSERT_RANGE(pos, 0, static_cast<int>(sizeof(FilePath)));
        return path;
    }

    MovieWriterFileSystem::FilePath MovieWriterFileSystem::GetFilePath(int index) const NN_NOEXCEPT
    {
        FilePath filename = GetFilename(index);
        FilePath path = {};
        char* p = path.value;
        int pos = 0;
        pos += nn::util::Strlcpy(p + pos, m_MountName.value, sizeof(FilePath) - pos);
        pos += nn::util::Strlcpy(p + pos, ":", sizeof(FilePath) - pos);
        pos += nn::util::Strlcpy(p + pos, filename.value, sizeof(FilePath) - pos);
        NN_SDK_ASSERT_RANGE(pos, 0, static_cast<int>(sizeof(FilePath)));
        return path;
    }

    int MovieWriterFileSystem::FindEmptyFileSlot() const NN_NOEXCEPT
    {
        for(int i = 0; i < FileCountMax; i++)
        {
            if(m_FileStateList[i] == FileState_Empty)
            {
                return i;
            }
        }
        return -1;
    }

    int MovieWriterFileSystem::FindFileSlot(nn::fs::FileHandle h) const NN_NOEXCEPT
    {
        for(int i = 0; i < FileCountMax; i++)
        {
            if(m_FileStateList[i] == FileState_Opened && m_FileHandleList[i].handle == h.handle)
            {
                return i;
            }
        }
        return -1;
    }

    nn::Result MovieWriterFileSystem::AttachMovieWriteStreamDataSection(
        nn::fs::FileHandle* pOutFileHandle,
        AlbumMovieWriteStreamHandle streamHandle,
        int64_t cacheChunkSize,
        int64_t cacheChunkCount,
        void* memory,
        size_t memorySize,
        util::optional<fs::PriorityRaw> pFsPriorityRaw
    ) NN_NOEXCEPT
    {
        NN_CAPSRV_LOG_MOVIEFS_MWFS("Attach()\n");
        NN_SDK_REQUIRES_GREATER(cacheChunkSize, 0);
        NN_SDK_REQUIRES_EQUAL(cacheChunkSize % AlbumMovieDataUnitSize, 0);
        NN_SDK_REQUIRES_GREATER_EQUAL(cacheChunkCount, 1);
        NN_SDK_REQUIRES_NOT_NULL(memory);
        NN_SDK_REQUIRES_ALIGNED(memory, GetRequiredMemoryAlignmentForFile());
        NN_SDK_REQUIRES_GREATER_EQUAL(memorySize, GetRequiredMemorySizeForFile(cacheChunkSize, cacheChunkCount));

        int index = FindEmptyFileSlot();
        NN_RESULT_THROW_UNLESS(index >= 0, nn::fs::ResultFileEntryFull());

        auto filename = GetFilename(index);
        m_Impl.AttachWriteStream(index, filename, streamHandle, cacheChunkSize, cacheChunkCount, memory, memorySize, pFsPriorityRaw);
        auto filepath = GetFilePath(index);

        nn::fs::FileHandle handle = {};
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&handle, filepath.value, nn::fs::OpenMode_Read | nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend));

        m_FileStateList[index] = FileState_Opened;
        m_FileHandleList[index] = handle;
        NN_CAPSRV_LOG_MOVIEFS_LINE("->h=%llX (%s)\n", handle.handle, filepath.value);
        *pOutFileHandle = handle;
        NN_RESULT_SUCCESS;
    }

    void MovieWriterFileSystem::DetachMovieWriteStreamDataSection(nn::fs::FileHandle handle) NN_NOEXCEPT
    {
        NN_CAPSRV_LOG_MOVIEFS_MWFS("Detach(h=%llX)\n", handle.handle);
        int index = FindFileSlot(handle);
        NN_ABORT_UNLESS_RANGE(index, 0, FileCountMax);

        nn::fs::CloseFile(handle);
        m_Impl.DetachWriteStream(index);
        m_FileStateList[index] = FileState_Empty;
        m_FileHandleList[index] = {};
    }

}}}
