﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

namespace nn{ namespace capsrv{ namespace capture{

    enum ImageFormat
    {
        // R8G8B8A8-Packed-Pitch
        //
        // Layout:
        //   Plane0: 4 x W x H bytes
        //     R(0,0) G(0,0) B(0,0) A(0,0) R(1,0) G(1,0) B(1,0) A(1,0) ... R(W-1,0) G(W-1,0) B(W-1,0) A(W-1,0)
        //     R(0,1) G(0,1) B(0,1) A(0,1) R(1,1) G(1,1) B(1,1) A(1,1) ... R(W-1,1) G(W-1,1) B(W-1,1) A(W-1,1)
        //       ...
        //     R(0,H-1) G(0,H-1) B(0,H-1) A(0,H-1)      ...        R(W-1,H-1) G(W-1,H-1) B(W-1,H-1) A(W-1,H-1)
        ImageFormat_Rgba8,

        // YUV444-Planar-Pitch
        //
        // Layout:
        //   Plane0: W x H bytes
        //     Y(0,0) Y(1,0) ... Y(W-1,0)
        //     Y(0,1) Y(1,1) ... Y(W-1,1)
        //       ...
        //     Y(0,H-1)   ...  Y(W-1,H-1)
        //   Plane1: W x H bytes
        //     U(0,0) U(1,0) ... U(W-1,0)
        //     U(0,1) U(1,1) ... U(W-1,1)
        //       ...
        //     U(0,H-1)   ...  U(W-1,H-1)
        //   Plane2: W x H bytes
        //     V(0,0) V(1,0) ... V(W-1,0)
        //     V(0,1) V(1,1) ... V(W-1,1)
        //       ...
        //     V(0,H-1)   ...  V(W/2-1,H-1)
        ImageFormat_Yuv444_Bt601_Planar,

        // YUV422-Planar-Pitch
        //
        // Layout:
        //   Plane0: W x H bytes
        //     Y(0,0) Y(1,0) ... Y(W-1,0)
        //     Y(0,1) Y(1,1) ... Y(W-1,1)
        //       ...
        //     Y(0,H-1)   ...  Y(W-1,H-1)
        //   Plane1: W/2 x H bytes
        //     U(0,0) U(1,0) ... U(W/2-1,0)
        //     U(0,1) U(1,1) ... U(W/2-1,1)
        //       ...
        //     U(0,H-1)   ...  U(W/2-1,H-1)
        //   Plane2: W/2 x H bytes
        //     V(0,0) V(1,0) ... V(W/2-1,0)
        //     V(0,1) V(1,1) ... V(W/2-1,1)
        //       ...
        //     V(0,H-1)   ...  V(W/2-1,H-1)
        ImageFormat_Yuv422_Bt601_Planar,

        // YUV440-Planar-Pitch
        //
        // Layout:
        //   Plane0: W x H bytes
        //     Y(0,0) Y(1,0) ... Y(W-1,0)
        //     Y(0,1) Y(1,1) ... Y(W-1,1)
        //       ...
        //     Y(0,H-1)   ...  Y(W-1,H-1)
        //   Plane1: W x H/2 bytes
        //     U(0,0) U(1,0) ... U(W-1,0)
        //     U(0,1) U(1,1) ... U(W-1,1)
        //       ...
        //     U(0,H/2-1)   ...  U(W-1,H/2-1)
        //   Plane2: W x H/2 bytes
        //     V(0,0) V(1,0) ... V(W-1,0)
        //     V(0,1) V(1,1) ... V(W-1,1)
        //       ...
        //     V(0,H/2-1)   ...  V(W-1,H/2-1)
        ImageFormat_Yuv440_Bt601_Planar,

        // YUV420(BT601)-Planar-Pitch
        //
        // Layout:
        //   Plane0: W x H bytes
        //     Y(0,0) Y(1,0) ... Y(W-1,0)
        //     Y(0,1) Y(1,1) ... Y(W-1,1)
        //       ...
        //     Y(0,H-1)   ...  Y(W-1,H-1)
        //   Plane1: W/2 x H/2 bytes
        //     U(0,0) U(1,0) ... U(W/2-1,0)
        //     U(0,1) U(1,1) ... U(W/2-1,1)
        //       ...
        //     U(0,H/2-1)   ...  U(W/2-1,H/2-1)
        //   Plane2: W/2 x H/2 bytes
        //     V(0,0) V(1,0) ... V(W/2-1,0)
        //     V(0,1) V(1,1) ... V(W/2-1,1)
        //       ...
        //     V(0,H/2-1)   ...  V(W/2-1,H/2-1)
        ImageFormat_Yuv420_Bt601_Planar,

        // Y plane of Yuv-Planar format.
        ImageFormat_Y_Bt601,
        // U plane of Yuv-Planar format.
        ImageFormat_U_Bt601,
        // V plane of Yuv-Planar format.
        ImageFormat_V_Bt601,

        // Y plane of NV12(Yuv-SemiPlanar) format.
        ImageFormat_Y_NV12,
        // UV plane of NV12(Yuv-SemiPlanar) format.
        ImageFormat_Uv_NV12,

        ImageFormat_NV12_Bt709,

        ImageFormat_SharedTexture,

    };


    static inline bool IsYuvFormat(ImageFormat format) NN_NOEXCEPT
    {
        return (ImageFormat_Yuv444_Bt601_Planar <= format && format <= ImageFormat_Yuv440_Bt601_Planar);
    }

    static inline uint32_t GetHorizontalSubsamplingShift(ImageFormat format) NN_NOEXCEPT
    {
        switch (format)
        {
        case ImageFormat_Yuv444_Bt601_Planar:
            return 0;
        case ImageFormat_Yuv422_Bt601_Planar:
            return 1;
        case ImageFormat_Yuv440_Bt601_Planar:
            return 0;
        case ImageFormat_Yuv420_Bt601_Planar:
            return 1;
        default: NN_UNEXPECTED_DEFAULT;
        }
    }

    static inline uint32_t GetVerticalSubsamplingShift(ImageFormat format) NN_NOEXCEPT
    {
        switch (format)
        {
        case ImageFormat_Yuv444_Bt601_Planar:
            return 0;
        case ImageFormat_Yuv422_Bt601_Planar:
            return 0;
        case ImageFormat_Yuv440_Bt601_Planar:
            return 1;
        case ImageFormat_Yuv420_Bt601_Planar:
            return 1;
        default: NN_UNEXPECTED_DEFAULT;
        }
    }

    static inline uint32_t GetHorizontalSubsampling(ImageFormat format) NN_NOEXCEPT
    {
        return (1 << GetHorizontalSubsamplingShift(format));
    }

    static inline uint32_t GetVerticalSubsampling(ImageFormat format) NN_NOEXCEPT
    {
        return (1 << GetVerticalSubsamplingShift(format));
    }

}}}
