﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/os.h>

#include "capsrv_DisplayCapture.h"

#include <binder/IBinder.h>
#include <gui/IGraphicBufferProducer.h>
#include <gui/IGraphicBufferConsumer.h>

namespace nn{ namespace capsrv{ namespace capture{

    class DisplayCaptureData
    {
    public:
        static const int BufferCount = DisplayCapture::BufferCount;

    public:
        struct DisplayBufferEntry
        {
        public:
            DisplayBufferEntry() NN_NOEXCEPT
                : pBuffer(nullptr)
                , displaySlotIndex(0)
                , isAcquiredInternal(false)
                , frameNumber(0)
                , pFence()
            {}

            void Clear() NN_NOEXCEPT
            {
                pBuffer = nullptr;
                displaySlotIndex = 0;
                isAcquiredInternal = false;
                frameNumber = 0;
                pFence.clear();
            }

            bool IsValid() const NN_NOEXCEPT
            {
                return pBuffer != nullptr;
            }

        public:
            DisplayBuffer* pBuffer;
            int displaySlotIndex;    // VirtualDisplay 上のインデックス
            bool isAcquiredInternal; // VirtualDisplay から Acquire しているか
            uint64_t frameNumber;
            android::sp<android::Fence> pFence;
        };

    public:
        DisplayCaptureData() NN_NOEXCEPT;

        void Initialize(
            CaptureModule* pModule,
            const char* name,
            nn::vi::LayerStack layerStack,
            int width,
            int height,
            ImageFormat format
            ) NN_NOEXCEPT;

        void Finalize() NN_NOEXCEPT;

        void SetCaptureBuffer(int index, DisplayBuffer* pBuffer) NN_NOEXCEPT;
        void UnsetCaptureBuffer(int index) NN_NOEXCEPT;


        // キャプチャを実行する。
        // キャプチャが正しく行えれば成功する。
        // IsRecoveryRequired() が true を返す場合、この関数は失敗する。
        //
        // キャプチャに成功した場合、キャプチャしたバッファを返す。
        // キャプチャに失敗した場合、nullptr を返す。
        // キャプチャに失敗した場合、 IsRecoveryRequired() が true を返すようになる。
        DisplayBuffer* AcquireCaptureBuffer(nn::TimeSpan timeout) NN_NOEXCEPT;
        DisplayBuffer* AcquireCaptureBuffer(nn::TimeSpan timeout, nn::vi::LayerStack layerStack) NN_NOEXCEPT;
        DisplayBuffer* AcquireCaptureBuffer(nn::TimeSpan timeout, nn::vi::LayerStack layerStack, int width, int height) NN_NOEXCEPT;
        void ReleaseCaptureBuffer(DisplayBuffer* pBuffer) NN_NOEXCEPT;

        // 復帰処理が必要かを返す。
        // AcquireCaptureBuffer() が失敗すると true を返すようになる。
        // RecoverCapture() に成功すると false を返すようになる。
        bool IsRecoveryRequired() const NN_NOEXCEPT;

        // 復帰処理を行う。
        //
        // IsRecovery() が true を返す場合、復帰処理を行い成功した場合 true を返す。
        // IsRecovery() が false を返す場合、何もせず true を返す。
        //
        // 復帰処理に成功した場合、 IsRecoveryRequired() が false を返すようになる。
        // 復帰処理に失敗した場合、 IsRecoveryRequired() は true を返すまま。
        bool RecoverCapture(nn::TimeSpan timeout) NN_NOEXCEPT;


    private:
        CaptureModule*     m_pModule;
        nn::vi::LayerStack m_LayerStack;
        int                m_Width;
        int                m_Height;
        ImageFormat        m_Format;
        DisplayBufferEntry m_DisplayBufferList[BufferCount];

        nn::os::SystemEventType m_FrameAvailableEvent;

        android::sp<android::IBinder>                m_pDisplay;
        android::sp<android::IGraphicBufferProducer> m_pProducer;
        android::sp<android::IGraphicBufferConsumer> m_pConsumer;
    };

}}}

