﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "capsrv_DisplayCapture.h"
#include "capsrv_DisplayCapture-module.nvnflinger.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/os/os_SdkSystemEventApi.h>
#include <nvgr.h>
#include <ui/PixelFormat.h>
#include <gui/SurfaceComposerClient.h>
#include <gui/BufferItem.h>
#include <ui/DisplayInfo.h>
#include <nn/vi/vi_LogicalResolution.h>

#include "../server/capsrvServer_Config.h"
#include "capsrv_CaptureModule-module.nvnflinger.h"
#include "capsrv_DisplayBuffer-module.nvnflinger.h"
#include "detail/capsrv_ImageFormatInfo-module.nvnflinger.h"

namespace nn{ namespace capsrv{ namespace capture{

    static const uint32_t BufferUsageBits = GRALLOC_USAGE_HW_COMPOSER | GRALLOC_USAGE_HW_VIDEO_ENCODER;

    DisplayCaptureData::DisplayCaptureData() NN_NOEXCEPT
        : m_pModule(nullptr)
        , m_LayerStack()
        , m_Width(0)
        , m_Height(0)
        , m_Format()
        , m_FrameAvailableEvent()
        , m_pDisplay()
        , m_pProducer()
        , m_pConsumer()
    {
    }

    void DisplayCaptureData::Initialize(
        CaptureModule* pModule,
        const char* name,
        nn::vi::LayerStack layerStack,
        int width,
        int height,
        ImageFormat format
        ) NN_NOEXCEPT
    {
        m_pModule = pModule;
        m_LayerStack = layerStack;
        m_Width   = width;
        m_Height  = height;
        m_Format  = format;
        for(int i = 0; i < BufferCount; i++)
        {
            m_DisplayBufferList[i].Clear();
        }

        android::status_t err = 0;
        m_pDisplay = pModule->GetClinet()->createDisplay(android::String8(name), false);
        NN_SDK_ASSERT_NOT_NULL(m_pDisplay);

        android::SurfaceComposerClient::openGlobalTransaction();
        pModule->GetClinet()->setDisplaySize(m_pDisplay, static_cast<uint32_t>(width), static_cast<uint32_t>(height));
        pModule->GetClinet()->setDisplayLayerStack(m_pDisplay, static_cast<uint32_t>(layerStack));
        pModule->GetClinet()->setDisplayProjection(m_pDisplay,
                                                   android::DISPLAY_ORIENTATION_0,
                                                   android::Rect(nn::vi::LogicalResolutionWidth, nn::vi::LogicalResolutionHeight),
                                                   android::Rect(width, height));
        android::SurfaceComposerClient::closeGlobalTransaction();

        err = pModule->GetClinet()->getDisplaySurface(m_pDisplay, m_pProducer, m_pConsumer);
        NN_SDK_ASSERT(err == android::NO_ERROR);
        NN_SDK_ASSERT_NOT_NULL(m_pProducer);
        NN_SDK_ASSERT_NOT_NULL(m_pConsumer);

        m_pProducer->setBufferCount(BufferCount);
        m_pConsumer->setConsumerUsageBits(BufferUsageBits);
        m_pConsumer->setDefaultBufferSize(static_cast<uint32_t>(width), static_cast<uint32_t>(height));
        m_pConsumer->setDefaultBufferFormat(detail::ImageFormatInfo::GetComposerImageFormat(format));

        // setup system event
        {
            nn::os::NativeHandle eventHandle = {};
            bool isEventManaged = {};
            err = m_pConsumer->getFrameAvailableEventHandle(eventHandle, isEventManaged);
            NN_SDK_ASSERT_EQUAL(err, android::NO_ERROR);

            nn::os::AttachReadableHandleToSystemEvent(&m_FrameAvailableEvent, eventHandle, isEventManaged, nn::os::EventClearMode_ManualClear);
        }
    }

    void DisplayCaptureData::Finalize() NN_NOEXCEPT
    {
        nn::os::DetachReadableHandleOfSystemEvent(&m_FrameAvailableEvent);

        m_pConsumer.clear();
        m_pProducer.clear();

        m_pModule->GetClinet()->destroySurface(m_pDisplay);
        m_pModule->GetClinet()->destroyDisplay(m_pDisplay);
        m_pDisplay.clear();

        for(int i = 0; i < BufferCount; i++)
        {
            m_DisplayBufferList[i].Clear();
        }
    }

    void DisplayCaptureData::SetCaptureBuffer(int index, DisplayBuffer* pBuffer) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(index, 0, static_cast<int>(BufferCount));
        NN_SDK_REQUIRES(!m_DisplayBufferList[index].IsValid());

        auto pBufferData = pBuffer->GetData();
        auto& pGraphicBuffer = pBufferData->GetGraphicBuffer();

        int slotIndex = 0;
        m_pConsumer->attachBuffer(&slotIndex, pGraphicBuffer);

        m_DisplayBufferList[index].pBuffer = pBuffer;
        m_DisplayBufferList[index].displaySlotIndex = slotIndex;
        m_DisplayBufferList[index].frameNumber = 0;
        m_DisplayBufferList[index].isAcquiredInternal = true;
    }

    void DisplayCaptureData::UnsetCaptureBuffer(int index) NN_NOEXCEPT
    {
        //NN_SDK_REQUIRES(!IsRecoveryRequired());
        NN_SDK_REQUIRES_RANGE(index, 0, static_cast<int>(BufferCount));

        if(m_DisplayBufferList[index].IsValid())
        {
            NN_ABORT_UNLESS_EQUAL(m_pConsumer->detachBuffer(m_DisplayBufferList[index].displaySlotIndex), android::NO_ERROR);
            m_DisplayBufferList[index].Clear();
        }
    }

    DisplayBuffer* DisplayCaptureData::AcquireCaptureBuffer(nn::TimeSpan timeout, nn::vi::LayerStack layerStack) NN_NOEXCEPT
    {
        if(IsRecoveryRequired())
        {
            return nullptr;
        }

        if (layerStack != m_LayerStack)
        {
            android::SurfaceComposerClient::openGlobalTransaction();
            m_pModule->GetClinet()->setDisplayLayerStack(m_pDisplay, static_cast<uint32_t>(layerStack));
            android::SurfaceComposerClient::closeGlobalTransaction();
            this->m_LayerStack = layerStack;
        }
        return AcquireCaptureBuffer(timeout);
    }

    DisplayBuffer* DisplayCaptureData::AcquireCaptureBuffer(nn::TimeSpan timeout, nn::vi::LayerStack layerStack, int width, int height) NN_NOEXCEPT
    {
        if(IsRecoveryRequired())
        {
            return nullptr;
        }

        if (layerStack != m_LayerStack || width != m_Width || height != m_Height)
        {
            android::SurfaceComposerClient::openGlobalTransaction();
            m_pModule->GetClinet()->setDisplaySize(m_pDisplay, static_cast<uint32_t>(width), static_cast<uint32_t>(height));
            m_pModule->GetClinet()->setDisplayLayerStack(m_pDisplay, static_cast<uint32_t>(layerStack));
            m_pModule->GetClinet()->setDisplayProjection(m_pDisplay,
                                                       android::DISPLAY_ORIENTATION_0,
                                                       android::Rect(nn::vi::LogicalResolutionWidth, nn::vi::LogicalResolutionHeight),
                                                       android::Rect(width, height));
            android::SurfaceComposerClient::closeGlobalTransaction();
            this->m_LayerStack = layerStack;
            this->m_Width = width;
            this->m_Height = height;
        }
        return AcquireCaptureBuffer(timeout);
    }

    DisplayBuffer* DisplayCaptureData::AcquireCaptureBuffer(nn::TimeSpan timeout) NN_NOEXCEPT
    {
        if(IsRecoveryRequired())
        {
            return nullptr;
        }

        // NOTE: This is for single buffer capturing
        NN_STATIC_ASSERT(BufferCount == 1);
        auto& bufferEntry = m_DisplayBufferList[0];

        if(!bufferEntry.IsValid())
        {
            return nullptr;
        }

        // Releasing buffer to capture current screen
        {
            bufferEntry.pBuffer->FlushCache();
            auto err = m_pConsumer->releaseBuffer(bufferEntry.displaySlotIndex, bufferEntry.frameNumber, EGL_NO_DISPLAY, EGL_NO_SYNC_KHR, android::Fence::NO_FENCE);
            NN_SDK_ASSERT_EQUAL(err, android::NO_ERROR);
            NN_UNUSED(err);
            bufferEntry.isAcquiredInternal = false;
        }

        // Acquiring newly captured buffer
        {
            android::status_t err = 0;
            android::IGraphicBufferConsumer::BufferItem item;

            if(!nn::os::TimedWaitSystemEvent(&m_FrameAvailableEvent, timeout))
            {
                NN_CAPSRV_LOG_WARN("[capture] capture timed out (buffer is not ready)\n");
                return nullptr;
            }

            err = m_pConsumer->acquireBuffer(&item, 0);
            if(err)
            {
                NN_CAPSRV_LOG_WARN("[capture] acquire capture buffer failed: %d\n", static_cast<int>(-err));
                return nullptr;
            }

            NN_SDK_ASSERT_EQUAL(item.mBuf, bufferEntry.displaySlotIndex);
            bufferEntry.isAcquiredInternal = true;
            bufferEntry.frameNumber = item.mFrameNumber;
            bufferEntry.pFence = item.mFence;

            if(bufferEntry.pFence != nullptr)
            {
                err = bufferEntry.pFence->wait(timeout.GetMilliSeconds());
                if(err)
                {
                    NN_CAPSRV_LOG_WARN("[capture] capture timed out (data is not ready)\n");
                    return nullptr;
                }
                bufferEntry.pFence.clear();
            }

            bufferEntry.pBuffer->GetData()->m_IsContentTrivial = ((item.mFlags & android::BufferItem::FLAG_NO_CONTENT) != 0);
        }

        return bufferEntry.pBuffer;
    }

    void DisplayCaptureData::ReleaseCaptureBuffer(DisplayBuffer* pBuffer) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pBuffer);

        // do nothing. keep buffer 'ACQUIRED'
        NN_UNUSED(pBuffer);
    }

    bool DisplayCaptureData::IsRecoveryRequired() const NN_NOEXCEPT
    {
        // NOTE: This is for single buffer capturing
        NN_STATIC_ASSERT(BufferCount == 1);
        auto& bufferEntry = m_DisplayBufferList[0];

        if(!bufferEntry.IsValid())
        {
            return false;
        }

        if(!bufferEntry.isAcquiredInternal)
        {
            return true;
        }

        if(bufferEntry.pFence != nullptr)
        {
            return true;
        }
        return false;
    }

    bool DisplayCaptureData::RecoverCapture(nn::TimeSpan timeout) NN_NOEXCEPT
    {
        if(!IsRecoveryRequired())
        {
            return true;
        }

        // NOTE: This is for single buffer capturing
        NN_STATIC_ASSERT(BufferCount == 1);
        auto& bufferEntry = m_DisplayBufferList[0];

        // RECOVERY:
        //   If previous capturing failed, reset capture status.
        //   We should start from 'ACQUIRED' and 'buffer is idle' state.
        android::status_t err = 0;
        android::IGraphicBufferConsumer::BufferItem item;

        // Wait for nvnflinger programming
        if(!bufferEntry.isAcquiredInternal)
        {
            if(!nn::os::TimedWaitSystemEvent(&m_FrameAvailableEvent, timeout))
            {
                NN_CAPSRV_LOG_ERR("[recovery] capture timed out (buffer is not ready)\n");
                return false;
            }

            err = m_pConsumer->acquireBuffer(&item, 0);
            if(err)
            {
                NN_CAPSRV_LOG_ERR("[recovery] acquire capture buffer failed: %d\n", static_cast<int>(-err));
                return false;
            }

            NN_SDK_ASSERT_EQUAL(item.mBuf, bufferEntry.displaySlotIndex);
            bufferEntry.isAcquiredInternal = true;
            bufferEntry.frameNumber = item.mFrameNumber;
            bufferEntry.pFence = item.mFence;
        }

        // Wait for hardware composition
        if(bufferEntry.pFence != nullptr)
        {
            err = bufferEntry.pFence->wait(timeout.GetMilliSeconds());
            if(err)
            {
                NN_CAPSRV_LOG_ERR("[recovery] capture timed out (data is not ready)\n");
                return false;
            }
            bufferEntry.pFence.clear();
        }


        // Discard recovery captured image
        ReleaseCaptureBuffer(bufferEntry.pBuffer);

        NN_CAPSRV_LOG_WARN("[recovery] success\n");
        return true;
    }

    //--------------------------------------

    DisplayCapture::DisplayCapture() NN_NOEXCEPT
    {
        NN_CAPSRV_CAPTURE_DATASTORAGE_CHECK(DataType, m_DataStorage);
        std::memset(this, 0, sizeof(*this));
    }

    void DisplayCapture::Initialize(
        CaptureModule* pModule,
        const char* name,
        nn::vi::LayerStack layerStack,
        int width,
        int height,
        ImageFormat format
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!IsInitialized());
        m_pData = new(&m_DataStorage) DataType();
        return m_pData->Initialize(pModule, name, layerStack, width, height, format);
    }

    void DisplayCapture::Finalize() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        m_pData->Finalize();
        m_pData->~DataType();
        m_pData = nullptr;
    }

    void DisplayCapture::SetCaptureBuffer(int index, DisplayBuffer* pBuffer) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        m_pData->SetCaptureBuffer(index, pBuffer);
    }

    void DisplayCapture::UnsetCaptureBuffer(int index) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        m_pData->UnsetCaptureBuffer(index);
    }

    DisplayBuffer* DisplayCapture::AcquireCaptureBuffer(nn::TimeSpan timeout) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        return m_pData->AcquireCaptureBuffer(timeout);
    }

    DisplayBuffer* DisplayCapture::AcquireCaptureBuffer(nn::TimeSpan timeout, vi::LayerStack layerStack) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        return m_pData->AcquireCaptureBuffer(timeout, layerStack);
    }

    DisplayBuffer* DisplayCapture::AcquireCaptureBuffer(nn::TimeSpan timeout, vi::LayerStack layerStack, int width, int height) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        return m_pData->AcquireCaptureBuffer(timeout, layerStack, width, height);
    }

    void DisplayCapture::ReleaseCaptureBuffer(DisplayBuffer* pBuffer) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        m_pData->ReleaseCaptureBuffer(pBuffer);
    }

    bool DisplayCapture::IsRecoveryRequired() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        return m_pData->IsRecoveryRequired();
    }

    bool DisplayCapture::RecoverCapture(nn::TimeSpan timeout) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        return m_pData->RecoverCapture(timeout);
    }

}}}

