﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_Optional.h>
#include <nn/sf/sf_ShimLibraryUtility.h>
#include <nn/capsrv/sf/capsrv_Services.sfdl.h>
#include <nn/capsrv/sf/capsrv_ServiceName.h>

#if defined(NN_CAPSRV_USE_HIPC)
#include <nn/sf/sf_HipcClientProxyByName.h>
#endif

#include "capsrv_Macro.h"

namespace nn{ namespace capsrv{

    template<typename TISessionType>
    class SessionHolder
    {
    public:
        typedef TISessionType ISessionType;
        typedef nn::sf::SharedPointer<ISessionType> SessionPointerType;

    public:
        SessionHolder() NN_NOEXCEPT
            : m_pSession()
        {
        }

        void StoreSession(SessionPointerType p) NN_NOEXCEPT
        {
            m_pSession = p;
        }

        void ReleaseSession() NN_NOEXCEPT
        {
            m_pSession = nullptr;
        }

        SessionPointerType GetSession() NN_NOEXCEPT
        {
            return m_pSession;
        }

    private:
        SessionPointerType m_pSession;
    };

    class NullSessionHolder
    {
    public:
        typedef void ISessionType;
        typedef void* SessionPointerType;
    public:
        void StoreSession(SessionPointerType) NN_NOEXCEPT {}
        void ReleaseSession() NN_NOEXCEPT {}
        SessionPointerType GetSession() NN_NOEXCEPT { return nullptr; }
    };

    template<typename TSelfType, typename TIServiceType, size_t ObjectCountMax, typename TSessionHolder = NullSessionHolder>
    class ServiceHolderBase
        : public TSessionHolder
    {
    public:
        typedef TIServiceType IServiceType;
        typedef nn::sf::SharedPointer<IServiceType> ServicePointerType;

    public:
        ServiceHolderBase() NN_NOEXCEPT
            : m_InitializeCount(0)
            , m_pService()
#if defined(NN_CAPSRV_USE_HIPC)
            , m_Manager(NN_SF_SIMPLE_ALL_IN_ONE_HIPC_CLIENT_MANAGER_INITIALIZER)
            , m_Holder(NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER)
#endif
        {
        }

        nn::Result Initialize(ServicePointerType pService) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_GREATER_EQUAL(m_InitializeCount, 0);
            if(m_InitializeCount > 0)
            {
                m_InitializeCount++;
                NN_CAPSRV_LOG_SHIM("Initialize(skipped) -> %d count\n", m_InitializeCount);
                NN_RESULT_SUCCESS;
            }

            bool isSuccess = false;

#if defined(NN_CAPSRV_USE_HIPC)
            {
                NN_RESULT_DO(m_Manager.InitializeShimLibraryHolder(&m_Holder, TSelfType::GetServiceName()));
                m_pService = m_Holder.GetObject();
            }
            NN_UTIL_SCOPE_EXIT{
                if(!isSuccess)
                {
                    m_pService.Reset();
                    m_Holder.FinalizeHolder();
                }
            };
#elif defined(NN_CAPSRV_USE_DIRECT_FUNCTION_CALL)
            {
                m_pService = pService;
            }
#endif

            m_InitializeCount++;
            NN_CAPSRV_LOG_SHIM("Initialize(done) -> %d count\n", m_InitializeCount);
            isSuccess = true;
            NN_RESULT_SUCCESS;
        }

        void Finalize() NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_GREATER(m_InitializeCount, 0);
            if(m_InitializeCount > 1)
            {
                m_InitializeCount--;
                NN_CAPSRV_LOG_SHIM("Finalize(skipped) -> %d count\n", m_InitializeCount);
                return;
            }

            TSessionHolder::ReleaseSession();
            m_pService.Reset();

#if defined(NN_CAPSRV_USE_HIPC)
            {
                m_Holder.FinalizeHolder();
            }
#endif

            m_InitializeCount--;
            NN_CAPSRV_LOG_SHIM("Finalize(done) -> %d count\n", m_InitializeCount);
        }

        bool IsInitialized() const NN_NOEXCEPT
        {
            return m_InitializeCount > 0;
        }

        int GetInitializeCount() const NN_NOEXCEPT
        {
            return m_InitializeCount;
        }

        ServicePointerType Get() NN_NOEXCEPT
        {
            return m_pService;
        }

    private:
        int                m_InitializeCount;
        ServicePointerType m_pService;

#if defined(NN_CAPSRV_USE_HIPC)
        nn::sf::SimpleAllInOneHipcClientManager<ObjectCountMax> m_Manager;
        nn::sf::ShimLibraryObjectHolder<IServiceType>           m_Holder;
#endif
    };


}}
