﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <assert.h>
#include <stdlib.h>
#include <pthread.h>
#include <mutex>
#include <AudioUnit/AudioUnit.h>
#include <AudioToolbox/AudioToolbox.h>
#include <winext/types.h>
#include "ai_dma.h"
#include "AudioBufferQueue.h"
#include "SoundEngineCoreAudio.h"

using namespace nw::internal::winext;

namespace {
    AudioUnit g_OutputAudioUnit;
    const int BuffersCount = CSoundEngineCoreAudio::BUFFER_COUNT;
    bool g_IsPlaying;
    const int BitsPerChannelCount     = 16;
    const int ChannelsPerFrameCount   = 2;
    const int FramesPerPacketCount    = 1;
    AudioBufferQueue g_AudioBufferQueue;
}

static OSStatus audioRenderCallback(void *inRefCon, AudioUnitRenderActionFlags *ioActionFlags, const AudioTimeStamp *inTimeStamp, UInt32 inBusNumber, UInt32 inNumberFrames, AudioBufferList *ioData)
{
    const int buffersCount = ioData->mNumberBuffers;

    // 転送元バッファを用意
    AudioBuffer *srcBuffer = g_AudioBufferQueue.Peek();
    if(srcBuffer == nullptr)
    {
        for(int i = 0 ; i < buffersCount; ++i)
        {
            const AudioBuffer *destBuffer = &ioData->mBuffers[i];
            memset(destBuffer->mData, 0, destBuffer->mDataByteSize);
        }
        // TODO: 適切なエラーコードがあるのであればそれに変更する
        return noErr;
    }

    for(int i = 0 ; i < buffersCount; ++i)
    {
        // 転送先バッファを用意
        const AudioBuffer *destBuffer = &ioData->mBuffers[i];
        UInt32 destBufferCopiedSize = 0;
        UInt32 requestSize = destBuffer->mDataByteSize;

        for(;;)
        {
            // 転送元バッファから転送先バッファへコピー
            const int copySize = std::min(srcBuffer->mDataByteSize, requestSize);
            memcpy((u8*)destBuffer->mData + destBufferCopiedSize, srcBuffer->mData, copySize);

            srcBuffer->mDataByteSize         -= copySize;
            requestSize                      -= copySize;
            srcBuffer->mData                 = static_cast<u8*>(srcBuffer->mData) + copySize;
            destBufferCopiedSize             += copySize;

            // 転送元バッファを使い切った時は新しいバッファに切り替える
            if(requestSize > 0)
            {
                // 転送元バッファが残っているか確認
                if(g_AudioBufferQueue.GetRemainBuffersCount() == 0)
                {
                    // TODO: 適切なエラーコードがあるのであればそれに変更する
                    return noErr;
                }

                srcBuffer = g_AudioBufferQueue.Dequeue();
                assert(srcBuffer != nullptr);

                // 波形生成を要求する
                CSoundEngineCoreAudio::Instance()->RequestRender();
            }
            else
            {
                break;
            }
        }
    }

    return noErr;
}

void AIDMA_Init(u32 sampleRate)
{
    OSStatus status;
    assert(sampleRate == 48000);

    /* RemoteIO コンポーネント (出力先) を取得 */
    AudioComponentDescription audioComponentDescription;
    AudioComponent audioComponent;

    audioComponentDescription.componentType = kAudioUnitType_Output;
    audioComponentDescription.componentSubType = kAudioUnitSubType_RemoteIO;
    audioComponentDescription.componentManufacturer = kAudioUnitManufacturer_Apple;
    audioComponentDescription.componentFlags = 0;
    audioComponentDescription.componentFlagsMask = 0;
    audioComponent = AudioComponentFindNext( NULL, &audioComponentDescription );
    assert(audioComponent != nullptr);

    /* 出力 AudioComponentInstance を作成 */
    status = AudioComponentInstanceNew( audioComponent, &g_OutputAudioUnit );
    assert(status == noErr);

    /* 出力 AudioUnit (AudioComponentInstane) を初期化 */
    status = AudioUnitInitialize( g_OutputAudioUnit );
    assert(status == noErr);

    /* 波形要求コールバック関数を設定 */
    AURenderCallbackStruct audioUnitRenderCallback;
    audioUnitRenderCallback.inputProc = audioRenderCallback;
    status = AudioUnitSetProperty(g_OutputAudioUnit,
                         kAudioUnitProperty_SetRenderCallback,
                         kAudioUnitScope_Global,
                         0, /* Bus number */
                         &audioUnitRenderCallback,
                         sizeof(AURenderCallbackStruct));
    assert(status == noErr);

    /* 波形のフォーマットを指定 */
    AudioStreamBasicDescription audioFormat;
    audioFormat.mFormatID           = kAudioFormatLinearPCM;
    audioFormat.mFormatFlags        = kAudioFormatFlagIsSignedInteger | kAudioFormatFlagIsPacked;
    audioFormat.mSampleRate         = sampleRate;
    audioFormat.mBitsPerChannel     = BitsPerChannelCount;
    audioFormat.mChannelsPerFrame   = ChannelsPerFrameCount;
    audioFormat.mFramesPerPacket    = FramesPerPacketCount;
    audioFormat.mBytesPerFrame      = audioFormat.mBitsPerChannel / 8 * audioFormat.mChannelsPerFrame;
    audioFormat.mBytesPerPacket     = audioFormat.mBytesPerFrame * audioFormat.mFramesPerPacket;
    audioFormat.mReserved           = 0;
    status = AudioUnitSetProperty(g_OutputAudioUnit,
                                  kAudioUnitProperty_StreamFormat,
                                  kAudioUnitScope_Input,
                                  0,
                                  &audioFormat,
                                  sizeof(audioFormat));

    assert(status == noErr);
}

s32  AIDMA_GetBufferQueueCapacity()
{
    return BuffersCount;
}

void AIDMA_ExecuteDMA(u8* address, u32 size)
{
    AudioBuffer *pAudioBuffer;

    pAudioBuffer = g_AudioBufferQueue.Enqueue(address, size);
    assert(pAudioBuffer != nullptr);

    // TODO: 再生開始のバッファサイズは要検討
    if(!g_IsPlaying && g_AudioBufferQueue.GetRemainBuffersCount() == BuffersCount)
    {
        AudioOutputUnitStart(g_OutputAudioUnit);
        g_IsPlaying = true;
    }
}

void AIDMA_StopDMA()
{
    AudioOutputUnitStop(g_OutputAudioUnit);
}

void AIDMA_Quit()
{
    AIDMA_StopDMA();
}

