﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*
 * Includes
 *---------------------------------------------------------------------------*/

#include <string.h>
#include <stdlib.h>
#include <stddef.h>

#include <winext/cafe/ai.h>
#include <winext/cafe/os/OSInterrupt.h>

#if defined(__APPLE__)
#include <TargetConditionals.h>
#endif

#if defined(ANDROID)
#include "SoundEngineSLES.h"
#elif TARGET_OS_IPHONE
#include "SoundEngineCoreAudio.h"
#else
#include "SoundEngineMME.h"
#endif

#include <winext/cafe/os.h>

namespace nw {
namespace internal {
namespace winext {

/*---------------------------------------------------------------------------*
 * Local types
 *---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*
 * Local definitions/declarations
 *---------------------------------------------------------------------------*/

#define USE_REVC_CODE
#undef  USE_REVC_CODE

#define AI_STREAM_START     0x00000001
#define AI_STREAM_STOP      0x00000000

#define DSP_WORD_SHIFT      16
#define DSP_WORD_MASK       0xffff

#define DSP_AIMA_MASK_HI    0x03ff
#define DSP_AIMA_MASK_LO    0xffe0

static  AIDCallback         __AID_Callback; // AI-FIFO DMA callback
static  u8*                 __CallbackStack;// stack for extended handlers
                                            // note that stacks grow DOWN,
                                            // so this must be the HIGH
                                            // address of the stack
static  u8*                 __OldStack;     // stack of context interrupted by
                                            // interrupt.

static u32 __DMABufferStartAddr = 0;
static u32 __DMABufferLength = 0;

// For AICheckInit()
static volatile BOOL        __AI_init_flag = FALSE;

// a flag for checking previous __AID_Callback is still acitve or not.
static volatile BOOL        __AID_Active = FALSE;

// Private Functions
// static void __AIDHandler                (__OSInterrupt interrupt, OSContext *context);
static void __AICallbackStackSwitch     (register void *cb);
static void __AI_SRC_INIT               (void);

#define DO_SRC32TO48
//#undef  DO_SRC32TO48

#ifdef DO_SRC32TO48
static void __tmp_aid_callback          (void);
void __do_src_32to48             (s16* output, s16* input, s32 num);
static void __init_src_32to48           (void);
#endif

/*---------------------------------------------------------------------------*
 * Debugging tools for AI SRC hardware bug
 *---------------------------------------------------------------------------*/

typedef struct STRUCT_TIMELOG
{
    OSTime t_start;
    OSTime t1;
    OSTime t2;
    OSTime t3;
    OSTime t4;
    OSTime t_end;

} sTIMELOG;

sTIMELOG profile;

sTIMELOG *__ai_src_get_time();

/*---------------------------------------------------------------------------*
 * Globals
 *---------------------------------------------------------------------------*/

/*===========================================================================*
 *                   F U N C T I O N    D E F I N I T I O N S
 *===========================================================================*/

/*---------------------------------------------------------------------------*
 * AI-FIFO DMA Controls
 *---------------------------------------------------------------------------*/

static void WaveOutCallbackFunc(
    int channels,
    signed short* buffer,
    unsigned long samples,
    int /* sampleRate */)
{
    if(__DMABufferStartAddr == 0)
    {
        return;
    }

    void* pDMABuffer = reinterpret_cast<void*>(__DMABufferStartAddr);
    u32 bufferLength = sizeof(buffer[0]) * samples * channels;
    //ASSERT(bufferLength <= __DMABufferLength);

    memcpy(pDMABuffer, buffer, bufferLength);

    if(__AID_Callback == NULL)
    {
        return;
    }

    __AID_Callback();
}

/*---------------------------------------------------------------------------*
 * Name        : AIRegisterDMACallback()
 *
 * Description : Registers the given function as the AIDINT interrupt callback.
 *
 * Arguments   : AIDCallback *callback
 *
 * Returns     : Returns the previously defined callback (NULL if there isn't
 *               one).
 *
 *---------------------------------------------------------------------------*/

AIDCallback AIRegisterDMACallback(AIDCallback callback)
{

    AIDCallback old_callback;
    BOOL        old;

        old_callback = __AID_Callback;

        old = OSDisableInterrupts();

        __AID_Callback = callback;

        OSRestoreInterrupts(old);

        return(old_callback);

} /* end of AIRegsiterDMACallback() */

AIDCallback AIDRCRegisterDMACallback   (AIDCallback /* callback */)
{
    return NULL;
}

void AIInitDMA(u32 start_addr, u32 length)
{
    __DMABufferStartAddr = start_addr;
    __DMABufferLength = length;
}

u32 AIGetDMAStartAddr()
{
    return __DMABufferStartAddr;
}

u32 AIDRCGetDMAStartAddr()
{
    return 0;
}

u32 AIGetDMALength()
{
    return __DMABufferLength;
}

u32 AIDRCGetDMALength()
{
    return 0;
}

void AIInit( unsigned int deviceId )
{
#if defined(ANDROID)
    CSoundEngineSLES *pSoundEngine = CSoundEngineSLES::Instance();
#elif TARGET_OS_IPHONE
    CSoundEngineCoreAudio *pSoundEngine = CSoundEngineCoreAudio::Instance();
#else
    CSoundEngineMME  *pSoundEngine = CSoundEngineMME::Instance();
#endif

    pSoundEngine->OpenStream( deviceId );

    if(pSoundEngine->GetWaveOutCallback() == NULL)
    {
        pSoundEngine->SetWaveOutCallback( WaveOutCallbackFunc );
    }

    __AI_init_flag = TRUE;
}

BOOL AICheckInit(void)
{
    return __AI_init_flag;
}

void AIReset()
{
#if defined(ANDROID)
    CSoundEngineSLES *pSoundEngine = CSoundEngineSLES::Instance();
#elif TARGET_OS_IPHONE
    CSoundEngineCoreAudio *pSoundEngine = CSoundEngineCoreAudio::Instance();
#else
    CSoundEngineMME  *pSoundEngine = CSoundEngineMME::Instance();
#endif

    pSoundEngine->CloseStream();
}

void AISetDSPSampleRate(u32 /* rate */)
{
}

} // namespace winext
} // namespace internal
} // namespace nw
