﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/ut/ut_MemoryFileStream.h>
#include <nw/ut/ut_Memory.h>

namespace nw {
namespace ut {

//---------------------------------------------------------------------------
MemoryFileStream::MemoryFileStream()
: m_Available( false ),
  m_StartPos( 0 ),
  m_CurPos( 0 ),
  m_Size( 0 )
{
}

//---------------------------------------------------------------------------
MemoryFileStream::MemoryFileStream( MemoryFileStream* fileStream )
: m_Available( true ),
  m_StartPos( fileStream->m_StartPos ),
  m_CurPos( fileStream->m_CurPos ),
  m_Size( fileStream->m_Size )
{
}

//---------------------------------------------------------------------------
bool
MemoryFileStream::Open( void* addr, u32 size )
{
    if ( m_Available )
    {
        Close();
    }

    m_StartPos = reinterpret_cast<u32>( addr );
    m_CurPos = m_StartPos;
    m_Size = size;
    m_Available = true;

    return true;
}

//---------------------------------------------------------------------------
void
MemoryFileStream::Close()
{
    m_Available = false;
}

//---------------------------------------------------------------------------
s32
MemoryFileStream::Read( void* buf, u32 length )
{
    NW_ASSERT( m_Available );

    u32 size = length;
    if ( m_CurPos + length > m_StartPos + m_Size )
    {
        size = m_Size - ( m_CurPos - m_StartPos );
    }

    nw::ut::MemCpy( buf, reinterpret_cast<void*>( m_CurPos ), size );

    m_CurPos += size;

    return size;
}

//---------------------------------------------------------------------------
s32
MemoryFileStream::Write( const void* buf, u32 length )
{
    NW_ASSERT( m_Available );

    u32 size = length;
    if ( m_CurPos + length > m_StartPos + m_Size )
    {
        size = m_Size - ( m_CurPos - m_StartPos );
    }

    nw::ut::MemCpy( reinterpret_cast<void*>( m_CurPos ), buf, size );

    m_CurPos += size;

    return size;
}

//---------------------------------------------------------------------------
u32
MemoryFileStream::GetSize() const
{
    NW_ASSERT( m_Available );

    return m_Size;
}

//---------------------------------------------------------------------------
bool
MemoryFileStream::Seek( s32 offset, u32 origin )
{
    NW_ASSERT( m_Available );

    u32 pos;

    switch( origin )
    {
    case FILE_STREAM_SEEK_BEGIN:
        pos = m_StartPos + offset;
        break;
    case FILE_STREAM_SEEK_CURRENT:
        pos = m_CurPos + offset;
        break;
    case FILE_STREAM_SEEK_END:
        pos = m_StartPos + m_Size - offset;
        break;
    default:
        NW_ASSERTMSG( false, "MemoryFileStream::Seek Invalid origin parameter." );
        return false;
    }

    if ( pos < m_StartPos || pos > m_StartPos + m_Size )
    {
        NW_WARNING( false, "MemoryFileStream::Seek is Failed: Invalid offset." );
        return false;
    }

    m_CurPos = pos;

    return true;
}

}   /* namespace ut */
}   /* namespace nw */

