﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/config.h>
#include <nw/ut/os/ut_MessageQueue.h>

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
#include <windows.h>
#include <nw/config/undef_windows.h>
#elif defined(NW_PLATFORM_ANDROID)
#include <errno.h>
#endif


namespace nw {
namespace ut {

//---------------------------------------------------------------------------
MessageQueue::MessageQueue()
{
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    // TODO: NintendoSdk 対応後、このコメントを削除してください。
    m_pMessageArray = NULL;
    m_MessageArraySize = 0;
    m_hMessageCountSemaphore = NULL;
    m_hFreeCountSemaphore = NULL;
    m_ReadPos = 0;
    m_WritePos = 0;
#elif defined(NW_PLATFORM_ANDROID)
    m_pMessageArray = NULL;
    m_MessageArraySize = 0;
    m_ReadPos = 0;
    m_WritePos = 0;
#elif defined(NW_PLATFORM_IOS)
    m_pMessageArray = NULL;
    m_MessageArraySize = 0;
    m_ReadPos = 0;
    m_WritePos = 0;
    m_hMessageCountSemaphore = nullptr;
    m_hFreeCountSemaphore = nullptr;
#elif defined(NW_PLATFORM_CAFE)
#endif
}

//---------------------------------------------------------------------------
MessageQueue::~MessageQueue()
{
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    // TODO: NintendoSdk 対応後、このコメントを削除してください。
    if ( m_hMessageCountSemaphore != NULL ) {
        CloseHandle(m_hMessageCountSemaphore);
    }
    if ( m_hFreeCountSemaphore != NULL ) {
        CloseHandle(m_hFreeCountSemaphore);
    }
#elif defined(NW_PLATFORM_ANDROID)
    int result;
    result = sem_destroy(&m_hMessageCountSemaphore);
    NW_ASSERT(result >= 0);

    result = sem_destroy(&m_hFreeCountSemaphore);
    NW_ASSERT(result >= 0);
    NW_UNUSED_VARIABLE(result);
#elif defined(NW_PLATFORM_IOS)
    Finalize();
#elif defined(NW_PLATFORM_CAFE)
#endif
}

//---------------------------------------------------------------------------
void
MessageQueue::Initialize(BufferType* pMessageArray, u32 messageSize)
{
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    // TODO: NintendoSdk 対応後、このコメントを削除してください。
    m_pMessageArray = pMessageArray;
    m_MessageArraySize = messageSize;
    m_hMessageCountSemaphore = CreateSemaphore(NULL, 0, messageSize, NULL);
    m_hFreeCountSemaphore = CreateSemaphore(NULL, messageSize, messageSize, NULL);
#elif defined(NW_PLATFORM_ANDROID)
    int result;
    m_pMessageArray = pMessageArray;
    m_MessageArraySize = messageSize;
    result = sem_init(&m_hMessageCountSemaphore, 0, 0);
    NW_ASSERT(result >= 0);

    result = sem_init(&m_hFreeCountSemaphore, 0, messageSize);
    NW_ASSERT(result >= 0);
    NW_UNUSED_VARIABLE(result);
#elif defined(NW_PLATFORM_IOS)
    m_pMessageArray = pMessageArray;
    m_MessageArraySize = messageSize;
    m_hMessageCountSemaphore = dispatch_semaphore_create(0);
    NW_ASSERT(m_hMessageCountSemaphore != nullptr);

    m_hFreeCountSemaphore = dispatch_semaphore_create(messageSize);
    NW_ASSERT(m_hFreeCountSemaphore != nullptr);
#elif defined(NW_PLATFORM_CAFE)
    OSInitMessageQueue(&m_MessageQueue, pMessageArray, messageSize);
#endif
}

//---------------------------------------------------------------------------
void
MessageQueue::Finalize()
{
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    // TODO: NintendoSdk 対応後、このコメントを削除してください。
    if ( m_hMessageCountSemaphore != NULL ) {
        CloseHandle(m_hMessageCountSemaphore);
        m_hMessageCountSemaphore = NULL;
    }
    if ( m_hFreeCountSemaphore != NULL ) {
        CloseHandle(m_hFreeCountSemaphore);
        m_hFreeCountSemaphore = NULL;
    }
#elif defined(NW_PLATFORM_ANDROID)
    int result;

    result = sem_destroy(&m_hMessageCountSemaphore);
    NW_ASSERT(result >= 0);

    result = sem_destroy(&m_hFreeCountSemaphore);
    NW_ASSERT(result >= 0);
    NW_UNUSED_VARIABLE(result);
#elif defined(NW_PLATFORM_IOS)
    if(m_hMessageCountSemaphore != nullptr)
    {
        dispatch_release(m_hMessageCountSemaphore);
        m_hMessageCountSemaphore = nullptr;
    }

    if(m_hFreeCountSemaphore != nullptr)
    {
        dispatch_release(m_hFreeCountSemaphore);
        m_hFreeCountSemaphore = nullptr;
    }
#elif defined(NW_PLATFORM_CAFE)
#endif
}

//---------------------------------------------------------------------------
bool
MessageQueue::Send(MessageType msg, bool blockFlag)
{
#if defined(NW_PLATFORM_WIN32) || defined( NW_USE_NINTENDO_SDK )
    // TODO: NintendoSdk 対応後、このコメントを削除してください。
    DWORD dwResult = WaitForSingleObject(m_hFreeCountSemaphore,blockFlag ? INFINITE : 0);
    if ( dwResult != WAIT_OBJECT_0 ) {
        return false;
    }
    {
        ut::ScopedLock<ut::CriticalSection> lock(m_CriticalSection);
        m_pMessageArray[m_WritePos] = msg;
        m_WritePos++;
        if ( m_WritePos>=m_MessageArraySize ) m_WritePos = 0;
    }
    ReleaseSemaphore(m_hMessageCountSemaphore,1,NULL);
    return true;
#elif defined(NW_PLATFORM_ANDROID)
    int result = 0;
    if (blockFlag)
    {
        result = sem_wait(&m_hFreeCountSemaphore);
        NW_ASSERT(result >= 0);
    }
    else
    {
        result = sem_trywait(&m_hFreeCountSemaphore);
        if (result < 0 && errno == EAGAIN)
        {
            return false;
        }
    }

    {
        ut::ScopedLock<ut::CriticalSection> lock(m_CriticalSection);
        m_pMessageArray[m_WritePos] = msg;
        m_WritePos++;
        if ( m_WritePos>=m_MessageArraySize ) m_WritePos = 0;
    }

    result = sem_post(&m_hMessageCountSemaphore);
    NW_ASSERT(result >= 0);

    return true;
#elif defined(NW_PLATFORM_IOS)
    long result;

    result = dispatch_semaphore_wait(m_hFreeCountSemaphore, blockFlag ? DISPATCH_TIME_FOREVER : 0);
    if(result != 0)
    {
      return false;
    }

    {
        ut::ScopedLock<ut::CriticalSection> lock(m_CriticalSection);
        m_pMessageArray[m_WritePos] = msg;
        m_WritePos++;
        if ( m_WritePos>=m_MessageArraySize ) m_WritePos = 0;
    }

    dispatch_semaphore_signal(m_hMessageCountSemaphore);

    return true;
#elif defined(NW_PLATFORM_CAFE)
    OSMessage msgStruct;
    msgStruct.message = msg;
    return OSSendMessage(&m_MessageQueue, &msgStruct, blockFlag ? OS_MESSAGE_BLOCK : OS_MESSAGE_NOBLOCK);
#endif
}

//---------------------------------------------------------------------------
bool
MessageQueue::Jam(MessageType msg, bool blockFlag)
{
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    // TODO: NintendoSdk 対応後、このコメントを削除してください。
    DWORD dwResult = WaitForSingleObject(m_hFreeCountSemaphore,blockFlag ? INFINITE : 0);
    if ( dwResult != WAIT_OBJECT_0 ) {
        return false;
    }
    {
        ut::ScopedLock<ut::CriticalSection> lock(m_CriticalSection);
        if ( m_ReadPos == 0 ) {
            m_ReadPos = m_MessageArraySize-1;
        }
        else {
            m_ReadPos--;
        }
        m_pMessageArray[m_ReadPos] = msg;
    }
    ReleaseSemaphore(m_hMessageCountSemaphore, 1, NULL);
    return true;
#elif defined(NW_PLATFORM_ANDROID)
    int result = 0;

    if (blockFlag)
    {
        result = sem_wait(&m_hFreeCountSemaphore);
        NW_ASSERT(result >= 0);
    }
    else
    {
        result = sem_trywait(&m_hFreeCountSemaphore);
        if (result < 0 && errno == EAGAIN)
        {
            return false;
        }
    }


    {
        ut::ScopedLock<ut::CriticalSection> lock(m_CriticalSection);
        if ( m_ReadPos == 0 ) {
            m_ReadPos = m_MessageArraySize-1;
        }
        else {
        }
        m_pMessageArray[m_ReadPos] = msg;
    }
    result = sem_post(&m_hMessageCountSemaphore);
    NW_ASSERT(result >= 0);
    return true;
#elif defined(NW_PLATFORM_IOS)
    long result;

    result = dispatch_semaphore_wait(m_hFreeCountSemaphore, blockFlag ? DISPATCH_TIME_FOREVER : 0);
    if(result != 0)
    {
      return false;
    }

    {
        ut::ScopedLock<ut::CriticalSection> lock(m_CriticalSection);
        if ( m_ReadPos == 0 ) {
            m_ReadPos = m_MessageArraySize-1;
        }
        else {
        }
        m_pMessageArray[m_ReadPos] = msg;
    }
    dispatch_semaphore_signal(m_hMessageCountSemaphore);

    return true;
#elif defined(NW_PLATFORM_CAFE)
    OSMessage msgStruct;
    msgStruct.message = msg;
    return OSJamMessage(&m_MessageQueue, &msgStruct, blockFlag ? OS_MESSAGE_BLOCK : OS_MESSAGE_NOBLOCK);
#endif
}

//---------------------------------------------------------------------------
bool
MessageQueue::Recv(MessageType* msg, bool blockFlag)
{
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    // TODO: NintendoSdk 対応後、このコメントを削除してください。
    DWORD dwResult = WaitForSingleObject(m_hMessageCountSemaphore, blockFlag ? INFINITE : 0);
    if ( dwResult != WAIT_OBJECT_0 ) {
        return false;
    }
    {
        ut::ScopedLock<ut::CriticalSection> lock(m_CriticalSection);
        *msg = m_pMessageArray[m_ReadPos];
        m_ReadPos++;
        if ( m_ReadPos>=m_MessageArraySize ) m_ReadPos = 0;
    }
    ReleaseSemaphore(m_hFreeCountSemaphore, 1, NULL);
    return true;
#elif defined(NW_PLATFORM_ANDROID)
    int result = 0;

    if (blockFlag)
    {
        result = sem_wait(&m_hMessageCountSemaphore);
        NW_ASSERT(result >= 0);
    }
    else
    {
        result = sem_trywait(&m_hMessageCountSemaphore);
        if (result < 0 && errno == EAGAIN)
        {
            return false;
        }
    }

    {
        ut::ScopedLock<ut::CriticalSection> lock(m_CriticalSection);
        *msg = m_pMessageArray[m_ReadPos];
        m_ReadPos++;
        if ( m_ReadPos>=m_MessageArraySize ) m_ReadPos = 0;
    }
    result = sem_post(&m_hFreeCountSemaphore);
    NW_ASSERT(result >= 0);

    return true;
#elif defined(NW_PLATFORM_IOS)
    long result;

    result = dispatch_semaphore_wait(m_hMessageCountSemaphore, blockFlag ? DISPATCH_TIME_FOREVER : 0);
    if(result != 0)
    {
      return false;
    }

    {
        ut::ScopedLock<ut::CriticalSection> lock(m_CriticalSection);
        *msg = m_pMessageArray[m_ReadPos];
        m_ReadPos++;
        if ( m_ReadPos>=m_MessageArraySize ) m_ReadPos = 0;
    }
    dispatch_semaphore_signal(m_hFreeCountSemaphore);

    return true;
#elif defined(NW_PLATFORM_CAFE)
    OSMessage msgStruct;

    BOOL result = OSReceiveMessage(&m_MessageQueue, &msgStruct, blockFlag ? OS_MESSAGE_BLOCK : OS_MESSAGE_NOBLOCK);
    *msg = msgStruct.message;

    return result;
#endif
}

} // namespace ut
} // namespace nw


