﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/types.h>
#include <nw/ut/ut_IoStream.h>
#include <nw/mcs/mcs_McsIoStream.h>

namespace nw {
namespace mcs {


#if defined(NW_MCS_ENABLE)

//---------------------------------------------------------------------------
/* ctor */
McsIoStream::McsIoStream() :
m_IsAvailable( false ),
m_Channel( 0 )
{
}



//---------------------------------------------------------------------------
/* ctor */
McsIoStream::McsIoStream( mcs::ChannelType channel, void* buf, u32 length ) :
m_IsAvailable( false ),
m_Channel( 0 )
{
    (void)this->Open( channel, buf, length );
}


//---------------------------------------------------------------------------
/* dtor */
McsIoStream::~McsIoStream()
{
    this->Close();
}


//---------------------------------------------------------------------------
bool
McsIoStream::Open( mcs::ChannelType channel, void* buf, u32 length )
{
    NW_NULL_ASSERT( buf );

    this->Close();

    mcs::Mcs_RegisterBuffer( channel, buf, length );

    m_Channel = channel;
    m_IsAvailable = true;

    return true;
}


//---------------------------------------------------------------------------
void
McsIoStream::Close()
{
    if ( !m_IsAvailable )
    {
        return;
    }

    mcs::Mcs_UnregisterBuffer( m_Channel );
    m_Channel = 0;
    m_IsAvailable = false;
}


//---------------------------------------------------------------------------
s32
McsIoStream::Read( void* buf, u32 length )
{
    NW_ASSERT_NOT_NULL( buf );

    u32 result = mcs::Mcs_Read( m_Channel, buf, length);

    if ( result == MCS_ERROR_SUCCESS )
    {
        return s32(length);
    }

    NW_ASSERT( s32( result ) > 0 );
    return - s32( result );
}


//---------------------------------------------------------------------------
s32
McsIoStream::Skip(u32 length)
{
    u32 result = mcs::Mcs_Skip( m_Channel, length);

    if ( result == MCS_ERROR_SUCCESS )
    {
        return s32(length);
    }

    NW_ASSERT( s32( result ) > 0 );
    return - s32( result );
}


//---------------------------------------------------------------------------
s32
McsIoStream::Peek( void* buf, u32 length )
{
    NW_ASSERT_NOT_NULL( buf );

    return s32( mcs::Mcs_Peek( m_Channel, buf, length) );
}


//---------------------------------------------------------------------------
s32
McsIoStream::Write( const void* buf, u32 length )
{
    NW_ASSERT_NOT_NULL( buf );

    u32 result = mcs::Mcs_Write( m_Channel, buf, length);

    if ( result == MCS_ERROR_SUCCESS )
    {
        return s32(length);
    }

    NW_ASSERT( s32(result) > 0 );
    return - s32( result );
}

//---------------------------------------------------------------------------
u32
McsIoStream::GetReadableBytes() const
{
    if ( !m_IsAvailable )
    {
        return 0;
    }

    return mcs::Mcs_GetReadableBytes( m_Channel );
}

//---------------------------------------------------------------------------
u32
McsIoStream::GetWritableBytes() const
{
    if ( !m_IsAvailable )
    {
        return 0;
    }

    u32 size = 0;
    mcs::Mcs_GetWritableBytes( m_Channel, &size );

    return size;
}


#endif  // #if defined(NW_MCS_ENABLE)

}   /* namespace mcs  */
}   /* namespace nw */

