﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/assert.h>
#include <nw/font/font_TagProcessorBase.h>
#include <nw/font/font_TextWriterBase.h>


namespace nw {
namespace font {






/* ------------------------------------------------------------------------
        コンストラクタ/デストラクタ
   ------------------------------------------------------------------------ */

template <typename CharType>
TagProcessorBase<CharType>::TagProcessorBase()
{
}

template <typename CharType>
TagProcessorBase<CharType>::~TagProcessorBase()
{
}




/* ------------------------------------------------------------------------
        タグ処理
   ------------------------------------------------------------------------ */

template <typename CharType>
typename TagProcessorBase<CharType>::Operation
TagProcessorBase<CharType>::Process(
    u16             code,
    ContextType*    pContext
)
{
    NW_ASSERT(code < ' ');
    NW_ASSERT_VALID_POINTER(pContext);

    switch (code)
    {
    case '\n':
        ProcessLinefeed(pContext);
        return OPERATION_NEXT_LINE;

    case '\t':
        ProcessTab(pContext);
        return OPERATION_NO_CHAR_SPACE;

    default:
        // nothing to do;
        break;
    }

    return OPERATION_DEFAULT;
}

template <typename CharType>
typename TagProcessorBase<CharType>::Operation
TagProcessorBase<CharType>::CalcRect(
    ut::Rect*       pRect,
    u16             code,
    ContextType*    pContext
)
{
    NW_ASSERT_VALID_POINTER(pRect);
    NW_ASSERT(code < ' ');
    NW_ASSERT_VALID_POINTER(pContext);

    switch (code)
    {
    case '\n':
        {
            TextWriterBase<CharType>& writer = *pContext->writer;

            pRect->right    = writer.GetCursorX();
            pRect->top      = writer.GetCursorY();

            ProcessLinefeed(pContext);

            pRect->left     = writer.GetCursorX();

            // ProcessLinefeed はカーソルを移動するだけで次の行の高さは含まれていないので
            // 次の行の高さとして FontHeight を足す。
            pRect->bottom   = writer.GetCursorY() + pContext->writer->GetFontHeight();

            pRect->Normalize();
        }
        return OPERATION_NEXT_LINE;

    case '\t':
        {
            TextWriterBase<CharType>& writer = *pContext->writer;

            pRect->left     = writer.GetCursorX();

            ProcessTab(pContext);

            pRect->right    = writer.GetCursorX();
            pRect->top      = writer.GetCursorY();
            pRect->bottom   = pRect->top + writer.GetFontHeight();

            pRect->Normalize();
        }
        return OPERATION_NO_CHAR_SPACE;

    default:
        // nothing to do;
        break;
    }

    return OPERATION_DEFAULT;
}

template <typename CharType>
void
TagProcessorBase<CharType>::BeginPrint(PrintContext<CharType>* /* pContext */)
{
}

template <typename CharType>
void
TagProcessorBase<CharType>::EndPrint(PrintContext<CharType>* /* pContext */)
{
}

template <typename CharType>
void
TagProcessorBase<CharType>::BeginCalcRect(PrintContext<CharType>* /* pContext */)
{
}

template <typename CharType>
void
TagProcessorBase<CharType>::EndCalcRect(PrintContext<CharType>* /* pContext */)
{
}

/* ------------------------------------------------------------------------
        private タグ処理
   ------------------------------------------------------------------------ */
template <typename CharType>
void
TagProcessorBase<CharType>::ProcessLinefeed(ContextType* pContext)
{
    NW_ASSERT_VALID_POINTER(pContext);

    TextWriterBase<CharType>& writer = *pContext->writer;

    const f32 x   = pContext->xOrigin;
    const f32 y   = writer.GetCursorY() + writer.GetLineHeight();

    writer.SetCursor(x, y);
}

template <typename CharType>
void
TagProcessorBase<CharType>::ProcessTab(ContextType* pContext)
{
    NW_ASSERT_VALID_POINTER(pContext);

    TextWriterBase<CharType>& writer = *pContext->writer;
    const int tabWidth = writer.GetTabWidth();

    if (tabWidth > 0)
    {
        const f32 aCharWidth  = writer.IsWidthFixed() ? writer.GetFixedWidth(): writer.GetFontWidth();
        const f32 dx          = writer.GetCursorX() - pContext->xOrigin;
        const f32 tabPixel    = tabWidth * aCharWidth;
        const int numTab      = static_cast<int>(dx / tabPixel) + 1;
        const f32 x           = pContext->xOrigin + tabPixel * numTab;

        writer.SetCursorX(x);
    }
}




/* ------------------------------------------------------------------------
        明示的実体化
   ------------------------------------------------------------------------ */

template class TagProcessorBase<char>;
template class TagProcessorBase<char16>;


}   // namespace font
}   // namespace nw
