﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/dev/dev_Pad.h>
#include <nw/dw/window/dw_WindowMenu.h>

namespace nw {
namespace internal {
namespace dw {
namespace detail {

// -----------------------------------------------------------
WindowMenu::WindowMenu( WindowList& windowList )
: Window( "WindowMenu", 300, 300 ),
  m_WindowList( windowList ),
  m_CurrentWindowIndex( 0 ),
  m_GrabWindowIndex( -1 ),
  m_ScrollOffset( 0 ),
  m_TextGfxBuffer( NULL ),
  m_TextBuffer( NULL )
{
    SetBackgroundColor( ut::Color4f( ut::Color4f::X_DIM_GRAY(), 0.8f ) );
}

// -----------------------------------------------------------
WindowMenu::~WindowMenu()
{
}

// -----------------------------------------------------------
void WindowMenu::OnDraw( IUIRenderContext& context, UIRenderer& renderer )
{
    renderer.BeginDraw();

    nw::math::VEC2 size = renderer.MeasureText(DrawTextArgs(), " ");

    ScrollToEnsureVisible( m_CurrentWindowIndex, size.y );

    f32 y = - m_ScrollOffset * size.y;
    int index = 0;

    for ( WindowList::ConstIterator itr = WindowList::const_iterator(m_WindowList.GetBeginIter());
          itr != WindowList::const_iterator(m_WindowList.GetEndIter()) ; itr++ )
    {
        const ut::Color4f color = itr->GetIsVisible() ? ut::Color4f::WHITE() : ut::Color4f::GRAY();
        f32 x = ( index == m_GrabWindowIndex ) ? size.x * 2.3f : size.x * 3.0f;

        renderer.DrawText(
            &context,
            DrawTextArgs().
            SetTopLeft(x, y).
            SetColor(color),
            itr->GetTitle());

        y += size.y;
        index++;
    }

    renderer.DrawText(
        &context,
        DrawTextArgs().
        SetTopLeft(0, (m_CurrentWindowIndex - m_ScrollOffset) * size.y).
        SetColor(nw::ut::Color4f::WHITE()),
        " >");

    DrawScrollMark( context, renderer, size.y );

    renderer.EndDraw();
}

// -----------------------------------------------------------
void WindowMenu::OnUpdateInputs(const nw::internal::dw::Inputs& inputs)
{
    if(inputs.GetPad() == NULL)
    {
        return;
    }

    const nw::dev::Pad& pad = *inputs.GetPad();

    if ( pad.IsHold( nw::dev::Pad::MASK_A ) )
    {
        m_GrabWindowIndex = m_CurrentWindowIndex;
    }
    else
    {
        m_GrabWindowIndex = -1;
    }

    if ( pad.IsRepeat( nw::dev::Pad::MASK_DOWN ) )
    {
        m_CurrentWindowIndex++;
        if ( m_CurrentWindowIndex > static_cast<int>(m_WindowList.GetSize()) - 1 ) {
            m_CurrentWindowIndex = 0;
        }
        if ( m_GrabWindowIndex >= 0 ) {
            ExchangeWindowIndex( m_CurrentWindowIndex, m_GrabWindowIndex );
        }
    }
    if ( pad.IsRepeat( nw::dev::Pad::MASK_UP ) )
    {
        m_CurrentWindowIndex--;
        if ( m_CurrentWindowIndex < 0 ) {
            m_CurrentWindowIndex = static_cast<int>(m_WindowList.GetSize()) - 1;
        }
        if ( m_GrabWindowIndex >= 0 ) {
            ExchangeWindowIndex( m_CurrentWindowIndex, m_GrabWindowIndex );
        }
    }
    if ( pad.IsTrig( nw::dev::Pad::MASK_B ) )
    {
        Window* window = GetWindow( m_CurrentWindowIndex );
        if ( window != NULL ) {
            bool flag = window->GetIsVisible();
            window->SetIsVisible( !flag );
        }
    }
}

// -----------------------------------------------------------
void WindowMenu::DrawScrollMark( IUIRenderContext& context, UIRenderer& renderer, f32 cyChar )
{
    (void)context;
    (void)renderer;
    (void)cyChar;

#if 0   // ★TODO : 未実装
    const Size clientSize = GetClientSize();
    int drawLines = clientSize.height / cyChar;
    int windowCount = static_cast<int>( m_WindowList.GetSize() );

    const int MARGIN_RIGHT = 4;
    const int MARGIN_HEIGHT = 4;
    const int WIDTH = 10;
    const int HEIGHT = 7;

    if ( m_ScrollOffset > 0 ) {
        // △印
        Point points[3]; // 上、右下、左下
        points[0].x = clientSize.width - MARGIN_RIGHT - WIDTH/2;
        points[0].y = MARGIN_HEIGHT;
        points[1].x = clientSize.width - MARGIN_RIGHT;
        points[1].y = MARGIN_HEIGHT + HEIGHT;
        points[2].x = clientSize.width - MARGIN_RIGHT - WIDTH;
        points[2].y = MARGIN_HEIGHT + HEIGHT;

        gfx->FillTriangle( points[0],points[1],points[2], ut::Color4u8::YELLOW );
    }
    if ( windowCount - m_ScrollOffset > drawLines ) {
        // ▽印
        Point points[3]; // 左上、右上、下
        points[0].x = clientSize.width - MARGIN_RIGHT - WIDTH;
        points[0].y = clientSize.height - MARGIN_HEIGHT - HEIGHT;
        points[1].x = clientSize.width - MARGIN_RIGHT;
        points[1].y = clientSize.height - MARGIN_HEIGHT - HEIGHT;
        points[2].x = clientSize.width - MARGIN_RIGHT - WIDTH/2;
        points[2].y = clientSize.height - MARGIN_HEIGHT;

        gfx->FillTriangle( points[0],points[1],points[2], ut::Color4u8::YELLOW );
    }
#endif
}

// -----------------------------------------------------------
void WindowMenu::ScrollToEnsureVisible( int windowIndex, f32 cyChar )
{
    const nw::math::Vector2 clientSize = GetClientSize();
    int drawLines = static_cast<int>(clientSize.y / cyChar);

    int lastIndex = m_ScrollOffset + drawLines-1;
    if ( windowIndex < m_ScrollOffset )
    {
        // 上スクロール
        m_ScrollOffset = windowIndex;
    }
    else if ( windowIndex > lastIndex )
    {
        // 下スクロール
        m_ScrollOffset = windowIndex - (drawLines-1);
    }
}

// -----------------------------------------------------------
void WindowMenu::ExchangeWindowIndex( int index1, int index2 )
{
    if ( index1 == index2 ) return;
    if ( index1 > index2 ) {
        // swap( index1, index2 );
        int tmp = index1; index1 = index2; index2 = tmp;
    }
    NW_ASSERT( index1 < index2 );

    Window* window1 = GetWindow( index1 );
    if ( window1 == NULL ) return;
    Window* window2 = GetWindow( index2 );
    if ( window2 == NULL ) return;

    WindowList::Iterator itr1 = m_WindowList.GetIteratorFromPointer( window1 );
    WindowList::Iterator itr2 = m_WindowList.GetIteratorFromPointer( window2 );

    itr2++; // window2の次のウィンドウを指す

    // window1 の前に window2 を挿入
    m_WindowList.Erase( window2 );
    m_WindowList.Insert( itr1, window2 );

    // NOTE: window2はwindow1より後方にあるため、itr2は有効のはず

    // windo2 の元の位置に window1 を挿入
    m_WindowList.Erase( window1 );
    m_WindowList.Insert( itr2, window1 );
}

// -----------------------------------------------------------
Window* WindowMenu::GetWindow( int windowIndex )
{
    int index = 0;
    for ( WindowList::Iterator itr = m_WindowList.GetBeginIter();
          itr != m_WindowList.GetEndIter() ; itr++ )
    {
        if ( index == windowIndex ) return &*itr;
        index++;
    }
    return NULL;
}

} // namespace nw::internal::dw::detail
} // namespace nw::internal::dw
} // namespace nw::internal
} // namespace nw

