﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/dw/window/dw_LogViewerWindow.h>
#include <nw/dw/system/dw_NwTypeUtility.h>
#include <nw/ut.h>

#include <cstdarg>
#include <cstdio>
#include <cstring>

namespace nw {
namespace internal {
namespace dw {

// -----------------------------------------------------------
LogViewerWindow::LogViewerWindow()
: Window( "LogViewer", 300, 300 ),
  m_AutoScrollFlag( true ),
  m_Buffer( NULL ),
  m_BufferSize ( 0 ),
  m_Width(0),
  m_LineMax(0),
  m_RingTop(0),
  m_PrintTop(0),
  m_PrintXPos(0),
  m_TextScale( DEFAULT_TEXT_SCALE ),
  m_Y(0),
  m_CyChar(0),
  m_AccelVertOrigin( 0.0f ),
  m_IsLocking(false)
{
}

// -----------------------------------------------------------
LogViewerWindow::~LogViewerWindow()
{
}

// -----------------------------------------------------------
void LogViewerWindow::AssignBuffer( void* buffer, unsigned long bufferSize, int width )
{
    m_Buffer = buffer;
    m_BufferSize = bufferSize;
    m_Width = width;
    m_LineMax = static_cast<int>(bufferSize) / ( width+1 );
}

// -----------------------------------------------------------
void LogViewerWindow::ReleaseBuffer()
{
    m_Buffer = NULL;
    m_BufferSize = 0;
}

// -----------------------------------------------------------
void LogViewerWindow::OnDraw( IUIRenderContext& context, UIRenderer& renderer )
{
    Window::OnDraw( context, renderer );

    if ( m_Buffer == NULL ) {
        renderer.DrawText( &context, DrawTextArgs(), "- not assign buffer -" );
        return;
    }

    nw::math::VEC2 size = renderer.MeasureText(DrawTextArgs(), " ");
    m_CyChar = static_cast<int>( size.y );

    int cyChar = static_cast<int>( (static_cast<float>( m_CyChar ) * std::pow( 1.1f, m_TextScale ) ) );

    const nw::math::Vector2 clientSize = GetClientSize();
    m_Y = ut::Clamp(
        m_Y,
        0.f,
        ut::Max( 0.f, cyChar * GetLineCount() - clientSize.y )
    );

    f32 scale    = std::pow( 1.1f,m_TextScale );
    int line     = m_RingTop;
    int drawLine = 0;

    const f32 drawOffsetX = 4.f;

    while ( line != m_PrintTop )
    {
        f32 y = static_cast<f32>( drawLine * cyChar ) - m_Y;
        if ( ( -cyChar <= y ) && ( y <= clientSize.y ) )
        {
            renderer.DrawText(
                &context,
                DrawTextArgs().
                SetTopLeft(drawOffsetX, y).
                SetColor(nw::ut::Color4f::WHITE()).
                SetScale(scale),
                GetTextPtr(line, 0));
        }

        ++drawLine;
        if ( ++line == m_LineMax )
        {
            line = 0;
        }
    }

    // 現在の行の描画
    if ( m_PrintXPos > 0 )
    {
        f32 y = static_cast<f32>( drawLine * cyChar ) - m_Y;
        if ( ( -cyChar <= y ) && ( y <= clientSize.y ) )
        {
            renderer.DrawText(
                &context,
                DrawTextArgs().
                SetTopLeft(drawOffsetX, y).
                SetColor(nw::ut::Color4f::WHITE()).
                SetScale(scale),
                GetTextPtr(m_PrintTop, 0));
        }
    }

    // スクロールバーの表示
    static const f32 scrollBarWidth = 8.f;
    renderer.FillRectangle(
        &context,
        DrawRectangleArgs().
            SetTopLeft(clientSize.x - scrollBarWidth, 0).
            SetSize(scrollBarWidth, static_cast<f32>(clientSize.y)).
            SetColor(NwTypeUtility::SRGBToLinear(nw::ut::Color4f::X_DIM_GRAY())));   // ★TODO : 配色を変更

    int lineCount = GetLineCount();
    if ( lineCount > 0 )
    {
        f32 x      = clientSize.x - scrollBarWidth;
        f32 y      = static_cast<f32>(m_Y * clientSize.y / (lineCount * cyChar));
        f32 width  = scrollBarWidth;
        f32 height = nw::ut::Min(
            clientSize.y,
            clientSize.y * clientSize.y / static_cast<f32>(lineCount * cyChar)
        );

        renderer.FillRectangle(
            &context,
            DrawRectangleArgs().
                SetTopLeft(x, y).
                SetSize(width, height).
                SetColor(NwTypeUtility::SRGBToLinear(nw::ut::Color4f::X_STEEL_BLUE())));   // ★TODO : 配色を変更
    }
}

// -----------------------------------------------------------
void
LogViewerWindow::OnUpdateInputs(const nw::internal::dw::Inputs& inputs)
{
    Window::OnUpdateInputs(inputs);

    if(inputs.GetPad() == NULL)
    {
        return;
    }

    const nw::dev::Pad& pad = *inputs.GetPad();

    if ( m_Buffer == NULL ) return;

    const nw::math::Vector2 clientSize = GetClientSize();

    float stickY = pad.GetLeftStick().y;
    if ( stickY != 0 )
    {
        m_Y -= static_cast<int>(stickY * 8);
    }
    int cyChar = static_cast<int>( (static_cast<float>( m_CyChar ) * std::pow( 1.1f, m_TextScale ) ) );
    if ( pad.IsRepeat( nw::dev::Pad::MASK_UP ) )
    {
        m_Y -= cyChar;
    }
    if ( pad.IsRepeat( nw::dev::Pad::MASK_DOWN ) )
    {
        m_Y += cyChar;
    }
    if ( pad.IsTrig( nw::dev::Pad::MASK_LEFT ) )
    {
        m_Y = 0;
    }
    if ( pad.IsTrig( nw::dev::Pad::MASK_RIGHT ) )
    {
        m_Y = cyChar * GetLineCount() - clientSize.y;
    }

    if ( pad.IsTrig( nw::dev::Pad::MASK_X ) )
    {
        m_TextScale++;
    }
    if ( pad.IsTrig( nw::dev::Pad::MASK_B ) )
    {
        m_TextScale--;
    }
    m_TextScale = ut::Clamp( m_TextScale, -5, 10 );
}

// -----------------------------------------------------------
void LogViewerWindow::Printf( const char* format, ... )
{
    if ( m_Buffer == NULL ) return;

    va_list vlist;
    va_start( vlist, format );
    VPrintf( format, vlist );
    va_end( vlist );
}

// -----------------------------------------------------------
void LogViewerWindow::VPrintf( const char* format, std::va_list vlist )
{
    // todo: NW4RのAutoInterruptLockの挙動がこれであってるのか不明
    if (m_IsLocking)
    {
        return;
    }

    if ( m_Buffer == NULL ) return;

    const unsigned long STRING_BUF_SIZE = 1024; // 出力文字列生成用のバッファサイズ
    static char s_StrBuf[ STRING_BUF_SIZE ];     // 出力文字列生成用のバッファ

    //ut::AutoInterruptLock lock;
    SelfLock lock(*this);

#ifdef NW_COMPILER_MSVC
#pragma warning(push)
#pragma warning(disable : 4996)
#endif
    (void)vsnprintf( s_StrBuf, STRING_BUF_SIZE, format, vlist );
#ifdef NW_COMPILER_MSVC
#pragma warning(pop)
#endif

    PutString( s_StrBuf );
}

// -----------------------------------------------------------
void LogViewerWindow::PutString( const char* str )
{
    // todo: NW4RのAutoInterruptLockの挙動がこれであってるのか不明
    if (m_IsLocking)
    {
        return;
    }

    NW_ASSERT_NOT_NULL( str );

    if ( m_Buffer == NULL ) return;

    const nw::math::Vector2 clientSize = GetClientSize();

    //ut::AutoInterruptLock lock;
    SelfLock lock(*this);

    int cyChar = static_cast<int>( (static_cast<float>( m_CyChar ) * std::pow( 1.1f, m_TextScale ) ) );
    bool canAutoScroll = false;
    if ( cyChar * GetLineCount() <= clientSize.y ) canAutoScroll = true;
    else if ( m_Y == cyChar * GetLineCount() - clientSize.y ) canAutoScroll = true;

    char* ptr = GetTextPtr( m_PrintTop, m_PrintXPos );

    // -------- 一文字ずつループ
    // ソース文字列が空になるまでコピー
    while ( *str )
    {
        bool newLineFlag = false;

        if ( *str == '\n' )
        {
            str++;
            ptr = NextLine();
        }
        else
        {
            u32 bytes = PutChar( str, ptr );

            if ( bytes > 0 )
            {
                str += bytes;
                ptr += bytes;
            }
            else
            {
                newLineFlag = true;
            }
        }
        if ( m_PrintXPos >= m_Width )
        {
            // 文字位置が端まで来ていたら改行フラグを立てる
            newLineFlag = true;
        }

        if ( newLineFlag )
        {
            // 次が改行文字であれば１つ読み飛ばす
            if ( *str == '\n' ) ++str;

            ptr = NextLine();
        }
    }

    // 終端にヌル文字付加
    if ( *str == '\0' )
    {
        *( GetTextPtr( m_PrintTop, m_PrintXPos ) ) = '\0';
    }

    // 表示位置を自動更新
    if ( m_AutoScrollFlag )
    {
        if ( canAutoScroll )
        {
            m_Y = ut::Max( 0.f, cyChar * GetLineCount() - clientSize.y );
        }
    }
}

// -----------------------------------------------------------
char* LogViewerWindow::GetTextPtr( int line, int xPos )
{
    return reinterpret_cast<char*>( m_Buffer ) + ( (m_Width + 1) * line + xPos );
}

// -----------------------------------------------------------
unsigned long LogViewerWindow::PutChar( const char* str, char* dstPtr )
{
    int codeWidth = static_cast<int>(
        ( (u8)*str >= 0x81 ) ? 2 : 1
    );

    if ( m_PrintXPos + codeWidth > m_Width ) return 0;

    m_PrintXPos += codeWidth;

    unsigned long cnt = codeWidth;
    while ( cnt > 0 )
    {
        *dstPtr++ = *str++;
        cnt--;
    }
    return codeWidth;
}

// -----------------------------------------------------------
char* LogViewerWindow::NextLine()
{
    // 現在の行末にNUL文字を挿入
    *( GetTextPtr(  m_PrintTop, m_PrintXPos ) ) = '\0';

    // 出力位置を変更
    m_PrintXPos = 0;
    m_PrintTop++;
    if ( m_PrintTop == m_LineMax )
    {
        m_PrintTop = 0;
    }

    if ( m_PrintTop == m_RingTop )
    {
        if ( ++m_RingTop == m_LineMax )
        {
            m_RingTop = 0;
        }
    }
    return GetTextPtr( m_PrintTop, 0 );
}

// -----------------------------------------------------------
int LogViewerWindow::GetLineCount()
{
    int lineCount = m_PrintTop - m_RingTop;
    if ( lineCount < 0 ) lineCount += m_LineMax;
    if ( m_PrintXPos > 0 ) lineCount += 1;
    return lineCount;
}

} // namespace nw::internal::dw
} // namespace nw::internal
} // namespace nw

